require 'rails_helper'

RSpec.describe "api/v1/claims", :type => :request do
  before do
    create(:hearing_loss_evaluation_spec)
    create(:elbow_and_forearm_evaluation_spec)
  end

  let(:example_request_sample) { File.open('spec/support/exam_request_sample.xml', 'rb').read }


  context "when the exam is assigned to a user that is not currently in the system" do
    it "should create that user and assign the exam to them" do
      expect { post api_v1_claims_path, example_request_sample, {'CONTENT_TYPE' => 'application/xml'} }.to change(User, :count).by(1)
    end
  end

  context "when the exam is assigned to an existing user" do
    before do
      user = create(:user, first_name: "Mae", last_name: "Jemison", email: "PII      ")
    end

    context "with xml that has a valid claim, contention and evaluations" do
      it "returns a created response status code" do
        post api_v1_claims_path, example_request_sample, {'CONTENT_TYPE' => 'application/xml'}
        expect(response).to have_http_status(:created)
      end

      it "stores a hash of the exam_request data with the claim" do
        post api_v1_claims_path, example_request_sample, {'CONTENT_TYPE' => 'application/xml'}
        expect(Claim.last.exam_request_data['ClaimInformation']['ClaimId']).to eq ('123xyz')
      end

      it "creates a claim for an existing user" do
        expect { post api_v1_claims_path, example_request_sample,
               {'CONTENT_TYPE' => 'application/xml'} }.to change(Claim, :count).by(1)
      end

      it "creates a contention for an existing user" do
        expect { post api_v1_claims_path, example_request_sample,
               {'CONTENT_TYPE' => 'application/xml'} }.to change(Contention, :count).by(1)
      end

      it "does not create a new user" do
        expect { post api_v1_claims_path, example_request_sample,
               {'CONTENT_TYPE' => 'application/xml'} }.not_to change(User, :count)
      end

      it "should log the ExamRequest" do
        expect { post api_v1_claims_path, example_request_sample, {'CONTENT_TYPE' => 'application/xml'} }.to change(ExamRequest, :count)
      end

      context "when there are three evaluations defined in the xml, but only two have CUI evaluation specs" do
        it "creates two evaluations" do
          expect { post api_v1_claims_path, example_request_sample,
                 {'CONTENT_TYPE' => 'application/xml'} }.to change(Evaluation, :count).by(2)
        end

        it "creates four log entries" do
          # Two for evaluations created, two for ExamRequests
          expect { post api_v1_claims_path, example_request_sample,
                 {'CONTENT_TYPE' => 'application/xml'} }.to change(EvaluationLog, :count).by(4)
        end
      end
    end
  end

  context "with invalid xml" do
    let(:invalid_xml) { '<?xml version="1.0" encoding="UTF-8" ?><feed xmlns="http://www.w3.org/2005/Atom" xsi:schemaLocation="http://www.w3.org/2005/Atom http://vler.DNS   /vler/schemas/atom/2005/vler/0.2/atom.xsd" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">
</feed>' }

    it "returns an unprocessable entity response status code" do
      post api_v1_claims_path, invalid_xml, {'CONTENT_TYPE' => 'application/xml'}
      expect(response).to have_http_status(:unprocessable_entity)
    end

    it "does not create a claim" do
      expect { post api_v1_claims_path, invalid_xml, {'CONTENT_TYPE' => 'application/xml'} }
             .to_not change(Claim, :count)
    end

    it "should log an error" do
      expect { post api_v1_claims_path, invalid_xml, {'CONTENT_TYPE' => 'application/xml'} }.to change(ExamRequest, :count)
    end
  end
end
