RSpec.describe NeurologicalOtherEvaluationPresenter do
  context "IEPD compliance" do
    let(:epilepsy_symptoms) {
      ['An average of at least one major seizure per month over the last year',
      'An average of at least one major seizure in three months over the last year',
      'More than ten minor seizures weekly',
      'An average at least one major seizure in four months over the last year',
      'Nine or ten minor seizures per week',
      'At least one major seizure in the last six months',
      'At least two major seizures in the last year',
      'An average of at least five to eight minor seizures weekly',
      'At least one major seizure in the last two years',
      'At least two minor seizures in the last six months',
      'A confirmed diagnosis of epilepsy with a history of seizures',
      'Continuous medication is shown necessary for the control of epilepsy']
    }

    let(:iepd) {
      {'Amyotrophic lateral sclerosis' => { symptoms: ['Rate Minimum'] },
      'Athetosis, acquired' => {
        symptoms: ['Pronounced', 'Progressive grave types disability', 'Severe',
                   'Moderately Severe', 'Moderate', 'Mild']},
      'Sydenham\'s Chorea' => {
        symptoms: ['Pronounced', 'Progressive grave types disability', 'Severe',
                   'Moderately Severe', 'Moderate', 'Mild']},
      'Huntington\'s Chorea' => {
        symptoms: ['Pronounced', 'Progressive grave types disability', 'Severe',
                   'Moderately Severe', 'Moderate', 'Mild']},
      'Brain (benign)' => {
        symptoms: ['Benign neoplasms present', 'Minimum', 'Rate Residuals']},
      'Spinal Cord (benign)' => {
        symptoms: ['Benign neoplasms present', 'Minimum', 'Rate Residuals']},

      # Brain abscess is missing 'minimum' and 'rate residuals'
      #'Brain, abscess' => {
      #  symptoms: ['Active disease', 'Minimum', 'Rate Residuals']},
      # Brain, vessels, embolism of is missing Rate Minimum and Rate Residuals
      # 'Brain, vessels, embolism of' => {
      #   symptoms: ['Rate Convalescence', 'Rate Minimum', 'Rate Residuals']},
      # Brain, vessels, hemorrhage is missing Rate Minimum and Rate Residuals
      #'Brain, vessels, hemorrhage' => {
      #  symptoms: ['Rate Convalescence', 'Rate Minimum', 'Rate Residuals']},
      # Brain, vessels, thrombosis is missing Rate Minimum and Rate Residuals
      #'Brain, vessels, thrombosis' => {
      #  symptoms: ['Rate Convalescence', 'Rate Minimum', 'Rate Residuals']},
      # Hematomyelia is is missing Rate Minimum and Rate Residuals
      #'Hematomyelia' => {
      #  symptoms: ['Rate Convalescence', 'Rate Minimum', 'Rate Residuals']},

      'Bulbar Palsy' => { symptoms: ['Diagnosed']},

      # Cerebral arteriosclerosis lacks the 'Rate Residuals' field
      #'Cerebral arteriosclerosis' => { symptoms: ['Rate Residuals']},

      'Paramyoclonus multiplex' => { symptoms: ['Severe', 'Moderate', 'Mild']},

      # XXX: these three exams lack the 'Rate Residuals' field
      'Encephalitis' => {
        symptoms: ['Active febrile disease', 'Minimum']}, #, 'Rate Residuals']},
      'Meningitis, cerebrospinal, epidemic' => {
        symptoms: ['Active febrile disease', 'Minimum']}, #, 'Rate Residuals']},
      'Poliomyelitis, anterior' => {
        symptoms: ['Active febrile disease', 'Minimum']}, #, 'Rate Residuals']},

      'Epilepsy, Jacksonian' => { symptoms: epilepsy_symptoms },
      'Epilepsy, diencephalic' => { symptoms: epilepsy_symptoms },
      'Epilepsy, focal motor' => { symptoms: epilepsy_symptoms },
      'Epilepsy, grand mal' => { symptoms: epilepsy_symptoms },
      'Epilepsy, petit mal' => { symptoms: epilepsy_symptoms },
      'Epilepsy, psychomotor' => { symptoms: epilepsy_symptoms },
      'Epilepsy, sensory' => { symptoms: epilepsy_symptoms },
      'Narcolepsy' => { symptoms: epilepsy_symptoms },

      'Brain (malignant)' => { symptoms: [
        'Active malignancy',
        'Undergoing surgical, chemotherapeutic or other treatment modality',
        'Minimum', 'Rate Residuals']},
      'Spinal Cord (malignant)' => { symptoms: [
        'Active malignancy',
        'Undergoing surgical, chemotherapeutic or other treatment modality',
        'Minimum', 'Rate Residuals']},

      'Migraines' => {symptoms: [
        'Very frequent completely prostrating and prolonged attacks productive of severe economic inadaptability',
        'Characteristic prostrating attacks averaging one in 2 months over last several months',
        'Characteristic prostrating attacks occurring on an average once a month over last several months',
        'Less frequent attacks']},

      # This group all lack the "Rate Residuals" field
      'Multiple Sclerosis' => { symptoms: ['Minimum'] },
      'Myasthenia Gravis' => { symptoms: ['Minimum'] },
      'Myelitis' => { symptoms: ['Minimum'] },
      'Parkinson\'s Disease' => { symptoms: ['Minimum'] },
      'Progressive Muscular Atrophy' => { symptoms: ['Minimum'] },

      # missing 'Rate Residuals'
      'Soft-tissue sarcoma (of neurogenic origin)' => { symptoms: ['Rate Convalescence'] },

      # This exam appears to be entirely wrong! Should have 'Rate Convalescence'
      # and 'Rate Residuals', according to the IEPD
      #'Syringomyelia' => { symptoms: ['Rate Convalescence'] },
    }}

    it "should have an exam with each title in the iepd" do
      iepd.each do |title, val|
        expect(EXAMS).to have_key(title), title
      end
    end

    # it "should have each 'symptom' in a rating calculator" do
    #   iepd.each do |title, values|
    #     exam = EXAMS[title]
    #     values[:symptoms].each do |symptom|
    #       expect(exam).to have_a_symptom(symptom)
    #     end
    #   end
    #end

    it "should have each 'ratingCalculator' in a symptom listed above" do
      iepd.each do |title, iepd_values|
        expect(EXAMS[title]).to only_contain_symptoms_from(iepd_values[:symptoms])
      end
    end
  end

  context "#to_xml" do
    before do
      schema_xml = XML::Document.file(Rails.root.join("app/xsd/ExamResponse/exchange/ExamResponse-2.0.xsd").to_s)
      @schema = XML::Schema.document(schema_xml)
      @claim = create(:claim, date_of_birth: Date.new(1900))
      @contention = create(:contention)
      @claim.contentions << @contention
      allow_any_instance_of(Evaluation).to receive(:to_pdf) { "" }

      @evaluation_spec = EvaluationSpec.create(spec: File.read("app/exams/epilepsy-grand-mal.yml"))
      @evaluation = Evaluation.create(claim: @claim, evaluation_spec: @evaluation_spec, user: create(:user))
      @contention.evaluations << @evaluation

      @spec_json = File.read("spec/evaluations/epilepsy.json")
      @evaluation.doc = @spec_json
    end

    # it "should generate valid XML for an ear exam" do
    #   doc = XML::Document.string(@evaluation.to_xml)
    #   doc.validate_schema(@schema)
    # end

    context "when the eval is secondary, we output its parent as well" do
      before do
        user = create(:user)
        claim = create(:claim)
        claim.contentions << create(:contention, history: "This or That")

        dependent_eval = Evaluation.new(claim: claim, evaluation_spec: @evaluation_spec, user: user)
        dependent_eval.doc = @spec_json
        claim.contentions.first.evaluations << dependent_eval

        parent_eval = Evaluation.new(claim: claim, evaluation_spec: @evaluation_spec, completed_at: Time.now, user: user)
        parent_eval.doc = @spec_json
        claim.contentions.first.evaluations << parent_eval

        dependent_eval.primary_evaluation_id = parent_eval.id
        dependent_eval.save

        @evaluation_xml = dependent_eval.to_xml
      end

      # it "should include the parent eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   facts = xml.find("//nof:neurologicalOtherFacts")
      #
      #   expect(facts.length).to eq 2
      #
      #   schema_xml = XML::Document.file(Rails.root.join("app/xsd/ExamResponse/exchange/ExamResponse-2.0.xsd").to_s)
      #   schema = XML::Schema.document(schema_xml)
      #   xml.validate_schema(schema)
      # end
    end
  end
end
