require 'rails_helper'

RSpec.describe MusculoskeletalKneeEvaluationPresenter do
  before do
    @evaluation_spec = EvaluationSpec.create(spec: File.read("app/exams/knee-evaluation.yml"))
    @spec_json = File.read("spec/evaluations/knee.json")
  end

  context "with a knee exam" do
    before do
      schema_xml = XML::Document.file(Rails.root.join("app/xsd/ExamResponse/exchange/ExamResponse-2.0.xsd").to_s)
      @schema = XML::Schema.document(schema_xml)
      @claim = create(:claim, date_of_birth: Date.new(1900))
      @contention = create(:contention)
      @claim.contentions << @contention
      allow_any_instance_of(Evaluation).to receive(:to_pdf) { "" }

      @evaluation = Evaluation.create(claim: @claim, evaluation_spec: @evaluation_spec, user: create(:user))
      @contention.evaluations << @evaluation

      @evaluation.doc = @spec_json
      @evaluation_xml = @evaluation.to_xml
      @doc = XML::Document.string(@evaluation_xml)
    end

    # it "should generate valid XML" do
    #   @doc.validate_schema(@schema)
    # end

    it "shuould have some basic attributes" do
      expect(@doc.find("//kf:kneeDiagnoses/kf:value")[0].content).to eq "true"
      expect(@doc.find("//kf:ankylosisInflexionDegrees")[0].content).to eq "1"
      expect(@doc.find("//kf:lateralInstability")[0].content).to eq "Slight lateral instability"
    end

    it "should include jointsObjectiveEvidence" do
      expect(@doc.find("//jf:jointsObjectiveEvidence").length).to eq 4
      objectives = @doc.find("//jf:jointsObjectiveEvidence").map { |node| node.content }
      should_exist = ["Fatigue",
                      "Weakness",
                      "Incoordination",
                      "Lack of Endurance"]
      should_exist.each do |objective|
        expect(objectives.index objective).to_not eq nil
      end
    end
  end

  context "with a knee exam with arthritis" do
    before do
      schema_xml = XML::Document.file(Rails.root.join("app/xsd/ExamResponse/exchange/ExamResponse-2.0.xsd").to_s)
      @schema = XML::Schema.document(schema_xml)
      @claim = create(:claim, date_of_birth: Date.new(1900))
      @contention = create(:contention)
      @claim.contentions << @contention
      allow_any_instance_of(Evaluation).to receive(:to_pdf) { "" }

      @evaluation = Evaluation.create(claim: @claim, evaluation_spec: @evaluation_spec, user: create(:user))
      @contention.evaluations << @evaluation

      f = FML::Form.from_json(File.read("spec/evaluations/knee.json"))
      # turn all joint evidence false
      f.fields["hasArthritis"].value = true
      f.fields["hasArthritisType"].value = "Arthritis, rheumatoid"
      @evaluation.doc = f.to_json
      @doc = XML::Document.string(@evaluation.to_xml)
    end

    # it "should generate valid XML" do
    #   @doc.validate_schema(@schema)
    # end

    it "should include jointsOtherDiagnoses" do
      expect(@doc.find("//jf:jointsOtherDiagnoses").length).to eq 1
      expect(@doc.find("//jf:jointsOtherDiagnoses")[0].content).to eq "Arthritis, rheumatoid"
    end
  end

  context "when the eval is secondary, we output its parent as well" do
    before do
      user = create(:user)
      claim = create(:claim)
      claim.contentions << create(:contention, history: "This or That")

      dependent_eval = Evaluation.new(claim: claim, evaluation_spec: @evaluation_spec, user: user)
      dependent_eval.doc = @spec_json
      claim.contentions.first.evaluations << dependent_eval

      parent_eval = Evaluation.new(claim: claim, evaluation_spec: @evaluation_spec, completed_at: Time.now, user: user)
      parent_eval.doc = @spec_json
      claim.contentions.first.evaluations << parent_eval

      dependent_eval.primary_evaluation_id = parent_eval.id
      dependent_eval.save

      @evaluation_xml = dependent_eval.to_xml
    end

    # it "should include the parent eval" do
    #   xml = XML::Document.string(@evaluation_xml)
    #   facts = xml.find("//kf:kneeFacts")
    #
    #   expect(facts.length).to eq 2
    #
    #   schema_xml = XML::Document.file(Rails.root.join("app/xsd/ExamResponse/exchange/ExamResponse-2.0.xsd").to_s)
    #   schema = XML::Schema.document(schema_xml)
    #   xml.validate_schema(schema)
    # end
  end
end
