require 'rails_helper'

RSpec.describe GenitourinaryEvaluationPresenter do
  before do
    schema_xml = XML::Document.file(Rails.root.join("app/xsd/ExamResponse/exchange/ExamResponse-2.0.xsd").to_s)
    @schema = XML::Schema.document(schema_xml)
    @claim = create(:claim, date_of_birth: Date.new(1900))
    @contention = create(:contention)
    @claim.contentions << @contention
    allow_any_instance_of(Evaluation).to receive(:to_pdf) { "" }
  end
  
  context "when the eval is tertiary, we output all three evals" do
    before do
      @user = create(:user)
      @claim = create(:claim)
      @claim.contentions << create(:contention, history: "This or That")
      
      def spec(title)
        <<-SPEC
    form:
      title: "#{title}"
      id: #{Digest::SHA1.hexdigest title}
      keywords: "Testing Keywords"
      version: "1.0"
      dependent: "Genitourinary"
      bodySystem: "Genitourinary"
      fieldsets:
        - fieldset:
          - field:
              name: "yesno"
              fieldType: "yes_no"
              label: "bananarama"
              isRequired: true
        SPEC
      end

      @tertiary_spec = create(:diabetes_spec, spec: spec("tertiary"))
      @tertiary_eval = Evaluation.new(claim: @claim, evaluation_spec: @tertiary_spec, user: @user)
      @tertiary_eval.doc = File.read("spec/evaluations/testis-disability.json")
      @claim.contentions.first.evaluations << @tertiary_eval

      @secondary_spec = create(:diabetes_spec, spec: spec("secondary"))
      @secondary_eval = Evaluation.new(claim: @claim, evaluation_spec: @secondary_spec, completed_at: Time.now, user: @user)
      @secondary_eval.doc = File.read("spec/evaluations/testis-disability.json")
      @claim.contentions.first.evaluations << @secondary_eval

      @primary_spec = create(:diabetes_spec, spec: spec("primary"))
      @primary_eval = Evaluation.new(claim: @claim, evaluation_spec: @primary_spec, completed_at: Time.now, user: @user)
      @primary_eval.doc = File.read("spec/evaluations/testis-disability.json")
      @claim.contentions.first.evaluations << @primary_eval

      @tertiary_eval.primary_evaluation_id = @secondary_eval.id
      @tertiary_eval.save

      @secondary_eval.primary_evaluation_id = @primary_eval.id
      @secondary_eval.save

      @evaluation_xml = @tertiary_eval.to_xml
    end

    # it "should include all parent evals" do
    #   xml = XML::Document.string(@evaluation_xml)
    #   facts = xml.find("//gf:genitourinaryFacts")
    #
    #   expect(facts.length).to eq 3
    #   expect(facts.first.children[1].children[1].content).to eq "tertiary"
    #   expect(facts[1].children[2].children[1].content).to eq "secondary"
    #   expect(facts[2].children[2].children[1].content).to eq "primary"
    #   expect(facts[1].children[0].content).to eq "tertiary"
    #   expect(facts[2].children[0].content).to eq "secondary"
    #
    #   schema_xml = XML::Document.file(Rails.root.join("app/xsd/ExamResponse/exchange/ExamResponse-2.0.xsd").to_s)
    #   schema = XML::Schema.document(schema_xml)
    #   xml.validate_schema(schema)
    # end
  end
  
  describe "kidney transplant" do
    before do
      @evaluation_spec = EvaluationSpec.create(spec: File.read("app/exams/kidney-transplant.yml"))
      @evaluation = Evaluation.create(claim: @claim, evaluation_spec: @evaluation_spec, user: create(:user))
      @contention.evaluations << @evaluation
    end      
  
    context "when the exam has been completed, and indicates that the discharge was in the last year" do
      before do
        @evaluation.doc = File.read("spec/evaluations/kidney-transplant-within-one-year.json")
        @evaluation_xml = @evaluation.to_xml
      end
        
      # it "should generate valid XML for the Kidney Transplant eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end
    
      it "should not output a 'Rate residuals' other diagnoses" do
        expect(@evaluation_xml).not_to match "Rate residuals"
      end
    end
  
    context "when the exam has been completed, and indicates that the discharge was not in the last year" do
      before do
        @evaluation.doc = File.read("spec/evaluations/kidney-transplant-more-than-one-year.json")
        @evaluation_xml = @evaluation.to_xml
      end
        
      # it "should generate valid XML for the Kidney Transplant eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end
    
      it "should output a 'Rate residuals' other diagnoses" do
        expect(@evaluation_xml).to match "Rate residuals"
      end
    end
  end
  
  describe "malignant neoplasms" do
    before do
      @evaluation_spec = EvaluationSpec.create(spec: File.read("app/exams/malignant-neoplasms.yml"))
      @evaluation = Evaluation.create(claim: @claim, evaluation_spec: @evaluation_spec, user: create(:user))
      @contention.evaluations << @evaluation
    end

    context "when the malignant neoplasm is in remission" do
      before do
        @evaluation.doc = File.read("spec/evaluations/malignant-neoplasms-in-remission.json")
        @evaluation_xml = @evaluation.to_xml
      end
        
      # it "should generate valid XML for the Malignant neoplams eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end
    
      it "should add a diagnoses of In remission" do
        expect(@evaluation_xml).to match "In remission"
      end
    end
  end
  
  describe "testis disability" do
    before do
      @evaluation_spec = EvaluationSpec.create(spec: File.read("app/exams/testis-disability.yml"))
      @evaluation = Evaluation.create(claim: @claim, evaluation_spec: @evaluation_spec, user: create(:user))
      @contention.evaluations << @evaluation
    end
    
    context "when values are selected for right and left testicle" do
      before do
        @evaluation.doc = File.read("spec/evaluations/testis-disability.json")
        @evaluation_xml = @evaluation.to_xml
      end
      
      # it "should generate valid XML for the testis disability eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end
      
      it "should include the ratingCalculator values for the selected values" do
        expect(@evaluation_xml).to match "Complete Atrophy"
        expect(@evaluation_xml).to match /No Disability \(Right\)/
      end
    end
  end

  describe "Hydronephrosis" do
    before do
      @evaluation_spec = EvaluationSpec.create(spec: File.read("app/exams/hydronephrosis.yml"))
      @evaluation = Evaluation.create(claim: @claim, evaluation_spec: @evaluation_spec, user: create(:user))
      @contention.evaluations << @evaluation
    end
    
    context "when no catheter drainage is required" do
      before do
        @evaluation.doc = File.read("spec/evaluations/hydronephrosis-no-catheter.json")
        @evaluation_xml = @evaluation.to_xml
      end
      
      # it "should generate valid XML for the Hydronephrosis eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end
      
      it "should include the ratingCalculator values no catheter drainage required" do
        expect(@evaluation_xml).to match "No catheter drainage required"
      end
    end

    context "when catheter drainage is required" do
      before do
        @evaluation.doc = File.read("spec/evaluations/hydronephrosis-catheter.json")
        @evaluation_xml = @evaluation.to_xml
      end
      
      # it "should generate valid XML for the Hydronephrosis eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end
      
      it "should include the ratingCalculator values no catheter drainage required" do
        expect(@evaluation_xml).to match "Catheter drainage required"
      end
    end

    context "when no signs of infection are present" do
      before do
        @evaluation.doc = File.read("spec/evaluations/hydronephrosis-no-infection.json")
        @evaluation_xml = @evaluation.to_xml
      end

      # it "should generate valid XML for the hydronephrosis eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end

      it "should include the ratingCalculator value for no infection noted" do
        expect(@evaluation_xml).to match "No infection noted"
      end
    end

    context "when signs of infection are present" do
      before do
        @evaluation.doc = File.read("spec/evaluations/hydronephrosis-infection.json")
        @evaluation_xml = @evaluation.to_xml
      end

      # it "should generate valid XML for the hydronephrosis eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end

      it "should include the ratingCalculator values for infection noted" do
        expect(@evaluation_xml).to match "Infection noted"
      end
    end
  end

  describe "ureter-stricture" do
    before do
      @evaluation_spec = EvaluationSpec.create(spec: File.read("app/exams/ureter-stricture.yml"))
      @evaluation = Evaluation.create(claim: @claim, evaluation_spec: @evaluation_spec, user: create(:user))
      @contention.evaluations << @evaluation
    end
    
    context "when no catheter drainage is required" do
      before do
        @evaluation.doc = File.read("spec/evaluations/ureter-stricture-no-catheter.json")
        @evaluation_xml = @evaluation.to_xml
      end
      
      # it "should generate valid XML for the ureter-stricture eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end
      
      it "should include the ratingCalculator values no catheter drainage required" do
        expect(@evaluation_xml).to match "No catheter drainage required"
      end
    end

    context "when catheter drainage is required" do
      before do
        @evaluation.doc = File.read("spec/evaluations/ureter-stricture-catheter.json")
        @evaluation_xml = @evaluation.to_xml
      end
      
      # it "should generate valid XML for the ureter-stricture eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end
      
      it "should include the ratingCalculator values no catheter drainage required" do
        expect(@evaluation_xml).to match "Catheter drainage required"
      end
    end

    context "when no signs of infection are present" do
      before do
        @evaluation.doc = File.read("spec/evaluations/ureter-stricture-no-infection.json")
        @evaluation_xml = @evaluation.to_xml
      end

      # it "should generate valid XML for the ureter-stricture eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end

      it "should include the ratingCalculator value for no infection noted" do
        expect(@evaluation_xml).to match "No infection noted"
      end
    end

    context "when signs of infection are present" do
      before do
        @evaluation.doc = File.read("spec/evaluations/ureter-stricture-infection.json")
        @evaluation_xml = @evaluation.to_xml
      end

      # it "should generate valid XML for the ureter-stricture eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end

      it "should include the ratingCalculator values for infection noted" do
        expect(@evaluation_xml).to match "Infection noted"
      end
    end
  end

  describe "nephrolithiasis" do
    before do
      @evaluation_spec = EvaluationSpec.create(spec: File.read("app/exams/nephrolithiasis.yml"))
      @evaluation = Evaluation.create(claim: @claim, evaluation_spec: @evaluation_spec, user: create(:user))
      @contention.evaluations << @evaluation
    end
    
    context "when no catheter drainage is required" do
      before do
        @evaluation.doc = File.read("spec/evaluations/nephrolithiasis-no-catheter.json")
        @evaluation_xml = @evaluation.to_xml
      end
      
      # it "should generate valid XML for the nephrolithiasis eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end
      
      it "should include the ratingCalculator values no catheter drainage required" do
        expect(@evaluation_xml).to match "No catheter drainage required"
      end
    end

    context "when catheter drainage is required" do
      before do
        @evaluation.doc = File.read("spec/evaluations/nephrolithiasis-catheter.json")
        @evaluation_xml = @evaluation.to_xml
      end
      
      # it "should generate valid XML for the nephrolithiasis eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end
      
      it "should include the ratingCalculator values no catheter drainage required" do
        expect(@evaluation_xml).to match "Catheter drainage required"
      end
    end

    context "when no signs of infection are present" do
      before do
        @evaluation.doc = File.read("spec/evaluations/nephrolithiasis-no-infection.json")
        @evaluation_xml = @evaluation.to_xml
      end

      # it "should generate valid XML for the nephrolithiasis eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end

      it "should include the ratingCalculator value for no infection noted" do
        expect(@evaluation_xml).to match "No infection noted"
      end
    end

    context "when signs of infection are present" do
      before do
        @evaluation.doc = File.read("spec/evaluations/nephrolithiasis-infection.json")
        @evaluation_xml = @evaluation.to_xml
      end

      # it "should generate valid XML for the nephrolithiasis eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end

      it "should include the ratingCalculator values for infection noted" do
        expect(@evaluation_xml).to match "Infection noted"
      end
    end
  end

  describe "ureterolithiasis" do
    before do
      @evaluation_spec = EvaluationSpec.create(spec: File.read("app/exams/ureterolithiasis.yml"))
      @evaluation = Evaluation.create(claim: @claim, evaluation_spec: @evaluation_spec, user: create(:user))
      @contention.evaluations << @evaluation
    end
    
    context "when no catheter drainage is required" do
      before do
        @evaluation.doc = File.read("spec/evaluations/ureterolithiasis-no-catheter.json")
        @evaluation_xml = @evaluation.to_xml
      end
      
      # it "should generate valid XML for the ureterolithiasis eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end
      
      it "should include the ratingCalculator values no catheter drainage required" do
        expect(@evaluation_xml).to match "No catheter drainage required"
      end
    end

    context "when catheter drainage is required" do
      before do
        @evaluation.doc = File.read("spec/evaluations/ureterolithiasis-catheter.json")
        @evaluation_xml = @evaluation.to_xml
      end
      
      # it "should generate valid XML for the ureterolithiasis eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end
      
      it "should include the ratingCalculator values no catheter drainage required" do
        expect(@evaluation_xml).to match "Catheter drainage required"
      end
    end


    context "when no signs of infection are present" do
      before do
        @evaluation.doc = File.read("spec/evaluations/ureterolithiasis-no-infection.json")
        @evaluation_xml = @evaluation.to_xml
      end

      # it "should generate valid XML for the ureterolithiasis eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end

      it "should include the ratingCalculator value for no infection noted" do
        expect(@evaluation_xml).to match "No infection noted"
      end
    end

    context "when signs of infection are present" do
      before do
        @evaluation.doc = File.read("spec/evaluations/ureterolithiasis-infection.json")
        @evaluation_xml = @evaluation.to_xml
      end

      # it "should generate valid XML for the ureterolithiasis eval" do
      #   xml = XML::Document.string(@evaluation_xml)
      #   xml.validate_schema(@schema)
      # end

      it "should include the ratingCalculator values for infection noted" do
        expect(@evaluation_xml).to match "Infection noted"
      end
    end
  end

  describe "penile disability" do 
    before do
      @evaluation_spec = EvaluationSpec.create(spec: File.read("app/exams/penile-disability.yml"))
      @evaluation = Evaluation.create(claim: @claim, evaluation_spec: @evaluation_spec, user: create(:user))
      @contention.evaluations << @evaluation
    end
    
    context "when there is a loss of erectile power" do
      before do
        @evaluation.doc = File.read("spec/evaluations/penile-disability-loss-of-erectile-power.json")
        @evaluation_xml = @evaluation.to_xml
        @xml = XML::Document.string(@evaluation_xml)
      end
      
      # it "should generate a valid XML document" do
      #   @xml.validate_schema(@schema)
      # end
      
      it "should list loss of erectile power as an other diagnoses" do
        expect(@xml.find("//gf:genitourinaryOtherDiagnoses/gf:value[text() = 'Loss of erectile power']").size).to eq 1
      end
    end
  end
end
