# == Schema Information
#
# Table name: referrals
#
#  id                   :integer          not null, primary key
#  coordinator_id       :integer          not null
#  referral_number      :string           not null
#  authorization_number :string           not null
#  content              :json
#  created_at           :datetime         not null
#  updated_at           :datetime         not null
#  consultation_id      :integer
#  referral_status_id   :integer
#  referral_type_id     :integer
#  referral_reason_id   :integer
#  provider_id          :integer
#

require 'rails_helper'

RSpec.describe Referral, :type => :model do

  context "when creating associations" do

    before(:context) do
      @veteran = create(:veteran)
      @consultation = create(:consultation, veteran: @veteran)
    end

    it "should have an associated veteran and consultation", cpp: true do
      referral = create(:referral, consultation: @consultation)
      expect(referral.consultation.consultation_number).to eq @consultation.consultation_number
      expect(referral.consultation.veteran.full_name).to eq @veteran.full_name
    end

  end # context "when creating associations"


  context "when validating referrals" do

    it "should not create multiple referrals with the same authorization number", cpp: true do
      create(:referral, authorization_number: 'TEST-67890')
      expect {
        create(:referral, authorization_number: 'TEST-67890')
      }.to raise_error(ActiveRecord::RecordInvalid)
    end

    it "should not create referrals with blank authorization numbers", cpp: true do
      expect {
        create(:referral, authorization_number: nil)
      }.to raise_error(ActiveRecord::RecordInvalid)
    end

    it "should create but not update referrals with no diagnosis codes", cpp: true do
      referral = create(:referral, diagnosis_codes: nil)
      expect {
        referral.save!
      }.to raise_error(ActiveRecord::RecordInvalid)
    end

    it "should create but not update referrals with an empty length of stay value", cpp: true do
      referral = create(:referral, length_of_stay: nil)
      expect {
        referral.save!
      }.to raise_error(ActiveRecord::RecordInvalid)
    end

    it "should create but not update referrals with an empty length of procedure value", cpp: true do
      referral = create(:referral, length_of_procedure: nil)
      expect {
        referral.save!
      }.to raise_error(ActiveRecord::RecordInvalid)
    end

    it "should create but not update referrals with an empty number of visits value", cpp: true do
      referral = create(:referral, number_of_visits: nil)
      expect {
        referral.save!
      }.to raise_error(ActiveRecord::RecordInvalid)
    end

    it "should not create referrals with a non-numeric number of visits value", cpp: true do
      expect {
        referral = create(:referral, number_of_visits: 'ABC')
      }.to raise_error(ActiveRecord::RecordInvalid)
    end

    it "should not allow referrals to change number of visits to a non-numeric value", cpp: true do
      referral = create(:referral, number_of_visits: 3)
      expect {
        referral.number_of_visits = 'ABC'
        referral.save!
      }.to raise_error(ActiveRecord::RecordInvalid)
    end

  end # context "when validating referrals"


  context "when performing filtering" do

    # creating test data used for referral filtering tests.
    # Test data is used for VHA CC unit tests, and appears when
    # performing non-VHA tests but generally functions as 'white noise'.
    before(:context) do
      @test_visn = Visn.where(name: 'Rocky Mountain Network').first
      @test_care_category = CareCategory.where(title: 'Neurology').first
      @test_referral_type = ReferralType.where(title: 'Radiology').first
      @test_provider_with_visn = create(:provider, :with_visn_by_name, visn_name: 'Rocky Mountain Network')

      @default_veteran = create(:veteran)
      @veteran_with_testname = create(:veteran, first_name: 'Marge', last_name: 'Simpson')
      @veteran_with_ssn = create(:veteran, first_name: 'Test2SSN', last_name: 'User2SSN', ssn: '777444111')

      @vha_cc_user = create(:vha_user)
      @non_vha_user = create(:community_provider)

      @consultations = {
        default:                  create(:consultation, veteran: @default_veteran),
        urgent:                   create(:consultation, :urgent, veteran: @default_veteran),
        recently_no_longer_valid: create(:consultation, :recently_no_longer_valid, veteran: @default_veteran),
        no_longer_valid:          create(:consultation, :no_longer_valid, veteran: @default_veteran),
        with_testname:            create(:consultation, veteran: @veteran_with_testname),
        with_test_ssn:            create(:consultation, veteran: @veteran_with_ssn),
        with_consult_number:      create(:consultation, veteran: @default_veteran,
                                    consultation_number: 'GHI-123'),
        with_care_category:       create(:consultation, veteran: @default_veteran,
                                    care_category: @test_care_category)
      }

      @referrals = {
        recently_no_longer_valid: create(:referral, :complete_no_visits,
                                    consultation: @consultations[:recently_no_longer_valid]),
        no_longer_valid:          create(:referral, :complete_no_visits,
                                    consultation: @consultations[:no_longer_valid]),
        with_testname:            create(:referral, consultation: @consultations[:with_testname]),
        with_test_ssn:            create(:referral, consultation: @consultations[:with_test_ssn]),
        with_consult_number:      create(:referral, consultation: @consultations[:with_consult_number]),
        with_care_category:       create(:referral, consultation: @consultations[:with_care_category]),
        with_test_referral_type:  create(:referral, consultation: @consultations[:default],
                                    referral_type: @test_referral_type),
        with_coordinator:         create(:referral, consultation: @consultations[:default],
                                    coordinator: @vha_cc_user),
        with_auth_number:         create(:referral, consultation: @consultations[:default],
                                    authorization_number: 'GHI-654-321'),
        with_test_provider:       create(:referral, consultation: @consultations[:default],
                                    provider: @test_provider_with_visn),
        review_pending:           create(:referral, :review_pending,
                                    consultation: @consultations[:default]),
        pending:                  create(:referral, :pending,
                                    consultation: @consultations[:default])
      }

      @visible_referrals = @referrals.reject do |key, referral|
        referral == @referrals[:no_longer_valid]
      end
    end


    context "for VHA CC users" do

      it "should return all referrals in collection when no filter params are passed in", cpp: true do
        results = Referral.filter({}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys(@visible_referrals.keys, @referrals)
      end

      it "should filter based on veteran name", cpp: true do
        results = Referral.filter({'first_name' => @veteran_with_testname.first_name}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_testname], @referrals)
      end

      it "should return no records when filtering on name not in database", cpp: true do
        results = Referral.filter({'first_name' => 'asdfrgasgasghadsfghafsg'}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should filter based on veteran SSN", cpp: true do
        results = Referral.filter({'ssn' => '777444111'}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_test_ssn], @referrals)
      end

      it "should return no records when filtering on SSN not in database", cpp: true do
        results = Referral.filter({'ssn' => '777000000'}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should filter based on authorization number", cpp: true do
        results = Referral.filter({'authorization_number' => 'GHI-654-321'}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_auth_number], @referrals)
      end

      it "should return no records when filtering on authorization number not in database", cpp: true do
        results = Referral.filter({'authorization_number' => 'NOT-VALID-AUTHNUM'}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should filter based on consultation number", cpp: true do
        results = Referral.filter({'consultation_number' => 'GHI-123'}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_consult_number], @referrals)
      end

      it "should return no records when filtering on consultation number not in database", cpp: true do
        results = Referral.filter({'consultation_number' => 'NOT-VALID-CONSULT_NUM'}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should filter based on VISN id", cpp: true do
        results = Referral.filter({'visn_id' => @test_visn.id.to_s}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_test_provider], @referrals)
      end

      it "should return no records when filtering on VISN without referrals", cpp: true do
        visn_not_found = Visn.where(name: 'Sierra Pacific Network').first
        expect(visn_not_found).not_to be_nil

        results = Referral.filter({'visn_id' => visn_not_found.id.to_s}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should filter based on provider id", cpp: true do
        results = Referral.filter({'provider_id' => @test_provider_with_visn.id.to_s}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_test_provider], @referrals)
      end

      it "should return no records when filtering on provider containing no referrals", cpp: true do
        dummy_provider_id = create(:provider).id
        results = Referral.filter({'provider_id' => dummy_provider_id}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should filter based on care category id", cpp: true do
        results = Referral.filter({'care_category_id' => @test_care_category.id.to_s}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_care_category], @referrals)
      end

      it "should return no records when filtering on care category id not found", cpp: true do
        care_category_not_found = CareCategory.where(title: 'Diabetes').first
        expect(care_category_not_found).not_to be_nil

        results = Referral.filter({'care_category_id' => care_category_not_found.id.to_s}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should filter based on coordinator id", cpp: true do
        results = Referral.filter({'coordinator_id' => @vha_cc_user.id.to_s}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:with_coordinator], @referrals)
      end

      it "should return no records when filtering on coordinator not in database", cpp: true do
        not_a_coordinator = create(:examiner)
        results = Referral.filter({'coordinator_id' => not_a_coordinator.id.to_s}, @vha_cc_user)
        expect(results).to be_empty
      end

      it "should filter closed referrals and only return referrals closed in the last 30 days", cpp: true do
        complete_status = ReferralStatus.where(name: 'Complete No Visits').first
        results = Referral.filter({'status' => [complete_status.id]}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:recently_no_longer_valid], @referrals)
      end

      it "should filter based on multiple statuses by returning active referrals with any of the included statuses", cpp: true do
        review_status = ReferralStatus.where(name: 'Review Pending').first
        complete_status = ReferralStatus.where(name: 'Complete No Visits').first
        results = Referral.filter({'status' => [review_status.id, complete_status.id]}, @vha_cc_user)
        expect(results).to include_only_values_with_listed_keys([:review_pending, :recently_no_longer_valid], @referrals)
      end


      context "performing custom sidebar queries" do

        before(:context) do
          @referrals_for_sidebar = {
            expired: create(:referral, :review_pending,
                        consultation: @consultations[:recently_no_longer_valid]),
            stat:    create(:referral, :review_pending,
                        consultation: @consultations[:urgent]),
            closed_not_stat: create(:referral, :complete_no_visits,
                        consultation: @consultations[:urgent]),
            with_cprs_appointment: create(:referral, :with_appointments_added_to_cprs,
                        consultation: @consultations[:default]),
            with_non_cprs_appointments: create(:referral, :with_new_appointments,
                        consultation: @consultations[:default]),
          }
          @referrals_including_sidebar = @referrals_for_sidebar.merge(@referrals)
        end

        it "should filter for Expired referrals", cpp: true do
          results = Referral.filter({'expired_data' => true}, @vha_cc_user)
          expect(results).to include_only_values_with_listed_keys([:expired], @referrals_including_sidebar)
        end

        it "should filter for STAT referrals", cpp: true do
          results = Referral.filter({'stat_data' => true}, @vha_cc_user)
          expect(results).to include_only_values_with_listed_keys([:stat], @referrals_including_sidebar)
        end

        it "should filter for referrals with new appointments", cpp: true do
          results = Referral.filter({'new_appointments' => true}, @vha_cc_user)
          expect(results).to include_only_values_with_listed_keys([:with_non_cprs_appointments], @referrals_including_sidebar)
        end

        it "should filter for referrals with appointments requiring medical record letters", cpp: true do
          referrals_including_old_appointments = @referrals.merge({
            one_week_ago_and_incomplete: create(:referral, :with_new_appointments,
                        referral_appointment_time: 1.week.ago,
                        consultation: @consultations[:default]),
            one_week_ago_and_complete: create(:referral, :complete_no_visits, :with_new_appointments,
                        referral_appointment_time: 1.week.ago,
                        consultation: @consultations[:default])
          })

          results = Referral.filter({'medical_record_letters' => true}, @vha_cc_user)
          expect(results).to include_only_values_with_listed_keys([:one_week_ago_and_incomplete], referrals_including_old_appointments)
        end

        it "should increment the STAT counter when adding STAT referrals, and decrement when destroying STAT referrals", cpp: true do
          old_count = Referral.stat_count(@vha_cc_user)

          stat_referral_added = create(:referral, :review_pending, consultation: @consultations[:urgent])
          expect(Referral.stat_count(@vha_cc_user)).to eq(old_count + 1)

          stat_referral_added.destroy!
          expect(Referral.stat_count(@vha_cc_user)).to eq(old_count)
        end

        it "should increment the expired counter when adding expired referrals, and decrement likewise", cpp: true do
          old_count = Referral.expired_count(@vha_cc_user)

          expired_referral_added = create(:referral, :review_pending, consultation: @consultations[:recently_no_longer_valid])
          expect(Referral.expired_count(@vha_cc_user)).to eq(old_count + 1)

          expired_referral_added.destroy!
          expect(Referral.expired_count(@vha_cc_user)).to eq(old_count)
        end

        it "should increment the new appointments counter when adding referrals with new appointments, and decrement likewise", cpp: true do
          old_count = Referral.with_new_appointments_count

          new_appointments_referral_added = create(:referral, :with_new_appointments,
            consultation: @consultations[:default])
          expect(Referral.with_new_appointments_count).to eq(old_count + 1)

          new_appointments_referral_added.referral_appointments.destroy_all
          expect(Referral.with_new_appointments_count).to eq(old_count)
        end

        it "should not increment the new appointments counter when adding referrals with appointments added to CPRS", cpp: true do
          old_count = Referral.with_new_appointments_count

          referral_with_cprs_appointments_added = create(:referral, :with_appointments_added_to_cprs,
            consultation: @consultations[:default])
          expect(Referral.with_new_appointments_count).to eq(old_count)
        end

        it "should increment the medical letter counter when adding referrals requiring medical letters, and decrement likewise", cpp: true do
          old_count = Referral.med_letter_count(@vha_cc_user)

          med_letter_referral_added = create(:referral, :with_new_appointments,
            referral_appointment_time: 1.week.ago,
            consultation: @consultations[:default])
          expect(Referral.med_letter_count(@vha_cc_user)).to eq(old_count + 1)

          med_letter_referral_added.referral_appointments.destroy_all
          expect(Referral.med_letter_count(@vha_cc_user)).to eq(old_count)
        end

      end # context "performing custom sidebar queries"

    end # context "for VHA CC users"


    context "for non-VHA community provider users" do

      before(:context) do
        @east_coast_visn = Visn.where(name: 'VA Mid-Atlantic Health Care Network').first
        @midsouth_visn   = Visn.where(name: 'VA MidSouth Healthcare Network').first

        @east_coast_provider = create(:provider, :with_visn_by_name, visn_name: 'VA Mid-Atlantic Health Care Network')
        @midsouth_provider = create(:provider, :with_visn_by_name, visn_name: 'VA MidSouth Healthcare Network')

        @east_coast_non_vha_user = create(:community_provider, providers: [@east_coast_provider])
        @midsouth_non_vha_user = create(:community_provider, providers: [@midsouth_provider])

        # Referrals visible by default for @east_coast_non_vha_user
        @east_coast_referrals = {
          recently_no_longer_valid: create(:referral, :complete_no_visits,
                                      consultation: @consultations[:recently_no_longer_valid],
                                      provider: @east_coast_provider),
          with_testname:            create(:referral, consultation: @consultations[:with_testname],
                                      provider: @east_coast_provider),
          with_test_ssn:            create(:referral, consultation: @consultations[:with_test_ssn],
                                      provider: @east_coast_provider),
          with_consult_number:      create(:referral, consultation: @consultations[:with_consult_number],
                                      provider: @east_coast_provider),
          with_care_category:       create(:referral, consultation: @consultations[:with_care_category],
                                      provider: @east_coast_provider)
        }

        # Referrals visible by default for @midsouth_non_vha_user
        @midsouth_referrals = {
          with_test_referral_type:  create(:referral, consultation: @consultations[:default],
                                      referral_type: @test_referral_type,
                                      provider: @midsouth_provider),
          with_auth_number:         create(:referral, consultation: @consultations[:default],
                                      authorization_number: 'GHI-987-654',
                                      provider: @midsouth_provider),
          review_pending:           create(:referral, :review_pending,
                                      consultation: @consultations[:default],
                                      provider: @midsouth_provider)
        }
        @referrals_for_both_users = @east_coast_referrals.merge(@midsouth_referrals)
      end

      it "should not display any referrals not from the user's designated provider", cpp: true do
        east_coast_user_results = Referral.filter({}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({}, @midsouth_non_vha_user)

        # @referrals are the referrals from the original context used by VHA CC tests
        expect(east_coast_user_results).to include_no_values_from_list(@referrals.values)
        expect(midsouth_user_results).to   include_no_values_from_list(@referrals.values)
      end

      it "should return all referrals visible to the user when no filter params are passed in", cpp: true do
        east_coast_user_results = Referral.filter({}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_only_values_with_listed_keys(
          @east_coast_referrals.keys, @referrals_for_both_users
        )
        expect(midsouth_user_results).to   include_only_values_with_listed_keys(
          @midsouth_referrals.keys, @referrals_for_both_users
        )
      end

      it "should not display referrals for consultations that are no longer valid", cpp: true do
        invalid_referral = create(:referral, :complete_no_visits,
                                  consultation: @consultations[:no_longer_valid],
                                  provider: @east_coast_provider)

        results = Referral.filter({}, @east_coast_non_vha_user)
        expect(results).to_not include(invalid_referral)
      end

      it "should not display referrals that are not yet active", cpp: true do
        pending_referral = create(:referral, :pending,
                                  consultation: @consultations[:default],
                                  provider: @east_coast_provider)

        results = Referral.filter({}, @east_coast_non_vha_user)
        expect(results).to_not include(pending_referral)
      end

      it "should filter based on veteran name, only on referrals visible to the user", cpp: true do
        east_coast_user_results = Referral.filter({'first_name' => @veteran_with_testname.first_name}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'first_name' => @veteran_with_testname.first_name}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_only_values_with_listed_keys([:with_testname], @referrals_for_both_users)
        expect(midsouth_user_results).to   include_no_values_from_list(@referrals_for_both_users.values)
      end

      it "should filter based on veteran SSN, only on referrals visible to the user", cpp: true do
        east_coast_user_results = Referral.filter({'ssn' => '777444111'}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'ssn' => '777444111'}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_only_values_with_listed_keys([:with_test_ssn], @referrals_for_both_users)
        expect(midsouth_user_results).to   include_no_values_from_list(@referrals_for_both_users.values)
      end

      it "should filter based on authorization number, only on referrals visible to the user", cpp: true do
        east_coast_user_results = Referral.filter({'authorization_number' => 'GHI-987-654'}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'authorization_number' => 'GHI-987-654'}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_no_values_from_list(@referrals_for_both_users.values)
        expect(midsouth_user_results).to   include_only_values_with_listed_keys([:with_auth_number], @referrals_for_both_users)
      end

      it "should filter based on consultation number, only on referrals visible to the user", cpp: true do
        east_coast_user_results = Referral.filter({'consultation_number' => 'GHI-123'}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'consultation_number' => 'GHI-123'}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_only_values_with_listed_keys([:with_consult_number], @referrals_for_both_users)
        expect(midsouth_user_results).to   include_no_values_from_list(@referrals_for_both_users.values)
      end

      it "should filter based on VISN id from corresponding provider, displaying all visible records", cpp: true do
        east_coast_user_results = Referral.filter({'visn_id' => @east_coast_visn.id.to_s}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'visn_id' => @midsouth_visn.id.to_s}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_only_values_with_listed_keys(
          @east_coast_referrals.keys, @referrals_for_both_users
        )
        expect(midsouth_user_results).to   include_only_values_with_listed_keys(
          @midsouth_referrals.keys, @referrals_for_both_users
        )
      end

      it "should filter based on VISN id not from corresponding provider, displaying no visible records", cpp: true do
        east_coast_user_results = Referral.filter({'visn_id' => @midsouth_visn.id.to_s}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'visn_id' => @east_coast_visn.id.to_s}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to be_empty
        expect(midsouth_user_results).to   be_empty
      end

      it "should filter based on provider id from corresponding provider, displaying all visible records", cpp: true do
        east_coast_user_results = Referral.filter({'provider_id' => @east_coast_provider.id.to_s}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'provider_id' => @midsouth_provider.id.to_s}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_only_values_with_listed_keys(
          @east_coast_referrals.keys, @referrals_for_both_users
        )
        expect(midsouth_user_results).to   include_only_values_with_listed_keys(
          @midsouth_referrals.keys, @referrals_for_both_users
        )
      end

      it "should filter based on provider id not from corresponding provider, displaying no visible records", cpp: true do
        east_coast_user_results = Referral.filter({'provider_id' => @midsouth_provider.id.to_s}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'provider_id' => @east_coast_provider.id.to_s}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to be_empty
        expect(midsouth_user_results).to   be_empty
      end

      it "should filter based on care category id, only on referrals visible to the user", cpp: true do
        east_coast_user_results = Referral.filter({'care_category_id' => @test_care_category.id.to_s}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'care_category_id' => @test_care_category.id.to_s}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_only_values_with_listed_keys([:with_care_category], @referrals_for_both_users)
        expect(midsouth_user_results).to   include_no_values_from_list(@referrals_for_both_users.values)
      end

      it "should filter based on referral status, only on referrals visible to the user", cpp: true do
        review_status = ReferralStatus.where(name: 'Review Pending').first

        east_coast_user_results = Referral.filter({'status' => [review_status.id]}, @east_coast_non_vha_user)
        midsouth_user_results   = Referral.filter({'status' => [review_status.id]}, @midsouth_non_vha_user)

        expect(east_coast_user_results).to include_no_values_from_list(@referrals_for_both_users.values)
        expect(midsouth_user_results).to   include_only_values_with_listed_keys([:review_pending], @referrals_for_both_users)
      end

    end # context "for non-VHA community provider users"


    context "and generating CSV documents based on filtered results" do

      before(:context) do
        @veteran_names = Veteran.all.map &:first_name
        @find_any_name = -> (data_row) do
          @veteran_names.any? {|name| data_row.include? name }
        end
      end

      it "should return all records in CSV file" do
        csv_rows = Referral.to_csv({}, @vha_cc_user).split("\n")
        # count includes header line
        expect(csv_rows.count).to be > 2

        csv_header = csv_rows.first
        csv_rows_without_header = csv_rows.drop 1

        expect(@find_any_name.call(csv_header)).to eq false
        csv_rows_without_header.each do |csv_row|
          expect(@find_any_name.call(csv_row)).to eq true
        end
      end

      it "should return one record in CSV file for one-record filter" do
        referral_for_csv = create(:referral, authorization_number: 'CSV-TEST-098')
        csv_rows = Referral.to_csv({'authorization_number' => 'CSV-TEST-098'}, @vha_cc_user).split("\n")
        # count includes header line
        expect(csv_rows.count).to eq 2

        # header line should not contain veteran names, and first line should
        # contain veteran whose name was passed into to_csv filter parameter.
        expect(@find_any_name.call(csv_rows[0])).to eq false
        expect(@find_any_name.call(csv_rows[1])).to eq true
        expect(csv_rows[1]).to include(referral_for_csv.consultation.veteran.first_name)
      end

    end # context "and generating CSV documents based on filtered results"

  end # context "when performing filtering"

end
