# == Schema Information
#
# Table name: referral_documents
#
#  id                        :integer          not null, primary key
#  referral_document_type_id :integer
#  uploader_id               :integer
#  approver_id               :integer
#  approved_at               :datetime
#  content                   :json
#  created_at                :datetime         not null
#  updated_at                :datetime         not null
#  referral_id               :integer
#

require 'rails_helper'

RSpec.describe ReferralDocument, :type => :model do

  context 'associations' do
    it "should belongs to a referral", cpp: true do
      referral_document = ReferralDocument.new
      referral = Referral.new
      referral.referral_documents << referral_document
      expect(referral_document.referral).to be referral
    end

    it "should belongs to a referral_document_type", cpp: true do
      referral_document = ReferralDocument.new
      referral_document_type = ReferralDocumentType.new
      referral_document_type.referral_documents << referral_document
      expect(referral_document.referral_document_type).to be referral_document_type
    end

    it "should belongs to a approver", cpp: true do
      referral_document = ReferralDocument.new
      expect(referral_document).to respond_to :approver
    end

    it "should belongs to a uploader", cpp: true do
      referral_document = ReferralDocument.new
      expect(referral_document).to respond_to :uploader
    end
  end

  context 'validations' do
    it "should not raise validation error", cpp: true do
      referral_document = create(:referral_document)
      expect(referral_document.valid?).to be_truthy
    end
  end

  describe "#approved?" do
    context "when the referral document has been approved?" do
      before do
        @referral_document = create(:approved_referral_document)
      end

      it "should return true", cpp: true do
        expect(@referral_document.approved?).to eq true
      end
    end
  end

  describe 'visible?(current_user)' do
    context "when the referral document visible to user." do
      it 'should return true when referral_document is approved?', cpp: true do
        community_provider = create(:community_provider) #not a VHA user
        referral_document = create(:referral_document)
        expect(referral_document.visible?(community_provider)).to eq true
      end

      it 'should return true when referral_document uploader and current user is vha_user?', cpp: true do
        vha_user = create(:vha_user)
        referral_document = create(:referral_document, approved_at: nil, approver: nil, uploader: vha_user)
        expect(referral_document.visible?(vha_user)).to eq true
      end

      it 'should return true when referral_status is REVIEW_PENDING and uploader is not VHA user', cpp: true do
        community_provider = create(:community_provider) #not a VHA user
        referral_status =  ReferralStatus.where(name: 'Review Pending').first
        referral = create(:referral, referral_status: referral_status)
        referral_document = create(:referral_document, uploader: community_provider, referral: referral)
        expect(referral_document.visible?(community_provider)).to eq true
      end

      it "should return community_provider.is_non_vha?", cpp: true do
        community_provider = create(:community_provider) #not a VHA user
        referral_document = create(:referral_document, uploader: community_provider)
        expect(referral_document.visible?(community_provider)).to eq community_provider.is_non_vha?
      end

    end
  end

  # Return :delete_own_documents if document is yours, you are not a VHA user,
  # and referral status indicates that delete link should be displayed.
  describe "visible_action_links(current_user)" do
    it 'should return :delete_own_documents if document is yours, you are not a VHA user', cpp: true do
      community_provider = create(:community_provider) #not a VHA user
      referral_status =  ReferralStatus.where(name: 'Scheduled').first
      referral = create(:referral, referral_status: referral_status)
      referral_document = create(:referral_document, approved_at: nil, approver: nil, uploader: community_provider, referral: referral)
      expect(referral_document.visible_action_links(community_provider)).to eq(:delete_own_documents)
    end
  end  

end