# == Schema Information
#
# Table name: evaluation_specs
#
#  id                       :integer          not null, primary key
#  title                    :text
#  version                  :text
#  spec                     :text
#  created_at               :datetime
#  updated_at               :datetime
#  body_system              :string
#  evaluation_builder_title :string
#  dependent                :string
#  spec_id                  :string
#  active                   :boolean          default(TRUE)
#

require 'rails_helper'
require 'categorized_selectors'

RSpec.describe EvaluationSpec, type: :model do
  describe "create" do
    context "when an evaluation spec is created with a spec" do
      context "when a body system is specified" do
        before do
          @evaluation_spec = create(:diabetes_spec)
        end

        it "should save properly with initialized body_type attribute" do
          expect(@evaluation_spec.body_system).to eq "Endocrine"
        end

        it "should set the evaluation builder title to the body system value" do
          expect(@evaluation_spec.evaluation_builder_title).to eq @evaluation_spec.body_system
        end
      end

      context "when a body system and an evaluation builder title are specified" do
        before do
          @evaluation_spec = create(:sample_spec)
        end

        it "should set the evaluation builder title to what is in the spec" do
          expect(@evaluation_spec.evaluation_builder_title).to eq "America"
        end
      end
    end
  end

  describe "id" do
    before do
      @diabetes_spec = create(:diabetes_spec)
    end

    it "should have a spec_id" do
      expect(@diabetes_spec.spec_id).to eq "84b50fd99fdb6a873d6fdfc678745f1ca36bc18c"
    end
  end

  describe "dependencies" do
    before do
      @evaluation_spec = create(:dependent_spec)
      @diabetes_spec = create(:diabetes_spec)
    end

    it "should return true from dependent? only if the eval is dependent" do
      expect(@evaluation_spec.dependent?).to eq true
      expect(@diabetes_spec.dependent?).to eq false
    end

    it "should return the dependency for a dependent eval" do
      expect(@evaluation_spec.dependent).to eq "endocrine"
    end
  end

  describe "self.group_by_body_system" do
    context "when no evaluation_specs exist" do
      it "should return an empty array" do
        cs = CategorizedSelectors.new(EvaluationSpec.group_by_body_system)
        expect(cs.groups).to eq []
      end
    end

    context "when evaluation_specs exist" do
      before do
        @evaluation_spec1 = create(:diabetes_spec)
        @evaluation_spec2 = create(:evaluation_spec)
        @evaluation_spec3 = create(:conditional_spec)
      end

      it "should generate a collection of evaluation_specs grouped by body_system" do
        cs = CategorizedSelectors.new(EvaluationSpec.group_by_body_system)
        expect(cs.groups).to eq ['Endocrine', 'Unclassified']
        expect(cs.group_selectors('Endocrine')).to eq [[@evaluation_spec1.title, @evaluation_spec1.id, {"data-keywords"=>""}]]
        expect(cs.group_selectors('Unclassified')). to eq [[@evaluation_spec3.title, @evaluation_spec3.id, {"data-keywords"=>""}],
                                                           [@evaluation_spec2.title, @evaluation_spec2.id, {"data-keywords"=>"Testing Keywords"}]]
      end
      
      context "when there are inactive specs" do
        before do
          @evaluation_spec1.deactivate!
          @new_evaluation_spec_1 = create(:diabetes_spec)
          expect(EvaluationSpec.where(body_system: "Endocrine").count).to eq 2
        end
        
        it "should not return inactive specs" do
          cs = CategorizedSelectors.new(EvaluationSpec.group_by_body_system)
          expect(cs.groups).to eq ['Endocrine', 'Unclassified']
          expect(cs.group_selectors('Endocrine').size).to eq 1
        end
      end
    end
  end

  describe "sync" do
    before do
      @out = StringIO.new()
    end

    it "should sync valid specs" do
      EvaluationSpec.sync_specs(nil, "spec/exams/simple/*.yml", @out)
      expect(EvaluationSpec.all.length).to eq 3
    end

    it "should ignore duplicated specs" do
      EvaluationSpec.sync_specs(nil, "spec/exams/simple/*.yml", @out)
      EvaluationSpec.sync_specs(nil, "spec/exams/simple_duplicate/*.yml", @out)
      expect(EvaluationSpec.all.all? {|spec| spec.active }).to eq true
      expect(EvaluationSpec.all.length).to eq 3
    end

    it "should deactivate old specs and create new updated specs" do
      EvaluationSpec.sync_specs(nil, "spec/exams/simple/*.yml", @out)
      EvaluationSpec.sync_specs(nil, "spec/exams/simple_updated/*.yml", @out)

      expect(EvaluationSpec.active_exams.length).to eq 3
      expect(EvaluationSpec.all.length).to eq 6
      EvaluationSpec.active_exams.each do |updated_spec|
        old_spec = EvaluationSpec.where(spec_id:updated_spec.spec_id, active:false)[0]
        expect(old_spec.active).to eq false
        expect(old_spec.inactive).to eq true
        expect(updated_spec.id).not_to eq old_spec.id
        expect(updated_spec.title).to eq "New #{old_spec.title} Title"
      end
    end

    it "should only import from a given body_system" do
      EvaluationSpec.sync_specs(["Skin"], "spec/exams/simple/*.yml", @out)
      expect(EvaluationSpec.all.length).to eq 1
    end
  end
end
