#!/bin/env ruby
# encoding: utf-8

# == Schema Information
#
# Table name: evaluations
#
#  id                            :integer          not null, primary key
#  evaluation_spec_id            :integer
#  doc                           :json
#  created_at                    :datetime
#  updated_at                    :datetime
#  claim_id                      :integer
#  completed_at                  :datetime
#  primary_evaluation_id         :integer
#  guid                          :string
#  reviewed_at                   :datetime
#  vha_user_vista_access_code    :string
#  vha_user_vista_verify_code    :string
#  vha_user_electronic_signature :string
#  user_id                       :integer
#  assigner_id                   :integer
#  examination_id                :integer
#

require 'rails_helper'

RSpec.describe Evaluation, type: :model do
  before do
    @user = create(:examiner)
  end

  describe "#user" do
    it "requires a user" do
      e = Evaluation.new(evaluation_spec: create(:evaluation_spec))
      result = e.save
      expect(result).to eq false
      expect(e.errors.has_key? :user).to eq true
    end
  end

  describe "#new" do
    before do
      evaluation_spec = create(:evaluation_spec)
      @evaluation = Evaluation.new(evaluation_spec: evaluation_spec, user: @user)
      @evaluation.save
    end

    it "has a guid" do
      expect(@evaluation.guid.length).to eq 36
    end
  end

  describe "send_to_backend" do
    subject {
      Evaluation.new(
        claim: create(:claim),
        evaluation_spec: create(:diabetes_spec),
        user: create(:user))
    }

    before do
      ENV["ENABLE_PHASE_2"] = "true"
    end

    after do
      ENV.delete "ENABLE_PHASE_2"
    end

    context "when the job should be sent to VLER DAS" do
      before do
        ENV["PHASE_2_BODY_SYSTEMS"] = JSON.dump(["Endocrine"])
      end

      after do
        ENV.delete "PHASE_2_BODY_SYSTEMS"
      end

      it "sends it to VLER DAS" do
        expect(VlerDASJob).to receive(:perform_later)
        subject.send_to_backend
      end
    end

    context "when the job should not be sent to VLER DAS" do
      before do
        ENV["PHASE_2_BODY_SYSTEMS"] = JSON.dump([])
      end

      after do
        ENV.delete "PHASE_2_BODY_SYSTEMS"
      end

      it "sends it to VLER DAS" do
        expect(VBMSJob).to receive(:perform_later)
        subject.send_to_backend
      end
    end

    context "when phase_2 is not set at all" do
      before do
        ENV.delete "PHASE_2_BODY_SYSTEMS"
      end

      it "should send to VBMS" do
        expect(ENV.has_key? "PHASE_2_BODY_SYSTEMS").to eq false
        expect(VBMSJob).to receive(:perform_later)
        subject.send_to_backend
      end
    end
  end

  describe "#review!" do
    it "sets the reviewed_at date" do
      evaluation_spec = create(:diabetes_spec)
      evaluation = Evaluation.new(evaluation_spec: evaluation_spec, doc: build(:diabetes_doc).doc, user: @user)
      expect(evaluation.reviewed_at).to eq nil
      expect(evaluation.review!)
      expect(evaluation.reviewed_at).not_to eq nil
    end
  end

  describe "complete!" do
    before do
      @evaluation_spec = create(:diabetes_spec)
      @evaluation = Evaluation.new(evaluation_spec: @evaluation_spec, doc: build(:diabetes_doc).doc, user: @user)
    end

    context "when the evaluation is signed" do
      before do
        @evaluation.update_attributes({vha_user_vista_access_code: 'a', vha_user_vista_verify_code: 'a', vha_user_electronic_signature: 'a'})
        @evaluation.complete! @user
      end

      it "should have a completed_at date" do
        expect(@evaluation.completed_at).not_to eq nil
      end

      it "should create an EvaluationLog" do
        logs = @evaluation.evaluation_logs.where(message: "Evaluation submitted for review by #{ @user.name }")
        expect(logs.length).to eq 1
      end
    end

    context "when the evaluation is unsigned unsigned" do
      it "should raise a CredentialError when not signed" do
        expect { @evaluation.complete! @user }.to raise_error(CredentialError)
      end
    end
  end

  describe "#uncomplete!" do
    before do
      @evaluation_spec = create(:diabetes_spec)
      @evaluation = Evaluation.new(evaluation_spec: @evaluation_spec,
                                   doc: build(:diabetes_doc).doc,
                                   completed_at: Time.now,
                                   user: @user)
      @evaluation.uncomplete! @user
    end

    it "should have a completed_at date" do
      expect(@evaluation.completed_at).to eq nil
    end

    it "should create an evaluation_log" do
      logs = @evaluation.evaluation_logs.where(message: "Contention rejected by #{ @user.name }, evaluation uncompleted")
      expect(logs.length).to eq 1
    end
  end


  describe "#complete?" do
    before do
      evaluation_spec = create(:diabetes_spec)
      @evaluation = Evaluation.new(evaluation_spec: evaluation_spec, doc: build(:diabetes_doc).doc, user: @user)
      allow(@evaluation).to receive(:signed?).and_return(true)
    end

    it "returns true when after a call to #complete!" do
      @evaluation.complete! @user
      expect(@evaluation.completed?).to eq true
    end

    it "returns false when #complete! has not been called" do
      expect(@evaluation.completed?).to eq false
    end
  end

  describe "dependencies" do
     before do
      allow_any_instance_of(Evaluation).to receive(:signed?).and_return(true)
      @evaluation_spec = create(:dependent_spec)
      @claim = create(:claim)
      @claim.contentions << create(:contention, history: "This or That")
      @evaluation = Evaluation.new(claim: @claim,
                                   evaluation_spec: @evaluation_spec,
                                   doc: build(:diabetes_doc).doc,
                                   completed_at: Time.now,
                                   user: @user)
      @claim.contentions.first.evaluations << @evaluation

      @simple_spec= create(:diabetes_spec)
      @claim2 = create(:claim)
      @claim2.contentions << create(:contention, history: "This or That")
      @independent_eval = Evaluation.new(claim: @claim2,
                                         evaluation_spec: @simple_spec,
                                         doc: build(:diabetes_doc).doc,
                                         completed_at: Time.now,
                                         user: @user)
      @claim2.contentions.first.evaluations << @independent_eval

      @claim3 = create(:claim)
      @claim3.contentions << create(:contention, history: "This or That")
      @satisfied_evaluation = Evaluation.new(claim: @claim3,
                                             evaluation_spec: @evaluation_spec,
                                             doc: build(:diabetes_doc).doc,
                                             completed_at: Time.now,
                                             user: @user)
      @claim3.contentions.first.evaluations << @satisfied_evaluation
      @satisfied_evaluation.primary_evaluation_id = @independent_eval.id
    end

    it "should return dependent body system" do
      expect(@evaluation.dependent).to eq "endocrine"
      expect(@independent_eval.dependent).to eq nil
      expect(@satisfied_evaluation.dependent).to eq "endocrine"
    end

    it "should correctly answer dependent?" do
      expect(@evaluation.dependent?).to eq true
      expect(@independent_eval.dependent?).to eq false
      expect(@satisfied_evaluation.dependent?).to eq true
    end

    it "should fail check_dependencies for both" do
      expect {@evaluation.check_dependencies}.to raise_exception DependencyError
      expect {@independent_eval.check_dependencies}.not_to raise_error
      expect {@satisfied_evaluation.check_dependencies}.not_to raise_error
    end

    it "should not allow you to complete an evaluation with unsatisfied dependencies" do
      expect {@evaluation.complete! @user}.to raise_exception DependencyError
      expect {@satisfied_evaluation.complete! @user}.not_to raise_error
    end
  end

  describe "#to_xml" do
    before do
      allow_any_instance_of(Evaluation).to receive(:signed?).and_return(true)
      @evaluation_spec = create(:diabetes_spec)
      @claim = create(:claim)
      @claim.contentions << create(:contention, history: "This or That")
      @evaluation = Evaluation.new(claim: @claim,
                                   evaluation_spec: @evaluation_spec,
                                   doc: build(:diabetes_doc).doc,
                                   completed_at: Time.now,
                                   user: @user)
      @evaluation.created_at = Time.now - 4.days
      @claim.contentions.first.evaluations << @evaluation
      @evaluation.complete! @user
      allow(@evaluation).to receive(:to_pdf).and_return "just some randomish data so that we don't have to render the pdf"
      @xml = @evaluation.to_xml
    end

    it "should put the guid in the xml" do
      expect(@xml).to match @evaluation.guid
    end

    it "should show the document title text based on the evaluation spec's body system or evaluation builder value" do
      expect(@xml).to match "VA Disability Assessment #{@evaluation.evaluation_spec.evaluation_builder_title}"
    end

    it "should not use the created_at time as the receipt date" do
      expect(@xml).not_to match @evaluation.created_at.utc.iso8601
    end

    it "should embed a Base64 encoded version of the pdf and txt in the xml" do
      expect(@xml).to match Base64.encode64(@evaluation.to_pdf)[0..20]
      expect(@xml).to match Base64.encode64(@evaluation.to_txt)[0..20]
    end
  end

  describe "#to_html" do
    before do
      @evaluation_spec = create(:diabetes_spec)
      @claim = create(:claim)
      @claim.contentions << create(:contention, history: "This or That", reviewed_military_service_treatment_records: true)
      @evaluation = Evaluation.new(claim: @claim,
                                   evaluation_spec: @evaluation_spec,
                                   doc: build(:diabetes_doc).doc,
                                   completed_at: Time.now,
                                   user: @user)
      @claim.contentions.first.evaluations << @evaluation
    end

    context "on a linked evaluation" do
      before do
        @parent_spec = create(:evaluation_spec)
        @parent_eval = Evaluation.new(claim: @claim,
                                      evaluation_spec: @parent_spec,
                                      completed_at: Time.now,
                                      user: @user)
        @parent_eval.doc = FML::Form.new(@parent_spec.spec).to_json
        @claim.contentions.first.evaluations << @parent_eval

        @evaluation.primary_evaluation_id = @parent_eval.id
        @evaluation.save
      end

      it "put the parent spec in the html" do
        html = @evaluation.to_html
        expect(html).to match "Evaluation: Simple Yes/No"
      end
    end

    context "on an unlinked evaluation" do
      it "has all necessary fields in the rendered html" do
        html = @evaluation.to_html
        output = Capybara.string(html)
        expect(output).to have_selector('#records_table li', count: 10)
        expect(output).to have_text("☐ Military service personnel records")
        expect(output).to have_text("☑ Military service treatment records")
        expect(output).to have_content 'Does the veteran currently have diabetes mellitus?'
        expect(output).to have_content 'Visible Checkbox'
        expect(output).to have_no_content 'Random glucose'
        expect(output).to have_content("Claim: Joe Veteran #{@claim.created_at.to_date}")
        expect(output).to have_text('1 January 2014')
        expect(output).to have_selector('label.formlabel', text: 'Please explain how you determined whether the veteran requires regulation:')
        expect(output).to have_selector('.formvalue', text: 'I asked him')
        expect(output).to have_selector('label', text: "Visible Checkbox")
        expect(output).to_not have_selector('label', text: "Hidden Checkbox")
      end

      context "when the veteran's claim folder has been reviewed" do
        before do
          @claim.contentions.first.update_attributes(claim_folder_reviewed: true)
        end

        it "should simply note that in the text" do
          html = @evaluation.to_html
          output = Capybara.string(html)
          expect(output).to have_text "Veteran’s claim folder was reviewed"
          expect(output).to_not have_text "☐ Military service personnel records"
        end
      end
    end

    context "when not all of the signature fields are filled in" do
      before do
        @evaluation.update_attributes(vha_user_vista_access_code: "123", vha_user_vista_verify_code: "123", vha_user_electronic_signature: nil)
      end

      it "should not indciate that the evaluation was electronically signed" do
        html = @evaluation.to_html
        output = Capybara.string(html)
        expect(output).to have_no_text "electronically signed"
      end
    end

    context "when all the signature fields have been provided" do
      before do
        @evaluation.update_attributes(vha_user_vista_access_code: "123", vha_user_vista_verify_code: "123", vha_user_electronic_signature: "123")
      end

      it "should not indciate that the evaluation was electronically signed" do
        html = @evaluation.to_html
        output = Capybara.string(html)
        expect(output).to have_text "electronically signed"
      end
    end
  end

  describe "#to_pdf" do
    before do
      @evaluation_spec = create(:diabetes_spec)
      @claim = create(:claim)
      @claim.contentions << create(:contention, history: "This or That")
      @evaluation = Evaluation.new(claim: @claim,
                                   evaluation_spec: @evaluation_spec,
                                   doc: build(:diabetes_doc).doc,
                                   completed_at: Time.now,
                                   user: @user)
      @claim.contentions.first.evaluations << @evaluation
    end

    context "when rendering as PDF" do
      before do
        allow(@evaluation).to receive(:render_html).and_return "<p>HTML</p>"
      end

      it "should render the html as a PDF" do
        wicked = double("WickedPdf")
        expect(WickedPdf).to receive(:new) { wicked }
        expect(wicked).to receive(:pdf_from_string).with("<p>HTML</p>").and_return(kind_of(String))
        @evaluation.to_pdf
      end
    end
  end

  describe "#to_txt" do
    before do
      @evaluation_spec = create(:diabetes_spec)
      @claim = create(:claim)
      @claim.contentions << create(:contention, history: "This or That", reviewed_military_service_treatment_records: true)
      @evaluation = Evaluation.new(claim: @claim,
                                   evaluation_spec: @evaluation_spec,
                                   doc: build(:diabetes_doc).doc,
                                   completed_at: Time.now,
                                   user: @user)
      @claim.contentions.first.evaluations << @evaluation
    end

    it "should output the exam information as a text file" do
      txt = @evaluation.to_txt
      expect(txt).to match "Claim: Joe Veteran #{Date.current}"
      expect(txt).to match "Contention: My knee hurts"
      expect(txt).to match /\[ \] Military service personnel records/
      expect(txt).to match /\[X\] Military service treatment records/
      expect(txt).to match "1 January 2014"
      expect(txt).to match "you determined whether the veteran requires regulation:"
      expect(txt).to match "I asked him"
      expect(txt).to match "Visible Checkbox"
      expect(txt).not_to match "Hidden Checkbox"
    end

    context "when the veteran's claim folder has been reviewed" do
      before do
        @claim.contentions.first.update_attributes(claim_folder_reviewed: true)
      end

      it "should simply note that in the text" do
        txt = @evaluation.to_txt
        expect(txt).to match "Veteran's claim folder was reviewed"
        expect(txt).not_to match /\[ \] Military service personnel records/
      end
    end
  end

  describe "#evaluation_logs" do
    before do
      @evaluation_spec = create(:diabetes_spec)
      @claim = create(:claim)
      @claim.contentions << create(:contention, history: "This or That", reviewed_military_service_treatment_records: true)
      @evaluation = Evaluation.new(claim: @claim,
                                   evaluation_spec: @evaluation_spec,
                                   doc: build(:diabetes_doc).doc,
                                   completed_at: Time.now,
                                   user: @user)
      @claim.contentions.first.evaluations << @evaluation

      10.times { |i| create(:evaluation_log, evaluation_id: @evaluation.id) }
    end

    it "has evaluation logs" do
      expect(@evaluation.evaluation_logs.length).to eq 11
    end
  end
end
