require 'rails_helper'
require 'mail'
HTTPI.log = false

RSpec.describe VlerDASJob, :type => :job do
  before do
    allow_any_instance_of(Evaluation).to receive(:signed?).and_return(true)
    @evaluation_spec = create(:tmj_spec)
    @claim = create(:claim)
    @user = create(:examiner)
    @claim.contentions << create(:contention, history: "This or That")
    @evaluation = Evaluation.new(claim: @claim,
                                 evaluation_spec: @evaluation_spec,
                                 doc: build(:tmj_eval).doc,
                                 completed_at: Time.now,
                                 user: @user)
    @claim.contentions.first.evaluations << @evaluation
    @evaluation.complete! @user
    allow_any_instance_of(Evaluation).to receive(:to_pdf) { "" }
    ENV['VLER_DAS_URL'] = "http://example.com/api/v1"
    WebMock.enable!
  end

  after do
    WebMock.disable!
  end

  it "does not create a request to send if the evaluation is not complete and logs the error" do
    evaluation_spec = create(:diabetes_spec)
    claim = create(:claim)
    user = create(:user, email: "PII        ")
    claim.contentions << create(:contention, history: "This or That")
    evaluation = Evaluation.new(claim: claim,
                                evaluation_spec: evaluation_spec,
                                doc: build(:diabetes_doc).doc,
                                user: user)
    claim.contentions.first.evaluations << evaluation
    ENV['VLER_DAS_URL'] = "http://example.com/api/v1"
    WebMock.enable!
    expect(subject).to_not receive(:build_request)
    expect{subject.perform(evaluation)}.to change{EvaluationLog.count}.by(1)
  end

  it "does not send data to vlerdas when evaluation xml is invalid, and logs the failure" do
    allow(@evaluation).to receive(:to_xml) { "meows" }
    expect(subject).to_not receive(:build_request)
    expect{subject.perform(@evaluation)}.to change{EvaluationLog.count}.by(1)
  end

  it "sends data to vlerdas" do
    stub_request(:post, ENV['VLER_DAS_URL']).with {|request|
      headers = request.headers.map {|k,v| "#{k}: #{v}"}.join("\r\n")
      msgbody = "#{headers}\r\n\r\n#{request.body}"

      # To test that it's a valid multipart mime message, use the mail module
      mail = Mail.new(msgbody)
      expect(mail.multipart?)
      expect(mail.content_type).to match "multipart/form-data"
      expect(mail.header["Accept"].value).to match "application/json"
      expect(mail.parts.length).to eq 1

      mp = mail.parts[0]
      expect(mp.content_type).to match "application/xml"
      expect(mp.content_disposition).to match "filename=cui-#{@evaluation.guid}.xml"
      expect(mp.content_disposition).to match "name=file"

      expect(request.uri.host).to eq "example.com"
      expect(request.uri.path).to eq "/api/v1"
      expect(request.body).to match '^--3e776445'
    }
    subject.perform(@evaluation)
  end

  it "logs to the evaluation log table" do
    stub_request(:post, ENV['VLER_DAS_URL']).with {|request|
      headers = request.headers.map {|k,v| "#{k}: #{v}"}.join("\r\n")
      msgbody = "#{headers}\r\n\r\n#{request.body}"
    }
    expect{subject.perform(@evaluation)}.to change{EvaluationLog.count}.by(1)
  end

  context "SSL Environment Variables" do
    it "doesn't use :peer if no keyfile is present" do
      ENV.delete("VLER_DAS_KEYFILE")

      allow(@evaluation).to receive(:render_html).and_return "<html>simple</html>"
      allow(@evaluation).to receive(:to_xml).and_return "<xml>simple</xml>"

      expect_any_instance_of(HTTPI::Auth::SSL).to receive(:verify_mode=).with(:none)

      stub_request(:post, ENV['VLER_DAS_URL'])
      allow(@evaluation).to receive(:validate_xml).and_return([])
      subject.perform(@evaluation)
    end

    it "uses a keyfile only if keyfile is present" do
      ENV['VLER_DAS_URL'] = "https://example.com/api/v1"
      ENV["VLER_DAS_KEYFILE"] = "/tmp/key.crt"
      ENV["VLER_DAS_KEYPASS"] = "password"
      ENV["VLER_DAS_CERT"] = "/a/cert/file"
      ENV["VLER_DAS_CACERT"] = "/a/ca/cert"

      allow(@evaluation).to receive(:render_html).and_return "<html>simple</html>"
      allow(@evaluation).to receive(:to_xml).and_return "<xml>simple</xml>"

      expect_any_instance_of(HTTPI::Auth::SSL).to receive(:cert_key_file=).with(ENV["VLER_DAS_KEYFILE"]) { ENV["VLER_DAS_KEYFILE"] }
      expect_any_instance_of(HTTPI::Auth::SSL).to receive(:cert_key_password=).with(ENV["VLER_DAS_KEYPASS"]) { ENV["VLER_DAS_KEYPASS"] }
      expect_any_instance_of(HTTPI::Auth::SSL).to receive(:cert_file=).with(ENV["VLER_DAS_CERT"]) { ENV["VLER_DAS_CERT"] }
      expect_any_instance_of(HTTPI::Auth::SSL).to receive(:ca_cert_file=).with(ENV["VLER_DAS_CACERT"]) { ENV["VLER_DAS_CACERT"] }

      stub_request(:post, ENV['VLER_DAS_URL'])
      allow(@evaluation).to receive(:validate_xml).and_return([])
      subject.perform(@evaluation)
    end
  end
end
