# == Schema Information
#
# Table name: referrals
#
#  id                   :integer          not null, primary key
#  coordinator_id       :integer          not null
#  referral_number      :string           not null
#  authorization_number :string           not null
#  content              :json
#  created_at           :datetime         not null
#  updated_at           :datetime         not null
#  consultation_id      :integer
#  referral_status_id   :integer
#  referral_type_id     :integer
#  referral_reason_id   :integer
#  provider_id          :integer
#

FactoryGirl.define do

  get_content = -> (n) do
    diagnosis_code = DiagnosisCode.find(n).version_code
    fourdigit = (n+6543) % 10000
    {
      diagnosis_codes: diagnosis_code,
      procedure_codes: "X#{fourdigit}",
      length_of_stay: "#{2+(n%7)} days",
      length_of_procedure: "#{2+(n%7)} days",
      treatment_frequency: "#{2+(n%7)} times per day",
      number_of_visits: 2+(n%7),
      cost_estimate: "#{fourdigit}.27",
      status_message: "Status Message #{n} for referral"
    }
  end

  # Default referral: use default (first) consultation, default referral status
  # ("Pending"), and factory-generated non-VHA user with standard default values.
  # ReferralStatus, ReferralType, and ReferralReason get preloaded when running `rake db:seed`.
  # NOTE: don't count on this referral having the first default veteran -- unit tests
  # are run in random order and the default might not get created first.
  factory :referral do
    consultation     { Consultation.first || association(:consultation) }
    coordinator      { FactoryGirl.create(:community_provider) }
    referral_status  { ReferralStatus.where(name: 'Scheduled').first }
    sequence(:referral_number)      {|n| sprintf("%02d-%04d-%04d", (n%100), ((n*257) % 10_000), ((n*701) % 10_000)) }
    sequence(:authorization_number) {|n| sprintf("ABC-%02d-%04d-%04d", (n%100), ((n*313) % 10_000), ((n*683) % 10_000)) }
    sequence(:referral_type)        {|n| ReferralType.where(title: 'MD Office Visit').first }
    sequence(:referral_reason)      {|n| ReferralReason.find(1 + (n % ReferralReason.count)) }
    sequence(:content)              {|n| get_content.call(n) }
  end

  trait :pending do
    referral_status { ReferralStatus.where(name: 'Pending').first }
  end

  trait :complete_no_visits do
    referral_status { ReferralStatus.where(name: 'Complete No Visits').first }
  end

  trait :review_pending do
    referral_status { ReferralStatus.where(name: 'Review Pending').first }
  end

  trait :with_new_appointments do
    transient do
      referral_appointment_time nil
    end
    after(:create) do |saved_referral, evaluator|
      3.times do
        FactoryGirl.create(:referral_appointment, referral: saved_referral,
                           appointment_time: evaluator.referral_appointment_time || 7.days.from_now)
      end
    end
  end

  trait :with_appointments_added_to_cprs do
    transient do
      referral_appointment_time nil
    end
    after(:create) do |saved_referral, evaluator|
      3.times do
        FactoryGirl.create(:referral_appointment_added_to_cprs, referral: saved_referral,
                           appointment_time: evaluator.referral_appointment_time || 7.days.from_now)
      end
    end
  end

  trait :with_new_documents do
    transient do
      doc_uploader nil
    end
    after(:create) do |saved_referral, evaluator|
      3.times do
        FactoryGirl.create(:referral_document, referral: saved_referral,
                           uploader: evaluator.doc_uploader)
      end
    end
  end

  trait :with_approved_documents do
    transient do
      doc_uploader nil
      doc_approver nil
    end
    after(:create) do |saved_referral, evaluator|
      3.times do
        FactoryGirl.create(:approved_referral_document, referral: saved_referral,
                           uploader: evaluator.doc_uploader, approver: evaluator.doc_approver)
      end
    end
  end

end
