# frozen_string_literal: true
require 'memoist'
require 'pry'

module SAML
  # This class is responsible for putting together a complete ruby-saml
  # SETTINGS object, meaning, our static SP settings + the IDP settings
  # which must be fetched once and only once via IDP metadata.
  class SettingsService
    class << self
      extend Memoist

      SAML_CONFIG = Rails.application.config_for(:saml).freeze

      METADATA_RETRIES = 3
      OPEN_TIMEOUT = 2
      TIMEOUT = 15

      def saml_settings
        parsed_setings = OneLogin::RubySaml::IdpMetadataParser.new.parse(metadata, settings: settings)
        Rails.logger.info "Settings: #{parsed_setings.inspect}"
        return parsed_setings
      rescue => e
        Rails.logger.error "SAML::SettingService failed to parse SAML metadata: #{e.message}"
        raise e
      end
      memoize :saml_settings

      private

      def connection
        Faraday.new(SAML_CONFIG['metadata_url']) do |conn|
          conn.options.open_timeout = OPEN_TIMEOUT
          conn.options.timeout = TIMEOUT
          conn.adapter :net_http
        end
      end
      memoize :connection

      def metadata
        attempt ||= 0
        response = connection.get
        raise SAML::InternalServerError, response.status if (400..504).cover? response.status.to_i
        response.body
      rescue StandardError => e
        attempt += 1
        msg = "Failed to load SAML metadata {msg = #{e.message}}: try #{attempt} of #{METADATA_RETRIES}"
        attempt >= METADATA_RETRIES ? Rails.logger.error(msg) : Rails.logger.warn(msg)
        if attempt < METADATA_RETRIES
          sleep attempt * 0.25
          retry
        end
      end

      def settings
        settings = OneLogin::RubySaml::Settings.new
        settings.certificate = SAML_CONFIG['certificate']
        settings.private_key = SAML_CONFIG['key']
        settings.issuer = SAML_CONFIG['issuer']
        settings.assertion_consumer_service_url = SAML_CONFIG['callback_url']
        settings.authn_context = 'http://idmanagement.gov/ns/assurance/loa/1'
        settings.security[:authn_requests_signed] = true
        settings.security[:logout_requests_signed] = true
        settings.security[:embed_sign] = false
        settings.security[:signature_method] = XMLSecurity::Document::RSA_SHA256
        settings.security[:digest_method] = XMLSecurity::Document::SHA256
        settings.idp_cert_fingerprint_algorithm = XMLSecurity::Document::RSA_SHA256
        return settings
      end
    end
  end
  class InternalServerError < StandardError
  end
end
