module DataTablePaginator

  # Creates JSON data for response based on query passed in.
  # The :start and :length params
  # Default for JSON-ification is to pass in as_json params.
  # Alternate method: pass block to JSON-ify to convert User to standard
  # Ruby Hash object which then gets JSON-ified.
  def datatables_json_for_query(query, json_data_params)
    total_record_count = query.count
    return {
      draw: params[:draw].to_i,
      recordsTotal: total_record_count,
      recordsFiltered: total_record_count,
      data: paginate_with_params(query).as_json(json_data_params)
    }
  end

  def datatables_json_for_query_using_func(query, json_transform)
    json_user_data = paginate_with_params(query).map {|u| json_transform.call(u) }
    total_record_count = query.count
    return {
      draw: params[:draw].to_i,
      recordsTotal: total_record_count,
      recordsFiltered: total_record_count,
      data: json_user_data
    }
  end

  def paginate_with_params(query)
    query.offset(params[:start]).limit(params[:length])
  end

  def apply_ordering_to_query(query, ordering_map, ordering_params = nil)
    order_by_selection = parse_ordering_params(ordering_map, ordering_params)
    query.order order_by_selection
  end

private

  # Expects DataTables ordering params (ordering_params) of the form:
  # {"0"=>{"column"=>"0", "dir"=>"asc"}, ...}
  # ..where the "column" value points to a column index to be referenced
  # in the ordering_map_by_index map passed in as the first parameter:
  # {0 => 'dbfield1', 1 => 'dbfield2', ...}
  # If an ordering column does not exist in the map, it will not get added.
  def parse_ordering_params(ordering_map_by_index, ordering_params = nil)
    ordering_params ||= {"0"=>{"column"=>"0", "dir"=>"asc"}}
    ordering_params.keys.sort.inject([]) do |arr, key|
      val = ordering_params[key].symbolize_keys
      colindex = val[:column].to_i
      if ordering_map_by_index.has_key? colindex
        field_or_fields = ordering_map_by_index[colindex]
        if field_or_fields.is_a?(Array)
          arr += field_or_fields.map {|field| "#{field} #{val[:dir]}" }
        else
          arr.push "#{field_or_fields} #{val[:dir]}"
        end
      end
      arr
    end
  end

end
