# lib/csv_seeder.rb

require 'smarter_csv'

# Support utility for db/seeds.rb
module CsvSeeder

  CHUNK_SIZE = 100

  # These next constants are used by backup_database_table
  DB_SUPERUSERS = ActiveRecord::Base.connection.execute("SELECT rolname FROM pg_roles where rolsuper='t';").to_a.map{|row| row['rolname']}

  I_AM_SUPER = DB_SUPERUSERS.include?(ENV['DBUSER'])
  I_AM_LOCAL = ENV['DBHOST'].include?('localhost')

  TIME_STAMP = Time.now.strftime('%Y%m%d_%H%M%S')

  # static CSV files have no headers; however, you can pass a header definition in as
  # the 'column_mappings' parameter
  #
  # NOTE: this method requires that one of the column names be 'id'

  # TODO: accept a header row from the CSV file; this will make the CSV file un-usable to
  #       an SQL COPY FROM ... csv command with PostgreSQL

  # Do common stuff before passing off to either the update or add method
  def load_static_text_table_from_csv(
      klass:,                    # the actual ActiveRecord Class object
      csv_file:         nil,     # The name of the csv file located in db/csv/*.csv
      column_mappings:  ['id','sequence','title'], # An array of mapped column names
      headers_in_file:  false,   # Our standard CSV files do not contain headers
      use_copy_from:    false,   # Makes use of the fastest data load process
      common_fields:    {}       # common field names and values to be added to each record
    )

    print "Loading static text for #{klass} #{use_copy_from ? 'using Copy From CSV SQL ' : ''}..."

    # NOTE: Can also use:    header = klass.columns.map{|column| column.name}.join(',')
    #       pr make simpler: header = klass.columns.map{|column| column.name.to_sym}
    unless klass.ancestors.include? ActiveRecord::Base
      Rails.logger.error "ERROR: expected an ActiveRecord::Base class but got: #{klass} a class #{klass.superclass}"
      return false
    end

    if use_copy_from  &&  !common_fields.empty?
      Rails.logger.error "ERROR: For class #{klass} you can not have 'use_copy_from: true' and also attempted to add common_fields"
      return false
    end

    if use_copy_from
      unless I_AM_LOCAL  &&  I_AM_SUPER
        Rails.logger.error "ERROR: For class #{klass} you can not have 'use_copy_from: true' when the DBHOST is not localhost and the DBUSER is not a superuser."
        return false
      end
    end


    if csv_file.nil?
      csv_file = klass.table_name + '.csv'
    end

    csv_path = Rails.root + "db/csv/#{csv_file}"

    unless csv_path.exist?
      Rails.logger.error "ERROR: Missing file: #{csv_path}"
      return false
    end

    if Rails.env.development?  &&  DBRESET
      # This environment means that we are most likely resetting the database
      # via a rake db:reset task
      add_static_text_table_from_csv(
          klass:           klass,
          csv_path:        csv_path,
          column_mappings: column_mappings,
          headers_in_file: headers_in_file,
          use_copy_from:   use_copy_from,
          common_fields:   common_fields
        )
    else
      # Other environments are mostly trying to update an existing database
      # content.
      update_static_text_table_from_csv(
          klass:           klass,
          csv_path:        csv_path,
          column_mappings: column_mappings,
          headers_in_file: headers_in_file,
          common_fields:   common_fields
        )
    end

  end # def load_static_text_table_from_csv


  # Updates an existing table of content.  Intended for use outside of the 'development' environment
  # to update production tables with changed values.
  #
  def update_static_text_table_from_csv(
      klass:,            # the actual ActiveRecord Class object
      csv_path:,         # the name of the CSV file to use from the db/csv directory
      column_mappings:,  # an array of attributes mapped to each CSV column
      headers_in_file:,  # boolean; does the CSV file contain a header record
      common_fields:     # hash of field names and values to be added to every record
    )

    # Make a backup of the table first
    backup_database_table(klass)

    # keep all keys symbolized for comparing/merging hashes
    common_fields.symbolize_keys!

    csv_processing_params = {
      headers_in_file:        headers_in_file,
      chunk_size:             CHUNK_SIZE,
      user_provided_headers:  column_mappings.map {|fieldname| fieldname.strip.to_sym }
    }

    # currently always using leftmost column as key field in CSV document.
    # This field is currently assumed to be unique for each record.
    key_column = column_mappings[0].to_sym

    existing_id_list = klass.pluck(key_column).map(&:to_s)
    counts = {insert: 0, update: 0, match: 0}

    # update records, one chunk at a time
    total_chunks = SmarterCSV.process(csv_path, csv_processing_params) do |chunk|
      print '.'

      if common_fields.present?
        chunk.each {|record| record.merge! common_fields }
      end
      records_by_action_type = separate_records_by_action_type(chunk, klass, key_column)

      counts[:match] += records_by_action_type[:match].count

      # bulk insert records not previously found in DB.
      if records_by_action_type[:insert].count > 0
        new_records = []
        records_by_action_type[:insert].each do |record_to_insert|
          new_records.push klass.new(record_to_insert)
        end
        klass.import new_records
        counts[:insert] += new_records.count
      end

      records_by_action_type[:update].each do |record_to_update|
        record_id = record_to_update.delete(:id)
        klass.update(record_id, record_to_update)
        counts[:update] += 1
      end

      # keys from CSV records read in should be removed from deleted list.
      existing_id_list = existing_id_list - records_by_action_type[:keys]

    end # SmarterCSV.process

    # soft-delete records not found in CSV document
    counts[:delete] = existing_id_list.count
    klass.where(id: existing_id_list).destroy_all

    counts_message = counts.map {|key, count| "#{key}=#{count}" }.join(', ')
    puts "\n#{klass.name} counts: #{counts_message}."

  end # def update_static_text_table_from_csv(klass, csv_file=nil)


  # Separate a chunk of CSV records passed in, into three categories:
  # records to add (INSERT), records to change (UPDATE), and records that
  # match and should be left alone.
  # Also return a list of stringified keys to remove from the to-delete list.
  def separate_records_by_action_type(chunk, klass, key_field)
    separated_records = {insert: [], update: [], match: []}
    keys_in_chunk = chunk.map {|record| record[key_field].to_s }
    separated_records[:keys] = keys_in_chunk

    # get records from query (which includes soft-deleted records),
    # then convert record array to key->record hash
    query_from_chunk_keys = klass.with_deleted.where({key_field => keys_in_chunk})
    chunk_ar_by_key = query_from_chunk_keys.each_with_object({}) do |record, obj|
      obj[record.send(key_field)] = record
    end

    chunk.each do |csv_record|
      record_key = csv_record[key_field]
      db_record = chunk_ar_by_key[record_key]
      if chunk_ar_by_key.key? record_key
        if compare_to_record(csv_record, db_record)
          separated_records[:match].push csv_record
        else
          # add the ID to records that need to be updated, if not present
          csv_record[:id] ||= db_record.id
          csv_record[:deleted_at] = nil
          separated_records[:update].push csv_record
        end
      else
        separated_records[:insert].push csv_record
      end
    end

    return separated_records
  end

  # Compare hashified data record (such as CSV object data used by SmarterCSV)
  # to existing ActiveRecord and retrurn true if the records match.
  # Data value type (Integer, String, etc.) is ignored as comparisons are
  # all with stringified objects.
  def compare_to_record(data_record, existing_active_record)
    data_record.all? do |fieldname, value|
      existing_active_record.send(fieldname).to_s == value.to_s
    end
  end

  # A header of :id is not required.  It will be deleted if passed as part of the header string
  def add_static_text_table_from_csv(
      klass:,           # active_record model's class name
      csv_path:,        # the name of the CSV file to use from the db/csv directory
      column_mappings:, # an array of attributes mapped to each CSV column
      headers_in_file:, # boolean; does the CSV file contain a header record
      common_fields:,   # hash of field names and values to be added to every record
      use_copy_from:    # boolean; to use or not to use <=- That is the question!
    )

    # Add the new records in chunks (array of hashes)
    if use_copy_from
      ActiveRecord::Base.connection.execute("COPY #{klass.table_name} FROM '#{csv_path}' CSV")
    else
      total_chunks = SmarterCSV.process(
          csv_path,
          {
            headers_in_file:        headers_in_file,
            chunk_size:             100,
            user_provided_headers:  column_mappings.map{|a_string| a_string.strip.to_sym}
          }
        ) do |chunk|
            print '.'
            chunk.each do |a_hash|
             a_hash.delete(:id)
             a_hash.merge!(common_fields) unless common_fields.empty?
            end
            klass.create(chunk)
         end
    end

    puts " done"

  end # def Add_static_text_table_from_csv(klass, csv_file=nil)


  # Backup a database table to a CSV file
  # PostgreSQL v9.3 documentation says:
  #
  # Files named in a COPY command are read or written directly by the server,
  # not by the client application. Therefore, they must reside on or be accessible
  # to the database server machine, not the client. They must be accessible to and
  # readable or writable by the PostgreSQL user (the user ID the server runs as),
  # not the client. Similarly, the command specified with PROGRAM is executed
  # directly by the server, not by the client application, must be executable by
  # the PostgreSQL user. COPY naming a file or command is only allowed to database
  # superusers, since it allows reading or writing any file that the server has
  # privileges to access.

  def backup_database_table(ar_model)
    table_name = ar_model.table_name
    csv_path = Rails.root + 'db' + 'backup' + "#{table_name}_#{TIME_STAMP}.csv"

    if I_AM_LOCAL  &&  I_AM_SUPER
      # NOTE: This is fast but has limitations.  See documentation on PostgreSQL v9.3 COPY
      #       Also this does not output a header row
      ActiveRecord::Base.connection.execute("COPY #{table_name} TO '#{csv_path}' CSV")
    else
      CSV.open(csv_path, "wb") do |csv|
        csv << ar_model.attribute_names
        ar_model.all.each do |row|
          csv << row.attributes.values
        end
      end
    end

  end

end # module CsvSeeder