#!/usr/bin/env ruby
######################################################
###
##  File: im_gateway.rb
##  Desc: Server for distributed Ruby (dRuby) gateway to hipchat,
##        slack or both instant messaging (IM) platforms.
##        Required for integration with Rails v4.2.3
##        due to an openSSL error within Rails.
#

# NOTE: v2 of the Hipchat server API is hardcoded.

require 'dotenv'
require 'awesome_print'

require 'drb/drb'
require 'pathname'

root          = Pathname.new(__FILE__).realpath.parent.parent
lib           = root + 'lib'
log           = root + 'log'

require lib + 'truth_or_consequences'

LOG = File.open(log+'im_gateway.log', 'a')
LOG.puts "Starting ..."

# dotenv uses a first set always set rule w/r/t SEV value; this allows
# the setting of an SEV on the command line to over-ride the values in the
# .env* files.
Dotenv.load(
  (root + '.env.local'),
  (root + ".env.#{ENV['RAILS_ENV'] || 'development'}"),
  (root + '.env')
)

LOG.puts "environment loaded:"

ENV.keys.select{ |key|
    key.include?('HIPCHAT') || key.include?('SLACK') || key.include?('_IM_')
  }.sort.each do |key|
  LOG.puts "\t#{key}\t#{ENV[key]}"
end

LOG.flush


truth_or_consequences 'enable_feedback'
truth_or_consequences 'enable_feedback_hipchat'
truth_or_consequences 'enable_feedback_slack'

exit unless ENABLE_FEEDBACK  &&  (ENABLE_FEEDBACK_HIPCHAT || ENABLE_FEEDBACK_SLACK)

LOG.puts "IM notification was enabled."
LOG.flush

require 'hipchat'       if ENABLE_FEEDBACK_HIPCHAT

if ENABLE_FEEDBACK_SLACK
  require 'slack-notify'
  require 'slacken'
end


class ImGateway

  def initialize_hipchat()
    @room     = ENV['FEEDBACK_HIPCHAT_ROOM']
    @server   = ENV['FEEDBACK_HIPCHAT_SERVER']
    @token    = ENV['FEEDBACK_HIPCHAT_TOKEN']

    unless @room && @token
      puts <<~EOS

        The following system environment variables are required for hipchat notification:

          FEEDBACK_HIPCHAT_ROOM .... current value: #{ENV['FEEDBACK_HIPCHAT_ROOM']}
          FEEDBACK_HIPCHAT_TOKEN ... current value: #{ENV['FEEDBACK_HIPCHAT_TOKEN']}

        The following system environment variable is optional:

          FEEDBACK_HIPCHAT_SERVER .. current value: #{ENV['FEEDBACK_HIPCHAT_SERVER']}

      EOS
      exit
    end

    if @server.nil? || @server.empty?
      # use the default server
      @hipchat_client = HipChat::Client.new(
        @token,
        :api_version => 'v2'
      )
    else
      @hipchat_client = HipChat::Client.new(
        @token,
        :api_version => 'v2',
        :server_url  => "https://#{@server}"
      )
    end

    LOG.puts "hipchat client initialized."
    LOG.flush

  end # def initialize_hipchat


  def initialize_slack

    webhook_url = ENV['FEEDBACK_SLACK_WEBHOOK_URL']
    channel     = ENV['FEEDBACK_SLACK_CHANNEL']       || '#feedback'
    username    = ENV['FEEDBACK_SLACK_USERNAME']      || 'feedbacker'


    unless webhook_url && channel && username
      puts <<~EOS

        The following system environment variables are required for slack notification:

          FEEDBACK_SLACK_WEBHOOK_URL ... current value: #{ENV['FEEDBACK_SLACK_WEBHOOK_URL']}
          FEEDBACK_SLACK_CHANNEL ....... current value: #{ENV['FEEDBACK_SLACK_CHANNEL']}
          FEEDBACK_SLACK_USERNAME ...... current value: #{ENV['FEEDBACK_SLACK_USERNAME']}

      EOS
      exit
    end

    @slack_client = SlackNotify::Client.new(
                      webhook_url:  webhook_url,
                      channel:      channel,
                      username:     username
                    )

    LOG.puts "slack client initialized."
    LOG.flush

  end # def initialize_slack


  def initialize
    initialize_hipchat  if ENABLE_FEEDBACK_HIPCHAT
    initialize_slack    if ENABLE_FEEDBACK_SLACK
  end


  def notify_hipchat(options={})
    LOG.puts "sending to hipchat ..."
    LOG.flush
    fromuser  = options["fromuser"]   || 'IM Gateway'
    fromuser  = fromuser[0,20] if fromuser.size > 20

    message   = options["message"]  || 'Integration Error'
    notify    = options["notify"]   || false
    color     = options["color"]    || 'yellow'

    result = @hipchat_client[@room].send(
      fromuser,
      message.gsub("\n",'<br />'), # NOTE: hipchat message render HTML
      notify: notify,
      color: color
    )

    LOG.puts "Result: #{result}"
    LOG.flush
  end # def notify_hipchat(options={})


  def notify_slack(options={})
    LOG.puts "sending to slack ..."
    LOG.flush
    message = options['message'].gsub("\n", "<br /><br />")
    @slack_client.notify(Slacken.translate(message)) # xlate HTML to markdown for slack
  end


  def notify(options={})
    LOG.puts "notify: #{options}"
    LOG.flush
    notify_hipchat(options)  if ENABLE_FEEDBACK_HIPCHAT
    notify_slack(options)    if ENABLE_FEEDBACK_SLACK
  end



end # class ImGateway

gateway = ImGateway.new

gateway_url = ENV['FEEDBACK_IM_GATEWAY'] || 'druby://localhost:9999'

LOG.puts "starting dRb service."
LOG.flush

DRb.start_service(gateway_url, gateway)
DRb.thread.join

LOG.puts "terminating ..."
LOG.flush
LOG.close

