module VX_API

  # Access the VX API PatientSearch service for veteran(s)
  class PatientSearch
    include ActiveModel::Model

    # return status code is NOT 200 when trying to access URL with search parameters
    class AccessError < RuntimeError; end

    URL = VX_API::BASE_URL + '/patient-search/global'

    # Standardized parameters before using the web-service
    SEARCH_WITH = {
      # from params[] => to external web service
        'first_name'  => 'name.first',
        'last_name'   => 'name.last',
        'birth_date'  => 'date.birth',
        'ssn'         => 'ssn'
    }


    class << self

      #######################################################
      ##                                                   ##
      ## PatientSearch is used to search for veterans      ##
      ## using at least two of the following fields:       ##
      ##                                                   ##
      ## name.first                                        ##
      ## name.last ... always required                     ##
      ## ssn                                               ##
      ## date.birth                                        ##
      ##                                                   ##
      #######################################################

      # Method triggered by a Search submit from the user
      def global(params)

        # remove unnecessary parameters; convert keys to API parameter names;
        # checks for argument errors
        vet_params  = clean_params(params)

        cookie      = VX_API.authentication_request
        results     = search_request(vet_params, cookie)
        response    = JSON.parse results.body

        # NOTE: will not return if there are access errors
        check_for_access_errors(response['status'], response['msg'])

        veterans = []

        response['data']['items'].each do |item|
          veterans << PatientSearch::Veteran.new(item)
        end

        return veterans
      end # def global(params)
      alias :search :global
      alias :where  :global


      private

        # Raise ArgumentError when the parameters are not perfect
        def check_for_argument_errors(parameters)
          unless parameters.include?('name.last') # parameter name as required by API
            raise ArgumentError, "last_name is a required parameter" # parm name used in this app
          end

          unless 2 >= parameters.size
            raise ArgumentError, "At least two fields required for search"
          end
        end


        # raise AccessError when there is an access error
        def check_for_access_errors(status, message)
          unless 200 == status
            raise AccessError, message
          else
            unless message.blank?  ||  message.downcase.start_with?('no result')
              raise AccessError, message
            end
          end
        end


        # Send search request to VistA.
        def search_request(params, cookie)
          request = Faraday.new(URL)

          response = request.post do |req|
            req.headers["Content-Type"] = "application/json"
            req.headers["Cookie"]       = cookie
            req.body                    = params.to_json
          end

          return response
        end


        # Remove improper parameters and convert remain to proper web-service API parameter name
        def clean_params(params)
          clean_params = {}
          params.stringify_keys!

          (SEARCH_WITH.keys & params.keys).each do |key|
            value = params[key]
            clean_params[SEARCH_WITH[key]] = value unless value.blank?
          end

          # NOTE: will not return if there are errors
          check_for_argument_errors(clean_params)

          return clean_params
        end

    end # class << self

  end # class PatientSearch

end # module VX_API
