# == Schema Information
#
# Table name: pointless_feedback_messages
#
#  id            :integer          not null, primary key
#  name          :string
#  email_address :string
#  topic         :string
#  description   :text
#  created_at    :datetime
#  updated_at    :datetime
#

# extending the model from the pointless_feedback gem

require 'awesome_print' # NOTE: uses #pretty_inspect to format hash.

# TODO: add user_id and make a relationship to the User model.

module PointlessFeedback
  class Message < ActiveRecord::Base
    attr_accessor :contact_info
    attr_accessor :user
    attr_accessor :request_data

    validates :name, :email_address, :topic, :description, :presence => true
    validates :topic, :inclusion => PointlessFeedback.message_topics
    validates :email_address, :format => /\A([^@\s]+)@((?:[-a-z0-9]+\.)+[a-z]{2,})\z/i
    validate :is_not_spam

    if ENABLE_FEEDBACK_HIPCHAT  ||  ENABLE_FEEDBACK_SLACK
      after_save :send_feedback_to_im, :unless => :honeypot_filled_in?
    end

    if ENABLE_FEEDBACK_EMAIL
      after_save :send_feedback_to_email, :unless => :honeypot_filled_in?
    end

    if PointlessFeedback.table_name.present?
      self.table_name = PointlessFeedback.table_name
    end

    private

    def add_signed_in_user_to_description
      description = <<~EOS
        Signed in as User.id(#{self.user.id}) :
        #{self.user.name} (<a href=mailto:#{self.user.email}>#{self.user.email}</a>)

        Who has the following roles defined: #{self.user.roles}

        Request Data:
        #{self.request_data.pretty_inspect}
      EOS
      return description
    end


    def format_feedback_message
      message = <<~EOS
        #{created_at}  Topic: #{topic}
        #{name}  (<a href=mailto:#{email_address}>#{email_address}</a>)
        said:
        #{description}

        #{add_signed_in_user_to_description}
      EOS
      return message
    end


    # sends feedback message to either Hipchat, Slack or Both
    def send_feedback_to_im
      options = {
        fromuser: topic,
        notify:   true,
        # TODO: Need better way to assign color to feedback topic
        color:    ( topic.include?('Comment') || topic.include?('Enhancement') ) ? 'green' : 'red',
        message:  format_feedback_message
      }

      ImNotificationWorker.perform_async(options)

    end # def send_feedback_to_im


    def send_feedback_to_email
      FeedbackEmailJob.set(wait: 2.seconds).perform_later(self, self.user, self.request_data)
    end

    # SMELL: not used; part of legacy pointless_feedback gem; can be deleted
    def export_feedback
      return unless PointlessFeedback.email_feedback

      # Support Rails < 4.2 and >= 4.2 delivery options
      mailer = FeedbackMailer.feedback(self)
      mailer.respond_to?(:deliver_now) ? mailer.deliver_now : mailer.deliver
    end

    def honeypot_filled_in?
      contact_info.present?
    end

    def is_not_spam
      if URI.extract(description || "").count > 20
        errors.add(:description, "can not contain more than 20 http links (often a sign of spam)")
      end
    end

  end # class Message < ActiveRecord::Base
end # module PointlessFeedback
