# == Schema Information
#
# Table name: general_questions
#
#  id                    :integer          not null, primary key
#  name                  :string
#  label                 :string
#  diagnosis_modifier_id :integer
#  minor_system_id       :integer
#  created_at            :datetime         not null
#  updated_at            :datetime         not null
#  position              :integer
#

class GeneralQuestion < ActiveRecord::Base
  include ContentManagerHelper
  has_many :qm_assignments, dependent: :delete_all
  has_many :question_modifiers, :through => :qm_assignments
  belongs_to :minor_system
  validates_presence_of :name
  validates_presence_of :label
  validates_uniqueness_of :name

  before_save :set_name
  before_update :set_name

  scope :order_by_position, -> () { includes(qm_assignments: :question_modifier).order("position ASC" )}

  FIELDS_FOR_RESULTSET = {
    0 => 'general_questions.name',
    1 => 'general_questions.label'
  }

  class << self
    def filter(params={})
      query = GeneralQuestion.all
      unless params[:search][:value].blank?
        query = filter_using_general_question(params, query)
      end
      return query
    end

    def filter_using_general_question(cp, q)
      q.where('name ilike :search OR label ilike :search', :search => "%#{cp[:search][:value]}%")
    end
  end

  def build_html(the_question_style_to_use='yes_no')   # the other value is 'checkbox'
    case the_question_style_to_use
      when 'checkbox'
        @bhtml = QUESTION_HTML.clone
        @bhtml.gsub!('[Question Name]', self.name)
        @bhtml.gsub!('[Question Label]', self.label)
      when 'yes_no'
        @bhtml = the_html_for_a_yes_no_style_question
      else
        raise 'An invalid question style was provided'
    end

    # qm_html will concat the html for qm_assignments that are to display on a Yes (general_question) selection by the user.
    # nqm_html will concat the html for qm_assignments that are to display on a No (general_question) selection by the user.
    qm_html = ""
    nqm_html = ""
    self.qm_assignments.order_by_position.each do |qma|
      # A true negative_show value means the am_assignment should only be displayed when the user selects No for the
      # general question.
      if qma.negative_show == false
        qm_html += qma.build_html(self.name)
      else
        nqm_html += qma.build_html(self.name)
      end
    end

    @bhtml.gsub!('[HTML for associated question modifiers]', qm_html.html_safe)
    @bhtml.gsub!('[HTML for associated negative question modifiers]', nqm_html.html_safe)
    return @bhtml
  end

  def base_html
    return QUESTION_HTML
  end

#  def get_qm_assignments(question_modifier)
#    if question_modifier.present?
#      return self.qm_assignments.find_all_by_question_modifier_id(question_modifier.id)
#    else
#      return false
#    end
#  end

  def get_qm_assignments(question_modifier)
    if question_modifier.present?
      return self.qm_assignments.where(question_modifier_id: question_modifier.id)
    else
      return false
    end
  end  

  def update_qma_names
    qm_assignments.includes(:question_modifier).each { |qma| qma.set_name; qma.save }
  end

private

  QUESTION_HTML =
    ' <div id="individual_question_container-[Question Name]" class="container question individual">
        <input id="question-[Question Name]" name="question-[Question Name]" type="checkbox" value = "[Question Name]" visibility_dependents="question_modifiers-[Question Name]" onchange="Set_Dependent_Visibility_Based_On(this);">
        <label  for="question-[Question Name]">
  	      [Question Label]
        </label>
      </div>
      <div id="question_modifiers-[Question Name]" class="container lab-results inner small-12 columns hidden">
 	      [HTML for associated question modifiers]
      </div>'

  def the_html_for_a_yes_no_style_question
    return  '<div id="individual_question_container-' + self.name + '" class="container question individual">'                                +
              '<input   id="question-' + self.name + '-yes"'                                                                                   +
                      ' type="radio"'                                                                                                           +
                      ' name="question-' + self.name + '-yes_no"'                                                                              +
                      ' visibility_dependents="question_modifiers-' + self.name + ',negative_question_modifiers-' + self.name + '"'                                                          +
                      ' onchange="Set_Dependent_Visibility_Based_On(this);"'                                                                    +
                      ' value="' + self.name + '">Yes '                                                                                         +
              '<input   id="question-' + self.name + '-no"'                                                                                    +
                      ' type="radio"'                                                                                                           +
                      ' name="question-' + self.name + '-yes_no"'                                                                              +
                      ' visibility_dependents="question_modifiers-' + self.name + ',negative_question_modifiers-' + self.name + '"'                                                          +
                      ' onchange="Set_Dependent_Visibility_Based_On(this);"'                                                                    +
#                      ' value="' + self.name + '">No '                                                                                          +
                      ' value="">No '                                                                                                           +
               ' - ' + self.label                                                                                                               +
              '<div id="question_modifiers-' + self.name + '" class="container lab-results inner small-12 columns hidden">'     +
 	              '[HTML for associated question modifiers]'                                                                                     +
              '</div>'                                                                                                                          +
              '<div id="negative_question_modifiers-' + self.name + '" class="container lab-results inner small-12 columns hidden">'     +
                '[HTML for associated negative question modifiers]'                                                                                     +
              '</div>'                                                                                                                          +
            '</div>'
  end


  def set_name
    self.name = clean_name(self.name)
  end

end
