# == Schema Information
#
# Table name: evaluations
#
#  id                            :integer          not null, primary key
#  evaluation_spec_id            :integer
#  doc                           :json
#  created_at                    :datetime
#  updated_at                    :datetime
#  claim_id                      :integer
#  completed_at                  :datetime
#  primary_evaluation_id         :integer
#  guid                          :string
#  reviewed_at                   :datetime
#  vha_user_vista_access_code    :string
#  vha_user_vista_verify_code    :string
#  vha_user_electronic_signature :string
#  user_id                       :integer
#  assigner_id                   :integer
#  examination_id                :integer
#

require 'securerandom'

class Evaluation < ActiveRecord::Base
  belongs_to :evaluation_spec
  belongs_to :claim
  belongs_to :primary_evaluation, class_name: "Evaluation"
  belongs_to :user
  belongs_to :assigner, class_name: 'User'
  has_and_belongs_to_many :contentions
  has_many :evaluation_logs, -> { order('created_at DESC') }

  belongs_to :examination
  
  before_validation :create_guid

  validates_presence_of :evaluation_spec
#cs 2016-09-13  validates_presence_of :user, message: "Assignee examiner can't be blank" #this is the assigned clinician

  after_commit :log_creation, on: :create

  def complete!(current_user)
    self.check_dependencies
    self.update_attributes(completed_at: Time.now)
    EvaluationLog.create(evaluation_id: self.id,
                               message: "Evaluation submitted for review by #{current_user.name}")
  end

  def uncomplete!(current_user)
    self.update_attributes(completed_at: nil)
    EvaluationLog.create(evaluation_id: self.id,
                               message: "Contention rejected by #{current_user.name}, evaluation uncompleted")
  end

  def complete?
    completed_at.present?
  end

  def dependent?
    return !self.evaluation_spec.dependent.nil?
  end

  def dependent
    return self.evaluation_spec.dependent
  end

  def body_system
    return self.evaluation_spec.body_system
  end

  def check_dependencies
    if !dependencies_satisfied?
      raise DependencyError.new("This evaluation requires a parent evaluation with body_system \"#{self.dependent}\" but none was found", self.form)
    end
    if !signed?
      raise CredentialError.new("You must enter VistA credentials before completing this evaluation")
    end
  end

  def form
    self.doc.nil? ? FML::Form.new(self.evaluation_spec.spec) : FML::Form.from_json(self.doc.to_json)
  end

  def review!
   self.update_attributes(reviewed_at: Time.now)
  end

  def form_validates?
    return false if self.doc.blank?
    form = FML::Form.from_json(self.doc.to_json)
    begin
      form.validate
      return true
    rescue FML::ValidationErrors => errors
      return false
    end
  end

  def reviewed?
    reviewed_at ? true : false
  end

  def completed?
    self.completed_at ? true : false
  end

  def draft?
    self.created_at != self.updated_at && self.doc && self.completed_at.nil? ? true : false
  end

  def status
    if completed?
      "Completed"
    elsif reviewed?
      "Reviewed"
    else
      "Draft"
    end
  end
  
  def signed?
    vha_user_vista_access_code.present? && vha_user_vista_verify_code.present? && vha_user_electronic_signature.present?
  end

  # Convert the Evaluation to XML
  #
  # render_pdf is a boolean representing whether to render the pdf or not. You
  #            want this to be true except in testing
  def to_xml(render_pdf=true)
    EvaluationPresenter.render(self, render_pdf)
  end

  def to_html
    render_html
  end

  def to_txt
    render_txt
  end

  def to_pdf
    @to_pdf ||= WickedPdf.new.pdf_from_string(self.to_html)
  end

  def temp_pdf_path
    Rails.root.join('tmp', "pdfs", "cui-#{self.guid}.pdf")
  end

  def send_to_backend
    phase_2_body_systems = JSON.load(ENV["PHASE_2_BODY_SYSTEMS"]) || []
    phase_2_enabled = ENV['ENABLE_PHASE_2'] == "true"
    if phase_2_enabled && phase_2_body_systems.include?(self.evaluation_spec.body_system)
      #VlerDASJob.perform_later(self)
      VlerDASJob.perform(self)
    else
      VBMSJob.perform_later(self)
      #VBMSJob.perform(self)
    end
  end

  def primary_evaluations
    evals = []
    parent = self.primary_evaluation
    while parent do
      evals << parent
      parent = parent.primary_evaluation
    end
    evals
  end
  
  def validate_xml
    xsd_path = Rails.root.join("app", "xsd", "ExamResponse", "exchange", "ExamResponse-4.0.xsd")
    xsd = Nokogiri::XML::Schema(File.open(xsd_path))
    doc = Nokogiri::XML(self.to_xml)
    xsd.validate(doc).map { |error| error.message }
  end    


protected

  def create_guid
    if( self.guid.nil? )
      self.guid = SecureRandom.uuid
    end
  end


private

  def dependencies_satisfied?
    return true if !self.dependent?
    return false if !self.primary_evaluation

    parent = self.primary_evaluation
    found = false
    while parent do
      found = true if parent.body_system.downcase == self.dependent.downcase && !parent.completed_at.nil?
      parent = parent.primary_evaluation
    end

    return found
  end

  def render_html
    action_view.render template: "evaluations/show.pdf.haml", layout: "layouts/pdf.html.haml",
      locals: { evaluation: self, claim: self.claim }
  end

  def render_txt
    action_view.render template: "evaluations/show.txt.erb", layout: nil,
      locals: { evaluation: self, claim: self.claim, form: FML::Form.from_json(self.doc.to_json) }
  end

  def action_view
    view = ActionView::Base.new
    view.view_paths = ActionController::Base.view_paths
    view.class_eval do
      include Rails.application.routes.url_helpers
      include ApplicationHelper
      include EvaluationsHelper
    end
    view
  end

#  def create_guid
#    if( self.guid.nil? )
#      self.guid = SecureRandom.uuid
#    end
#  end

  def log_creation
    creator = self.assigner ? self.assigner.name : "exam request"
#cs Creation and assignment are now two different steps::    message = "#{creator} created evaluation #{self.id} and assigned it to #{self.user.email}"
    message = "#{creator} created evaluation #{self.id}"
    EvaluationLog.create(message: message, evaluation_id: self.id)
  end
end

class CredentialError<Exception
end

# DependencyError follows the same interface as an FML::ValidationError
# since it's used in the same way
class DependencyErrorMessage<Exception
  attr :debug_message, :field_name
  def initialize(msg)
    super(msg)
    @debug_message = msg
    @field_name = nil
  end
end

# DependencyErrors follows the same interface as an FML::ValidationErrors
# since it's used in the same way
class DependencyError<Exception
  attr :errors, :form

  def initialize(msg, form)
    super(msg)
    @errors = [DependencyErrorMessage.new(msg)]
    @form = form
  end
end
