# == Schema Information
#
# Table name: diagnoses
#
#  id                    :integer          not null, primary key
#  name                  :string
#  label                 :string
#  code                  :string
#  diagnosis_modifier_id :integer
#  created_at            :datetime         not null
#  updated_at            :datetime         not null
#  minor_system_id       :integer
#  position              :integer
#

class Diagnosis < ActiveRecord::Base
  include ContentManagerHelper
  has_many :dm_assignments, dependent: :delete_all
  has_many :diagnosis_modifiers, :through => :dm_assignments
  has_many :symp_diag_relations, dependent: :delete_all
  has_many :symptoms, :through => :symp_diag_relations
  belongs_to :minor_system
  validates_presence_of :name
  validates_presence_of :label
  validates_presence_of :code
  validates_uniqueness_of :name

  before_save :set_name
  before_update :set_name

  scope :order_by_position, -> () { includes(dm_assignments: :diagnosis_modifier).order("position ASC" )}

  FIELDS_FOR_RESULTSET = {
    0 => 'diagnoses.name',
    1 => 'diagnoses.label',
    2 => 'diagnoses.code'
  }

  class << self
    def filter(params={})

      query = Diagnosis.all

      unless params[:search][:value].blank?
        query = filter_using_diagnosis(params, query)
      end

      return query
    end

    def filter_using_diagnosis(cp, q)
      q.where('name ilike :search OR label ilike :search OR code ilike :search', :search => "%#{cp[:search][:value]}%")
    end
  end

  def build_html(the_diagnosis_style_to_use='yes_no')   # the other value is 'checkbox'
    case the_diagnosis_style_to_use
      when 'checkbox'
        @bhtml = DIAGNOSIS_HTML.clone
        @bhtml.gsub!('[Diagnosis Name]', self.name)
        @bhtml.gsub!('[Diagnosis Code]', self.code)
        @bhtml.gsub!('[Diagnosis Label]', self.label)
      when 'yes_no'
        @bhtml = the_html_for_a_yes_no_style_diagnosis
      else
        raise 'An invalid diagnosis style was provided'
    end

    # dm_html will concat the html for dm_assignments that are to display on a Yes (diagnosis) selection by the user.
    # ndm_html will concat the html for dm_assignments that are to display on a No (diagnosis) selection by the user.
    dm_html = ""
    ndm_html = ""

    self.dm_assignments.order_by_position.each do |dma|
      # A true negative_show value means the dm_assignment should only be displayed when
      # the user selects No for the diagnosis.
      if dma.negative_show == false
        dm_html += dma.build_html(self.code)
      else
        ndm_html += dma.build_html(self.code)
      end
    end
    @bhtml.gsub!('[HTML for associated diagnosis modifiers]', dm_html.html_safe)
    @bhtml.gsub!('[HTML for associated negative diagnosis modifiers]', ndm_html.html_safe)
    return @bhtml
  end


  def base_html
    return DIAGNOSIS_HTML
  end

#  def get_dm_assignments(diagnosis_modifier)
#    if diagnosis_modifier.present?
#      return self.dm_assignments.find_all_by_diagnosis_modifier_id(diagnosis_modifier.id)
#    else
#      return false
#    end
#  end

  def get_dm_assignments(diagnosis_modifier)
    if diagnosis_modifier.present?
      return self.dm_assignments.where(diagnosis_modifier_id: diagnosis_modifier.id)
    else
      return false
    end
  end

  def update_dma_names
    dm_assignments.includes(:diagnosis_modifier).each { |dma| dma.set_name; dma.save }
  end

private

  DIAGNOSIS_HTML =
    ' <div id="individual_diagnosis_container-[Diagnosis Name]" class="container diagnosis individual">
        <input id="diagnosis-[Diagnosis Name]" name="diagnosis-[Diagnosis Name]" type="checkbox" value = "[Diagnosis Code]" visibility_dependents="diagnosis_modifiers-[Diagnosis Name]" onchange="Set_Dependent_Visibility_Based_On(this);">
        <label  for="diagnosis-[Diagnosis Name]">
  	      [Diagnosis Label]
        </label>
      </div>
      <div id="diagnosis_modifiers-[Diagnosis Name]" class="container lab-results inner small-12 columns hidden">
 	      [HTML for associated diagnosis modifiers]
      </div>'

  def the_html_for_a_yes_no_style_diagnosis
    return  '<div id="individual_diagnosis_container-' + self.name + '" class="container diagnosis individual">'                                +
              '<input   id="diagnosis-' + self.name + '-yes"'                                                                                   +
                      ' type="radio"'                                                                                                           +
                      ' name="diagnosis-' + self.name + '-yes_no"'                                                                              +
                      ' visibility_dependents="diagnosis_modifiers-' + self.name + ',negative_diagnosis_modifiers-' + self.name + '"'           +
                      ' onchange="Set_Dependent_Visibility_Based_On(this);"'                                                                    +
                      ' value="' + self.code + '">Yes '                                                                                         +
              '<input   id="diagnosis-' + self.name + '-no"'                                                                                    +
                      ' type="radio"'                                                                                                           +
                      ' name="diagnosis-' + self.name + '-yes_no"'                                                                              +
                      ' visibility_dependents="diagnosis_modifiers-' + self.name + ',negative_diagnosis_modifiers-' + self.name + '"'           +
                      ' onchange="Set_Dependent_Visibility_Based_On(this);"'                                                                    +
#                      ' value="' + self.code + '">No '                                                                                          +
                      ' value="">No '                                                                                                           +
               ' - ' + self.label                                                                                                               +
              '<div id="diagnosis_modifiers-' + self.name + '" class="container lab-results inner small-12 columns hidden">'                    +
 	              '[HTML for associated diagnosis modifiers]'                                                                                     +
              '</div>'                                                                                                                          +
              '<div id="negative_diagnosis_modifiers-' + self.name + '" class="container lab-results inner small-12 columns hidden">'           +
                '[HTML for associated negative diagnosis modifiers]'                                                                            +
              '</div>'                                                                                                                          +
            '</div>'
  end


  def set_name
    self.name = clean_name(self.name)
  end

end
