# == Schema Information
#
# Table name: contentions
#
#  id                                              :integer          not null, primary key
#  claim_id                                        :integer
#  resolved_at                                     :datetime
#  vba_diagnostic_code                             :string
#  created_at                                      :datetime
#  updated_at                                      :datetime
#  name                                            :text
#  history                                         :text
#  no_diagnosis                                    :boolean
#  claim_folder_reviewed                           :boolean
#  reviewed_military_service_treatment_records     :boolean
#  reviewed_military_service_personnel_records     :boolean
#  reviewed_military_enlistment_examination        :boolean
#  reviewed_military_separation_examination        :boolean
#  reviewed_military_post_deployment_questionnaire :boolean
#  reviewed_dod_form_214_separation_documents      :boolean
#  reviewed_vha_treatment_records                  :boolean
#  reviewed_civilian_medical_records               :boolean
#  reviewed_interviews_with_collateral_witnesses   :boolean
#  reviewed_at                                     :datetime
#  reviewed_by                                     :integer
#  assigner_id                                     :integer
#  reviewed_not_indicated                          :boolean
#  exam_related_contention_id                      :string
#  clarification_requested                         :boolean
#  is_contention_cancelled                         :boolean
#  reason_for_cancellation                         :text
#  cancellation_details                            :text
#  identifier                                      :string
#  insufficient                                    :boolean          default(FALSE)
#  previously_worked_contention_id                 :string
#

class Contention < ActiveRecord::Base
  belongs_to :claim
  belongs_to :assigner, class_name: 'User'
  belongs_to :reviewer, class_name: 'User', foreign_key: :reviewed_by
  has_and_belongs_to_many :evaluations, -> {order "created_at ASC"}
  has_many :rejections, -> {order "created_at ASC"}
  has_and_belongs_to_many :examinations

  has_many :contention_details
  has_many :clarification_details
  has_many :exam_prioritization_special_issues
  has_many :rework_reasons
  has_many :rework_reason_free_texts

  auto_strip_attributes :name

  default_scope { order('created_at ASC') }
  scope :reviewable, -> { where.not(resolved_at: nil).where(reviewed_at: nil) }

  accepts_nested_attributes_for :evaluations

  delegate :patient_name, :patient_ssn, :vbms_claim_id, :benefit_type, to: :claim

  RECORD_FIELD_LABEL_MAP = {
    :reviewed_not_indicated => "Not indicated",
    :reviewed_military_service_treatment_records => "Military service treatment records",
    :reviewed_military_service_personnel_records => "Military service personnel records",
    :reviewed_military_enlistment_examination => "Military enlistment examination",
    :reviewed_military_separation_examination => "Military separation examination",
    :reviewed_military_post_deployment_questionnaire => "Military post-deployment questionnaire",
    :reviewed_dod_form_214_separation_documents => "Department of Defense Form 214 Separation Documents",
    :reviewed_vha_treatment_records => "Veterans Health Administration medical records (VA treatment records)",
    :reviewed_civilian_medical_records => "Civilian medical records",
    :reviewed_interviews_with_collateral_witnesses => "Interviews with collateral witnesses (family and others who have known the veteran before and after military service)"
  }

  def complete?
    self.no_diagnosis || (self.evaluations.count > 0 && self.evaluations.map(&:completed?).all?)
  end

  def resolve!(current_user)
    return false if history.blank? || !has_valid_review?
    self.update_attributes(resolved_at: Time.now)
    self.evaluations.each do |evaluation|
      EvaluationLog.create(evaluation_id: evaluation.id,
                                 message: "Evaluation accepted by #{ current_user.name }")
    end
  end

  def has_valid_review?
    claim_folder_reviewed || one_or_more_reviewed_checkboxes_selected?
  end

  def unresolve!(current_user)
    self.update_attributes(resolved_at: nil)
    self.evaluations.each { |e| e.uncomplete! current_user }
  end

  def resolved?
    resolved_at.nil? ? false : true
  end

  def reviewed?
    reviewed_at.nil? ? false : true
  end

  # return nil or the most recent rejection
  def rejection
    self.rejections.nil? ? nil : self.rejections[-1]
  end

  def review!(reviewed_by = nil)
    self.update_attributes(reviewed_at: Time.now, reviewed_by: reviewed_by) unless self.reviewed?
  end

  def send_evaluations
    self.evaluations.each{|evaluation| evaluation.send_to_backend }
  end

  def one_or_more_reviewed_checkboxes_selected?
    RECORD_FIELD_LABEL_MAP.keys.any? { |k| self[k] }
  end

  def exam_date
    updated_at.strftime("%b %d %Y, %I:%M %P")
  end

  def contention_description
    contention_details.first.description
  end

  def previous_contention_exams
    con_details = []
    contention = Contention.find_by_exam_related_contention_id_and_claim_id previously_worked_contention_id, claim.id
    contention.examinations.each do |exam|
      clinician = User.find(exam.clinician.to_i)
      con_details << {title: exam.title, provider: clinician.first_name+" "+clinician.last_name, exam_date: exam.exam_date.strftime("%b %d %Y, %I:%M %P")}
    end
    con_details
  end

  def special_issues
    exam_prioritization_special_issues.pluck(:special_issue)
  end

  def rework_reason_list
    rework_reasons.pluck(:reason)
  end

  def rework_reason_free_text_list
    rework_reason_free_texts.pluck(:reason)
  end
end
