# == Schema Information
#
# Table name: consultations
#
#  id                     :integer          not null, primary key
#  consultation_number    :string           not null
#  ordering_provider_id   :integer
#  valid_from             :datetime
#  valid_to               :datetime
#  content                :json
#  created_at             :datetime         not null
#  updated_at             :datetime         not null
#  care_category_id       :integer
#  consultation_order_id  :integer
#  consultation_status_id :integer
#  consultation_type_id   :integer
#  veteran_id             :integer
#

# The Consultation model is currently backed by a database record.
# However in the objective system it will be some kind of
# ActiveModel backed by a web-service.

class Consultation < ActiveRecord::Base

  belongs_to :care_category
  belongs_to :consultation_order
  belongs_to :consultation_status
  belongs_to :consultation_type
  belongs_to :ordering_provider, class_name: 'Provider'
  belongs_to :veteran

  has_many :referrals
  has_many :facilities, through: :referrals   # SMELL: there is also a provider_id in the referral model
  has_many :visns,      through: :referrals

  autowire_content_fields   :urgent_priority,
                            :ordering_physician_name,
                            :requesting_provider_telephone,
                            :requesting_provider_fax,
                            :requesting_provider_contact_name

  validates :care_category_id, presence: true
  validates :consultation_order, presence: true
  validates :veteran_id, presence: true
  validates :consultation_number, presence: true, uniqueness: true
  validate do |consultation|
    if !(consultation.urgent_priority == true || consultation.urgent_priority == false)
      consultation.errors.add :base, 'Request Priority cannot be blank'
    end
  end
  validates :valid_to, presence: true
  validates :valid_from, presence: true
  validates :ordering_physician_name, presence:true

  # SMELL: What kind of category is this?

  def has_referral_category?(referral_category)
    referral_categories.include? referral_category
  end


  ################################################################
  ## Decorators

  def human_readable_request_priority
    urgent_priority ? 'Urgent' : 'Routine'
  end


  ################################################################
  ## Access Helpers

  ###################
  ## Veteran model ##
  ###################


  def first_name
    veteran.first_name
  end


  def last_name
    veteran.last_name
  end


  def ssn
    veteran.ssn
  end


  ########################
  ## CareCategory model ##
  ########################


  def care_category_title
    care_category.title
  end


  ####################
  ## Referral model ##
  ####################


  # SMELL: why would we every want a list of the
  #        authorization numbers from the referrals?
  #        This may not be used.

  def coordinator_id
    referrals.pluck(:coordinator_id)
  end

  ##############################
  ## ConsultationStatus model ##
  ##############################


  def status_name
    consultation_status.title
  end


  #############################
  ## ConsultationOrder model ##
  #############################


  def order_name
    consultation_order.title
  end


  ####################
  ## Provider model ##
  ####################


  # SMELL: a provider could be in multiple facilities in the objective system.

  def facility_name
    ordering_provider.facility.name
  end


  # Arel mappings for resultset field names, based on SQL (not ActiveRecord)
  # table and column names.  Array fields are added as multiple ORDER BY parameters.
  FIELDS_FOR_RESULTSET = {
    0 => ['veterans.first_name', 'veterans.middle_name', 'veterans.last_name'],
    1 => 'veterans.ssn',
    2 => 'care_categories.title',
    3 => 'consultation_statuses.title',
    5 => 'consultations.created_at'
  }

  ########################################################################
  ###
  ##    Eigen Class
  #

  class << self

    ########################################################################
    ## Filter / Search Support

    VALID_FILTER_PARAM_KEYS = %w[

      care_category_id
      consultation_number
      status
      created_at
      from_date
      to_date

      first_name
      last_name
      ssn

      ordering_provider_id
      visn_id
      facility_id

      authorization_number
      coordinator_id
    ]

    def clean_the_params(params)
      return params if params.blank?

      clean_params = params.select{|k,v| VALID_FILTER_PARAM_KEYS.include? k}
                           .delete_if{|k,v| v.nil? || v.empty?}
                           .symbolize_keys

      # TODO: make sure to validate user input of either a from_date or both but not just a to_date
      clean_params[:created_at] = DateHelpers.create_range(
                  clean_params[:from_date],
                  clean_params.has_key?(:to_date) ? clean_params[:to_date] : ''
            )  if clean_params.has_key?(:from_date)

      # NOTE: allows user to enter ssn like: xxx-xx-xxxx
      clean_params[:ssn].gsub!(/\D/,'') if clean_params.has_key? :ssn

      return clean_params
    end # def clean_the_params(params)


    ########################################################################
    # Returns an instance of the class ActiveRecord_Relation (AREL)

    def filter(params={})
      filter_params = clean_the_params(params)

      # Join on all associated tables needed for ORDER BY clause (sorting by row)
      query = all.joins(:veteran).joins(:care_category).joins(:consultation_status)

      unless filter_params.blank?
        query = filter_using_self(filter_params,      query)
        query = filter_using_veteran(filter_params,   query)
        query = filter_using_provider(filter_params,  query)
        query = filter_using_referral(filter_params,  query)
      end

      # select_fields = FIELDS_FOR_RESULTSET.map do |field_obj|
      #   "#{field_obj[:datafield]} as #{field_obj[:alias]}"
      # end
      # query.select(select_fields)

      return query
    end


    ########################################################################
    # Add query components that deal with attributes that are an inherent
    # part of the main record and any static text sub-records.

    def filter_using_self(fp, q = all)
      return q if ([:care_category_id, :consultation_number, :status, :created_at] & fp.keys).empty?

      if fp.has_key? :care_category_id
        q = q.where(care_category_id: fp[:care_category_id])
      end

      if fp.has_key? :consultation_number
        q = q.where("consultations.consultation_number ilike ?", "#{fp[:consultation_number]}%")
      end

      if fp.has_key? :status
        q = q.where('consultation_status_id in (?)', fp[:status])
      end

      if fp.has_key? :created_at
        q = q.where(created_at: fp[:created_at])
      end

      return q
    end # def filter_using_self(fp, query = all)


    ########################################################################
    # Add query components that deal with attributes that are in the Veteran model.
    # TODO: consider a way to cross-use filter methods from different models; for example,
    #       instead of having this method in the Consultation model, make use of
    #       Veteran.filter(some_param_set, query)

    def filter_using_veteran(fp, q)
      return q if ([:first_name, :last_name, :ssn] & fp.keys).empty?

      # NOTE: supports case insensitive starts_with?
      if fp.has_key? :first_name
        q = q.where("veterans.first_name ilike ?", "#{fp[:first_name]}%")
      end

      # NOTE: supports case insensitive starts_with?
      if fp.has_key? :last_name
        q = q.where("veterans.last_name ilike ?", "#{fp[:last_name]}%")
      end

      # NOTE: supports searching for last 4 digits
      #       This is an IMPORTANT Personal Privacy issue when
      #       dealing with people over the phone - they do not
      #       want to give out their full SSN because other
      #       people may be in the room eavesdropping.

      if fp.has_key? :ssn
        q = q.where("veterans.ssn like ?", "%#{fp[:ssn]}")
      end

      return q
    end # def filter_using_veteran(fp, q)


    ########################################################################
    # Add query components that deal with attributes that are in the Provider model.
    # FIXME: when the ordering physician requirement gets settled.
    # SMELL: The facility/visn components need to be clarified as to whether they are
    #        based upon the ordering physician or the referral provider.

    def filter_using_provider(fp, q)
      return q if ([:ordering_provider_id, :facility_id, :visn_id] & fp.keys).empty?

      q = q.includes(referrals: [:facility, :visn])

      # NOTE: this next query is for the provider_id of the
      #       consultation - the ordering physician

      # FIXME: when the ordering physician/provider
      #        requirement gets settled.

      if fp.has_key? :ordering_provider_id
        q = q.where("ordering_provider_id = ?", fp[:ordering_provider_id])
      end

      # NOTE: The geolocation queries are based upon
      #       the referral's provider not the ordering
      #       physician.

      # FIXME: when the ordering physician requirement
      #        gets settled.

      # NOTE: no reason to search the facility since we
      #       already have the provider.

      unless  fp.has_key?(:ordering_provider_id)

        if fp.has_key? :facility_id
          q = q.where('facilities.id = ?', fp[:facility_id])
               .references(:facilities)
        elsif fp.has_key? :visn_id
          # NOTE: no reason to search for the region since we
          #       already have the facility.
          q = q.where('visns.id = ?', fp[:visn_id]).references(:visns)
        end

      end # unless  fp.has_key?(:provider_id)

      return q

    end # def filter_using_provider(fp, q)


    ########################################################################
    # Add query components that deal with attributes that are in the Referral model.

    def filter_using_referral(fp, q)
      return q if ([:authorization_number,:coordinator_id] & fp.keys).empty?

      q = q.includes(:referrals).references(:referrals)

      if fp.has_key? :authorization_number
        # The coordinator_id can be ignored here as authorization_number is a unique field
        q = q.where('referrals.authorization_number = ?',
                    fp[:authorization_number])
      end

      if !fp.has_key? (:authorization_number) and fp.has_key? (:coordinator_id)
        unique_ids = q.where('referrals.coordinator_id = ?',
                  fp[:coordinator_id]).pluck(:id).uniq
        q = q.where(id: unique_ids)
      end

      return q
    end # def filter_using_referral(fp, q)


    ########################################################################
    # Provide data for download based upon the last filter action.
    # NOTE: The options hash configures the CSV.generate method

    def to_csv(a_hash, options = {})

      an_arel = filter(a_hash)

      result = CSV.generate(options) do |csv|
        csv << ["Patient Name", "SSN", "Episode of Care", "Status", "Community Provider", "Date"]

        an_arel.each do |consultation|
          csv <<  [
                    consultation.veteran.patient_name,
                    consultation.veteran.formatted_ssn,
                    consultation.care_category.title,
                    consultation.status_name,
                    ProviderDataPresenter.new(consultation).name,
                    consultation.created_at.strftime('%F')
                  ]
        end
      end # result = CSV.generate(options) do |csv|

      return result
    end # def to_csv(options = {})

  end # class << self

end # class Consultation < ActiveRecord::Base
