# == Schema Information
#
# Table name: alerts
#
#  id                :integer          not null, primary key
#  description       :text
#  date_completed    :datetime
#  created_at        :datetime         not null
#  updated_at        :datetime         not null
#  notification_type :string
#  site              :integer
#  active            :boolean          default(TRUE)
#  requested_by      :integer          not null
#  completed_by      :integer
#

class Alert < ActiveRecord::Base

  REQUEST_CUI_MSG  = " has requested access to CUI".freeze
  REQUEST_CUI_TYPE = "CUI Access Request".freeze

  REQUEST_SITE_MSG = " has requested access to one or more sites".freeze
  REQUEST_SITE_TYPE = "Site Access Request".freeze

  scope :active, -> () { Alert.where(active: true).limit(10) }

  def self.complete_access_request(customer, fulfiller)
    Alert.where(
                 requested_by: customer,
                 notification_type: REQUEST_CUI_TYPE,
                 active:true
    )
    .update_all({
                  completed_by: fulfiller,
                  active: false,
                  date_completed:Time.now
    })
  end

  def self.complete_site_request(customer, fulfiller, site_requested)
    Alert.where(
                  requested_by: customer,
                  notification_type: REQUEST_SITE_TYPE,
                  active:true,
                  site: site_requested
    )
    .update_all({
                  completed_by: fulfiller,
                  active: false,
                  date_completed:Time.now
    })
  end

  def self.create_access_request(customer)
    Alert.create(
        :requested_by      => customer.id,
        :description       => customer.email + Alert::REQUEST_CUI_MSG,
        :notification_type => Alert::REQUEST_CUI_TYPE
    )
  end

  def self.create_site_request(customer, requested_site)
    Alert.create(
        :requested_by      => customer.id,
        :description       => customer.email + Alert::REQUEST_SITE_MSG,
        :notification_type => Alert::REQUEST_SITE_TYPE,
        :site              => requested_site
    )
  end

  def self.get_destination(alert)
    case alert.notification_type
      when REQUEST_CUI_TYPE  then "/user/#{alert.requested_by}/access_request"
      when REQUEST_SITE_TYPE then "/site/#{alert.site}/user/#{alert.requested_by}/site_request"
    end
  end

  def deactivate
    self.update_attributes(active: false)
  end


end
