# The Referrals controller provides user access to
# the Referral model.
class ReferralsController < ApplicationController
  include DataTablePaginator

  before_action :require_access, only: [:show]
  before_action :set_referral, only: [:show, :return_va, :edit, :update, :destroy, :update_status, :submit_for_va_review]
  after_action  :destroy_session, only: [:export_referrals]

  # The before_action methods below are nested; methods should only call one action below.
  before_action :set_referral_prefs, only: :index
  before_action :set_filtered_referrals, only: :paged_list
  before_action :set_filtered_referrals_and_user_prefs, only: :search

  # hook up authorization and user permissions
  check_authorization
  authorize_resource

  DEFAULT_REFERRAL_TAB = 'overview'


  # GET /referrals
  # GET /referrals.json
  def index
    set_global_counts
    # setting pre-loaded dropdown options before index page is rendered
    if @referral_filter['facility_id'].blank?
      @facilities = Facility.none
      @providers = Provider.none
    else
      @facilities = Visn.find(@referral_filter["visn_id"]).facilities
      @providers = Facility.find(@referral_filter["facility_id"]).providers
    end
  end


  def filter_by_quicklink
    selected_status_id = ReferralStatus.find_by_name(params[:filter]).id.to_s
    save_to_user_preference({status: selected_status_id})
    render json: {success: @saved}
  end


  def reset_referral_filter
    save_to_user_preference({})
    render json: {success: @saved}
  end


  def search
    render json: {success: @saved}
  end


  def export_referrals
    respond_to do |format|
      format.csv { send_data Referral.to_csv(current_user.user_preference.referral_filter, current_user) }
    end
  end

  # Action for Non-VHA users to submit records for VHA review.
  def submit_for_va_review
    unless @referral.referral_status.non_vha_add_document_phase?
      raise Exception, "Invalid Request: Referral cannot be submitted to VA."
    end
    @referral.referral_status = ReferralStatus.va_review_phase

    #Notification Email to the VA Coordinator
    ReferralNotificationEmailJob.set(wait: 2.seconds).perform_later(@referral, current_user, params)

    @referral.status_message = "Submitted for VA review by #{current_user.name}."
    redirected_tabpage = current_user.is_vha_cc? ? 'status' : 'overview'
    respond_to do |format|
      if @referral.save
        format.html do
          flash[:notice] = 'Referral has been submitted for VA review.'
          redirect_to referral_path(@referral, tabpage: 'documents')
        end
        format.json { head :no_content }
      else
        format.html do
          flash[:alert] = @referral.errors.full_messages.join(" ")
          redirect_to referral_path(@referral, tabpage: 'documents')
        end
        format.json { render json: @referral.errors, status: :unprocessable_entity }
      end
    end
  end

  # GET /referrals/1
  # GET /referrals/1.json
  def show
    @referral_note        = @referral.referral_notes.new
    @referral_appointment = @referral.referral_appointments.new
    @referral_document    = @referral.referral_documents.new
  end


  # GET /referrals/new
  def new
    @referral = Referral.new
  end


  # GET /referrals/1/edit
  def edit
  end


  # POST /referrals
  # POST /referrals.json
  def create
    @referral = Referral.new(referral_params)

    respond_to do |format|
      if @referral.save
        format.html { redirect_to @referral, notice: 'Referral was successfully created.' }
        format.json { render action: 'show', status: :created, location: @referral }
      else
        format.html { render action: 'new' }
        format.json { render json: @referral.errors, status: :unprocessable_entity }
      end
    end
  end


  # PATCH/PUT /referrals/1
  # PATCH/PUT /referrals/1.json
  def update
    respond_to do |format|
      if @referral.update(referral_params)
        format.html { redirect_to @referral, notice: 'Referral was successfully updated.' }
        format.json { head :no_content }
      else
        format.html { redirect_to @referral, flash: { error: @referral.errors.full_messages }}
        format.json { render json: @referral.errors, status: :unprocessable_entity }
      end
    end
  end


  # method specific to updating status.  Only users with vha_cc or admin access can perform this action.
  # PATCH /referrals/1/update_status
  # PATCH /referrals/1/update_status.json
  def update_status
    respond_to do |format|
      if @referral.update(referral_params_with_status)
        format.html do
          flash[:notice] = 'Referral status was successfully updated.'
          if @referral.referral_status.name == 'Request Information'

            #Email to Provider when a referral status is set to Request Information
            ProviderNotificationEmailJob.set(wait: 2.seconds).perform_later(@referral)

          end
          if @referral.referral_status.name == 'Scheduled'

            #Email to Provider when a referral status is set to Request Information
            ProviderScheduledNotificationJob.set(wait: 2.seconds).perform_later(@referral)

          end
          redirect_to referral_path(@referral, tabpage: 'status')
        end
        format.json { head :no_content }
      else
        format.html do
          flash[:alert] = @referral.errors.full_messages.join(" ")
          redirect_to referral_path(@referral, tabpage: 'status')
        end
        format.json { render json: @referral.errors, status: :unprocessable_entity }
      end
    end
  end

  def return_va
    #Empty method for return_VA
  end

  # DELETE /referrals/1
  # DELETE /referrals/1.json
  def destroy
    @referral.destroy
    respond_to do |format|
      format.html { redirect_to referrals_url }
      format.json { head :no_content }
    end
  end


  # Returns a paginated list of Referral records, based on DataTables
  # parameters passed in.  Params keys include:
  # :start - the offset when querying users in the database
  # :length - the number of records being queried
  # :draw - DataTables internal counter userd to sort async responses
  def paged_list
    Rails.logger.debug "PARAMS >>> #{params}"

    if current_user.is_vha_cc?
      referral_to_fields_func = -> (referral) do
        provider_data = ProviderDataPresenter.new(referral)
        {
          id:            referral.id,
          patient_name:  referral.veteran.first.full_name,
          referral:      referral.referral_type.title,
          ssn:           referral.veteran.first.formatted_ssn,
          date:          referral.created_at.strftime('%F'),
          status:        referral.referral_status.name,
          coordinator:   referral.coordinator.name,
          provider:      provider_data.name,
          visn_facility: provider_data.facility.try(:visn_region_prepended_name) || 'N/A'
        }
      end
      ordered_referrals = apply_ordering_to_query(
        @filtered_referrals, Referral::FIELDS_FOR_RESULTSET[:vha], params['order']
      )
    else
      referral_to_fields_func = -> (referral) do
        {
          id:                   referral.id,
          patient_name:         referral.veteran.first.full_name,
          referral:             referral.referral_type.title,
          date:                 referral.created_at.strftime('%F'),
          status:               referral.referral_status.name,
          coordinator:          referral.coordinator.name,
          authorization_number: referral.authorization_number,
          consultation_number:  referral.consultation.consultation_number
        }
      end
      ordered_referrals = apply_ordering_to_query(
        @filtered_referrals, Referral::FIELDS_FOR_RESULTSET[:non_vha], params['order']
      )
    end
    render json: datatables_json_for_query_using_func(
      ordered_referrals, referral_to_fields_func
    )
  end


  #########################################################################
  ## Don't Peek at my Privates!!

private

  # Basic before_action: only set referral filter preferences.
  def set_referral_prefs
    @referral_filter = current_user.user_preference.referral_filter
  end

  # Set global counts for all visible referrals
  def set_global_counts
    @basic_filtered_referrals = Referral.basic_filter(current_user)
    @referral_status_counts = @basic_filtered_referrals.group_by_status_and_count(current_user)
    @referral_stat_count = @basic_filtered_referrals.stat_count(current_user)
    @referral_med_letter_count = @basic_filtered_referrals.med_letter_count(current_user)
    if current_user.is_vha_cc?
      @referral_expired_count = Referral.expired_count(current_user)
      @referral_with_new_appointments_count = @basic_filtered_referrals.with_new_appointments_count
    end
  end

  # Set referral filter preferences and filtered list.
  def set_filtered_referrals
    set_referral_prefs
    @filtered_referrals = Referral.filter(@referral_filter, current_user)
    logger.debug "@FILTER -> #{@filtered_referrals.count} records found."
  end

  # Restricting the unauthrozied providers to view referrals
  def require_access
    if current_user.is_non_vha?
      provider_ids = current_user.providers.pluck(:id)
      if Referral.exists?(id: params[:id])
        provider_id = Referral.where(id: params[:id]).pluck(:provider_id).first
        return(true) if provider_ids.include?(provider_id)
      end
      # NOTE: falls through to here if the record is not found or if its not authorized for the current user.
      respond_to do |format|
        flash[:alert] = 'Not authorized to view selected referral'
        format.html { redirect_to referrals_url }
        format.json { head :no_content }
      end
    end
  end

  # Save search params to user preferences first, then set referral filter
  # preferences and filtered list.
  def set_filtered_referrals_and_user_prefs
    save_to_user_preference
    set_filtered_referrals
  end

  # Save parameters (passed in or from HTTP request hash) to user preferences.
  # The 'params' hash passed in from the request is used as the default.
  # Pass in an empty hash to reset prefs.
  def save_to_user_preference(referral_params = nil)
    referral_params ||= referral_filter_params.to_h
    referral_params['status'] ||= []
    @saved = current_user.user_preference.save_filter(:referral, referral_params)
    current_user.reload if @saved
  end

  # Use callbacks to share common setup or constraints between actions.
  def set_referral
    @referral = Referral.find id_from_params
    @consultation = @referral.consultation
    @veteran = @consultation.veteran
    @provider = ProviderDataPresenter.new(@referral)
    @ordering_provider = ProviderDataPresenter.new(@consultation)
  end

  REFERRAL_DEFAULT_PARAMS = [
    :consultation_id,
    :referral_type_id, :referral_reason_id, :coordinator_id, :provider_id,
    :referral_number, :authorization_number, :diagnosis_codes, :procedure_codes,
    :length_of_stay, :length_of_procedure, :treatment_frequency,
    :number_of_visits, :cost_estimate, :referral_reason_id
  ]

  def id_from_params
    params.has_key?(:referral_id) ? params[:referral_id] : params[:id]
  end


  # Never trust parameters from the scary internet, only allow the white list through.
  # Does not include referral status.  Use referral_params_with_status to include status.
  def referral_params
    params.require(:referral).permit(*REFERRAL_DEFAULT_PARAMS)
  end

  # Filter out parameters allowed when performing Filter action.
  # Includes both filter form fields and Referral Queue link options.
  def referral_filter_params
    params.permit(:from_date, :to_date, :first_name, :last_name,
                  :ssn, :coordinator_id, :authorization_number,
                  :consultation_number, :care_category_id, :referral_type_id,
                  :visn_id, :facility_id, :provider_id, {status: []},
                  :expired_data, :new_appointments, :medical_record_letters, :stat_data)
  end

  def referral_params_with_status
    params.require(:referral).permit(:referral_status_id, :status_message, *REFERRAL_DEFAULT_PARAMS)
  end


  def destroy_session
    session.delete(:referrals)
  end

end # class ReferralsController < ApplicationController
