class ReferralDocumentsController < ApplicationController
  include ApplicationHelper
  include ReferralDocumentsHelper

  before_action :set_referral, only: [:create]
  before_action :set_referral_and_document, only: [:approve, :reject, :delete, :update]
  check_authorization
  authorize_resource

  FILE_UPLOAD_DIR = 'fake_docstore'
  FILE_UPLOAD_PATH = File.join(Rails.root, "public/#{FILE_UPLOAD_DIR}")

  def create
    if (referral_document_params[:document_name].nil? or referral_document_params[:document_name].empty?)
      flash[:alert] = "No Document attached"
      redirect_to referral_path(@referral, tabpage: 'documents')
    else
      move_file_to_uploads_dir
      @referral_document = @referral.referral_documents.new(referral_document_params)
      @referral_document.location = "/#{FILE_UPLOAD_DIR}"
      @referral_document.uploader_id = current_user.id
      if current_user.is_vha_cc?
        @referral_document.approved_at = Time.now
      end
      respond_to do |format|
        if @referral.save
          format.html do
            flash[:notice] = 'Document was successfully created.'
            redirect_to referral_path(@referral, tabpage: 'documents')
          end
          format.json { render json: @referral, status: :created }
        else
          format.html do
            flash[:alert] = @referral.errors.full_messages.join(" ")
            redirect_to referral_path(@referral, tabpage: 'documents')
          end
          format.json { render json: @referral.errors, status: :unprocessable_entity }
        end
      end
    end
  end

  def approve
    @referral_document.approved_at = Time.now
    @referral_document.approver_id = current_user.id
    respond_to do |format|
      if @referral_document.save
        format.html do
          notice_text = "Document '#{@referral_document.document_name}' was successfully approved."
          redirect_to @referral, notice: notice_text
        end
        format.json do
          result_html = reviewed_by_field(@referral_document, current_user)
          render json: {reviewed_by: result_html}, status: :ok
        end
      else
        format.html do
          flash[:alert] = @referral_document.errors.full_messages.join(" ")
          redirect_to @referral
        end
        format.json { render json: @referral_document.errors, status: :unprocessable_entity }
      end
    end
  end

  def reject
    destroy(:rejected)
  end

  def delete
    if (@referral_document.uploader_id == current_user.id)
      destroy(:deleted)
    end
  end

  def update
    request_body = JSON.parse request.body.read
    respond_to do |format|
      format.json do
        if @referral_document.update_attributes(request_body)
          render json: @referral_document, status: :no_content
        else
          render json: @referral_document.errors, status: :unprocessable_entity
        end
      end
    end
  end

private

  def destroy(action_verb)
    @referral_document.destroy
    respond_to do |format|
      format.html do
        flash[:notice] = "Document was successfully #{action_verb}."
        redirect_to referral_path(@referral, tabpage: 'documents')
      end
      format.json { head :no_content }
    end
  end

  # Use callbacks to share common setup or constraints between actions.
  def set_referral
    @referral = Referral.find(params[:referral_id])
  end

  def set_referral_and_document
    set_referral
    if params.has_key?(:id)
      @referral_document = ReferralDocument.find(params[:id])
    elsif params.has_key?(:referral_document_id)
      @referral_document = ReferralDocument.find(params[:referral_document_id])
    else
      raise Exception, "Referral Document ID not found."
      return false
    end
    # check if referral document is contained in referral
    unless @referral.referral_documents.include? @referral_document
      raise Exception, "Invalid Referral Document ##{@referral_document.id} for referral ##{@referral.id}."
      return false
    end
  end

  # Never trust parameters from the scary internet, only allow the white list through.
  def referral_document_params
    params.require(:referral_document).permit(:document_name, :referral_document_type_id)
  end

  def referral_doc_file_from_params
    doc = params[:referral_document][:document]
    params[:referral_document].extract! :document
    doc
  end

  def move_file_to_uploads_dir
    @uploaded_file = referral_doc_file_from_params
    file_name = @uploaded_file.original_filename
    FileUtils.mv @uploaded_file.tempfile, File.join(FILE_UPLOAD_PATH, file_name)
  end

end
