﻿using NVCC.Models;
using NVCC.WebUI.Utilities;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;

namespace NVCC.WebUI.Infrastructure
{
    public class PatientProfileService : IPatientProfileService
    {
        private readonly IPatientRepository _patientRepository;
        private readonly IViaRepository _viaRepository;
        private readonly IUserSessionManager _userSessionManager;

        public PatientProfileService(IPatientRepository patientRepository, 
                                     IViaRepository viaRepository, 
                                     IUserSessionManager userSessionManager)
        {
            _patientRepository = patientRepository;
            _viaRepository = viaRepository;
            _userSessionManager = userSessionManager;
        }

        public PatientProfile GetPatientProfile(Patient patient, ViaUser viaUser)
        {          
            if (patient == null)
                throw new NullReferenceException("patient");
            if (viaUser == null)
                throw new NullReferenceException("queryBean");

            var vaUser = _userSessionManager.UserProfile;
            var patientProfile = new PatientProfile {VistaUser = viaUser};
            if (vaUser == null)
            return patientProfile;

            var viaProfile = GetViaData(viaUser.QueryBean, patient.PatientIen);
            if (viaProfile.VistaUser.Fault)
            {
                viaProfile.VistaUser.QueryBean = viaUser.QueryBean;
                return viaProfile;
            }
                
            patientProfile.RadiologyReports = viaProfile.RadiologyReports;
            patientProfile.Authorizations = viaProfile.Authorizations;
            patientProfile.Notes = viaProfile.Notes;
            foreach (var note in viaProfile.Notes.ToList())
            {

                if (note.NoteTitle == "CONSENT FOR LONG-TERM OPIOIDS FOR PAIN")
                {
                    patientProfile.OpioidNote.NoteTitle = note.NoteTitle;
                    patientProfile.OpioidExist = true;
                    patientProfile.OpiodNoteText = note.NoteText;
                    patientProfile.OpioidNote.NoteAuthor = note.NoteAuthor;
                    patientProfile.OpioidNote.NoteTimestamp = note.NoteTimestamp;
                    patientProfile.OpioidNote.NoteID = note.NoteID;
                    
                    //remove the opiod note from progress Notes section because it's been showed under 
                    //medication list. 
                    patientProfile.Notes.Remove(note);
                }

            }
            patientProfile.Allergies = _patientRepository.GetAllergies(patient.PatientSid);
            patientProfile.Appointments = _patientRepository.GetAppointments(patient.PatientSid);
            patientProfile.Labs = _patientRepository.GetLabs(patient.PatientSid);
            patientProfile.Medications = _patientRepository.GetMedications(patient.PatientSid);
            patientProfile.ProblemDiagnoses = _patientRepository.GetProblemList(patient.PatientSid);
            patientProfile.NonVaMedications = _patientRepository.GetNonVaMedications(patient.PatientSid);
            patientProfile.Consults = _patientRepository.GetConsults(patient.PatientSid);
            patientProfile.NextOfKin = _patientRepository.GetNextOfKin(patient.PatientSid);
            return patientProfile;
            }

        private PatientProfile GetViaData(ViaQueryBean myQueryBean, string patientIen)
        {
            var patientProfile = new PatientProfile();
            var date1 = DateTime.Now.Add(TimeSpan.FromDays(-180));
            var date2 = DateTime.Now;
            date2 = date2.AddDays(1);

            var patient = _viaRepository.GetPatient(myQueryBean, patientIen);
            if (patient.Fault)
            {
                patientProfile.VistaUser.FaultMessage = patient.FaultMessage;
                patientProfile.VistaUser.Fault = true;
                return patientProfile;
            }

            myQueryBean.ViaPatient.LocalPid = patient.LocalPid;
            myQueryBean.ViaPatient.MpiPid = patient.MpiPid;
            myQueryBean.ViaPatient.LocalSiteId = patient.LocalSiteId;
            IList<Note> notes = _viaRepository.GetProgressNotes(date1, date2, 50, myQueryBean);
            IList<Authorization> authorizations = notes
                .Where(n =>
                    ((n.NoteTitle != null) &&
                        (Authorization.IdentifyingNoteStrings.Any(s => n.NoteTitle.Contains(s)))) ||
                    ((n.NoteText != null) &&
                        (Authorization.IdentifyingNoteStrings.Any(s => n.NoteText.Contains(s)))))
                .Select(n => new Authorization
                {
                    AuthorizationId = n.NoteID,
                    AuthorizationTitle = n.NoteTitle,
                    AuthorizationText = n.NoteText
                })
                .ToList();
            var reports = _viaRepository.GetRadiologyReports(date1, date2, 10, myQueryBean);
            patientProfile.Authorizations = authorizations;
            patientProfile.Notes = notes;
            patientProfile.RadiologyReports = reports;

            return patientProfile;
        }

        public Patient GetPatient(string patientSsn, short sta3n)
        {
            if(patientSsn==null)
                throw new NullReferenceException("patientSsn");
            var patient = _patientRepository.GetPatient(patientSsn, sta3n);
            return patient;
        }

        public void LogItem(Stopwatch stopwatch, Patient patient, string currentUser, string action)
        {
            if(patient==null)
                throw new NullReferenceException("patient");
            if (stopwatch == null)
                throw new NullReferenceException("stopwatch");

            var hostName = HttpContextManager.Current.Request.Url != null
                ? HttpContextManager.Current.Request.Url.Host : null;
            var logitem = new LogItem
            {
                HostName = hostName,
                UserID = currentUser,
                AccessDateTime = DateTime.Now,
                Action = action,
                PatientSid = patient.PatientSid,
                ElapsedTime = stopwatch.Elapsed.TotalSeconds,
                DatabaseServerName = "SERVER"
            };
            _patientRepository.WriteLog(logitem);
        }
    }
}