﻿using System;
using System.Collections.Generic;
using System.Linq;
using NVCC.WebUI.Models;
using NVCC.Models;

namespace NVCC.WebUI.Infrastructure
{
    public class AodService : IAodService
    {
        private readonly IAodRepository _aodRepository;



        public AodService(IAodRepository aodRepository)
        {
            _aodRepository = aodRepository;
        }

        public void RecordAodInfo(PatientProfileViewModel patientProfile)
        {
            if (patientProfile.DisclosureInfo == null)
            {
                throw new ApplicationException("There is no Accounting of Disclosure Info in the PatientProfileViewModel object");
            }

            patientProfile.DisclosureInfo.DateTimeOfDisclosure = DateTime.Now;
            patientProfile.DisclosureInfo.PatientIcn = patientProfile.PatientIcn;
            patientProfile.DisclosureInfo.ReferralType = patientProfile.ReferralType;

            //Remove additional objects with null or empty values, if they exist in the list
            HandleAdditionalObjects(patientProfile);

            AddNatures(patientProfile);

            AddAuthorizations(patientProfile);

            AddAppointments(patientProfile);

            //According to current requirements, we are not
            //saving Problems/Diagnoses, even sensitive ones. 
            //I'm leaving the table in the database as well as the properties in the model and the method below in.
            //Just commenting out the calling of the method here:
            //AddProblemDiagnoses(patientProfile);

            AddNotes(patientProfile);

            AddRadiology(patientProfile);

            AddConsult(patientProfile);

            _aodRepository.SaveAodInfo(patientProfile.DisclosureInfo);

        }

        private void HandleAdditionalObjects(PatientProfileViewModel patientProfile)
        {
            if ((patientProfile.DisclosureInfo.AdditionalObjects != null) && (patientProfile.DisclosureInfo.AdditionalObjects.Any(item => item.ObjectDateString == null) || patientProfile.DisclosureInfo.AdditionalObjects.Any(item => item.ObjectType == null)))
            {
                IList<AdditionalObject> objects = new List<AdditionalObject>();
                foreach (AdditionalObject obj in patientProfile.DisclosureInfo.AdditionalObjects)
                {
                    if (obj.ObjectDateString != null && obj.ObjectType != null)
                    {
                        obj.DateOfObject = Convert.ToDateTime(obj.ObjectDateString);
                        objects.Add(obj);
                    }
                }
                patientProfile.DisclosureInfo.AdditionalObjects = objects;
            }
            else if (patientProfile.DisclosureInfo.AdditionalObjects != null && patientProfile.DisclosureInfo.AdditionalObjects.Any())
            {
                foreach (var obj in patientProfile.DisclosureInfo.AdditionalObjects)
                {
                    obj.DateOfObject = Convert.ToDateTime(obj.ObjectDateString);
                }
            }
        }

        private void AddNatures(PatientProfileViewModel patientProfile)
        {
            const string ProblemListNatureName = "Problem list";
            const string AppointmentListNatureName = "Appointment list";
            const string MedicationListNatureName = "Medication list";
            const string LabsListNatureName = "Labs";

            var natures = _aodRepository.GetAllNatures();



            if (patientProfile.ReportOptions.OmitApptList)
            {
                natures.RemoveAll(n => n.NatureName == AppointmentListNatureName);
            }
            if (patientProfile.ReportOptions.OmitLabList)
            {
                natures.RemoveAll(n => n.NatureName == LabsListNatureName);
            }
            if (patientProfile.ReportOptions.OmitMedList)
            {
                natures.RemoveAll(n => n.NatureName == MedicationListNatureName);
            }
            if (patientProfile.ReportOptions.OmitProblemList)
            {
                natures.RemoveAll(n => n.NatureName == ProblemListNatureName);
            }

            patientProfile.DisclosureInfo.Natures = natures;

        }

        private void AddAuthorizations(PatientProfileViewModel patientProfile)
        {
            if (patientProfile.PatientProfile.Authorizations != null && patientProfile.AuthorizationSelected != null)
            {
                //I just realized that maybe there will only ever be one authorization selected so, maybe, it's a one-to-one between Disclosure and DisclosedAuthorization.
                //Because I haven't seen what it looks like when a patient has authorizations, I was unsure.
                var disclosedAuthorizations = new List<DisclosedAuthorization>();
                var authorization = new DisclosedAuthorization(patientProfile.PatientProfile.Authorizations[(int)patientProfile.AuthorizationSelected]);
                disclosedAuthorizations.Add(authorization);
                patientProfile.DisclosureInfo.DisclosedAuthorizations = disclosedAuthorizations;
            }
        }

        private void AddAppointments(PatientProfileViewModel patientProfile)
        {
            if (!patientProfile.ReportOptions.OmitApptList && patientProfile.PatientProfile.Appointments != null && patientProfile.PatientProfile.Appointments.Count > 0 && patientProfile.AppointmentsSelected != null && patientProfile.AppointmentsSelected.Count > 0)
            {
                int appointmentCount = patientProfile.AppointmentsSelected.Count(selected => selected);
                if (appointmentCount > 0)
                {
                    var disclosedAppointments = new List<DisclosedAppointment>();
                    for (int i = 0; i < patientProfile.PatientProfile.Appointments.Count(); i++)
                    {
                        if (patientProfile.AppointmentsSelected[i])
                        {
                            disclosedAppointments.Add(new DisclosedAppointment(patientProfile.PatientProfile.Appointments[i]));
                        }
                    }
                    patientProfile.DisclosureInfo.DisclosedAppointments = disclosedAppointments;
                }
            }
        }

        private void AddProblemDiagnoses(PatientProfileViewModel patientProfile)
        {
            if (!patientProfile.ReportOptions.OmitProblemList && patientProfile.PatientProfile.ProblemDiagnoses != null && patientProfile.PatientProfile.ProblemDiagnoses.Count > 0 && patientProfile.ProblemDiagnosesSelected != null && patientProfile.ProblemDiagnosesSelected.Count > 0)
            {
                int problemCount = patientProfile.ProblemDiagnosesSelected.Count(selected => selected);
                if (problemCount > 0)
                {
                    var disclosedProblems = new List<DisclosedProblemDiagnosis>();

                    for (int i = 0; i < patientProfile.PatientProfile.ProblemDiagnoses.Count(); i++)
                    {
                        if (patientProfile.ProblemDiagnosesSelected[i])
                        {
                            disclosedProblems.Add(new DisclosedProblemDiagnosis(patientProfile.PatientProfile.ProblemDiagnoses[i]));
                        }
                    }
                    patientProfile.DisclosureInfo.DisclosedProblemDiagnoses = disclosedProblems;
                }
            }
        }

        private void AddNotes(PatientProfileViewModel patientProfile)
        {
            var disclosedNotes = new List<DisclosedProgressNote>();

            if (patientProfile.PatientProfile.Notes != null && patientProfile.PatientProfile.Notes.Count > 0 && patientProfile.NotesSelected != null && patientProfile.NotesSelected.Count > 0)
            {
                int NotesCount = patientProfile.NotesSelected.Count(selected => selected);


                if (NotesCount > 0)
                {

                    for (int i = 0; i < patientProfile.PatientProfile.Notes.Count(); i++)
                    {
                        if (patientProfile.NotesSelected[i])
                        {
                            disclosedNotes.Add(new DisclosedProgressNote(patientProfile.PatientProfile.Notes[i]));
                        }
                    }
                    patientProfile.DisclosureInfo.DisclosedProgressNotes = disclosedNotes;
                }
            }

            //Save the opioid note if the user deceide to disclose.
            if (!patientProfile.ReportOptions.OmitOpioidList && patientProfile.PatientProfile.OpioidExist)
            {

                disclosedNotes.Add(patientProfile.PatientProfile.OpioidNote);
                patientProfile.DisclosureInfo.DisclosedProgressNotes = disclosedNotes;


            }
        }

        private void AddRadiology(PatientProfileViewModel patientProfile)
        {
            if (patientProfile.PatientProfile.RadiologyReports != null && patientProfile.PatientProfile.RadiologyReports.Count > 0 && patientProfile.RadiologyReportsSelected != null && patientProfile.RadiologyReportsSelected.Count > 0)
            {
                int radiologyReportsCount = patientProfile.RadiologyReportsSelected.Count(selected => selected);

                if (radiologyReportsCount > 0)
                {
                    var disclosedRadiology = new List<DisclosedRadiologyReport>();
                    for (int i = 0; i < patientProfile.PatientProfile.RadiologyReports.Count(); i++)
                    {
                        if (patientProfile.RadiologyReportsSelected[i])
                        {
                            disclosedRadiology.Add(new DisclosedRadiologyReport(patientProfile.PatientProfile.RadiologyReports[i]));
                        }
                    }
                    patientProfile.DisclosureInfo.DisclosedRadiologyReports = disclosedRadiology;
                }
            }
        }

        private void AddConsult(PatientProfileViewModel patientProfile)
        {
            if (patientProfile.PatientProfile.Consults != null && patientProfile.ConsultSelected != null)
            {
                patientProfile.DisclosureInfo.DisclosedConsults = new List<DisclosedConsult>();
                var consult = new DisclosedConsult(patientProfile.PatientProfile.Consults[(int)patientProfile.ConsultSelected]);
                patientProfile.DisclosureInfo.DisclosedConsults.Add(consult);
            }
        }


    }
}