﻿using NVCC.WebUI.Infrastructure;
using NVCC.WebUI.Models;
using Rotativa;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.Mvc;
using NVCC.Models;

namespace NVCC.WebUI.Controllers
{
    public class RoiController : Controller
    {
        private readonly IRoiService _roiService;
        private IUserService _userService;

        private static readonly string MaxNumberOfAttachedPdfsErrorMessage = string.Format("required");

        public RoiController(IRoiService roiService,
                             IUserService userService)
        {
            _roiService = roiService;
            _userService = userService;

        }

        [HttpGet]
        public ActionResult Index()
        {
            return View(new RoiSearhViewModel());
        }

        [HttpPost]
        [ValidateAntiForgeryToken]
        public ActionResult Index(RoiSearhViewModel model)
        {
            if (ModelState.IsValid)
            {
                model.SearchResults = _roiService.RoiSearch(model.LastFourOfSsn, model.LastName, model.FirstFiveOfSsn, model.FirstName, model.StartDate, model.EndDate);
            }
            return View(model);
        }

        [HttpPost]
        [Authorize(Roles = "ROIAdmin")]
        [ValidateAntiForgeryToken]
        public ActionResult Admin(RoiAdminSearchViewModel model)
        {
            //facility is a required field.
            if (model.facilityId == 0)
            {
                model.visnList = _userService.GetVisn().ToList();
                GetFacilitiesById(model);
                model.UserList = new List<UserByFacility>();
                ModelState.AddModelError("facilityId", "Required");
                return View(model);
            }
            else
            {

                //username is optional
                if (model.selectedUserName == "0")
                {
                    model.selectedUserName = null;
                }

                model.roiAdminSearchList = _roiService.GetRoiAdminSearchResults(model.facilityId, model.selectedVisnId, model.StartDate, model.EndDate, model.selectedUserName);
                model.visnList = _userService.GetVisn().ToList();

                //load facilities
                int selectedVisnId = model.selectedVisnId;
                model.facilityList = _userService.GetFacilityByVisnId().Where(p => p.VISN == selectedVisnId).ToList();

                //load username list
                int selectedfacilityId = model.facilityId;
                model.UserList = _roiService.GetUsersByFacilityId(selectedfacilityId);

                const string roiAdminSessionKey = "RoiAdminSearch";
                if (HttpContextManager.Current.Session != null)
                {
                    HttpContextManager.Current.Session[roiAdminSessionKey] = model;
                }

                return View(model);
            }
        }

        [HttpGet]
        public ActionResult PatientRoiPdf(int disclosureId, string lastFourOfSsn, string patientName, DateTime? dateOfBirth)
        {
            var model = new PatientRoiPdfViewModel
            {
                DateOfBirth = dateOfBirth,
                PatientLastFourOfSsn = lastFourOfSsn,
                PatientName = patientName
            };

            model.Disclosure = _roiService.GetDisclosureById(disclosureId);

            var patientRoiFilename = string.Format("{0}-{1}-{2}.pdf", patientName, lastFourOfSsn, model.Disclosure.DateTimeOfDisclosure.ToString("MM-dd-yyyy"));

            return new ViewAsPdf("PatientRoiPdf", model)
            {
                FileName = patientRoiFilename,
                PageSize = Rotativa.Options.Size.Letter,
                CustomSwitches = string.Format("--footer-left \"{0} DOB: {1}\" --footer-center \"Document Created: {2}\" --footer-right \"Page [page] of [toPage]\" --footer-line --footer-font-size \"8\" --footer-spacing \"3\"", patientName, string.Format("{0:MM/dd/yyyy}", dateOfBirth), string.Format("{0:MM/dd/yyyy HH:mm}", DateTime.Now))
            };
        }

        [HttpGet]
        [Authorize(Roles = "ROIAdmin")]
        public ActionResult RoiAdminPdf()
        {
            var model = RoiAdminSearchViewModel.retrieveFromSession();

            var adminRoiFilename = string.Format("{0}-{1}-{2}-{3}-to-{4}.pdf",
                model.selectedVisnId,
                model.selectedFacilityName,
                string.IsNullOrWhiteSpace(model.selectedUserName) ? "All Users" : model.selectedUserName,
                model.StartDate.HasValue ? model.StartDate.Value.ToString("MM-dd-yyyy") : "Unknown",
                model.EndDate.HasValue ? model.EndDate.Value.ToString("MM-dd-yyyy") : "Unknown");

            return new ViewAsPdf("RoiAdminPdf", model)
            {
                FileName = adminRoiFilename,
                PageSize = Rotativa.Options.Size.Letter,
                CustomSwitches = string.Format("--footer-left \"VISN: {0}, Facility: {1}-{2}, User: {3}  -  Document Created: {4}\" --footer-right \"Page [page] of [toPage]\" --footer-line --footer-font-size \"8\" --footer-spacing \"3\"",
                model.selectedVisnId, model.facilityId, model.selectedFacilityName,
                string.IsNullOrWhiteSpace(model.selectedUserName) ? "All Users" : model.selectedUserName,
                DateTime.Now.ToString("MM/dd/yyyy HH:mm"))
            };
        }

        [HttpGet]
        [Authorize(Roles = "ROIAdmin")]
        public ActionResult Admin()
        {
            var roiAdmin = new RoiAdminSearchViewModel();
            roiAdmin.visnList = _userService.GetVisn().ToList();
            roiAdmin.facilityList = new List<Facility>();
            roiAdmin.UserList = new List<UserByFacility>();

            return View(roiAdmin);
        }

        [HttpGet]
        public ActionResult GetFacilitiesById(int id)
        {
            var roiAdmin = new RoiAdminSearchViewModel();
            roiAdmin.facilityList = _userService.GetFacilityByVisnId().Where(p => p.VISN == id).ToList();
            return Json(roiAdmin.facilityList, JsonRequestBehavior.AllowGet);
        }

        [HttpPost]
        public ActionResult GetFacilitiesById(RoiAdminSearchViewModel model)
        {
            int selectedVisnId = model.selectedVisnId;
            model.facilityList = _userService.GetFacilityByVisnId().Where(p => p.VISN == selectedVisnId).ToList();
            return View(model);
        }


        [HttpGet]
        public ActionResult GetUserNameByStation(int id)
        {
            var roiAdmin = new RoiAdminSearchViewModel();
            roiAdmin.UserList = _roiService.GetUsersByFacilityId(id);

            return Json(roiAdmin.UserList, JsonRequestBehavior.AllowGet);
        }

        [HttpPost]
        public ActionResult GetUserNameByStation(RoiAdminSearchViewModel model)
        {
            int selectedfacilityId = model.facilityId;
            model.UserList = _roiService.GetUsersByFacilityId(selectedfacilityId);
            return View(model);
        }

        [ValidateAntiForgeryToken]
        [HttpPost]
        public ActionResult EditDisclosureMadeTo(Disclosure disclosure)
        {
            if (ModelState.IsValidField("ProviderNameOrGroup") 
                && ModelState.IsValidField("ProviderStreetAddress1")
                && ModelState.IsValidField("ProviderStreetAddress2")
                && ModelState.IsValidField("ProviderCity")
                && ModelState.IsValidField("ProviderState")
                && ModelState.IsValidField("ProviderZip")
                && ModelState.IsValidField("ProviderPhone"))
            {
                try
                {
                    var isRoiAdmin = System.Web.HttpContext.Current.User.IsInRole("ROIAdmin");
                    var userId = System.Web.HttpContext.Current.User.Identity.Name;
                    disclosure = _roiService.EditDisclosureMadeTo(disclosure, isRoiAdmin, userId);
                }
                catch (Exception e)
                {
                    ModelState.AddModelError(string.Empty, e.Message);
                }
                return PartialView(disclosure);
            }
            else
            {
                return PartialView(disclosure);
            }

        }

    }
}