﻿using NVCC.Models;
using NVCC.Repos.PatientRepository;
using NVCC.WebUI.Infrastructure;
using NVCC.WebUI.Models;
using Rotativa;
using System;
using System.Collections.Generic;
using System.Configuration;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Web.Mvc;

namespace NVCC.WebUI.Controllers
{
    [NvccAuthorize]
    [OutputCache(NoStore = true, Duration = 0, VaryByParam = "*")]
    public class PatientProfileController : Controller
    {
        private readonly IPatientProfileService _patientProfileService;
        private readonly IUserService _userService;
        private readonly IAodService _aodService;
        private readonly ICombinePdfService _combinePdfService;

        private static readonly int MaxTotalSizeOfAttachedPdfsInMB = int.Parse(ConfigurationManager.AppSettings["MaxTotalSizeOfAttachedPdfs"]) / 1048576;
        private static readonly int MaxSizeOfAttachedPdfInMB = int.Parse(ConfigurationManager.AppSettings["MaxSizeOfAttachedPdf"]) / 1048576;
        private static readonly int MaxNumberOfAttachedPdfs = int.Parse(ConfigurationManager.AppSettings["MaxNumberOfAttachedPdfs"]);
        private static readonly string MaxTotalSizeOfPdfsErrorMessage = string.Format("The total size of all attached PDFs has exceeded the {0}MB limit", MaxTotalSizeOfAttachedPdfsInMB);
        private static readonly string MaxSizeOfIndividualPdfErrorMessage = string.Format("Attached PDFs must be {0}MB or less each.", MaxSizeOfAttachedPdfInMB);
        private static readonly string MaxNumberOfAttachedPdfsErrorMessage = string.Format("The maximum number of additional PDFs that can be attached to a RefDoc Package is {0}.", MaxNumberOfAttachedPdfs);



        public PatientProfileController(IPatientProfileService patientProfileService,
                                        IUserService userService,
                                        IAodService aodService,
                                        ICombinePdfService combinePdfService)
        {
            _patientProfileService = patientProfileService;
            _userService = userService;
            _aodService = aodService;
            _combinePdfService = combinePdfService;
        }

        [HttpGet]
        public  ActionResult Index(int? patientSid)
        {
            if (patientSid == null)
                return RedirectToAction("Index", "Home");

            var patient = HttpContextManager.Current.Session!=null ? 
                          HttpContextManager.Current.Session[patientSid.ToString()] as Patient: null;
            if (patient == null)
                return RedirectToAction("Index", "Home");

            if(patient.PatientSid!=patientSid)
                return RedirectToAction("Index", "Home");

            User user = _userService.GetUser();
            ViaUser viaUser = _userService.GetViaUser(patient.Station);

            if (viaUser == null)
            {
                TempData["InfoMessage"] = "Unable to identifer user.";
                return RedirectToAction("Login", "User");
            }

            var stopwatch = new Stopwatch();
            stopwatch.Start();
            PatientProfile patientProfile;
            try
            {
                patientProfile = _patientProfileService.GetPatientProfile(patient, viaUser);
                patientProfile.UserInfo = user;
                              
                if (patientProfile.VistaUser.Fault && patientProfile.VistaUser.FaultMessage != null)
                {
                    string viaFaultMessage=null;

                    if (patientProfile.VistaUser.FaultMessage.Contains("The queryBean.provider.userId value has expired")
                        ||
                        patientProfile.VistaUser.FaultMessage.Contains("The queryBean.provider.userId value is not an encrypted valid value")
                        ||
                        patientProfile.VistaUser.FaultMessage.Contains("queryBean.provider.userId cannot be null"))
                        viaFaultMessage = "Your User Session Expired. Please submit credentials again.";
                    else if(patientProfile.VistaUser.FaultMessage.Contains("Division specified is invalid for user"))
                    {
                        TempData["InfoMessage"] = "Division specified is invalid for user";
                        return RedirectToAction("Division", "User", new { station = patient.Station });
                    }
                    else if (patientProfile.VistaUser.FaultMessage.Contains("Division") && patientProfile.VistaUser.FaultMessage.Contains("not supported"))
                    {
                        TempData["InfoMessage"] = "Selected division is not supported for this station";
                        return RedirectToAction("Division", "User", new { station = patient.Station });
                    }
                    else if (patientProfile.VistaUser.FaultMessage.Contains("InstitutionMappingNotFoundException"))
                    {
                        TempData["InfoMessage"] = "Selected divison does not correspond to any recognized station. Please select a different one or submit a work order to have your division in that station's VistA instance updated to match the station you are in.";
                        return RedirectToAction("Division", "User", new { station = patient.Station });
                    }
                    else
                        viaFaultMessage = patientProfile.VistaUser.FaultMessage;
                    TempData["InfoMessage"] = viaFaultMessage;
                    return RedirectToAction("Login", "User");
                }
            }
          
            catch (PatientTimeoutException)
            {
                return View("Timeout",
                    new PatientTimeoutViewModel {
                        PatientSid = patientSid,
                        PatientName = patient.PatientName
                    });
            }
            catch (Exception)
            {
                throw;
            }
            
            stopwatch.Stop();
            patientProfile.Patient = patient;

            HttpContextManager.Current.Session[patientProfile.Patient.PatientIcn] = patientProfile;
            var patientProfileViewModel = new PatientProfileViewModel(patientProfile);

            string id = patientProfileViewModel.cacheInSession();
            TempData["ElapsedTime"] = ViewBag.ElapsedTime;

            _patientProfileService.LogItem(stopwatch, patient, user.DomainPlusNetworkUserName, "PatientProfile");

            return RedirectToAction("Select", new { id });
        }

        [HttpPost]
        public ActionResult Timeout([Bind(Include = "PatientSid, PatientName")] PatientTimeoutViewModel patient)
        {
            return View(patient);
        }

        [HttpGet]
        public ActionResult Select(string id)
        {
            PatientProfileViewModel patientProfileViewModel = PatientProfileViewModel.retrieveFromSession(id);
            if (patientProfileViewModel == null)
            {
                return RedirectToAction("Index", "Home");
            }
            ViewBag.ElapsedTime = TempData["ElapsedTime"];
            ViewBag.PdfErrorMessage = TempData["PdfErroMessage"];

            

            //var aod = new Disclosure();
            
            if (patientProfileViewModel.DisclosureInfo == null)
            {
                patientProfileViewModel.DisclosureInfo = new Disclosure();
            }
            if (patientProfileViewModel.DisclosureInfo.AdditionalObjects == null)
            {
                patientProfileViewModel.DisclosureInfo.AdditionalObjects = new List<AdditionalObject>();
                patientProfileViewModel.DisclosureInfo.AdditionalObjects.Add(new AdditionalObject { ObjectDateString = "", ObjectType = "", OtherTypeText = "" });
                patientProfileViewModel.DisclosureInfo.AdditionalObjects.Add(new AdditionalObject { ObjectDateString = "", ObjectType = "", OtherTypeText = "" });
                patientProfileViewModel.DisclosureInfo.AdditionalObjects.Add(new AdditionalObject { ObjectDateString = "", ObjectType = "", OtherTypeText = "" });
                patientProfileViewModel.DisclosureInfo.AdditionalObjects.Add(new AdditionalObject { ObjectDateString = "", ObjectType = "", OtherTypeText = "" });
            }

            


            return View("Index", patientProfileViewModel);
        }

        /*ValidateInput(false) fixes a bug that caused app to crash when 
         * html-like bracket tags were inserted into text*/
        [HttpPost]
        [ValidateInput(false)]
        [ValidateAntiForgeryToken]
        public ActionResult Display([Bind(Include = "PatientIcn, ReportOptions, AppointmentsSelected, AuthorizationSelected, ConsultSelected, ProblemDiagnosesSelected, RadiologyReportsSelected, AdditionalLabs, ProgressNotes, AdditionalRecords, AdditionalAuthorizations, AdditionalConsults, AdditionalRadiology, AdditionalOrders, ReferralType, NotesSelected, PdfsToAttach, AttachedPdfs, DisclosureInfo, AdditionalObject, AdditionalObjects")] PatientProfileViewModel patientProfileViewModel)
        {
            // The PatientProfile component of the patientProfileViewModel is not passed back 
            // from the form, so restore it to the view model by reloading it from the Session.
            if (!patientProfileViewModel.RefreshPatientProfile())
            {
                TempData["Message"] = "Unable to retrieve cached patient data. Please search for patient again.";
                return RedirectToAction("Error", "Home");
            }

            //Remove additional objects with Removed flag equals true, if they exist in the list
            if (patientProfileViewModel.DisclosureInfo.AdditionalObjects!=null && patientProfileViewModel.DisclosureInfo.AdditionalObjects.Any(item => item.Removed == true))
            {
                IList<AdditionalObject> objects = new List<AdditionalObject>();
                foreach (AdditionalObject obj in patientProfileViewModel.DisclosureInfo.AdditionalObjects)
                {
                    if (!obj.Removed)
                    {
                        objects.Add(obj);
                    }
                }
                patientProfileViewModel.DisclosureInfo.AdditionalObjects = objects;
            }
            // if refresh is false, there is an error and patientProfileViewModel is not consistent; need to indicate this somehow and to do something about it.
            // This is one place that a PatientProfileViewModel is created from scratch and it is cached with a handle (ID) to be able to get it back

            var id = patientProfileViewModel.cacheInSession();
            return RedirectToAction("Display", new { id });
        }

        [HttpGet]
        public ActionResult Display(string id)
        {
            PatientProfileViewModel patientProfileViewModel = PatientProfileViewModel.retrieveFromSession(id);
            if (patientProfileViewModel == null)
            {
                return RedirectToAction("Index", "Home");
            }

            const string nonPdfErrorMessage = "Only PDFs can be attached to a RefDoc Package";
            const string pdfCorruptedErrorMessage = "One or more attached PDFs were corrupted and have been removed.";

            if (patientProfileViewModel.PdfsToAttach != null && patientProfileViewModel.PdfsToAttach.Any())
            {
                patientProfileViewModel.PdfsToAttach.RemoveAt(patientProfileViewModel.PdfsToAttach.Count() - 1);
                if (!patientProfileViewModel.AttachmentsArePdfs())
                {
                    TempData["PdfErrorMessage"] = nonPdfErrorMessage;
                    return RedirectToAction("Select", new { id });
                }
                if (!patientProfileViewModel.SizeOfEachPdfIsWithinLimit())
                {
                    TempData["PdfErrorMessage"] = MaxSizeOfIndividualPdfErrorMessage;
                    return RedirectToAction("Select", new { id });
                }
                if (!patientProfileViewModel.TotalSizeOfAttachedPdfsIsWithinLimit())
                {
                    TempData["PdfErrorMessage"] = MaxTotalSizeOfPdfsErrorMessage;
                    return RedirectToAction("Select", new { id });
                }
                if (!patientProfileViewModel.NumberOfPdfsIsWithinLimit())
                {
                    TempData["PdfErrorMessage"] = MaxNumberOfAttachedPdfsErrorMessage;
                    return RedirectToAction("Select", new { id });
                }
                TempData["PdfErrorMessage"] = null;                           
            }
            patientProfileViewModel.RemoveSelectedAttachedPdfs();

            patientProfileViewModel.AttachPdfs();

            if (!patientProfileViewModel.AttachedPdfsAreReadable())
            {
                TempData["PdfErrorMessage"] = pdfCorruptedErrorMessage;
                patientProfileViewModel.RemoveSelectedAttachedPdfs();
                return RedirectToAction("Select", new { id });
            }

            return View(patientProfileViewModel);
        }

        [HttpGet]
        public ActionResult Pdf(String id)
        {
            User user = _userService.GetUser();
            PatientProfileViewModel patientProfileViewModel = PatientProfileViewModel.retrieveFromSession(id);
            if (patientProfileViewModel == null)
            {
                return RedirectToAction("Index", "Home");
            }
            
            //On generate PDF save the AOD information.
            patientProfileViewModel.DisclosureInfo.UserId = user.DomainPlusNetworkUserName;
            patientProfileViewModel.DisclosureInfo.Sta3n = user.CurrentDefaultFacility;
            if (user.Facilities.ContainsKey(user.CurrentDefaultFacility))
            {
                patientProfileViewModel.DisclosureInfo.VISN = user.Facilities[user.CurrentDefaultFacility].VISN;
                patientProfileViewModel.DisclosureInfo.StationName = user.Facilities[user.CurrentDefaultFacility].StationName;
            }
            _aodService.RecordAodInfo(patientProfileViewModel);

            Stopwatch stopwatch = new Stopwatch();
            stopwatch.Start();
            var stationInfo = _userService.GetStationInfo(patientProfileViewModel.PatientProfile.Patient.Station);
            ViewBag.stationInfo = stationInfo;
            stopwatch.Stop();
            _patientProfileService.LogItem(stopwatch, patientProfileViewModel.PatientProfile.Patient, user.DomainPlusNetworkUserName, "PDF");

            var refDocFileName = patientProfileViewModel.PatientProfile.Patient.PatientName + "-" + patientProfileViewModel.PatientProfile.Patient.PatientSsn.Substring(5, 4) + ".pdf";

            //return View("PDF", patientProfileViewModel);
            var refDocPdfView = new ViewAsPdf("PDF", patientProfileViewModel)
            {
                FileName = refDocFileName,
                PageSize = Rotativa.Options.Size.Letter,
                CustomSwitches = "--footer-left \"" + patientProfileViewModel.PatientProfile.Patient.PatientName + " DOB: " + String.Format("{0:MM/dd/yyyy}", patientProfileViewModel.PatientProfile.Patient.DateOfBirth) + "\" --footer-center \"Document Created: " + String.Format("{0:MM/dd/yyyy HH:mm}", DateTime.Now) + "\" --footer-right \"Page [page] of [toPage]\" --footer-line --footer-font-size \"8\" --footer-spacing \"3\""
            };

            if (patientProfileViewModel.AttachedPdfs == null || !patientProfileViewModel.AttachedPdfs.Any())
            {
                return refDocPdfView;
            }

            var refDocTempPath = Path.GetTempFileName();

            var byteArray = refDocPdfView.BuildPdf(ControllerContext);
            var fileStream = new FileStream(refDocTempPath, FileMode.Create, FileAccess.Write);
            fileStream.Write(byteArray, 0, byteArray.Length);
            fileStream.Close();

            var combinedPdfPath = _combinePdfService.CombinePdfs(refDocTempPath, patientProfileViewModel.AttachedPdfs);
            
            var combinedPdf = File(System.IO.File.ReadAllBytes(combinedPdfPath), "application/pdf", refDocFileName);

            var allTempPaths = new List<string>() { combinedPdfPath, refDocTempPath };
            allTempPaths.AddRange(patientProfileViewModel.AttachedPdfs.Select(p => p.FullPath));
            ThreadPool.QueueUserWorkItem(o => DeleteFiles(allTempPaths));

            return combinedPdf;
        }

        private static readonly TimeSpan DelayDeleteTime = TimeSpan.FromMinutes(double.Parse(ConfigurationManager.AppSettings["MinutesToDelayDeletingTempPdfs"]));

        private static async void DeleteFiles(IEnumerable<string> paths)
        {
            ///We delay deleting of the temp files because the user may click to download the PDF
            ///more than once.
            await Task.Delay(DelayDeleteTime).ContinueWith(_ =>
            {
                foreach (var path in paths)
                {
                    try
                    {
                        System.IO.File.Delete(path);
                    }
                    catch
                    {
                        ///Not sure what to do in this catch block. Basically,
                        ///I want execution to continue even if Delete fails.
                    }
                }
            });
        }
    }
}
