﻿using NUnit.Framework;
using System.Linq;


namespace NVCC.Repos.UserRepository.Tests
{
    [TestFixture]
    public class UserRepositoryGetUserTests
    {
        private UserRepository userRepository;

        [OneTimeSetUp]
        public void SetUp()
        {
            userRepository = new UserRepository(new UserContext());
        }

        [Test]
        public void UserRepositoryGetUser_USER()
        {
            // Arrange
            // Act
            var result = userRepository.GetUser("DOMAIN\\USER");
            // Assert
            Assert.That(result, Is.Not.Null);
            Assert.That(result.Name, Is.Not.Null);
            Assert.That(result.Name, Is.Not.Empty);
            Assert.That(result.Name, Is.EqualTo("First, Last"));
            Assert.That(result.DomainPlusNetworkUserName, Is.EqualTo("DOMAIN\\USER").IgnoreCase);
            Assert.That(result.Facilities, Is.Not.Null);
            Assert.That(result.Facilities, Is.Not.Empty);
            Assert.That(result.Facilities.Count, Is.GreaterThan(100));
            Assert.That(result.Facilities.Values.Select(f => f.sta3n), Has.Some.EqualTo(result.CurrentDefaultFacility));
        }

        [Test]
        public void UserRepositoryGetUser_USER()
        {
            // Arrange
            // Act
            var result = userRepository.GetUser("DOMAIN\\USER");
            // Assert
            Assert.That(result, Is.Not.Null);
            Assert.That(result.Name, Is.Not.Null);
            Assert.That(result.Name, Is.Not.Empty);
            Assert.That(result.Name, Is.EqualTo("Wilmer, Elizabeth"));
            Assert.That(result.DomainPlusNetworkUserName, Is.EqualTo("DOMAIN\\USER").IgnoreCase);
            Assert.That(result.Facilities, Is.Not.Null);
            Assert.That(result.Facilities, Is.Empty);
            Assert.That(result.CurrentDefaultFacility, Is.EqualTo((short)0));
        }

        [Test]
        public void UserRepositoryGetUser_VHAMASER_VA_NSOC_EAS_TESTER1()
        {
            // Arrange
            // Act
            var result = userRepository.GetUser("DOMAIN\\VA-NSOC-EAS-TESTER1");
            // Assert
            Assert.That(result, Is.Not.Null);
            Assert.That(result.Name, Is.Not.Null);
            Assert.That(result.Name, Is.Not.Empty);
            Assert.That(result.Name, Is.EqualTo("Tester1, NSOC").IgnoreCase);
            Assert.That(result.DomainPlusNetworkUserName, Is.EqualTo("DOMAIN\\VA-NSOC-EAS-TESTER1").IgnoreCase);
            Assert.That(result.Facilities, Is.Not.Null);
            Assert.That(result.Facilities, Is.Empty);
            Assert.That(result.CurrentDefaultFacility, Is.EqualTo((short)0));
        }

        [Test]
        public void UserRepositoryGetUser_DOMAIN_USER()
        {
            // Arrange
            // Act
            var result = userRepository.GetUser("DOMAIN\\USER");
            // Assert
            Assert.That(result, Is.Not.Null);
            Assert.That(result.Name, Is.Not.Null);
            Assert.That(result.Name, Is.Not.Empty);
            Assert.That(result.Name, Is.EqualTo("First, Last"));
            Assert.That(result.DomainPlusNetworkUserName, Is.EqualTo("DOMAIN\\USER").IgnoreCase);
            Assert.That(result.Facilities, Is.Not.Null);
            Assert.That(result.Facilities, Is.Not.Empty);
            Assert.That(result.Facilities.Values.Select(f => f.sta3n), Has.Some.EqualTo(result.CurrentDefaultFacility));
            Assert.That(result.CurrentDefaultFacility, Is.EqualTo((short)544));
            Assert.That(result.Facilities.ContainsKey(result.CurrentDefaultFacility));
            Assert.That(result.Facilities[result.CurrentDefaultFacility].PHIPII, Is.True);
            Assert.That(result.Facilities[result.CurrentDefaultFacility].StationName, Does.Contain("Columbia"));
            Assert.That(result.Facilities.Count, Is.EqualTo(1));
        }

        [Test]
        public void UserRepositoryGetUser_DOMAIN_USER()
        {
            // Arrange
            // Act
            var result = userRepository.GetUser("DOMAIN\\USER");
            // Assert
            Assert.That(result, Is.Not.Null);
            Assert.That(result.Name, Is.Not.Null);
            Assert.That(result.Name, Is.Not.Empty);
            Assert.That(result.Name, Is.EqualTo("First, Last"));
            Assert.That(result.DomainPlusNetworkUserName, Is.EqualTo("DOMAIN\\USER").IgnoreCase);
            Assert.That(result.Facilities, Is.Not.Null);
            Assert.That(result.Facilities, Is.Not.Empty);
            Assert.That(result.Facilities.Values.Select(f => f.sta3n), Has.Some.EqualTo(result.CurrentDefaultFacility));
            Assert.That(result.CurrentDefaultFacility, Is.EqualTo((short)528));
            Assert.That(result.Facilities.ContainsKey(result.CurrentDefaultFacility));
            Assert.That(result.Facilities[result.CurrentDefaultFacility].PHIPII, Is.True);
            Assert.That(result.Facilities[result.CurrentDefaultFacility].StationName, Does.Contain("Upstate New York").IgnoreCase);
            Assert.That(result.Facilities.Count, Is.EqualTo(1));
        }

        [Test]
        public void UserRepositoryGetUser_DOMAIN_SERVER()
        {
            // Arrange
            // Act
            var result = userRepository.GetUser("DOMAIN\\SERVER");
            // Assert
            Assert.That(result, Is.Not.Null);
            Assert.That(result.Name, Is.Not.Null);
            Assert.That(result.Name, Is.Not.Empty);
            Assert.That(result.Name, Is.EqualTo(", ")); // Argubly not the proper thing
            Assert.That(result.DomainPlusNetworkUserName, Is.EqualTo("DOMAIN\\SERVER").IgnoreCase);
            Assert.That(result.Facilities, Is.Not.Null);
            Assert.That(result.Facilities, Is.Not.Empty);
            Assert.That(result.Facilities.Values.Select(f => f.sta3n), Has.Some.EqualTo(result.CurrentDefaultFacility));
            Assert.That(result.Facilities.ContainsKey(result.CurrentDefaultFacility));
            Assert.That(result.Facilities[result.CurrentDefaultFacility].PHIPII, Is.True);
            Assert.That(result.Facilities.Count, Is.EqualTo(8));
        }

        [Test]
        public void UserRepositoryGetUser_null()
        {
            var result = userRepository.GetUser(null);
            Assert.That(result, Is.Null);
        }

        [Test]
        public void UserRepositoryGetUser_UsernameDoesNotExist()
        {
            var result = userRepository.GetUser("a");
            Assert.That(result, Is.Null);
        }

        [Test]
        public void UserRepositoryGetUser_EmptyString()
        {
            var result = userRepository.GetUser("");
            Assert.That(result, Is.Null);
        }

        [Test]
        public void UserRepositoryGetUser_SingleSlash()
        {
            var result = userRepository.GetUser("\\");
            Assert.That(result, Is.Null);
        }

    }
}
