﻿using NVCC.Models;
using NVCC.Repos.MdwsRepository.EmrService;
using NVCC.Repos.MdwsRepository.Utilities;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Configuration;

namespace NVCC.Repos.MdwsRepository
{
    public class ViaRepository : IViaRepository, IDisposable
    {
        private readonly EmrSvcInterfaceClient _emrClient;
        private readonly queryBean _queryBean;

        public ViaRepository()
        {
            _emrClient = new EmrSvcInterfaceClient();
            _queryBean = new queryBean
            {
                patient = new patient(),
                requestingApp = "CITCom",
                consumingAppToken = ConfigurationManager.ConnectionStrings["VIAtoken"].ConnectionString,
                consumingAppPassword = ConfigurationManager.ConnectionStrings["VIApassword"].ConnectionString
            };
        }

        public ViaUser Login(string siteCode, string accessCode, string verifyCode)
        {
            var viaUser = new ViaUser();
            var viaQueryBean = new ViaQueryBean();

            userTO personsTo = _emrClient.loginVIA(siteCode, accessCode, verifyCode, _queryBean);


            if (personsTo == null)
            {
                viaUser.Fault = true;
                viaUser.FaultMessage = "The VistA Web Service failed to return data at this time. Please try again.";
                return viaUser;
            }

            if (personsTo.divisions != null)
            {
                viaUser.Divisions = personsTo.divisions.Select(d => new Division { Sta6a = d.siteId, DivisionName = d.name }).ToList();
            } 
            else
            {
                viaUser.Divisions = new List<Division>();
            }
               
            if (personsTo.fault == null)
                viaUser.ViaBApproved = true;
            if (personsTo.fault != null)
            {
                viaUser.Fault = true;
                viaUser.FaultMessage = personsTo.fault.message;
                if(personsTo.fault.message.Contains("VIAB WEB SERVICES OPTION"))
                    viaUser.ViaBApproved = false;
                else if (personsTo.fault.message.Contains("Not a valid ACCESS CODE/VERIFY CODE pair"))
                    viaUser.InvalidAccessVerifyCode = true;
                else if (personsTo.fault.message.Contains("VERIFY CODE must be changed before continued use"))
                    viaUser.InvalidAccessVerifyCode = true;
            }

            viaQueryBean.ViaProvider = new ViaProvider
            {
                //all are required fields, cannot be null
                UserId = personsTo.vistaDUZ, 
                Name = personsTo.name,
                LoginSiteCode = (viaUser.Divisions.Any(d => d.Sta6a == personsTo.siteId)) ? personsTo.siteId : ((viaUser.Divisions.FirstOrDefault() != null) ? viaUser.Divisions.First().Sta6a : siteCode),
                UserSessionId = personsTo.DUZ //ViaSessionId
            };

            viaUser.QueryBean = viaQueryBean;
            return viaUser;
        }

        public ViaPatient GetPatient(ViaQueryBean myQueryBean, string patientIen)
        {        
            if (myQueryBean == null)
                throw new NullReferenceException("myQueryBean");
            if (patientIen == null)
                throw new NullReferenceException("patientIen");

            var viaPatient = new ViaPatient();
            
            myQueryBean.ViaPatient.LocalPid = patientIen;
            _queryBean.patient.localPid = patientIen;
            _queryBean.patient.localSiteId = myQueryBean.ViaProvider.LoginSiteCode;
            _queryBean.provider = new provider
            {
                loginSiteCode = myQueryBean.ViaProvider.LoginSiteCode,
                userId = myQueryBean.ViaProvider.UserSessionId,
                name = myQueryBean.ViaProvider.Name
            };

            var patientTo = _emrClient.select(_queryBean);
            if(patientTo.fault!=null)
            {
                viaPatient.Fault = true;
                viaPatient.FaultMessage = patientTo.fault.message;
                return viaPatient;

            }
            viaPatient.LocalSiteId = _queryBean.patient.localSiteId;
            viaPatient.LocalPid = patientTo.localPid;
            viaPatient.MpiPid = patientTo.mpiPid;
            return viaPatient;
        }

        public void GetOrders()
        {
            var orders = _emrClient.getAllOrders(_queryBean);
           
            var orders2 = orders.arrays[0];
            var orders3 = orders2.items;
            var ordersActive = orders3.Where(m => m.status.ToLower().Contains("active"));
            var ordersPending = orders3.Where(m => m.status.ToLower().Contains("pending"));
            var ordersScheduled = orders3.Where(m => m.status.ToLower().Contains("scheduled"));
            var ordersCompleted = orders3.Where(m => m.status.ToLower().Contains("complete"));
            var ordersCancelled = orders3.Where(m => m.status.ToLower().Contains("cancelled"));
            var orderCounts = new Dictionary<string, int>
            {
                {"Active", ordersActive.Count()},
                {"Pending", ordersPending.Count()},
                {"Scheduled", ordersScheduled.Count()},
                {"Completed", ordersCompleted.Count()},
                {"Cancelled", ordersCancelled.Count()}
            };

            foreach (var orderCount in orderCounts)
                Debug.Print(orderCount.Key + ": " + orderCount.Value);
        }

        public IList<Note> GetProgressNotes(DateTime fromDate, DateTime toDate, int numResults,
           ViaQueryBean myQueryBean)
        {
            string fromDateString = MdwsDataUtilities.DatetoCprsFormat(fromDate);
            string toDateString = MdwsDataUtilities.DatetoCprsFormat(toDate);

            _queryBean.startDate = fromDateString;
            _queryBean.endDate = toDateString;
            _queryBean.maxRecords = numResults.ToString();
            _queryBean.patient = new patient
            {
                localPid = myQueryBean.ViaPatient.LocalPid,
                localSiteId = myQueryBean.ViaPatient.LocalSiteId,
                mpiPid = myQueryBean.ViaPatient.MpiPid
            };
            taggedNoteArrays notes = _emrClient.getNotesWithText(_queryBean);

            if (notes.arrays == null || notes.arrays[0] == null || notes.arrays[0].notes == null)
                return new List<Note>();

            var transformed = notes.arrays[0].notes
                // getNotesWithText can return "extra" notes which are missing many of the "required"
                // fields. Not sure what these notes are. They don't show up in CPRS. They don't
                // have titles and/or timestamps and (at least the two I saw for the same patient)
                // id's. For that patient, 52 were returned even though only 50 were asked for.
                // Filter these out using the null id's to get "real" notes.
                .Where(n => n.id != null)
                .Select(n => new Note
                {
                    NoteID = n.id,
                    NoteTitle = n.localTitle,
                    NoteAuthor = n.author.name,
                    NoteText = n.text,
                    NoteTimestamp = MdwsDataUtilities.CprsFormattoDate(n.timestamp)
                })
                .ToList();
            return transformed;
        }

        public IList<RadiologyReport> GetRadiologyReports(DateTime fromDate, DateTime toDate, int numResults, ViaQueryBean myQueryBean)
        {
            string fromDateString = MdwsDataUtilities.DatetoCprsFormat(fromDate);
            string toDateString = MdwsDataUtilities.DatetoCprsFormat(toDate);
            _queryBean.startDate = fromDateString;
            _queryBean.endDate = toDateString;
            _queryBean.maxRecords = numResults.ToString();
            var reportArrays = _emrClient.getRadiologyReports(_queryBean);
            if (reportArrays.fault != null)
                return new List<RadiologyReport>();

            if (reportArrays.arrays == null)
                return new List<RadiologyReport>();

            var taggedRadiologyReportArray = reportArrays.arrays[0];
            if (taggedRadiologyReportArray == null)
                return new List<RadiologyReport>();

            var reports = taggedRadiologyReportArray.rpts;

            if (reports != null)
                return reports
                    .Select(p => new RadiologyReport { RadiologyReportId = p.id, ExamDate = p.timestamp, ReportText = p.text, ImpressionText = p.impression, ProcedureName = p.title })
                    .ToList();

            return new List<RadiologyReport>();
        }


        public IList<MdwsConsult> GetConsults(ViaQueryBean myQueryBean)
        {
            var consultArrays = _emrClient.getConsultsForPatient(_queryBean);
            if (consultArrays.fault != null)
                throw new ApplicationException(consultArrays.fault.message);

            if (consultArrays.arrays == null)
                return new List<MdwsConsult>();

            var taggedConsultArray = consultArrays.arrays[0];
            var consults = taggedConsultArray.consults;

            if (consults == null) return new List<MdwsConsult>();
            var finalConsults = consults
                .Where(c => c.status.Equals("Active") || c.status.Equals("Pending"))
                .Select(p => new MdwsConsult { ConsultId = p.id, ConsultDate = p.timestamp, ConsultText = p.text, ConsultTitle = p.title, Status = p.status, StartDate = p.startDate, StopDate = p.stopDate })
                .ToList();
            return finalConsults;
        }

        #region IDisposable Support
        private bool disposedValue = false; // To detect redundant calls

        protected virtual void Dispose(bool disposing)
        {
            if (!disposedValue)
            {
                if (disposing)
                {
                    // TODO: dispose managed state (managed objects).
                    ((IDisposable)_emrClient).Dispose();
                }

                // TODO: free unmanaged resources (unmanaged objects) and override a finalizer below.
                // TODO: set large fields to null.

                disposedValue = true;
            }
        }

        // TODO: override a finalizer only if Dispose(bool disposing) above has code to free unmanaged resources.
        // ~ViaRepository() {
        //   // Do not change this code. Put cleanup code in Dispose(bool disposing) above.
        //   Dispose(false);
        // }

        // This code added to correctly implement the disposable pattern.
        public void Dispose()
        {
            // Do not change this code. Put cleanup code in Dispose(bool disposing) above.
            Dispose(true);
            // TODO: uncomment the following line if the finalizer is overridden above.
            // GC.SuppressFinalize(this);
        }
        #endregion

    }
}
