﻿using NVCC.Models;
using System;
using System.Collections.Generic;
using System.Data.Entity;
using System.Data.SqlClient;
using System.Linq;
using System.Security.Authentication;

namespace NVCC.Repos.AodRepository
{
    public class AodRepository : IAodRepository
    {
        private readonly AodContext _context;

        public AodRepository(AodContext context)
        {
            _context = context;
        }

        public List<Nature> GetAllNatures()
        {
            return _context.Natures.ToList();
        }

        public Disclosure EditDisclosureMadeTo(Disclosure editedDisclosure, bool isRoiAdmin, string userId)
        {
            var disclosure = _context.Disclosures.FirstOrDefault(d => d.DisclosureId == editedDisclosure.DisclosureId);
            if (disclosure == null)
            {
                throw new ApplicationException("No disclosure could be found with the given Disclosure ID");
            }
            if (!isRoiAdmin && disclosure.UserId != userId)
            {
                throw new AuthenticationException("Only an ROI Admin or the user who created this disclosure may edit the Provider Information.");
            }
            disclosure.ProviderNameOrGroup = editedDisclosure.ProviderNameOrGroup;
            disclosure.ProviderStreetAddress1 = editedDisclosure.ProviderStreetAddress1;
            disclosure.ProviderStreetAddress2 = editedDisclosure.ProviderStreetAddress2;
            disclosure.ProviderCity = editedDisclosure.ProviderCity;
            disclosure.ProviderState = editedDisclosure.ProviderState;
            disclosure.ProviderZip = editedDisclosure.ProviderZip;
            disclosure.ProviderPhone = editedDisclosure.ProviderPhone;

            _context.SaveChanges();
            return disclosure;
        }

        public Disclosure GetDisclosureById(int disclosureId)
        {
            return _context.Disclosures.Where(disclosure => disclosure.DisclosureId == disclosureId)
                .Include(d => d.AdditionalObjects)
                .Include(d => d.DisclosedAppointments)
                .Include(d => d.DisclosedAuthorizations)
                .Include(d => d.DisclosedConsults)
                //.Include(d => d.DisclosedProblemDiagnoses)
                .Include(d => d.DisclosedProgressNotes)
                .Include(d => d.DisclosedRadiologyReports)
                .FirstOrDefault();
        }

        public List<Disclosure> GetPatientDisclosures(string patientICN, DateTime? startDate = default(DateTime?), DateTime? endDate = default(DateTime?))
        {
            if (startDate.HasValue)
            {
                startDate = startDate.Value.Date;
            }

            if (endDate.HasValue)
            {
                endDate = endDate.Value.Date.AddDays(1);
            }
            var results = _context.Disclosures
                .Where(d => d.PatientIcn == patientICN && (!startDate.HasValue || d.DateTimeOfDisclosure >= startDate.Value) && (!endDate.HasValue || d.DateTimeOfDisclosure < endDate.Value))
                .Include(d => d.AdditionalObjects)
                .Include(d => d.DisclosedAppointments)
                .Include(d => d.DisclosedAuthorizations)
                .Include(d => d.DisclosedConsults)
                //.Include(d => d.DisclosedProblemDiagnoses)
                .Include(d => d.DisclosedProgressNotes)
                .Include(d => d.DisclosedRadiologyReports);
            return results.ToList();
        }

        public void SaveAodInfo(Disclosure disclosure)
        {
            _context.Disclosures.Add(disclosure);
            _context.SaveChanges();
        }

        public IEnumerable<ROIAdminSearch> GetRoiAdminSearchResults(short sta3n, short visn, DateTime? startDate, DateTime? endDate, string userId)
        {
            var nonNullableStartDate = startDate.HasValue ? startDate.Value : DateTime.MinValue;
            var nonNullableEndDate = endDate.HasValue ? endDate.Value : DateTime.MaxValue;
            userId = userId ?? string.Empty;
            var uspAdminResultList = _context.Database.SqlQuery<ROIAdminSearch>(
                 "EXEC App.NVCC_usp_GetROIAdminSearch @sta3n, @visn, @startDate, @endDate, @userId",
                                                                new SqlParameter("sta3n", sta3n),
                                                                new SqlParameter("visn", visn),
                                                                new SqlParameter("startDate", nonNullableStartDate),
                                                                new SqlParameter("endDate", nonNullableEndDate),
                                                                new SqlParameter("userId", userId)).ToList();
            return uspAdminResultList;
        }

        public List<UserByFacility> GetUsersByFacilityId(int sta6a)
        {
            return _context.GetUsersByFacilityId(sta6a);
        }
    }
}
