﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using NVCC.Models;
using NVCC.WebUI.Infrastructure;
using System.Web.ModelBinding;

namespace NVCC.WebUI.Models
{
    [Serializable]
    public class PatientProfileViewModel
    {
        private string id;
        [BindNever]
        public string ID
        {
            get {
                return id;
            }
        }
        private void resetID()
        {
            id = null;
            id = String.Concat(PatientProfile.Patient.PatientSid, '-', this.GetHashCode().ToString("X8"));
        }
        public string PatientIcn { get; set; }
        [BindNever]
        public PatientProfile PatientProfile { get; set; }
        public ReportOptionsModel ReportOptions { get; set; }
        public List<bool> AppointmentsSelected { get; set; }
        public Int32? AuthorizationSelected { get; set; }
        public Int32? ConsultSelected { get; set; }
        public List<bool> ProblemDiagnosesSelected { get; set; }
        public List<bool> RadiologyReportsSelected { get; set; }
        public List<bool> NotesSelected { get; set; }
        public String AdditionalLabs { get; set; }
        [AllowHtml]
        public String ProgressNotes { get; set; }
        [AllowHtml]
        public String AdditionalRecords { get; set; }
        [AllowHtml]
        public String AdditionalAuthorizations { get; set; }
        [AllowHtml]
        public String AdditionalConsults { get; set; }
        [AllowHtml]
        public String AdditionalRadiology { get; set; }
        [AllowHtml]
        public String AdditionalOrders { get; set; }
        [BindNever]
        public IList<AuthorizationViewModel> AuthorizationViewModels { get; set; }
        [BindNever]
        public IList<ConsultViewModel> ConsultViewModels { get; set; }
        public string ReferralType { get; set; }

        public PatientProfileViewModel() { }
        public PatientProfileViewModel(PatientProfile patientProfile)
        {
            AuthorizationViewModels = new List<AuthorizationViewModel>();
            ConsultViewModels = new List<ConsultViewModel>();
            PatientProfile = patientProfile;
            PatientIcn = patientProfile.Patient.PatientIcn;
            ReportOptions = new ReportOptionsModel();
            AppointmentsSelected = PatientProfile.Appointments == null ? new List<bool>() : Enumerable.Repeat(false, PatientProfile.Appointments.Count()).ToList();
            ProblemDiagnosesSelected = PatientProfile.ProblemDiagnoses == null ? new List<bool>() : Enumerable.Repeat(true, PatientProfile.ProblemDiagnoses.Count()).ToList();
            RadiologyReportsSelected = PatientProfile.RadiologyReports == null ? new List<bool>() : Enumerable.Repeat(false, PatientProfile.RadiologyReports.Count()).ToList();
            NotesSelected = PatientProfile.Notes == null ? new List<bool>() : Enumerable.Repeat(false, PatientProfile.Notes.Count()).ToList();

            PopulateViewHelpers();
        }
        /// <summary>
        /// Fills the parts of PatientProfileViewModel which are transforms of PatientProfile
        /// </summary>
        /// <remarks>
        /// The PatientProfileViewModel object contains some members which are translations of the
        /// data which is in the PatientProfile, specifically, the AuthorizationViewModels which is
        /// a translation of PatientProfile.Authorizations and ConsultViewModels which is a
        /// transformation of PatientProfile.MdwsConsults. When the PatientProfile is assigned
        /// or restored, theses need to be (re)initialized as well. Note that the selection members
        /// of the models do not need to be restored. They either come from the form submission or
        /// are created when the object is first created.
        /// </remarks>
        private void PopulateViewHelpers()
        {
            AuthorizationViewModels = PatientProfile.Authorizations.Select(auth => new AuthorizationViewModel(auth)).ToList();
            ConsultViewModels = PatientProfile.Consults.Select(cons => new ConsultViewModel(cons)).ToList();
        }
        public bool RefreshPatientProfile()
        {
            if (PatientIcn == null)
            {
                // not enough information to extract or rebuild PatientProfile (need at least ICN)
                return false;
            }
            if (PatientProfile == null || PatientProfile.Patient == null || PatientProfile.Patient.PatientSid == 0)
            {
                // need to refresh
                // have an ICN, so can refresh
                // Get it from the session object, if exists
                if (HttpContextManager.Current.Session != null)
                {
                    PatientProfile pp = HttpContextManager.Current.Session[PatientIcn] as PatientProfile;
                    if (pp != null) {
                        PatientProfile = pp;
                        PopulateViewHelpers();
                        return true;
                    }
                    else
                    {
                        // want to re-query the patient profile, which is something like the following (pulled from HomeController and PatientProfileController), but it is mising all the repository and service objects.
                        //Patient patient = _patientRepository.GetPatient(viewModel.PatientSSN, viewModel.Station);
                        //PatientProfile patientProfile = await _patientProfileService.GetPatientProfileAsync(patient, userIen);
                        //HttpContextManager.Current.Session["patientIcn"] = patientProfile;
                        //PatientProfile = patientProfile;
                        //return true;
                        return false;
                    }
                }
            }
            else
            {
                // no need to refresh.
                return true;
            }
            return false;
        }

        /// <summary>
        /// Stores the PatientProfileViewModel in the Session.
        /// </summary>
        /// <remarks>
        /// Recomputes a (potentially) new ID and uses that as the key in the Session collection. That new ID is stored both in the object (before it is stored in the Session) and returned from the method.
        /// </remarks>
        /// <returns>a string which is the key which can be used to look up the object in Session at a later time.</returns>
        public string cacheInSession()
        {
            resetID();
            if (HttpContextManager.Current.Session != null)
            {
                HttpContextManager.Current.Session[ID] = this;
                return ID;
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// Gets the associated PatientProfileViewModel from the Session object based on the ID
        /// </summary>
        /// <remarks>
        /// The compliment of <c>cacheInSession()</c>. This will get the object associated with the handle that <c>cacheInSession()</c> returns.</remarks>
        /// <param name="ID">The key of the PatientProfileViewModel to be retrieved. Value from <c>cacheInSession()</c>.</param>
        /// <returns>The assocated PatientProfileViewModel object</returns>
        public static PatientProfileViewModel retrieveFromSession(string ID)
        {
            if (string.IsNullOrEmpty(ID)) return null;
            if (HttpContextManager.Current.Session == null) return null;
            return HttpContextManager.Current.Session[ID] as PatientProfileViewModel;
        }
    }
}