﻿using NVCC.Models;
using NVCC.Repos.UserRepository.dtoModels;
using System;
using System.Collections.Generic;
using System.Data.Entity;
using System.Data.SqlClient;
using System.Linq;

namespace NVCC.Repos.UserRepository
{
    public class UserRepository : IUserRepository
    {
        private readonly UserContext _context;

        public UserRepository(UserContext context)
        {
            _context = context;
        }

        public User GetUser(string DomainPlusNetworkUserName)
        {
            if (string.IsNullOrEmpty(DomainPlusNetworkUserName)) return null;
            string NetworkUserName;
            try
            {
                NetworkUserName = GetStationNamePart(DomainPlusNetworkUserName);
            }
            catch (ArgumentNullException)
            {
                return null;
            }
            // need to do something with a thrown SqlException
            XdwUser xdwUser;
            try
            {
                xdwUser = _context.GetXdwUserByNetworkName(NetworkUserName).FirstOrDefault();
            }
            catch (Exception)
            {
                // if that call fails, then can't get any farther anyway, so just return null
                xdwUser = null;
            }
            if (xdwUser == null)
            {
                return null; // username does not exist in active directory
            }
            IEnumerable<XdwFacility> xdwFacilities;
            if (xdwUser.LCustomerID != null)
            {
                try
                {
                    xdwFacilities = _context.GetXdwFacilitiesByLCustomerID(xdwUser.LCustomerID);
                }
                catch (Exception)
                {
                    xdwFacilities = new List<XdwFacility>();
                }
            }
            else
            {
                xdwFacilities = new List<XdwFacility>();
            }
            DtoUser dtoUser;
            try
            {
                dtoUser = _context.GetUser(DomainPlusNetworkUserName);
            }
            catch (Exception)
            {
                dtoUser = null;
            }
            if (dtoUser == null)
            {
                // If this is null, then there is not anything else that can be done, so return null
                return null;
            }
            IEnumerable<DtoUserFacility> dtoUserFacilities;
            try
            {
                // This may need to be replaced with stored procedure call instead of Linq calls through Entity Framework(?)
                dtoUserFacilities = _context.UserFacility.Where(u => u.UserId == dtoUser.UserId).ToList();
            }
            catch (Exception)
            {
                dtoUserFacilities = new List<DtoUserFacility>();
            }

            // merge the data in xdwFacilities and dtoUserFacility together into UserFacility by sta3n, even if there is no entry for that sta3n in one side

            Dictionary<short, XdwFacility> xdwFacilityDictionary = new Dictionary<short, XdwFacility>();
            foreach (var xdwFacility in xdwFacilities)
            {
                xdwFacilityDictionary.Add(xdwFacility.FacilityID, xdwFacility);
            }
            Dictionary<short, DtoUserFacility> dtoUserFacilityDictionary = new Dictionary<short, DtoUserFacility>();
            foreach (var dtoUserFacility in dtoUserFacilities)
            {
                dtoUserFacilityDictionary.Add(dtoUserFacility.sta3n, dtoUserFacility);
            }
            IDictionary<short, UserFacility> userFacilities = new Dictionary<short, UserFacility>();
            foreach (short sta3n in xdwFacilityDictionary.Keys.Union(dtoUserFacilityDictionary.Keys))
            {
                XdwFacility xdwFacility;
                if (xdwFacilityDictionary.ContainsKey(sta3n))
                {
                    xdwFacility = xdwFacilityDictionary[sta3n];
                }
                else
                {
                    xdwFacility = null;
                }
                DtoUserFacility dtoUserFacility;
                if (dtoUserFacilityDictionary.ContainsKey(sta3n))
                {
                    dtoUserFacility = dtoUserFacilityDictionary[sta3n];
                }
                else
                {
                    dtoUserFacility = null;
                }
                string sta6a = (dtoUserFacility != null) ? dtoUserFacility.sta6a : sta3n.ToString();
                string stationName;
                if (xdwFacility != null)
                {
                    stationName = xdwFacility.FacilityName;
                }
                else
                {
                    try
                    {
                        var facility = _context.GetXdwFacilityBySta3n(sta3n);
                        if (facility != null)
                        {
                            stationName = facility.FacilityName;
                        }
                        else
                        {
                            stationName = "";
                        }
                    }
                    catch(Exception)
                    {
                        // if the call fails, we don't care why; leaving the name blank is acceptable
                        stationName = "";
                    }
                }
                string locationName;
                if ((dtoUserFacility != null) && !string.IsNullOrWhiteSpace(dtoUserFacility.sta6a))
                {
                    try
                    {
                        var division = GetDivision(dtoUserFacility.sta6a);
                        if (division != null)
                        {
                            locationName = division.DivisionName;
                        }
                        else
                        {
                            locationName = "";
                        }
                    }
                    catch(Exception)
                    {
                        // if the call fails, we don't care why; leaving the name blank is acceptable
                        locationName = "";
                    }
                }
                else
                {
                    locationName = "";
                }
                userFacilities[sta3n] = new UserFacility
                {
                    sta3n = sta3n,
                    sta6a = sta6a,
                    StationName = stationName,
                    LocationName = locationName,
                    LastAuthDateTime = (dtoUserFacility != null) ? dtoUserFacility.LastAuthDateTime : null,
                    ViaB = (dtoUserFacility != null) ? dtoUserFacility.ViaB : null,
                    PHIPII = (xdwFacility != null), // if present there, then that means PHI/PII access is granted
                    VIAToken = "",
                    VIATokenExpiration = null,
                    Admin = (dtoUserFacility != null) ? dtoUserFacility.admin : false
                };
            }
            short defaultFacility = (userFacilities.Count() > 0) ? userFacilities.Values.OrderByDescending(f => f.LastAuthDateTime).First().sta3n : (short)0;
            return new User
            {
                ApplicationUserID = dtoUser.UserId,
                DomainPlusNetworkUserName = DomainPlusNetworkUserName,
                Name = xdwUser.LastName + ", " + xdwUser.FirstName,
                Facilities = userFacilities,
                CurrentDefaultFacility = defaultFacility
            };
        }

        public void UpdateUserFacility(int applicationUserID, UserFacility userFacility)
        {
            var existing = _context.UserFacility.Where(uf => uf.UserId == applicationUserID && uf.sta3n == userFacility.sta3n);
            DtoUserFacility dtoUserFacility = new DtoUserFacility { UserId = applicationUserID, sta3n = userFacility.sta3n, sta6a = userFacility.sta6a, LastAuthDateTime = userFacility.LastAuthDateTime, ViaB = userFacility.ViaB, admin = userFacility.Admin };
            if (existing.Count() == 0)
            {
                _context.UserFacility.Add(dtoUserFacility);
            }
            else
            {
                _context.Entry(dtoUserFacility).State = EntityState.Modified;
            }
            _context.SaveChanges();
        }

        private string GetStationNamePart(string domainPlusUserName)
        {
            if (string.IsNullOrEmpty(domainPlusUserName))
                throw new ArgumentNullException("domainPlusUserName");

            var loginResult = domainPlusUserName.Split('\\');
            return loginResult.Length == 2 ? loginResult[1] : domainPlusUserName;
        }

        public void SaveStationInfo(StationInfo stationInfo)
        {
            _context.Entry(stationInfo).State = EntityState.Modified;
            _context.SaveChanges();
        }

        public StationInfo GetStationInfo(short station)
        {
            var stationInfo = _context.StationInfo.FirstOrDefault(s => s.StationInfoId == station);
            return stationInfo;

        }

        public IEnumerable<Division> GetDivisions(int station)
        {
            var uspDivisions = _context.Database.SqlQuery<Division>(
                "EXEC App.NVCC_usp_GetDivisions @sta3n", new SqlParameter("sta3n", station));
            return uspDivisions.ToList();
        }

        public Division GetDivision(string sta6a)
        {
            if (string.IsNullOrEmpty(sta6a))
                return null;
            var uspDivision = _context.Database.SqlQuery<Division>(
                "EXEC App.NVCC_usp_GetDivision @sta6a", new SqlParameter("sta6a", sta6a));
            return uspDivision.FirstOrDefault();
        }
    }
}
