
-- =============================================
-- Author:		Charles Demosthenes, MD
--              Atlanta VA Medical Center Clinical Informatics
-- Create date: 08/13/2015
-- Revised:     08/24/2015, 9/10/2015 (modified problem list for CDW changes)
--              09/30/2015 (Modified for ICD10)
-- Description:	creates multiple stored procedures
--              to support NVA care data capture
-- =============================================

use S508_ATL_Projects;   --Your database name here
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
IF not EXISTS (
SELECT  schema_name
FROM    information_schema.schemata
WHERE   schema_name = 'nva' )   -- Your schema name here if different
BEGIN
EXEC sp_executesql N'Create Schema nva';
END

--Schema may be different, but procedure names must be these.
IF OBJECT_ID('nva.CheckConnection', 'P') IS NOT NULL
    DROP PROCEDURE nva.CheckConnection;
GO
create PROCEDURE nva.CheckConnection
as
begin
select 'connected' as status;
return 11;
end;
go 


IF OBJECT_ID('nva.GetPatient', 'P') IS NOT NULL
    DROP PROCEDURE nva.GetPatient;
GO

Create Procedure nva.GetPatient
	@station int, --station for pt, restricts queries local VA station
	@ssn varchar(9) -- this is the patient's SSN
AS
BEGIN
	-- SET NOCOUNT ON added to prevent extra result sets from
	-- interfering with SELECT statements.
	SET NOCOUNT ON;
SELECT TOP 1000 [PatientSID]
      ,cast(patientsid as varchar) as strPatientSID
      ,[Sta3n]
      ,[PatientName]
      ,[PatientLastName]
      ,[PatientFirstName]
      ,[PatientSSN]
      ,[StreetAddress1]
      ,[StreetAddress2]
      ,[StreetAddress3]
      ,[City]
      ,[Zip]
      ,[Zip4]
      ,[PostalCode]
      ,[Province]
      ,[Country]
      ,[PhoneResidence]
      ,[PhoneWork]
      ,[PhoneCellular]
      ,[DateOfDeath]
      ,[BadAddressIndicator]
      ,[TemporaryAddressActiveFlag]
      ,[Gender]
      ,[Age]
      ,[State]
      ,[Deceased]
      ,[Eligibility]
      ,[DateOfBirth]
      ,[IneligibleDate]
      ,[EligibilityStatus]
      ,[EligibilityVerificationSource]
  FROM [LSV].[SPatient].[SPatient]
where PatientSSN like @ssn and sta3n=@station;
END;
go

IF OBJECT_ID('nva.GetAppts', 'P') IS NOT NULL
    DROP PROCEDURE nva.GetAppts;
GO
create PROCEDURE nva.GetAppts
	-- Add the parameters for the stored procedure here
	@station int, --station for search,
	              --this restricts queries to the local VA station
	@pid bigint -- this is the patientsid
AS
BEGIN
	-- SET NOCOUNT ON added to prevent extra result sets from
	-- interfering with SELECT statements.
	SET NOCOUNT ON;
SELECT TOP (100) loc.LocationName,
  cast(ap.AppointmentDateTime as datetime) as appointmentdatetime,
  ap.AppointmentSID, ap.PatientSID, ap.AppointmentTypeSID,
  convert(varchar(12), (iif(desiredappointmentdate is null,
    cast(dateadd(day,1,appointmentmadedatetime) as date)
	,desiredappointmentdate)),101) as desireddate,
  ap.CancelNoShowCode, ap.LocationSID, ap.PurposeOfVisit, 
  ap.SchedulingRequestType, ap.CancelDateTime,
  ap.AppointmentCancelledFlag
FROM lsv.Appt.Appointment AS ap INNER JOIN
  lsv.dim.Location AS loc ON ap.LocationSID = loc.LocationSID
  where ap.appointmentdatetime > dateadd(month,-8,getdate()) 
  and ap.patientsid=@pid and ap.sta3n=@station;
end;
go

IF OBJECT_ID('nva.GetRadOrders', 'P') IS NOT NULL
    DROP PROCEDURE nva.GetRadOrders;
GO

create PROCEDURE nva.GetRadOrders
	-- Add the parameters for the stored procedure here
	@station int, --station for search, 
	              --this restricts queries to the local VA station
	@pid bigint -- this is the patientsid
AS
BEGIN
	-- SET NOCOUNT ON added to prevent extra result sets from
	-- interfering with SELECT statements.
	SET NOCOUNT ON;
SELECT ord.RadNucMedOrdersIEN
  ,ord.Sta3n, ord.PatientSID, ord.ReasonForStudy, 
  ord.RadNucMedProceduresIEN, ord.ImagingTypeIEN, ord.CategoryOfExam, 
  ord.RequestStatus, 
  (select top 1 o.OrderStatus from lsv.dim.OrderStatus o
    where o.OrderstatusIEN like ord.RequestStatus and o.sta3n=@station) as OrderStatus,
  ord.RequestUrgency, ord.CPRSOrderSID, ord.CPRSOrderIEN, 
  ord.RadNucMedReasonIEN, ord.OriginalRequestChanged, ord.Pregnant, 
  ord.RequestingPhyStaffSID, ord.RequestEnteredDateTime, 
  ord.LastActivityDateTime, ord.LastActivityDateTimeSID, 
  ord.ImagingLocationsIEN, 
  ord.DateDesiredNotGuaranteed as DesiredDate, ord.LocationSID, ord.LocationIEN, 
  ord.CancelDescription,
  (select top 1 i.TypeOfImaging from lsv.V_Dim.ImagingType AS i 
      where i.imagingtypeIEN like ord.ImagingTypeIEN and i.sta3n=@station) as ImagingType,
  (select top 1 i.Abbreviation from lsv.V_Dim.ImagingType AS i 
      where i.imagingtypeIEN like ord.ImagingTypeIEN and i.sta3n=@station) as Abbreviation,
  (select top 1 [proc].ProcedureName from lsv.V_Dim.RadNucMedProcedures AS [proc] 
      where [proc].RadNucMedProceduresIEN like ord.RadNucMedProceduresIEN
	   and [proc].sta3n=@station) as ProcedureName
  , s.SignatureBlockName, s.SignatureBlockTitle 
from lsv.V_RadNucMed.RadNucMedOrders AS ord inner join lsv.sstaff.sstaff s
  on  ord.RequestingPhyStaffSID = s.Staffsid
  where patientsid=@pid and 
  ord.sta3n=@station and
  ord.DateDesiredNotGuaranteed > dateadd(year,-1,getdate()) 
  and ord.RequestStatus in 
     (select OrderStatusIEN from lsv.dim.OrderStatus 
        where OrderStatus not in ('DISCONTINUED','COMPLETE')) ;
end;
go
--exec nva.GetRadOrders @station=508,@pid=1717444

IF OBJECT_ID('nva.GetLabs', 'P') IS NOT NULL
    DROP PROCEDURE nva.GetLabs;
GO

Create PROCEDURE nva.GetLabs
	-- Add the parameters for the stored procedure here
	@station int, --station for pt, restricts queries local VA station
	@pid bigint -- this is the patientsid
AS
BEGIN
	-- SET NOCOUNT ON added to prevent extra result sets from
	-- interfering with SELECT statements.
	SET NOCOUNT ON;
SELECT TOP (1000) lab.LabChemSID, tst.LabChemTestName, tst.LabChemPrintTestName, 
 lab.LabChemSpecimenDateTime, lab.LabChemResultValue, 
 iif(lab.Units is null,' ',lab.Units) as units, iif(lab.Abnormal is null,' ',lab.Abnormal) as abnormal,
 iif(lab.RefHigh is null,' ',lab.RefHigh) as RefHigh, iif(lab.RefLow is null,' ',lab.RefLow) as RefLow, 
 lab.LabChemResultNumericValue, lab.LabChemTestSID, lab.PatientSID
FROM  lsv.Chem.LabChem AS lab INNER JOIN
 lsv.dim.LabChemTest AS tst ON lab.LabChemTestSID = tst.LabChemTestSID
 where lab.patientsid = @pid and lab.sta3n = @station 
 and lab.LabChemSpecimenDateTime > DATEADD(month, -15, GETDATE())
 order by lab.LabChemTestSID asc, lab.labchemspecimendatetime desc;
end
go
--exec nva.GetLabs @station=508,@pid=1717444

IF OBJECT_ID('nva.GetVAMeds', 'P') IS NOT NULL
    DROP PROCEDURE nva.GetVAMeds;
GO

Create PROCEDURE nva.GetVAMeds
	-- Add the parameters for the stored procedure here
	@station int, --station for pt, restricts queries local VA station
	@pid bigint -- this is the patientsid
AS
BEGIN
	-- SET NOCOUNT ON added to prevent extra result sets from
	-- interfering with SELECT statements.
	SET NOCOUNT ON;
SELECT TOP 1000 [RxOutpatFillSID]
      ,[RxOutpatSID]
      ,[FillType]
      ,[FillNumber]
      ,[RxStatus]
      ,[PatientSID]
      ,[ProviderSID]
      ,[LocalDrugNameWithDose]
      ,[IssueDate]
      ,[FillDateTime]
      ,[ReleaseDateTime]
      ,[ExpirationDate]
      ,[DispensedDate]
      ,[Qty]
      ,[QtyNumeric]
      ,[DaysSupply]
into #mydrugs  FROM [LSV].[RxOut].[RxOutpatFill] where patientsid = @pid 
  and (dispenseddate > dateadd(year,-1,getdate())) and sta3n=@station
  order by rxoutpatsid, FillDateTime desc;

With mdr
as (Select *,Row_Number()Over(partition by [RxOutpatSID] Order by [FillDateTime] desc) Rn From #mydrugs)
Select *into #druglist From mdr where rn=1;

select dl.rxoutpatsid,dl.localdrugnamewithdose
  , cast(dl.filldatetime as date) as filldate
  , 'Qty:'+dl.qty as qty
  , dl.FillNumber, sig.sig, dl.issuedate, dl.dayssupply
into #drugswsig from #druglist dl inner join lsv.rxout.rxoutpatsig sig 
on dl.Rxoutpatsid=sig.RxOutpatSID;

select  dl.rxoutpatsid,dl.localdrugnamewithdose, dl.filldate, dl.qty, 
 dl.FillNumber,iif(rx.maxrefills is null,'0',rx.maxrefills) as maxrefills, 
 dl.sig, dl.issuedate, dl.dayssupply, rx.RxStatus, 
 iif(rx.NextPossibleFillDate is null,'N/A',
 cast(rx.NextPossibleFillDate as varchar)) as nextpossiblefilldate
from #drugswsig dl inner join lsv.rxout.rxoutpat rx 
on dl.Rxoutpatsid=rx.RxOutpatSID order by filldate desc;

drop table #mydrugs;
drop table #druglist;
drop table #drugswsig;
END;

go
--exec nva.GetVAMeds @station=508,@pid=1717444

IF OBJECT_ID('nva.GetNVAMeds', 'P') IS NOT NULL
    DROP PROCEDURE nva.GetNVAMeds;
GO

Create PROCEDURE nva.GetNVAMeds
	@station int, --station for pt, restricts queries local VA station
	@pid bigint -- this is the patientsid
AS
BEGIN
	-- SET NOCOUNT ON added to prevent extra result sets from
	-- interfering with SELECT statements.
	SET NOCOUNT ON;

  	SELECT d.LocalDrugNameWithDose, n.MedicationRoute, 
	  n.Schedule 
	FROM  lsv.NonVAMed.NonVaMed n INNER JOIN
	  lsv.dim.LocalDrug d ON n.LocalDrugSID = d.LocalDrugSID
	  where patientsid = @pid and n.DiscontinuedDateTime is null and n.sta3n=@station;
end;
go

--check using a test patient
--exec nva.GetNVAMeds @station=508, @pid=1717444

IF OBJECT_ID('nva.GetConsults', 'P') IS NOT NULL
    DROP PROCEDURE nva.GetConsults;
GO

create PROCEDURE nva.GetConsults
	-- Add the parameters for the stored procedure here
	@station int, --station for pt, restricts queries local VA station
	@pid bigint -- this is the patientsid
AS
BEGIN
	-- SET NOCOUNT ON added to prevent extra result sets from
	-- interfering with SELECT statements.
	SET NOCOUNT ON;

SELECT TOP 1000 ToRequestServiceName, cast(RequestDateTime as date) as RequestDate, 
	  CPRSStatus, CPRSOrderSID,  Consultsid
	  ,ConsultIEN, PatientSID, ToRequestServiceSID, RequestType
	  ,OrderStatusSID,  RecentActivityTypeSID, RecentActivityType
	  ,SendingStaffSID, Urgency,  InpatOutpat
	  ,iif(ProvisionalDiagnosis is null,' ',ProvisionalDiagnosis) as provdx
	  ,iif(ProvisionalDiagnosisCode is null,' ',ProvisionalDiagnosisCode) as provdxcode
	  ,TIUDocumentSID, EarliestDate
	into #cons FROM LSV.Con.Consult 
	where RequestDateTime > dateadd(day,-180,getdate()) and sta3n=@station
	  and ToRequestServiceName not like 'NON-FORM%' 
	  and CPRSStatus not in ('DISCONTINUED','CANCELLED','COMPLETE') 
	  and patientsid = @pid order by requestdatetime;
select #cons.*,r.ConsultReason from #cons
  inner join lsv.SPatient.SConsultReason r on #cons.ConsultIEN=r.ConsultIEN
order by RequestDateTime;
drop table #cons;
END
go
--exec nva.GetConsults @station=508,@pid=1717444

IF OBJECT_ID('nva.GetRadRpts', 'P') IS NOT NULL
    DROP PROCEDURE nva.GetRadRpts;
GO

create PROCEDURE nva.GetRadRpts
	-- Add the parameters for the stored procedure here
	@station int, --station for pt, restricts queries local VA station
	@pid bigint -- this is the patientsid
AS
BEGIN
	-- SET NOCOUNT ON added to prevent extra result sets from
	-- interfering with SELECT statements.
	SET NOCOUNT ON;
--declare @station as int;set @station=508;declare @pid as bigint; set @pid=1717444;
SELECT  TOP (1000) r.PatientSID, p.ProcedureName, p.TypeOfProcedureExt,
   p.TypeOfProcedureInt, r.RegisteredExamsIEN, r.ExaminationsIEN,
   r.RadNucMedProceduresIEN, 
   r.ExaminationStatusIEN, r.RequestedDate, r.RadNucMedOrdersIEN, 
   r.RadNucMedReportsIEN, p.ImagingTypeIEN
into #exams FROM  lsv.V_RadNucMedSub.RegisteredExams_Examinations AS r INNER JOIN
  lsv.V_Dim.RadNucMedProcedures AS p 
ON r.RadNucMedProceduresIEN like p.RadNucMedProceduresIEN
WHERE (r.RequestedDate > DATEADD(year, - 4, GETDATE())) and r.patientsid = @pid
  and r.sta3n=@station and p.sta3n=@station;

--declare @station int; set @station=508;
SELECT r.PatientSID, r.ProcedureName, r.TypeOfProcedureExt,
   r.TypeOfProcedureInt, r.RegisteredExamsIEN, r.ExaminationsIEN,
   r.RadNucMedProceduresIEN, 
   r.ExaminationStatusIEN, r.RequestedDate, r.RadNucMedOrdersIEN, 
   r.RadNucMedReportsIEN, r.ImagingTypeIEN
  ,(select rpt.ExamDateTime from lsv.V_RadNucMed.RadNucMedReports rpt
     where rpt.RadNucMedReportsIEN like r.RadNucMedReportsIEN and rpt.sta3n=@station) as ExamDateTime
  ,(select rpt.ReportText from lsv.V_RadNucMed.RadNucMedReports rpt
     where rpt.RadNucMedReportsIEN like r.RadNucMedReportsIEN and rpt.sta3n=@station) as ReportText
  ,(select rpt.IsReportTextTruncated from lsv.V_RadNucMed.RadNucMedReports rpt
     where rpt.RadNucMedReportsIEN like r.RadNucMedReportsIEN and rpt.sta3n=@station) as IsReportTextTruncated
  ,(select rpt.ImpressionText from lsv.V_RadNucMed.RadNucMedReports rpt
     where rpt.RadNucMedReportsIEN like r.RadNucMedReportsIEN and rpt.sta3n=@station) as ImpressionText
  ,(select rpt.IsImpressionTextTruncated from lsv.V_RadNucMed.RadNucMedReports rpt
     where rpt.RadNucMedReportsIEN like r.RadNucMedReportsIEN and rpt.sta3n=@station) as IsImpressionTextTruncated
into #rpt FROM  #exams r;

select r.PatientSID, r.ProcedureName, r.TypeOfProcedureExt,
   r.TypeOfProcedureInt, r.RegisteredExamsIEN, r.ExaminationsIEN,
   r.RadNucMedProceduresIEN, 
   r.ExaminationStatusIEN, r.RequestedDate, r.RadNucMedOrdersIEN, 
   r.RadNucMedReportsIEN, r.ImagingTypeIEN
  ,r.ExamDateTime
  ,iif(r.ReportText is null,'No Report',ReportText) as ReportText
  ,r.IsReportTextTruncated
  ,iif(r.ImpressionText is null,'No Impression',r.ImpressionText) as ImpressionText
  ,r.IsImpressionTextTruncated, 
  (select top 1 o.ReasonForStudy from lsv.V_RadNucMed.RadNucMedOrders o
     where o.radnucmedordersien like r.radnucmedordersien and o.sta3n=@station) as ReasonForStudy
  from #rpt r where ExamDateTime is not null;

drop table #rpt;
drop table #exams;
END;
go
--exec nva.GetRadRpts @station=508,@pid=1717444

IF OBJECT_ID('nva.GetSensDx', 'P') IS NOT NULL
    DROP PROCEDURE nva.GetSensDx;
GO
create PROCEDURE nva.GetSensDx
	-- Add the parameters for the stored procedure here
	@station int, --station for pt, restricts queries local VA station
	@pid bigint -- this is the patientsid
AS
BEGIN
--declare @station int; set @station=508; declare @pid bigint; set @pid=1717444;
	-- SET NOCOUNT ON added to prevent extra result sets from
	-- interfering with SELECT statements.
	SET NOCOUNT ON;

declare @flag int;
set @flag=0;
SELECT s.Code
  , i.ICD10SID
  , s.DiagnosisText
  , s.AbuseDx
  , s.SickleDx
  , s.HIVDx
into #sens10
FROM  nva.SensitiveDxCodes AS s INNER JOIN lsv.dim.ICD10 AS i
ON s.code like i.icd10code where s.codingsystem like 'ICD10' and i.sta3n = @station;

SELECT s.Code
  , i.ICD9SID
  , s.DiagnosisText
  , s.AbuseDx
  , s.SickleDx
  , s.HIVDx
into #sens9
FROM  nva.SensitiveDxCodes s INNER JOIN lsv.dim.ICD9 AS i
ON s.code like i.icd9code where s.codingsystem like 'ICD9' and i.sta3n = @station;

SELECT TOP 3000 
 iif(p.icd10SID>0,p.ICD10SID,p.ICD9SID) as code,
 iif(p.icd10sid>0,'ICD10','ICD9') as codingsystem,
 iif(p.icd10sid>0,(select top 1 s.AbuseDx from #sens10 s where p.icd10sid=s.icd10sid),
   (select top 1 t.AbuseDx from #sens9 t where p.icd9sid=t.icd9sid)) as AbuseDx,
 iif(p.icd10sid>0,(select top 1 s.SickleDx from #sens10 s where p.icd10sid=s.icd10sid),
   (select top 1 t.SickleDx from #sens9 t where p.icd9sid=t.icd9sid)) as SickleDx,
 iif(p.icd10sid>0,(select top 1 s.HIVDx from #sens10 s where p.icd10sid=s.icd10sid),
   (select top 1 t.HivDx from #sens9 t where p.icd9sid=t.icd9sid)) as HIVDx
 into #ptsdx FROM LSV.Outpat.VDiagnosis p where patientsid=@pid and sta3n=@station; 

select s.code, s.CodingSystem
  ,iif(s.AbuseDx is null, 0, s.AbuseDx) as AbuseDx
  ,iif(s.SickleDx is null, 0, s.SickleDx) as SickleDx
  ,iif(s.HIVDx is null, 0, s.HIVDx) as HIVDx
into #vdx from #ptsDx s ;

if ((select sum(abusedx) from #vdx) > 0)  set @flag=@flag+1;
if ((select sum(HIVdx) from #vdx) > 0)  set @flag=@flag+4;
if ((select sum(sickledx) from #vdx) > 0)  set @flag=@flag+16;

SELECT 
  prb.ICD9SID, prb.ICD10SID, prb.PatientSID,s.AbuseDx,s.SickleDx,s.HIVDx 
into #ptsprb9 FROM lsv.Outpat.ProblemList AS prb INNER JOIN
     #sens9 s on prb.ICD9SID=s.icd9SID where 
 patientsid=@pid and prb.icd10SID <= 0  and prb.ActiveFlag like 'A'
 and prb.problemlistcondition not like 'H' ;

SELECT 
  prb.ICD9SID, prb.ICD10SID, prb.PatientSID,s.AbuseDx,s.SickleDx,s.HIVDx 
into #ptsprb10 FROM lsv.Outpat.ProblemList AS prb INNER JOIN
     #sens10 s on prb.ICD10SID=s.icd10SID where 
 patientsid=@pid and prb.icd10SID > 0
  and prb.ActiveFlag like 'A'
 and prb.problemlistcondition not like 'H' ;

if (iif((select sum(abusedx) from #ptsprb9) > 0 or (select sum(abusedx)  from #ptsprb10) > 0,1,0) ) >0 set @flag=@flag+2;
if (iif((select sum(HIVDx) from #ptsprb9)   > 0 or (select sum(HIVDx)    from #ptsprb10) > 0,1,0) ) >0 set @flag=@flag+8;
if (iif((select sum(SickleDx) from #ptsprb9)> 0 or (select sum(SickleDx) from #ptsprb10) > 0,1,0) ) >0 set @flag=@flag+32;

drop table #sens9;
drop table #sens10;
drop table #ptsdx;
drop table #vdx;
drop table #ptsprb10;
drop table #ptsprb9;

select @flag as sensdxflag;
--exec nva.GetSensDx @station=508, @pid=1717444;
end;
go
--exec nva.GetSensDx @station=508,@pid=1717444

IF OBJECT_ID('nva.GetProbs', 'P') IS NOT NULL
    DROP PROCEDURE nva.GetProbs;
GO
create PROCEDURE [nva].[GetProbs]
	-- Add the parameters for the stored procedure here
	@station int, --station for pt, restricts queries local VA station
	@pid bigint -- this is the patientsid
AS
BEGIN
--declare @station int; set @station=508; declare @pid bigint; set @pid=1717444; declare @sflag int; set @sflag=1;
	-- SET NOCOUNT ON added to prevent extra result sets from
	-- interfering with SELECT statements.
	SET NOCOUNT ON;

select 
  p.PatientSID, 
  p.ICD9SID,
  p.ICD10SID
into #ptprobs FROM lsv.Outpat.ProblemList AS p 
where p.patientsid=@pid and p.ActiveFlag like 'A'
     and problemlistcondition not like 'H'  
	 and p.sta3n=@station;

Select
  p.icd9SID,
  i.icd9code,
  j.icd9diagnosis,
  p.icd10SID,
  k.icd10code,
  l.icd10diagnosis
into #probs FROM #ptprobs AS p
left outer join lsv.dim.icd9 i on p.icd9sid=i.icd9sid
left outer join lsv.dim.ICD9DiagnosisVersion j on p.icd9sid=j.icd9sid
left outer join lsv.dim.icd10 k on p.icd10sid=k.icd10sid
left outer join lsv.dim.ICD10DiagnosisVersion l on p.icd10sid=l.icd10sid;

select iif(icd9sid>0,icd9code,'') as icd9code,
       iif(icd10sid>0,icd10code,'') as icd10code,
  cast(iif(icd10sid>0,
     left(icd10diagnosis,50)+ ' (ICD10 '+icd10code+')',
	 left(icd9diagnosis,50) + ' (ICD9 ' +icd9code+')') 
   as varchar(75)) as Problem,
   cast(1 as int) as dxinclude
  from #probs;

drop table #ptprobs;
drop table #probs;

end;

--exec nva.getProbs @station=508, @pid=1717444;
--exec nva.getSensDxCodes;
