﻿using NVCC.Models;
using NVCC.Repos.PatientRepository;
using NVCC.WebUI.Infrastructure;
using NVCC.WebUI.Models;
using Rotativa;
using System;
using System.Diagnostics;
using System.Web.Mvc;

namespace NVCC.WebUI.Controllers
{
    [NvccAuthorize]
    public class PatientProfileController : Controller
    {
        private readonly IPatientProfileService _patientProfileService;
        private readonly IUserService _userService;

        public PatientProfileController(IPatientProfileService patientProfileService,
                                        IUserService userService)
        {
            _patientProfileService = patientProfileService;
            _userService = userService;
        }

        [HttpGet]
        public  ActionResult Index(int? patientSid)
        {
            if (patientSid == null)
                return RedirectToAction("Index", "Home");

            var patient = HttpContextManager.Current.Session!=null ? 
                          HttpContextManager.Current.Session[patientSid.ToString()] as Patient: null;
            if (patient == null)
                return RedirectToAction("Index", "Home");

            if(patient.PatientSid!=patientSid)
                return RedirectToAction("Index", "Home");

            User user = _userService.GetUser();
            ViaUser viaUser = _userService.GetViaUser(patient.Station);

            if (viaUser == null)
            {
                TempData["InfoMessage"] = "Unable to identifer user.";
                return RedirectToAction("Login", "User");
            }

            var stopwatch = new Stopwatch();
            stopwatch.Start();
            PatientProfile patientProfile;
            try
            {
                patientProfile = _patientProfileService.GetPatientProfile(patient, viaUser);
                if (patientProfile.VistaUser.Fault && patientProfile.VistaUser.FaultMessage != null)
                {
                    string viaFaultMessage=null;

                    if (patientProfile.VistaUser.FaultMessage.Contains("The queryBean.provider.userId value has expired")
                        ||
                        patientProfile.VistaUser.FaultMessage.Contains("The queryBean.provider.userId value is not an encrypted valid value")
                        ||
                        patientProfile.VistaUser.FaultMessage.Contains("queryBean.provider.userId cannot be null"))
                        viaFaultMessage = "Your User Session Expired. Please submit credentials again.";
                    else if(patientProfile.VistaUser.FaultMessage.Contains("Division specified is invalid for user"))
                        return RedirectToAction("Division", "User", new { station = patientProfile.Patient.Station });
                    else if (patientProfile.VistaUser.FaultMessage.Contains("Division") && patientProfile.VistaUser.FaultMessage.Contains("not supported"))
                        return RedirectToAction("Division", "User", new { station = patientProfile.Patient.Station });
                    else
                        viaFaultMessage = patientProfile.VistaUser.FaultMessage;
                    TempData["InfoMessage"] = viaFaultMessage;
                    return RedirectToAction("Login", "User");
                }
            }
          
            catch (PatientTimeoutException)
            {
                return View("Timeout",
                    new PatientTimeoutViewModel {
                        PatientSid = patientSid,
                        PatientName = patient.PatientName
                    });
            }
            catch (Exception)
            {
                throw;
            }
            
            stopwatch.Stop();
            patientProfile.Patient = patient;

            HttpContextManager.Current.Session[patientProfile.Patient.PatientIcn] = patientProfile;
            var patientProfileViewModel = new PatientProfileViewModel(patientProfile);

            string id = patientProfileViewModel.cacheInSession();
            TempData["ElapsedTime"] = ViewBag.ElapsedTime;

            _patientProfileService.LogItem(stopwatch, patient, user.DomainPlusNetworkUserName, "PatientProfile");

            return RedirectToAction("Select", new { id });
        }

        [HttpPost]
        public ActionResult Timeout([Bind(Include = "PatientSid, PatientName")] PatientTimeoutViewModel patient)
        {
            return View(patient);
        }

        [HttpGet]
        public ActionResult Select(string id)
        {
            PatientProfileViewModel patientProfileViewModel = PatientProfileViewModel.retrieveFromSession(id);
            if (patientProfileViewModel == null)
            {
                return RedirectToAction("Index", "Home");
            }
            ViewBag.ElapsedTime = TempData["ElapsedTime"];
            return View("Index", patientProfileViewModel);
        }

        /*ValidateInput(false) fixes a bug that caused app to crash when 
         * html-like bracket tags were inserted into text*/
        [HttpPost]
        [ValidateInput(false)]
        [ValidateAntiForgeryToken]
        public ActionResult Display([Bind(Include = "PatientIcn, ReportOptions, AppointmentsSelected, AuthorizationSelected, ConsultSelected, ProblemDiagnosesSelected, RadiologyReportsSelected, AdditionalLabs, ProgressNotes, AdditionalRecords, AdditionalAuthorizations, AdditionalConsults, AdditionalRadiology, AdditionalOrders, ReferralType, NotesSelected")] PatientProfileViewModel patientProfileViewModel)
        {
            // The PatientProfile component of the patientProfileViewModel is not passed back 
            // from the form, so restore it to the view model by reloading it from the Session.
            if (!patientProfileViewModel.RefreshPatientProfile())
            {
                return RedirectToAction("Error", "Home", new { vistaUserFaultMessage = "Unable to retrieve cached patient data. Please search for patient again." });
            }
            // if refresh is false, there is an error and patientProfileViewModel is not consistent; need to indicate this somehow and to do something about it.
            // This is one place that a PatientProfileViewModel is created from scratch and it is cached with a handle (ID) to be able to get it back
            var id = patientProfileViewModel.cacheInSession();
            return RedirectToAction("Display", new { id });
        }

        [HttpGet]
        public ActionResult Display(string id)
        {
            PatientProfileViewModel patientProfileViewModel = PatientProfileViewModel.retrieveFromSession(id);
            if (patientProfileViewModel == null)
            {
                return RedirectToAction("Index", "Home");
            }

            return View(patientProfileViewModel);
        }

        [HttpGet]
        public ActionResult Pdf(String id)
        {
            User user = _userService.GetUser();
            PatientProfileViewModel patientProfileViewModel = PatientProfileViewModel.retrieveFromSession(id);
            if (patientProfileViewModel == null)
            {
                return RedirectToAction("Index", "Home");
            }

            Stopwatch stopwatch = new Stopwatch();
            stopwatch.Start();
            var stationInfo = _userService.GetStationInfo(patientProfileViewModel.PatientProfile.Patient.Station);
            ViewBag.stationInfo = stationInfo;
            stopwatch.Stop();
            _patientProfileService.LogItem(stopwatch, patientProfileViewModel.PatientProfile.Patient, user.DomainPlusNetworkUserName, "PDF");

            //return View("PDF", patientProfileViewModel);
            return new ViewAsPdf("PDF", patientProfileViewModel)
            {
                FileName = patientProfileViewModel.PatientProfile.Patient.PatientName + "-" + patientProfileViewModel.PatientProfile.Patient.PatientSsn.Substring(5, 4) + ".pdf",
                PageSize = Rotativa.Options.Size.Letter,
                CustomSwitches = "--footer-left \"" + patientProfileViewModel.PatientProfile.Patient.PatientName + " DOB: " + String.Format("{0:MM/dd/yyyy}", patientProfileViewModel.PatientProfile.Patient.DateOfBirth) + "\" --footer-center \"Document Created: " + String.Format("{0:MM/dd/yyyy HH:mm}", DateTime.Now) + "\" --footer-right \"Page [page] of [toPage]\" --footer-line --footer-font-size \"8\" --footer-spacing \"3\""
            };
        }
    }
}
