{******************************************************************************}
{ Package:      Clinical Case Registries Custom Components                     }
{ Date Created: November 18, 2004                                              }
{ Site Name:    Hines OIFO                                                     }
{ Developers:   Sergey Gavrilov                                                }
{ Description:  Common utilities.                                              }
{ Note:                                                                        }
{******************************************************************************}

unit uROR_Utilities;

{$I Components.inc}

interface

uses
  Classes, ComCtrls, Controls, Dialogs, ExtCtrls, Forms, Graphics, StdCtrls,
  SysUtils;

type
  {============================== TCCRGridDataType =============================
    Overview:     Formatting options for FileMan date/time values.
    SeeAlso:      TCCRGridField.FMDTOptions; FMDateTimeStr
  }
  TFMDateTimeMode = set of (

    fmdtDateOnly,   { Ignore time part of the value. }

    fmdtShortDate,  { Short date (digits only) }

    fmdtYear2,      { 2 digit year }

    fmdtTimeOnly,   { Ignore date part of the value. }

    fmdtShortTime,  { Ignore seconds. }

    fmdtTime24      { Military time format }

  );

{- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Overview:     Returns printable representation of a boolean value.
  SeeAlso:      StringToBoolean
  Description:
    BooleanToString converts and formats a boolean value passed via the
    <i>aValue</i> parameter according to the value of the <i>aFormat</i>
    parameter.
    <p>
    Value of the optional <i>aFormat</i> parameter should contain printable
    representations for True and False separated by semicolon (e.g. 'Y;N').
    If it is omitted or empty, then Windows regional values are used . If
    components were compiled without OrpheusLite support (the NOORPHEUS symbol
    was defined), then default 'T;F' string is used.
}
function BooleanToString(const aValue: Boolean;
  const aFormat: String = ''): String;

{- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Overview:     Returns date/time format string.
  SeeAlso:      DateTimeToString; TFMDateTimeMode
  Description:
    CCRDateTimeFormat returns date/time format string that corresponds to the
    value of the <i>Mode</i> parameter. This string can be used to format
    date/time values with the DateTimeToString function. The function takes
    Windows regional settings into account and uses either 'month first' or
    'day first' format.
}
function CCRDateTimeFormat(Mode: TFMDateTimeMode = []): String;

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
function CmdLineSwitch(SwLst: array of String): Boolean; overload;
{
  Overview:     Checks for the command-line switch(es).
  SeeAlso:      ParamStr
  Description:
    CmdLineSwitch returns True if at least one form of the command-line switch
    passed via the <i>SwLst</i> parameter is present in the application command
    line. Value of the switch is returned via the optional <i>SwVal</i>
    parameter.
    <p>
    The function checks if any of the command-line parameters starts from one of
    the strings stored in the <i>SwLst</i>. The comparison is case-insensitive.
}
function CmdLineSwitch(SwLst: array of String;
  var SwVal: String): Boolean; overload;

{- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Overview:     Converts an internal FileMan date/time into its external format.
  SeeAlso:      TFMDateTimeMode
  Description:
    FMDateTimeStr formats the internal FileMan date/time passed via the
    <i>DateTime</i> parameter according to options passed via the optional
    <i>Mode</i> parameter. If the <i>DateTime</i> is empty, then the function
    returns value of the optional <i>DfltValue</i> parameter (empty string,
    by default). This function takes Windows regional settings into account
    and uses either 'month first' or 'day first' format.
}
function FMDateTimeStr(DateTime: String; Mode: TFMDateTimeMode = [];
  DfltValue: String = ''): String;

{- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Overview:     Indicates if a string contains one of substrings.
  SeeAlso:      AnsiContainsStr; AnsiContainsText; StartString
  Description:
    InString returns True if the string passed via the <i>S</i> parameter
    contains at least one of the substrings passed via the <i>SubStrs</i>.
    Use optional <i>CaseSensitive</i> parameter to control case-sensitivity of
    the search (case-sensitive, by default).
}
function InString(S: String; SubStrs: array of String;
  CaseSensitive: Boolean = True): Boolean;

{- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Overview:     Indicates if a string starts with one of substrings.
  SeeAlso:      AnsiStartsStr; AnsiStartsText; InString
  Description:
    StartString returns True if the string passed via the <i>S</i> parameter
    starts with one of the substrings passed via the <i>SubStrs</i>. Use
    optional <i>CaseSensitive</i> parameter to control case-sensitivity of the
    search (case-sensitive, by default).
}
function StartString(S: String; SubStrs: array of String;
  CaseSensitive: Boolean = True): Boolean;

{- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Overview:     Customized version of the MessageDlg function.
  SeeAlso:      MessageDlg; TModalResult
  Description:
    MessageDialog is a customized version of the MessageDlg function. It has 2
    additional parameters. Use the <i>ACaption</i> parameter to change caption
    of the message dialog box. Assign one of the TModalResult values to the
    <i>DefButton</i> parameter to change the default button for the dialog.
}
function MessageDialog(const ACaption: string; const Msg: string;
  DlgType: TMsgDlgType; Buttons: TMsgDlgButtons; DefButton: Integer;
  HelpCtx: Longint): Word;

{- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Overview:     Customized version of the MessageDlg function for 508 compliance
  SeeAlso:      MessageDlg; TModalResult
  Description:
    MessageDlg508 is a customized version of the standard MessageDlg function
    that fixes several 508 related issues (text not being read, etc). Use the
    <i>ACaption</i> parameter to change caption of the message dialog box.
}
function MessageDlg508(const ACaption: string; const Msg: widestring;
  DlgType: TMsgDlgType; Buttons: TMsgDlgButtons; HelpCtx: Longint): Integer;

{- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Overview:     Customized version of the Windows ShowMessage for 508 compliance
  SeeAlso:      MessageDlg; TModalResult
  Description:
    Display a message.
}
procedure Showmessage(const Msg: widestring);

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
function Piece(Value, Delimiter: String;
  StartPiece: Integer = 1): String; overload;
{
  Overview:     Returns the specified substring from the specified string.
  SeeAlso:      ExtractStrings
  Description:
    Piece emulates the M (MUMPS) $PIECE function. If a string passed via the
    <i>Value</i> parameter is composed of a series of substrings separated by a
    common delimiter passed via the <i>Delimiter</i> parameter, then the
    two-argument format of Piece returns the substring located before the first
    occurrence of delimiter. That is, it returns the first substring in the
    string.
    <p>
    The three-argument format of the function returns the <i>StartPiece</i>
    substring in the string. That is, Piece returns the substring located
    between the <i>StartPiece</i>-1 and <i>StartPiece</i> occurrences of the
    delimiter.
    <p>
    The four-argument format of Piece returns the substring beginning with the
    piece specified by the <i>StartPiece</i> and ending with the piece specified
    by the value of the <i>EndPiece</i>. The substring includes any intermediate
    occurrences of the delimiter.

    Keep the following points in mind when you use the Piece function:
    <p>* The two-argument format is equivalent to the three-argument format when
    the value of <i>StartPiece</i> is 1.
    <p>* If the <i>StartPiece</i> is less than 1, Piece returns an empty string.
    <p>* If the <i>StartPiece</i>-1 is greater than the number of occurrences of
    the delimiter in the string, Piece returns an empty string.
    <p>* If the <i>EndPiece</i> is greater than the number of occurrences of
    the delimiter in the string, Piece returns all substrings beginning with
    the piece specified by the <i>StartPiece</i>.
    <p>* If <i>StartPiece</i> is greater than <i>EndPiece</i>, the function
    returns an empty string.
    <p>* If there are fewer than the <i>StartPiece</i>-1 instances of the
    delimiter in the string, Piece returns an empty string.
    <p>* If the string does not contain the <i>Delimiter</i> and the
    <i>StartPiece</i> is 1, the function returns the entire string (unless the
    <i>EndPiece</i> is less than 1).
}
function Piece(Value, Delimiter: String;
  StartPiece, EndPiece: Integer): String; overload;

{- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Overview:     Returns the boolean value of a string
  SeeAlso:      BooleanToString
  Description:
    StringToBoolean makes every attempt possible to convert a string passed via
    the <i>aValue</i> parameter to its boolean equivalent.
    <p>
    Firstly, if the <i>aFormat</i> parameter is defined and not empty, the
    function performs case-insensitive comparison of the string to the first
    ';'-piece of the <i>aFormat</i>. If they are the same, then True is returned.
    <p>
    Otherwise, the function checks if the string starts from one of the
    following (all checks are case-insensitive): Windows regional 'true' and
    'yes' characters (if components were compiled with the OrpheusLite support),
    'T', and 'Y'. If there is a match, True is returned.
    <p>
    Otherwise, StrToIntDef function is used to get numeric value of the string
    (0 is used in case of failed conversion). If the result is not zero, then
    True is returned.
    <p>
    Otherwise, the function returns False.
}
function StringToBoolean(const aValue: String;
  const aFormat: String = ''): Boolean;

const
  {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    Overview:     Custom set of FileMan date/time formatting options.
    SeeAlso:      CCRDateTimeFormat; FMDateTimeStr; TFMDateTimeMode
    Description:
      fmdtShortDateTime combines those FileMan date/time formatting options
      that produce short date/time representation (i.e. MM/DD/YYYY@HHMM or
      DD/MM/YYYY@HHMM depending on Windows regional settings).
  }
  fmdtShortDateTime = [fmdtShortDate, fmdtShortTime, fmdtTime24];

implementation

uses
  Windows,
  {$IFNDEF NOORPHEUS}OvcIntl,{$ENDIF}
  StrUtils;

var
  FMDateFormat: array[1..2,1..3] of String;

function BooleanToString(const aValue: Boolean;
  const aFormat: String = ''): String;
begin
  if aFormat <> '' then
    begin
      if aValue then
        Result := Piece(aFormat, ';', 1)
      else
        Result := Piece(aFormat, ';', 2);
    end
  else if aValue then
    {$IFNDEF NOORPHEUS}
    Result := OvcIntlSup.TrueChar
    {$ELSE}
    Result := 'T'
    {$ENDIF}
   else
    {$IFNDEF NOORPHEUS}
     Result := OvcIntlSup.FalseChar;
    {$ELSE}
    Result := 'F';
    {$ENDIF}
end;

function CCRDateTimeFormat(Mode: TFMDateTimeMode = []): String;
var
  tfrm: String;
begin
  Result := '';

  if Not (fmdtTimeOnly in Mode) then
    begin
      if fmdtShortDate in Mode then
        Result := FMDateFormat[2][1]
      else
        Result := FMDateFormat[1][1];

      if fmdtYear2 in Mode then
        Result := StringReplace(Result, 'YYYY', 'YY', []);
    end;

  if Not (fmdtDateOnly in Mode) then
    begin
      if fmdtTime24 in Mode then
        if fmdtShortTime in Mode then
          tfrm := 'HH:NN'
        else
          tfrm := 'HH:NN:SS'
      else
        if fmdtShortTime in Mode then
          tfrm := 'T'
        else
          tfrm := 'TT';

      if Result <> '' then
        Result := Result + '  ' + tfrm
      else
        Result := tfrm;
    end;
end;

function CmdLineSwitch(SwLst: array of String): Boolean;
var
  swval: String;
begin
  Result := CmdLineSwitch(SwLst, swval);
end;

function CmdLineSwitch(SwLst: array of String; var SwVal: String): Boolean;
var
  i: Integer;
begin
  Result := False;
  SwVal := '';
  for i := 1 to ParamCount do
    if StartString(ParamStr(i), SwLst, False) then
      begin
        SwVal := Piece(ParamStr(i), '=', 2);
        Result := True;
        break;
      end;
end;

function FMDateTimeStr(DateTime: String; Mode: TFMDateTimeMode;
         DfltValue: String): String;
var
  buf, format: String;
  day, month, year: Word;
  hour, min, sec: Word;
  date, time: TDate;
  dateType, datePart: Integer;
begin
  if DateTime = '' then
    begin
      Result := DfltValue;
      Exit;
    end;
  Result := '';

  buf := Piece(DateTime, '.', 1);
  if( Not (fmdtTimeOnly in Mode)) and (buf <> '') then
    begin
      year  := StrToIntDef(Copy(buf, 1, 3), 0) + 1700;
      month := StrToIntDef(Copy(buf, 4, 2), 0);
      day   := StrToIntDef(Copy(buf, 6, 2), 0);

      if fmdtShortDate in Mode then
        dateType := 2
      else
        dateType := 1;

      datePart := 1;
      if day = 0 then
        begin
          day := 1;
          datePart := 2;
          if month = 0 then
            begin
              month := 1;
              datePart := 3;
            end;
        end;

      format := FMDateFormat[dateType][datePart];
      if fmdtYear2 in Mode then
        format := StringReplace(format, 'YYYY', 'YY', []);

      date := EncodeDate(year, month, day);
      Result := FormatDateTime(format, date);
    end;

  buf := Piece(DateTime, '.', 2);
  if (Not (fmdtDateOnly in Mode)) And (buf <> '') then
    begin
      buf := Copy(buf + '000000', 1, 6);

      hour := StrToIntDef(Copy(buf, 1, 2), 0);
      min  := StrToIntDef(Copy(buf, 3, 2), 0);
      sec  := StrToIntDef(Copy(buf, 5, 2), 0);

      if hour >= 24 then
        begin
          hour := 23; min := 59; sec := 59;
        end
      else if min >= 60 then
        begin
          min := 59; sec := 59;
        end
      else if sec >= 60 then
        sec := 59;

      time := EncodeTime(hour, min, sec, 0);

      if fmdtTime24 in Mode then
        if fmdtShortTime in Mode then
          format := 'HH:NN'
        else
          format := 'HH:NN:SS'
      else
        if fmdtShortTime in Mode then
          format := 'T'
        else
          format := 'TT';

      if Result <> '' then
         Result := Result + '  ' + FormatDateTime(format, time)
      else
         Result := FormatDateTime(format, time);
    end;
end;

procedure FMInitFormatArray;
var
  format: String;
begin
  FormatSettings := TFormatSettings.Create(GetThreadLocale);
  format := UpperCase(FormatSettings.ShortDateFormat);
  if Pos('M', format) > Pos('D', format) then
    begin
      FMDateFormat[1][1] := 'DD MMM YYYY';
      FMDateFormat[1][2] := 'MMM YYYY';
      FMDateFormat[1][3] := 'YYYY';

      FMDateFormat[2][1] := 'DD/MM/YYYY';
      FMDateFormat[2][2] := 'MM/YYYY';
      FMDateFormat[2][3] := 'YYYY';
    end
  else
    begin
      FMDateFormat[1][1] := 'MMM DD, YYYY';
      FMDateFormat[1][2] := 'MMM YYYY';
      FMDateFormat[1][3] := 'YYYY';

      FMDateFormat[2][1] := 'MM/DD/YYYY';
      FMDateFormat[2][2] := 'MM/YYYY';
      FMDateFormat[2][3] := 'YYYY';
    end;
end;

function InString(S: String; SubStrs: array of String;
  CaseSensitive: Boolean = True): Boolean;
var
  i: integer;
begin
  Result := False;
  for i:=0 to High(SubStrs) do
    if CaseSensitive then
      begin
        if AnsiContainsStr(S, SubStrs[i]) then
          begin
            Result := True;
            Break;
          end;
      end
    else
      begin
        if AnsiContainsText(S, SubStrs[i]) then
          begin
            Result := True;
            Break;
          end;
      end;
end;

{-----------------------------------------------------------------------
   Source:   Torry's Delphi page
   Author:   Thomas Stutz
   Homepage: http://www.swissdelphicenter.ch
 -----------------------------------------------------------------------}

function MessageDialog(const ACaption: string; const Msg: string;
  DlgType: TMsgDlgType; Buttons: TMsgDlgButtons; DefButton: Integer;
  HelpCtx: Longint): Word;
var
  i: Integer;
  btn: TButton;
begin
  with CreateMessageDialog(Msg, DlgType, Buttons) do
    try
      Caption := ACaption;
      HelpContext := HelpCtx;
      for i := 0 to ComponentCount - 1 do
      begin
        if (Components[i] is TButton) then
          begin
            btn := TButton(Components[i]);
            btn.default := btn.ModalResult = DefButton;
            if btn.default then ActiveControl := btn;
          end;
      end;
      Result := ShowModal;
    finally
      Free;
    end;
end;

{-----------------------------------------------------------------------
   Section 508 related fixes.
 -----------------------------------------------------------------------}
{$WARNINGS OFF}

function MessageDlg508(const ACaption: string; const Msg: widestring;
  DlgType: TMsgDlgType; Buttons: TMsgDlgButtons; HelpCtx: Integer): Integer;
var
  caption: PWideChar;
  uType: Integer;
begin
  uType := 0;
  case DlgType of
    mtWarning:
      begin
        caption := 'Warning';
        uType := MB_ICONWARNING;
      end;
    mtError:
      begin
        caption := 'Error';
        uType := MB_ICONERROR;
      end;
    mtInformation:
      begin
        caption := 'Information';
        uType := MB_ICONINFORMATION;
      end;
    mtConfirmation:
      begin
        caption := 'Confirm';
        uType := MB_ICONQUESTION;
      end;
  end;
  if length(ACaption) > 0 then
    caption := PWideChar(ACaption)
  else if uType = 0 then
    caption := PWideChar(Application.Title);

  if mbYes in Buttons then
  begin
    if mbCancel in Buttons then
    begin
      uType := uType + MB_YESNOCANCEL;
    end
    else begin
      uType := uType + MB_YESNO;
    end;
  end;
  if mbOk in Buttons then
  begin
    if mbCancel in Buttons then
    begin
      uType := uType + MB_OKCANCEL;
    end
    else begin
      uType := uType + MB_OK;
    end;
  end;
  if (mbAbort in Buttons) or (mbIgnore in Buttons) then
    uType := uType + MB_ABORTRETRYIGNORE;
  if (mbRetry in Buttons) and (mbCancel in Buttons) then
    uType := uType + MB_RETRYCANCEL + MB_DEFBUTTON2;
  uType := uType + MB_APPLMODAL + MB_TOPMOST;
  Result := Windows.MessageBox(Application.Handle, PWideChar(Msg), caption, uType);
end;
{$WARNINGS ON}

function MessageDlg(const Msg: string; DlgType: TMsgDlgType;
  Buttons: TMsgDlgButtons; HelpCtx: Longint): Integer;
begin
  Result := MessageDlg508('', Msg, DlgType, Buttons, HelpCtx);
end;

procedure Showmessage(const Msg: widestring);
begin
{$WARNINGS OFF}
  Windows.MessageBox(Application.Handle, PWideChar(Msg), PWideChar(Application.Title), MB_OK);
{$WARNINGS ON}
end;

function StartString(S: String; SubStrs: array of String;
  CaseSensitive: Boolean = True): Boolean;
var
  i, n: integer;
begin
  Result := False;
  n := High(SubStrs);
  if CaseSensitive then
    begin
      for i:=0 to n do
        if AnsiStartsStr(SubStrs[i], S) then
          begin
            Result := True;
            break;
          end
    end
  else
    begin
      for i:=0 to n do
        if AnsiStartsText(SubStrs[i], S) then
          begin
            Result := True;
            break;
          end;
    end;
end;

function Piece(Value, Delimiter: string; StartPiece: Integer = 1): string;
begin
  Result := Piece(Value, Delimiter, StartPiece, StartPiece);
end;

function Piece(Value, Delimiter: string; StartPiece, EndPiece: Integer): string;
var
  dlen, i, pnum: Integer;
  buf: String;
begin
  Result := '';
  if (Value <> '') And (StartPiece > 0) And (EndPiece >= StartPiece) then
    begin
      dlen := Length(Delimiter);
      i := Pos(Delimiter, Value) - 1;
      if i >= 0 then
        begin
          buf := Value;
          pnum := 1;
          repeat
            if pnum > EndPiece then
              break;
            if i < 0  then
              i := Length(buf);
            if pnum = StartPiece then
              Result := Copy(buf, 1, i)
            else if pnum > StartPiece then
              Result := Result + Delimiter + Copy(buf, 1, i);
            Delete(buf, 1, i+dlen);
            i := Pos(Delimiter, buf) - 1;
            Inc(pnum);
          until (i < 0) And (buf = '');
        end
      else if StartPiece = 1 then
        Result := Value;
    end;
end;

function StringToBoolean(const aValue: String;
  const aFormat: String = ''): Boolean;
begin
  Result := False;
  if (aFormat <> '') and (aValue = Piece(aFormat, ';')) then Result := True
  {$IFNDEF NOORPHEUS}
  else if AnsiStartsText(aValue, OvcIntlSup.TrueChar)   then Result := True
  else if AnsiStartsText(aValue, OvcIntlSup.YesChar)    then Result := True
  {$ELSE}
  else if AnsiStartsText(aValue, 'T')                   then Result := True
  else if AnsiStartsText(aValue, 'Y')                   then Result := True
  {$ENDIF}
  else if StrToIntDef(aValue, 0) <> 0                   then Result := True;
end;

initialization
  FMInitFormatArray;

end.
