unit uROR_ParamsPanel;
{$I Components.inc}

interface

uses
  Forms, SysUtils, Classes, Controls, ExtCtrls, StdCtrls, Types, Messages,
  uROR_Classes;

type
  TCustomReportParameters = class;

  TCCRPanelClearParamsEvent = procedure (const aReportCode: String;
    aList: TCustomReportParameters) of object;
  TCCRPanelGetEvent = procedure (const aReportCode: String;
    aList: TCustomReportParameters; aMsgLst: TStrings; var aField: TWinControl) of object;
  TCCRPanelSetEvent = procedure (const aReportCode: String;
    aList: TCustomReportParameters) of object;
  TCCRPanelSetupEvent = procedure (const aReportCode: String) of object;

{$REGION ' TCCRParamsPanel Class Definition '}
  TCCRParamsPanel = class(TCustomPanel)
  private
    fBevel:         TBevel;
    fCaption:       String;
    fBevelText:     TStaticText;
    fOnClearParams: TCCRPanelClearParamsEvent;
    fOnGetValues:   TCCRPanelGetEvent;
    fOnSetup:       TCCRPanelSetupEvent;
    fOnSetValues:   TCCRPanelSetEvent;
    fScreenReader:  TCCRSingleton;

  protected
    procedure AdjustClientRect(var aRect: TRect); override;
    procedure AlignControls(aControl: TControl; var aRect: TRect); override;
    procedure CreateHandle; override;
    procedure DoClearParams(const aReportCode: String;
      aList: TCustomReportParameters); virtual;
    procedure Paint; override;
    procedure Resize; override;
    procedure setCaption(const aValue: String); virtual;
    procedure SetEnabled(Value: Boolean); override;
    procedure SetName(const Value: TComponentName); override;
    procedure updateHeader(const aCaption: String); virtual;

  public
    constructor Create(anOwner: TComponent); override;
    destructor  Destroy; override;

    function  GetFieldValues(const aReportCode: String;
      aList: TCustomReportParameters; aMsgLst: TStrings): TWinControl;
    procedure SetFieldValues(const aReportCode: String; aList: TCustomReportParameters);
    procedure Setup(const aReportCode: String); virtual;

  published
    property Align;
    property Alignment default taLeftJustify;
    property Anchors;
    //property AutoSize;
    //property BevelInner;
    //property BevelOuter;
    //property BevelWidth;
    property BiDiMode;
    //property BorderWidth;
    //property BorderStyle;
    property Color;
    property Constraints;
    //property Ctl3D;
    property UseDockManager default True;
    property DockSite;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property FullRepaint;
    property Font;
    property Locked;
    property ParentBiDiMode;
    {$IFDEF VERSION7}
    property ParentBackground;
    {$ENDIF}
    property ParentColor;
    //property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnCanResize;
    property OnClick;
    property OnConstrainedResize;
    property OnContextPopup;
    property OnDockDrop;
    property OnDockOver;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnGetSiteInfo;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;
    property OnStartDock;
    property OnStartDrag;
    property OnUnDock;

    property Caption: String                        read   fCaption
                                                    write  setCaption;

    property OnClearParams: TCCRPanelClearParamsEvent read fOnClearParams
                                                    write  fOnClearParams;

    property OnGetValues: TCCRPanelGetEvent         read   fOnGetValues
                                                    write  fOnGetValues;

    property OnSetup: TCCRPanelSetupEvent           read   fOnSetup
                                                    write  fOnSetup;

    property OnSetValues: TCCRPanelSetEvent         read   fOnSetValues
                                                    write  fOnSetValues;

  end;
{$ENDREGION}

  TCustomReportParameters = class(TPersistent)
  end;

implementation

uses
  Graphics, uROR_Resources;

type

{$REGION ' TParamsLabel Class Definition '}
  {================================ TParamsLabel ===============================
    Overview:     Text used as a heading in the panel
    Description:
      Used by the panel as a heading.
  }
  TParamsLabel = class(TStaticText)
  protected
    procedure SetName(const Value: TComponentName); override;
  end;
{$ENDREGION}

//////////////////////////////// TCCRParamsPanel \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRParamsPanel.Create(anOwner: TComponent);
var
  tReader: TCCRScreenReader;
begin
  inherited;

  tReader       := TCCRScreenReader.Create;
  fScreenReader := tReader;

  Alignment  := taLeftJustify;
  BevelInner := bvNone;
  BevelOuter := bvNone;
  Ctl3D      := False;

  fBevel     := nil;
  fCaption   := '';
  fBevelText := nil;
end;

destructor TCCRParamsPanel.Destroy;
begin
  FreeAndNil(fScreenReader);
  inherited;
end;

procedure TCCRParamsPanel.AdjustClientRect(var aRect: TRect);
begin
  inherited;
  if Assigned(fBevelText) then
    aRect.Top := fBevelText.Height;
end;

procedure TCCRParamsPanel.AlignControls(aControl: TControl; var aRect: TRect);
begin
  if Assigned(fBevelText) then
    aRect.Top := fBevelText.Height;
  inherited;
end;

procedure TCCRParamsPanel.CreateHandle;
begin
  inherited;
  Resize;
end;

procedure TCCRParamsPanel.DoClearParams(const aReportCode: String;
  aList: TCustomReportParameters);
begin
  if Assigned(OnClearParams) then
    OnClearParams(aReportCode, aList);
end;

function TCCRParamsPanel.GetFieldValues(const aReportCode: String;
  aList: TCustomReportParameters; aMsgLst: TStrings): TWinControl;
begin
  Result := nil;
  if not Enabled then
    DoCLearParams(aReportCode, aList)
  else if Assigned(OnGetValues) then
    OnGetValues(aReportCode, aList, aMsgLst, Result);
end;

procedure TCCRParamsPanel.Paint;
var
  saveStyle: TPenStyle;
begin
  inherited;

  if not Assigned(fBevelText) and (csDesigning in ComponentState) then
    with Canvas do
      begin
        MoveTo(0, 0);
        saveStyle := Pen.Style;
        Pen.Style := psDash;
        LineTo(Self.ClientWidth, 0);
        Pen.Style := saveStyle;
      end;
end;

procedure TCCRParamsPanel.Resize;
const
  DX = 16;
begin
  inherited;

  if Assigned(fBevel) then
    fBevel.Width := ClientWidth;

  if Assigned(fBevelText) then
    case Alignment of
      taRightJustify:
        fBevelText.Left := ClientWidth - fBevelText.Width - DX;
      taCenter:
        fBevelText.Left := (ClientWidth - fBevelText.Width) div 2;
      else
        fBevelText.Left := DX;
    end;
end;

procedure TCCRParamsPanel.setCaption(const aValue: String);
begin
  if aValue <> fCaption then
    begin
      fCaption := aValue;
      if aValue <> '' then
        updateHeader('  ' + aValue + '  ')
      else
        updateHeader('');
    end
end;

procedure TCCRParamsPanel.SetEnabled(Value: Boolean);
begin
  inherited;
  if Assigned(fBevelText) then
    fBevelText.Enabled := Value;
end;

procedure TCCRParamsPanel.SetName(const Value: TComponentName);
begin
  inherited;
  inherited Caption := '';
end;

procedure TCCRParamsPanel.SetFieldValues(const aReportCode: String;
  aList: TCustomReportParameters);
begin
  if Assigned(OnSetValues) then
    OnSetValues(aReportCode, aList);
end;

procedure TCCRParamsPanel.Setup(const aReportCode: String);
begin
  if Assigned(OnSetup) then
    OnSetup(aReportCode);
end;

procedure TCCRParamsPanel.updateHeader(const aCaption: String);
var
  rf: Boolean;
begin

  if aCaption <> '' then
    begin
      rf := False;

      if not Assigned(fBevel) then
        begin
          rf := True;
          fBevel := TBevel.Create(Self);
          with fBevel do
            begin
              SetSubComponent(True);
              Parent := Self;

              Height   := 3;
              Left     := 0;
              Name     := 'HeaderBevel';
              Shape    := bsTopLine;
            end;
        end;

      if not Assigned(fBevelText) then
        begin
          rf := True;
          fBevelText := TParamsLabel.Create(Self);
          // NOTE:  The tab order for the text MUST be set at design time so it
          //        is the first control in the tab order. When the TabStop is
          //        set, the Height gets set to 4 so we set the Height after we
          //        set the TabStop.
          with fBevelText do
            begin
              SetSubComponent(True);
              Parent := Self;

              AutoSize := True;
              Name     := 'HeaderLabel';
              Top      := 0;

              Font.Style := Font.Style + [fsBold];
              if (not (csDesigning in ComponentState)) and CCRScreenReaderActive then
                TabStop := true;
              Height   := 16;
            end;
        end;

      if rf then
        begin
          fBevel.Top := (fBevelText.Height - fBevel.Height) div 2;
          Resize;
        end;

      fBevelText.Transparent := FALSE;
      fBevelText.Caption := aCaption;
    end
  else
    begin
      FreeAndNil(fBevel);
      FreeAndNil(fBevelText);
    end;
end;

///////////////////////////////// TParamsLabel \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TParamsLabel.SetName(const Value: TComponentName);
begin
  inherited;
  Caption := '';
end;

end.
