{******************************************************************************}
{ Package:      Clinical Case Registries Custom Components                     }
{ Date Created: November 18, 2004                                              }
{ Site Name:    Hines OIFO                                                     }
{ Developers:   Sergey Gavrilov                                                }
{ Description:  This unit defines classes for custom CCR grid.                 }
{ Note:                                                                        }
{******************************************************************************}
{ Updates:                                                                     }
{               2012. ICD-10 Remediation. vhaishandria                         }
{******************************************************************************}

unit uROR_GridView;

{$I Components.inc}

interface

uses
  ComCtrls, Controls, Classes, Graphics, Variants, uROR_Utilities,
  {$IFNDEF NOORPHEUS}OvcFiler,{$ENDIF}
  uROR_CustomListView, Messages, CommCtrl;

type
  TCCRGridItem = class;
  TCCRGridView = class;

  {=========================== TCCRColumnResizeEvent ===========================
    Overview:     Event type for the grid view column resize events.
    SeeAlso:      TCCRGridView.OnColumnResize
    Description:
      This event is generated after width of a grid view column is changed by
      the user or programmatically. See the OnColumnResize property of the
      TCCRGridView for more details.
  }
  TCCRColumnResizeEvent = procedure(aSender: TCCRGridView;
    aColumn: TListColumn) of object;

  {========================= TCCRFormatFieldValueEvent =========================
    Overview:     Event type for the grid view cell values custom formatting.
    SeeAlso:      TCCRGridView.OnFieldValueFormat
    Description:
      This event is generated when a grid view needs a formatted printable
      value of a data field. See the OnFieldValueFormat property of the
      TCCRGridView for more details.
  }
  TCCRFormatFieldValueEvent =  procedure (aSender: TObject;
    anItem: TCCRGridItem; const aFieldIndex: Integer; var aResult: String;
    var Default: Boolean) of object;

  {============================== TCCRGridDataType =============================
    Overview:     Codes of field data types supported by custom grid views.
    SeeAlso:      TCCRGridField.DataType; TCCRGridView.OnFieldValueFormat
  }
  TCCRGridDataType = (

    gftUnknown,   { This code is used internally. If you assign this type to a
                    field, then neither default conversion nor default formatting
                    will be performed. If the OnFieldValueFormat event handler is
                    not defined for the grid either, then the corresponding grid
                    column will always be empty. }

    gftString,    { This code indicates that the data field stores String
                    values. }

    gftInteger,   { This code indicates that the data field stores Integer
                    values. }

    gftDateTime,  { This code indicates that the data field stores TDateTime
                    values. }

    gftDouble,    { This code indicates that the data field stores Double
                    values. }

    gftFMDate,    { This code indicates that the data field stores date/time
                    values in FileMan internal format (YYYMMDD.HHMMSS).
                    Internally, they are stored as Double values. }

    gftBoolean,   { This code indicates that the data field stores Boolean
                    values. }

    gftMUMPS      { This code indicates that values of this data field should
                    be handled according to the M (MUMPS) rules. Internally,
                    they are stored either as String or as Double values. }

  );

  {$WARNINGS OFF}

  {=============================== TCCRFieldData ===============================
    Overview:     Internal storage for a field value.
    SeeAlso:      TCCRFieldDataArray; TCCRGridDataType; TCCRGridItem
    Description:
      Arrays of TCCRFieldData instances are used by grid view items to store
      internal field values.
  }
  TCCRFieldData = record

    { Value of a string field (gftString) or a default printable representation
      for other field types. }
    VString: String;

    { Code that indicates type of the actual value of a M data field (gftMUMPS).
      The value can be gftUnknown, gftDouble, or gftString. }
    MType: TCCRGridDataType;

    { Overlaid storage for internal values of different types. }
  case TCCRGridDataType of
    gftBoolean:  (VBoolean: Boolean);
    gftDateTime: (VDateTime: TDateTime);
    gftDouble:   (VDouble: Double);
    gftInteger:  (VInteger: Integer);

  end;

  {$WARNINGS ON}

  TCCRFieldDataArray = array of TCCRFieldData;
  PCCRFieldDataArray = ^TCCRFieldDataArray;

{$REGION ' TCCRGridField Class Definition '}
  {=============================== TCCRGridField ===============================
    Overview:     TCCRGridField represents a data field of a custom grid.
    SeeAlso:      TCCRGridFields; TCCRGridView; TListColumn
    Description:
      The TCCRGridField class encapsulates details of a data field definition.
      By default, all fields are visible and are associated with the TListColumn
      objects that represent columns of the grid view.
  }
  TCCRGridField = class(TCollectionItem)
  private

    fAlignment:    TAlignment;
    fAllowResize:  Boolean;
    fAllowSort:    Boolean;
    fAutoSize:     Boolean;
    fCaption:      String;
    fColIndex:     Integer;
    fDataType:     TCCRGridDataType;
    fFMDTOptions:  TFMDateTimeMode;
    fFormat:       String;
    fMaxWidth:     Integer;
    fMinWidth:     Integer;
    fTag:          Longint;
    fVisible:      Boolean;
    fWidth:        Integer;

    function  GetGridView: TCCRGridView;

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specialized AssignTo.
      SeeAlso:      TCCRGridField.Assign; TPersistent.AssignTo
      Keywords:     AssignTo,TCCRGridField
      Description:
        Do not call protected AssignTo method. The Assign method of a persistent
        object calls AssignTo if it is passed the TCCRGridField as a Source and
        the persistent object does not know how to copy the properties of the
        field definition object. The <i>aDest</i> parameter is the persistent
        object that should have its properties copied from the field definition.
    }
    procedure AssignTo(aDest: TPersistent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Keywords:     GetDisplayName,TCCRGridField
    }
    function  GetDisplayName: String; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Checks if the column index is valid.
      Description:
        IsValidColumn returns True if the zero-based column index passed via
        the <i>aColumnIndex</i> parameter is valid.
    }
    function IsValidColumn(const aColumnIndex: Integer): Boolean;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the Alignment property.
      SeeAlso:      TCCRGridField.Alignment
      Keywords:     SetAlignment,TCCRGridField
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
    }
    procedure SetAlignment(const aValue: TAlignment); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the AutoSize property.
      SeeAlso:      TCCRGridField.AutoSize
      Keywords:     SetAutoSize,TCCRGridField
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
    }
    procedure SetAutoSize(const aValue: Boolean); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the Caption property.
      SeeAlso:      TCCRGridField.Caption
      Keywords:     SetCaption,TCCRGridField
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
    }
    procedure SetCaption(const aValue: String); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the DataType property.
      SeeAlso:      TCCRGridField.DataType
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
    }
    procedure SetDataType(const aValue: TCCRGridDataType); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the FMDTOptions property.
      SeeAlso:      TCCRGridField.FMDTOptions
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
    }
    procedure SetFMDTOptions(const aValue: TFMDateTimeMode); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the Format property.
      SeeAlso:      TCCRGridField.Format
      Keywords:     SetFormat,TCCRGridField
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
    }
    procedure SetFormat(const aValue: String); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the MaxWidth property.
      SeeAlso:      TCCRGridField.MaxWidth
      Keywords:     SetMaxWidth,TCCRGridField
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
    }
    procedure SetMaxWidth(const aValue: Integer); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the MinWidth property.
      SeeAlso:      TCCRGridField.MinWidth
      Keywords:     SetMinWidth,TCCRGridField
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
    }
    procedure SetMinWidth(const aValue: Integer); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the Tag property.
      SeeAlso:      TCCRGridField.Tag
      Keywords:     SetTag,TCCRGridField
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
    }
    procedure SetTag(const aValue: Longint); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the Visible property.
      SeeAlso:      TCCRGridField.Visible
      Keywords:     SetVisible,TCCRGridField
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
    }
    procedure SetVisible(const aValue: Boolean); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the Width property.
      SeeAlso:      TCCRGridField.Width
      Keywords:     SetWidth,TCCRGridField
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
    }
    procedure SetWidth(const aValue: Integer); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Index of the corresponding grid view column.
      SeeAlso:      TCustomListView.Columns; TCCRGridField.Visible
      Description:
        ColIndex property indicates zero-based index of the TListColumn object
        associated with a visible data field. If the Visible property of the
        field is False, then it has no corresponding column and the ColIndex
        value will be less than zero.
    }
    property ColIndex: Integer  read fColIndex  write fColIndex;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRGridView.
      Keywords:     Create,TCCRGridField
      Description:
        Create calls the inherited constructor and assigns default values to
        properties.
    }
    constructor Create(aCollection: TCollection); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Copies the properties of a grid view column or another
                    field definition.
      SeeAlso:      TCCRGridField.AssignTo
      Keywords:     Assign,TCCRGridField
      Description:
        If the <i>Source</i> parameter is a TCCRGridField object, Assign copies
        its Alignment, AllowResize, AllowSort, AutoSize, Caption, ColIndex,
        DataType, FMDTOptions, Format, MaxWidth, MinWidth, Tag, Visible, and
        Width properties.
        <p>
        If the <i>Source</i> parameter is a TListColumn object, Assign copies
        its Alignment, AutoSize, Caption, MaxWidth, MinWidth, and Width
        properties.
        <p>
        Otherwise, Assign calls the inherited method so that any object that
        copies properties to a field definition in its AssignTo method can do so.
    }
    procedure Assign(Source: TPersistent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies the grid view that contains the field definition.
      SeeAlso:      TCCRGridFields; TCCRGridView.Fields
      Keywords:     GridView,TCCRGridField
      Description:
        GridView refers to the grid view that the field definition implicitly
        (via the TCCRGridFields object) belongs to.
    }
    property GridView: TCCRGridView  read GetGridView;

  published

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies how all text is aligned within the grid column.
      SeeAlso:      TListColumn.Alignment
      Keywords:     Alignment,TCCRGridField
      Description:
        Use Alignment to specify the horizontal placement of text in the column
        associated with a data field. The text includes the Caption that appears
        at the top of the column and the text labels of individual grid items
        that appear in the column.
    }
    property Alignment: TAlignment
      read fAlignment  write SetAlignment  default taLeftJustify;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies whether the grid column can be resized by user.
      SeeAlso:      TCCRGridField.MaxWidth; TCCRGridField.MinWidth;
                    TCCRGridField.Width
      Description:
        Use AllowResize to get or set whether the width of the column associated
        with a data field can be changed by dragging its border.
    }
    property AllowResize: Boolean
      read fAllowResize  write fAllowResize  default True;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies whether the user can sort the grid by the column.
      SeeAlso:      TCustomListView.ColumnClick
      Description:
        Use AllowSort to get or set whether the user can sort the grid by
        clicking the header of the column associated with a data field.
        <p>
        Sorting can also be enabled or disabled for all columns at once by the
        ColumnClick property of the grid view.
    }
    property AllowSort: Boolean  read fAllowSort  write fAllowSort  default True;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies whether the grid column automatically sizes
                    itself to the width of its text.
      SeeAlso:      TListColumn.AutoSize
      Keywords:     AutoSize,TCCRGridField
      Description:
        Use AutoSize to get or set whether the grid column associated with a
        data field automatically sizes itself to the width of its text.
    }
    property AutoSize: Boolean  read fAutoSize  write SetAutoSize default False;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies the text that appears at the top of the column.
      SeeAlso:      TListColumn.Caption
      Keywords:     Caption,TCCRGridField
      Description:
        Use Caption to label the column associated with a data field.
    }
    property Caption: String  read fCaption  write SetCaption;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies type of internal field values.
      SeeAlso:      TCCRGridField.FMDTOptions
      Description:
        Use DataType property to get or set type of internal field values. It
        defines how they are stored, sorted, and displayed. Do not change the
        data types for already populated grid; results are unpredictable.
    }
    property DataType: TCCRGridDataType
      read fDataType  write SetDataType  default gftString;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies formatting options for FileMan date/time values.
      SeeAlso:      TCCRGridField.DataType; TCCRGridDataType
      Description:
        Use FMDTOptions to get or set formatting options for a FileMan date/time
        field (DataType = gftFMDate).
    }
    property FMDTOptions: TFMDateTimeMode
      read fFMDTOptions  write SetFMDTOptions  default fmdtShortDateTime;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies format string for field values.
      SeeAlso:      TCCRGridField.DataType; TCCRGridField.FMDTOptions;
                    TCCRGridView.OnFieldValueFormat
      Keywords:     Format,TCCRGridField
      Description:
        Use Format to get or set format string for a field. Grid view formats
        internal field values according to this string (when it needs their
        printable representations) if the OnFieldValueFormat event handler
        is not defined or it requests default formatting.

        String values (gftString) are formatted by the Format function if the
        Format property has a non-empty value. Otherwise, they are shown as is.
        <p>
        Integer values (gftInteger) are formatted by the Format function if the
        Format property has a non-empty value. Otherwise, the IntToStr function
        is used.
        <p>
        Date/time values (gftDateTime) are formatted by the FormatDateTime
        function.
        <p>
        Double values (gftDouble) are formatted by the Format function.
        <p>
        Value of the Format property is ignored for FileMan date/time fields
        (gftFMDate). They are formatted by the FMDateTimeStr function according
        to the value of the FMDTOptions property.
        <p>
        Format property of a Boolean field (gftBoolean) can contain printable
        representations of internal True and False values separated by
        semicolon (e.g. 'Yes;No'). Otherwise, if CCR custom components were
        complied with the OrpheusLite support (default), then Windows regional
        values are used . If usage of OrpheusLite components was disabled by
        the NOORPHEUS symbol, then default 'T;F' string is used.
        <p>
        Formatting of an M value (gftMUMPS) depends on its actual type. String
        values are formatted by the Format function if the Format property has
        a non-empty value. Otherwise, they are shown as is. Numeric values are
        converted to strings by the IntToStr function and then treated in the
        same way as string values.
    }
    property Format: String  read fFormat  write SetFormat;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies the maximum width of the grid column.
      SeeAlso:      TCCRGridField.AllowResize; TCCRGridField.MinWidth;
                    TCCRGridField.Visible
      Keywords:     MaxWidth,TCCRGridField
      Description:
        Use MaxWidth to get or set the maximum width, in pixels, for the column
        associated with a data field. If MaxWidth = MinWidth, the column cannot
        be resized at runtime. Setting MaxWidth to 0 is an alternate way to
        hide a column.
    }
    property MaxWidth: Integer  read fMaxWidth  write SetMaxWidth  default 0;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies the minimum width of the grid column.
      SeeAlso:      TCCRGridField.AllowResize; TCCRGridField.MaxWidth
      Keywords:     MinWidth,TCCRGridField
      Description:
        Use MinWidth to get or set the minimum width, in pixels, for the column
        associated with a data field. If MinWidth = MaxWidth, the column cannot
        be resized at runtime.
    }
    property MinWidth: Integer  read fMinWidth  write SetMinWidth  default 0;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Stores an integer value as part of a field definition.
      Keywords:     Tag,TCCRGridField
      Description:
        Tag has no predefined meaning. The Tag property is provided for the
        convenience of developers. It can be used for storing an additional
        integer value or it can be typecast to any 32-bit value such as a
        component reference or a pointer.
    }
    property Tag: Longint  read fTag  write SetTag  default 0;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies whether a field has corresponding grid column.
      SeeAlso:      TCCRGridField.MaxWidth; TCCRGridView.ColumnIndex;
                    TCCRGridView.FieldIndex
      Keywords:     Visible,TCCRGridField
      Description:
        Use Visible to define whether a field has the corresponding grid view
        column. Setting MaxWidth to 0 is an alternate way to hide a column.
        <p>
        When a field is hidden, column indexes become different from field
        indexes for the following fields. Always use the ColumnIndex and
        FieldIndex methods of the grid view to get the column index for a field
        and vice versa.
    }
    property Visible: Boolean  read fVisible  write SetVisible  default True;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies the width of the grid column.
      SeeAlso:      TCCRGridField.AllowResize; TCCRGridField.MaxWidth;
                    TCCRGridField.MinWidth; TListColumn.Width
      Keywords:     Width,TCCRGridField
      Description:
        Use Width property to get or set the width of the column associated
        with a data field, in pixels. Grid columns can also be resized at
        runtime by dragging their borders.
    }
    property Width: Integer  read fWidth  write SetWidth  default 50;

  end;
{$ENDREGION}

  TCCRGridFieldClass = class of TCCRGridField;

{$REGION ' TCCRGridFields Class Definition '}
  {=============================== TCCRGridFields ==============================
    Overview:     TCCRGridFields is a collection of grid field definitions.
    SeeAlso:      TCustomListView.Columns; TCCRGridField; TCCRGridView.Fields
    Description:
      TCCRGridFields represents a collection of field definitions in a grid
      view. At design time, use the grid view controls Fields editor to add,
      remove, or modify fields.
  }
  TCCRGridFields = class(TOwnedCollection)
  private

    function  GetItem(anIndex: Integer): TCCRGridField;
    function  GetGridView: TCCRGridView;

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Responds when items are added to or removed from the
                    collection.
      SeeAlso:      TCCRGridView.Fields; TCollection.Notify;
                    TCustomListView.Columns
      Keywords:     Notify,TCCRGridFields
      Description:
        Notify is called automatically when the items in the collection change.
        <i>anItem</i> is the item that was just added to or that is about to be
        removed from the collection. It can be safely typecasted to
        TCCRGridField. <i>anAction</i> indicates whether item was added, is
        about to be removed, or is about to be deleted.
        <p>
        TCCRGridFields overrides this method in order to update the collection
        of columns (see the Columns property of the TCustomListView) according to
        changes to the field collection (see the Fields property of the
        TCCRGridView).
    }
    procedure Notify(anItem: TCollectionItem;
      anAction: TCollectionNotification); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Updates the collection to reflect changes to its items.
      SeeAlso:      TCCRGridView.Fields; TCollection.Update
                    TCustomListView.Columns
      Keywords:     Update,TCCRGridFields
      Description:
        Override Update in a descendant class to make any necessary changes
        when the items in the collection change. This method is called
        automatically when an update is complete.
        <p>
        <i>anItem</i> identifies the item that changed. It can be safely
        typecasted to TCCRGridField. If the <i>Item</i> parameter is nil, then
        the change affects more than one item in the collection.
        <p>
        TCCRGridFields overrides this method in order to update columns (see
        the Columns property of the TCustomListView) associated with modified
        field definitions (see the Fields property of the TCCRGridView).
    }
    procedure Update(anItem: TCollectionItem); override;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes a TCCRGridFields instance.
      SeeAlso:      TCCRGridField; TCollection.Add
      Keywords:     Create,TCCRGridFields
      Description:
        The Create method takes two parameters: the name of an instance object
        descended from TCCRGridView and the name of a TCCRGridField descendant
        class. The first parameter is the grid view that owns the field
        definitions. The second parameter determines the class of the items
        created by the Add method. By default, instances of the TCCRGridField
        are created.
    }
    constructor Create(anOwner: TCCRGridView;
      anItemClass: TCCRGridFieldClass = nil);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the column object referenced by the provided index.
      SeeAlso:      TCustomListView.Columns
      Description:
        Use GetColumn to get the column object referenced by the zero-based
        index passed via the <i>aColumnIndex</i> parameter. If the field
        collection does not have an owner (a grid view) or an invalid index
        is passed, the function returns nil.
    }
    function GetColumn(aColumnIndex: Integer): TListColumn;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies the grid view that owns the field collection.
      SeeAlso:      TCCRGridView.Fields
      Keywords:     GridView,TCCRGridFields
      Description:
        GridView refers to the grid view that field definitions belongs to.
    }
    property GridView: TCCRGridView  read GetGridView;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Lists field definitions items in the collection.
      SeeAlso:      TCollection.Count
      Keywords:     Items,TCCRGridFields
      Description:
        Use Items to access individual field definitions in the collection.
        The value of the <i>anIndex</i> parameter corresponds to the Index
        property of TCCRGridField. It represents the position of the field
        definition in the collection.
        <p>
        Items is the default property. So, if Fields references an instance of
        the TCCRGridFields, then you can access the first field either as
        Fields.Items[0] or as Fields[0].
    }
    property Items[anIndex: Integer]: TCCRGridField  read GetItem;  default;

  end;
{$ENDREGION}

{$REGION ' TCCRGridItem Class Definition '}
  {================================ TCCRGridItem ===============================
    Overview:     TCCRGridItem represents an item of the custom grid view.
    SeeAlso:      TCCRGridItems; TCCRGridView;
    Description:
      The TCCRGridItem encapsulates functionality of a grid view item that has
      storage for internal field values. Usually, each item stores the same
      number of data fields.
      <p>
      Internal value of each field is stored according to the field type:
      gftBoolean - as Boolean, gftDateTime - as TDateTime, gftDouble and
      gftFMDate - as Double, gftInteger - as Integer, and gftString - as String.
      An M (MUMPS) type value is stored either as Double or as String according
      to the actual value type that is determined by the As...[] property
      that was used to assign the value.
  }
  TCCRGridItem = class(TCCRCustomListItem)
  private

    fFieldData: array of TCCRFieldData;

    function  GetAsBoolean(aFieldIndex: Integer): Boolean;
    function  GetAsDateTime(aFieldIndex: Integer): TDateTime;
    function  GetAsDouble(aFieldIndex: Integer): Double;
    function  GetAsInteger(aFieldIndex: Integer): Integer;
    function  GetAsString(aFieldIndex: Integer): String;
    function  GetFormatted(aFieldIndex: Integer): String;
    function  GetListView: TCCRGridView;

    procedure SetAsBoolean(aFieldIndex: Integer; const aValue: Boolean);
    procedure SetAsDateTime(aFieldIndex: Integer; const aValue: TDateTime);
    procedure SetAsDouble(aFieldIndex: Integer; const aValue: Double);
    procedure SetAsInteger(aFieldIndex: Integer; const aValue: Integer);
    procedure SetAsString(aFieldIndex: Integer; const aValue: String);

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns formatted printable representation of field value.
      SeeAlso:      TCCRGridField.DataType; TCCRGridField.FMDTOptions;
                    TCCRGridField.Format; TCCRGridView.OnFieldValueFormat
      Description:
        FormatFieldValue converts and/or formats the value of the field that is
        referenced by the zero-based index passed via the <i>aFieldIndex</i>
        parameter, and returns the result.
        <p>
        If the OnFieldValueFormat event handler of the grid view is defined,
        then it is called to convert and/or format the value.
        <p>
        If the handler is not defined or it requests default processing, the
        FormatFieldValue formats the value according to the field type
        (DataType), date/time formatting options (FMDTOptions), and format
        string (Format) stored in the field definition.
    }
    function FormatFieldValue(aFieldIndex: Integer): String;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns internal field value and its default printable
                    representation.
      SeeAlso:      TCCRCustomListView.GetFieldValue
      Keywords:     GetFieldValue,TCCRGridItem
      Description:
        The GetFieldValue is called by a grid view to get internal field values
        of an item and their default printable counterparts. TCCRGridItem
        overrides this method to implement internal storage for field values.
        <p>
        Zero-based index of the field is passed via the <i>aFieldIndex</i>
        input parameter. Internal value of the field is returned via the
        <i>anInternalValue</i> output parameter. Printable field value is
        returned as the function result.
    }
    function GetFieldValue(const aFieldIndex: Integer;
      var anInternalValue: Variant): String; override;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRGridItem.
      SeeAlso:      TObject.Free
      Keywords:     Destroy,TCCRGridItem
      Description:
        Do not call Destroy directly. Instead, use Delete, which removes the
        item from the grid view window before freeing the associated memory.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Copies the properties of a grid item.
      SeeAlso:      TListItem.Assign
      Keywords:     Assign,TCCRGridItem
      Description:
        TCCRGridItem overrides the Assign method in order to copy internal
        field values (in addition to properties copied by the inherited method
        of the TListItem).
    }
    procedure Assign(Source: TPersistent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Parses the string and assigns field values.
      SeeAlso:      Piece; TCCRGridItem.GetRawData;
                    TCCRGridItems.AppendRawData; TCCRGridItems.AssignRawData
      Keywords:     AssignRawData,TCCRGridItem
      Description:
        AssignRawData parses the <i>RawData</i> value for pieces separated by
        the <i>Separator</i> (using the Piece function) and assigns pieces
        referenced by elements of the <i>anIndexList</i> array to the
        corresponding fields. If the optional <i>Separator</i> parameter is
        omitted, then the default '^' separator is used.
        <p>
        Elements of the zero-based <i>anIndexList</i> array correspond to the
        grid item fields. Element values are positions of the data pieces in
        the <i>RawData</i> string. If a value is not greater than 0, then the
        corresponding field is not changed. If the number of elements in
        the <i>anIndexList</i> array is less than the number of fields, then
        remaining fields are not changed. If the number of elements is greater
        than number of fields, then remaining data pieces are ignored.
        <p>
        For example, if the grid item gi has 3 fields, then
        gi.AssignRawData('ABC^DEF^GHI', [1,-1,2]) will assign 'ABC' to the first
        field (index=0) and 'DEF' to the last field (index=2).
    }
    procedure AssignRawData(const RawData: String; anIndexList: array of Integer;
      const Separator: String = '^');
    // ROR*1.5*19 2012-01-09 adding default value of system
    procedure AssignRawDataDef(const RawData: String;
      anIndexList: array of Integer; const Separator: String;
      const Default:String; const DefaultID: Integer =-99);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns code of the field data type.
      SeeAlso:      TCCRGridField.DataType; TCCRGridView.Fields
      Description:
        GetDataType returns the data type code of the grid field referenced by
        the zero-based index passed via the <i>aFieldIndex</i> parameter.
        If the item is not associated with the grid view or an invalid index is
        passed, the function returns gftUnknown.
    }
    function GetDataType(aFieldIndex: Integer): TCCRGridDataType;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the raw data string built from the field values.
      SeeAlso:      TCCRGridItem.AssignRawData; TCCRGridItems.GetRawData
      Keywords:     GetRawData,TCCRGridItem
      Description:
        GetRawData gets field values referenced by elements of the
        <i>anIndexList</i> array and concatenates them into the raw data string.
        Data pieces are separated by the string passed via the optional
        <i>Separator</i> parameter. If this parameter is omitted, then the
        default '^' separator is used.
        <p>
        Elements of the zero-based <i>anIndexList</i> array correspond to the
        data pieces of the raw data string (0->Piece1, 1->Piece2, ...). Element
        values are zero-based field indexes. If a value is less than 0 or not
        less than the number of fields, then an empty string is assigned to the
        corresponding data piece of the result.
        <p>
        For example, if the grid item gi has 3 fields that have 'ABC', 'DEF',
        and 'GHI' values, then the gi.GetRawData([1,-1,2,0], '*') will return
        the following string: 'DEF**GHI*ABC'.
    }
    function GetRawData(anIndexList: array of Integer;
      const Separator: String = '^'): String;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Updates StringValues according to corresponding internal
                    field values.
      SeeAlso:      TCCRCustomListItem.StringValues;
                    TCCRCustomListItem.UpdateStringValues
      Keywords:     UpdateStringValues,TCCRGridItem
      Description:
        UpdateStringValues method is called internally (e.g. by the EndUpdate)
        in order to update element(s) of the StringValues property with new
        formatted field value(s). TCCRGridItem overrides this method to adapt
        it to the internal field data storage of grid items. This method also
        updates default printable representation of the field(s).
        <p>
        If a valid field index is passed as the value of the optional
        <i>aFieldIndex</i> parameter, then only the column that corresponds
        to this field is updated. Otherwise, the whole StringValues array is
        updated.
    }
    procedure UpdateStringValues(const aFieldIndex: Integer); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Boolean representation of an internal field value.
      SeeAlso:      TCCRGridItem.AsDateTime; TCCRGridItem.AsDouble;
                    TCCRGridItem.AsInteger; TCCRGridItem.AsString
      Keywords:     AsBoolean,TCCRGridItem
      Description:
        Use AsBoolean to get or set Boolean value of the field referenced by
        the <i>anIndex</i>. If an invalid field index is passed, then this
        property returns False value (get) or ignores the assignment (set).

        The following rules are used to convert an internal field value into
        the Boolean result:
        <p>gftBoolean  - As is.
        <p>gftDateTime - True if the value is greater than 0, otherwise False.
        <p>gftDouble   - True if the value is not zero, otherwise False.
        <p>gftFMDate   - True if the value is greater than 0, otherwise False.
        <p>gftInteger  - True if the value is not zero, otherwise False.
        <p>gftMUMPS    - Treat it either as String or as Double according to
                         the actual value type.
        <p>gftString   - Result of the StrToBoolDef function (False if the
                         conversion fails).

        The following rules are used to assign a Boolean value to a grid field:
        <p>gftBoolean  - As is.
        <p>gftDateTime - EConvertError exception is raised.
        <p>gftDouble   - 1 for True or 0 for False.
        <p>gftFMDate   - EConvertError exception is raised.
        <p>gftInteger  - 1 for True or 0 for False.
        <p>gftMUMPS    - Treat it as Double and set the actual value type to
                         gftDouble.
        <p>gftString   - Result of the BoolToStr function.
    }
    property AsBoolean[anIndex: Integer]: Boolean
      read GetAsBoolean  write SetAsBoolean;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     TDateTime representation of an internal field value.
      SeeAlso:      TCCRGridItem.AsBoolean; TCCRGridItem.AsDouble;
                    TCCRGridItem.AsInteger; TCCRGridItem.AsString
      Keywords:     AsDateTime,TCCRGridItem
      Description:
        Use AsDateTime to get or set TDateTime value of the field referenced by
        the <i>anIndex</i>. If an invalid field index is passed, then this
        property returns 0 (get) or ignores the assignment (set).

        The following rules are used to convert an internal field value into
        the TDateTime result:
        <p>gftBoolean  - EConvertError exception is raised.
        <p>gftDateTime - As is.
        <p>gftDouble   - Typecast to TDateTime.
        <p>gftFMDate   - Typecast to TDateTime.
        <p>gftInteger  - Typecast to TDateTime.
        <p>gftMUMPS    - Treat it either as String or as Double according to
                         the actual value type.
        <p>gftString   - Result of the StrToFloatDef function
                         (0 if the conversion fails).
        <p>
        If the result is less than 0, then the EConvertError exception is
        raised.

        The following rules are used to assign a TDateTime value to a grid
        field:
        <p>gftBoolean  - True if not zero, otherwise - False.
        <p>gftDateTime - As is.
        <p>gftDouble   - Typecast to Double and assign the result.
        <p>gftFMDate   - As is.
        <p>gftInteger  - Result of the Trunc function.
        <p>gftMUMPS    - Treat it as Double and set the actual value type to
                         gftDouble.
        <p>gftString   - Result of the FloatToStr function.
    }
    property AsDateTime[anIndex: Integer]: TDateTime
      read GetAsDateTime  write SetAsDateTime;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Double representation of an internal field value.
      SeeAlso:      TCCRGridItem.AsBoolean; TCCRGridItem.AsDateTime;
                    TCCRGridItem.AsInteger; TCCRGridItem.AsString
      Keywords:     AsDouble,TCCRGridItem
      Description:
        Use AsDouble to get or set Double value of the field referenced by
        the <i>anIndex</i>. If an invalid field index is passed, then this
        property returns 0 (get) or ignores the assignment (set).

        The following rules are used to convert an internal field value into
        the Double result:
        <p>gftBoolean  - 0 for False or 1 for True.
        <p>gftDateTime - Typecast to Double.
        <p>gftDouble   - As is.
        <p>gftFMDate   - As is.
        <p>gftInteger  - Typecast to Double.
        <p>gftMUMPS    - Treat it either as String or as Double according to
                         the actual value type.
        <p>gftString   - Result of the StrToFloatDef function
                         (0 if the conversion fails).

        The following rules are used to assign a Double value to a grid field:
        <p>gftBoolean  - True if not zero, otherwise - False.
        <p>gftDateTime - Typecast to TDateTime and assign the result.
        <p>gftDouble   - As is.
        <p>gftFMDate   - As is.
        <p>gftInteger  - Result of the Trunc function.
        <p>gftMUMPS    - Treat it as Double and set the actual value type to
                         gftDouble.
        <p>gftString   - Result of the FloatToStr function.
    }
    property AsDouble[anIndex: Integer]: Double
      read GetAsDouble  write SetAsDouble;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Integer representation of an internal field value.
      SeeAlso:      TCCRGridItem.AsBoolean; TCCRGridItem.AsDateTime;
                    TCCRGridItem.AsDouble; TCCRGridItem.AsString
      Keywords:     AsInteger,TCCRGridItem
      Description:
        Use AsInteger to get or set Integer value of the field referenced by
        the <i>anIndex</i>. If an invalid field index is passed, then this
        property returns 0 (get) or ignores the assignment (set).

        The following rules are used to convert an internal field value into
        the Integer result:
        <p>gftBoolean  - 0 for False or 1 for True.
        <p>gftDateTime - Result of the Trunc function.
        <p>gftDouble   - Result of the Trunc function.
        <p>gftFMDate   - Result of the Trunc function.
        <p>gftInteger  - As is.
        <p>gftMUMPS    - Treat it either as String or as Double according to
                         the actual value type.
        <p>gftString   - Result of the StrToIntDef function
                         (0 if the conversion fails).

        The following rules are used to assign a Integer value to a grid field:
        <p>gftBoolean  - True if not zero, otherwise - False.
        <p>gftDateTime - Typecast to TDateTime and assign the result.
        <p>gftDouble   - Typecast to Double and assign the result.
        <p>gftFMDate   - Typecast to Double and assign the result.
        <p>gftInteger  - As is.
        <p>gftMUMPS    - Treat it as Double and set the actual value type to
                         gftDouble.
        <p>gftString   - Result of the IntToStr function.
    }
    property AsInteger[anIndex: Integer]: Integer
      read GetAsInteger  write SetAsInteger;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     String representation of an internal field value.
      SeeAlso:      TCCRGridItem.AsBoolean; TCCRGridItem.AsDateTime;
                    TCCRGridItem.AsDouble; TCCRGridItem.AsInteger
      Keywords:     AsString,TCCRGridItem
      Description:
        Use AsString to get or set String value of the field referenced by
        the <i>anIndex</i>. If an invalid field index is passed, then this
        property returns empty string (get) or ignores the assignment (set).

        The following rules are used to convert an internal field value into
        the String result:
        <p>gftBoolean  - Result of the BoolToStr function.
        <p>gftDateTime - Result of the FloatToStr function.
        <p>gftDouble   - Result of the FloatToStr function.
        <p>gftFMDate   - Result of the FloatToStr function.
        <p>gftInteger  - Result of the IntToStr function.
        <p>gftMUMPS    - Treat it either as String or as Double according to
                         the actual value type.
        <p>gftString   - As is.

        The following rules are used to assign a String value to a grid field:
        <p>gftBoolean  - Result of the StrToBoolDef function
                         (False if the conversion fails).
        <p>gftDateTime - Result of the StrToDateTimeDef function
                         (0 if the conversion fails).
        <p>gftDouble   - Result of the StrToFloatDef function
                         (0 if the conversion fails).
        <p>gftFMDate   - Result of the StrToFloatDef function
                         (0 if the conversion fails).
        <p>gftInteger  - Result of the StrToIntDef function
                         (0 if the conversion fails).
        <p>gftMUMPS    - Try to convert the value to Double using the StrToFloat
                         function. In case of successful conversion (no
                         exception), store the result and set the actual value
                         type to gftDouble. Otherwise, assign the string as is
                         and set the actual value type to gftString.
        <p>gftString   - As is.
    }
    property AsString[anIndex: Integer]: String
      read GetAsString  write SetAsString;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Represent formatted field values.
      SeeAlso:      TCCRGridItem.FormatFieldValue; TCCRGridItem.StringValues
      Description:
        Use Formatted[] property to get formatted value of a field. For visible
        fields, this property has the same values as corresponding values of
        the StringValues property. For hidden fields, the FormatFieldValue
        method is called every time the property is accessed (this can cause
        performance degradation).
    }
    property Formatted[anIndex: Integer]: String  read GetFormatted;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates the grid view control that displays the item.
      SeeAlso:      TCCRGridView
      Keywords:     ListView,TCCRGridItem
      Description:
        Do not confuse the ListView that displays the item with the Owner of
        the item. The Owner is the TCCRGridViewItems object that manages the
        collection of all grid view items. The grid view is the Owner of that
        TCCRGridViewItems object.
        <p>
        TCCRGridItem redefines this property in order to typecast it to
        TCCRGridView.
    }
    property ListView: TCCRGridView  read GetListView;

  end;
{$ENDREGION}

{$REGION ' TCCRGridItems Class Definition '}
  {=============================== TCCRGridItems ===============================
    Overview:     TCCRGridItems maintains the collection of items that appear
                  in a grid view control.
    SeeAlso:      TCCRGridItem; TCCRGridView
    Description:
      Use the properties and methods of TCCRGridItems to manipulate the list of
      items displayed by a grid view control.
  }
  TCCRGridItems = class(TCCRCustomListItems)
  private

    function  GetItem(anIndex: Integer): TCCRGridItem;
    procedure SetItem(anIndex: Integer; const aValue: TCCRGridItem);

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates a new grid item and adds it to the grid view control.
      SeeAlso:      TCCRGridItems.AddItem; TCCRGridItems.Insert
      Keywords:     Add,TCCRGridItems
      Description:
        Call Add to add a new grid item to the end of the list. Add returns
        the newly created TCCRGridItem object.
        <p>
        To create an item and insert it into the beginning or middle of the
        list, use the Insert method instead.
    }
    function Add: TCCRGridItem;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Adds an item in a specified location.
      SeeAlso:      TCCRGridItems.Add; TCCRGridItems.Insert
      Keywords:     AddItem,TCCRGridItems
      Description:
        Call AddItem to add grid item referenced by the <i>anItem</i> parameter
        at any place in the list. If <i>anItem</i> is not nil, its properties
        are duplicated. <i>anIndex</i> is location of the added item; if the
        index is negative, the TCCRGridItem is appended to the end of the list.
        <p>
        AddItem returns the TCCRGridItem that was added.
    }
    function AddItem(anItem: TCCRGridItem; anIndex: Integer = -1): TCCRGridItem;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Parses the raw data and appends it to the end of the list.
      SeeAlso:      TCCRGridItems.AssignRawData
      Description:
        Essentially, the AppendRawData does the same as the AssignRawData except
        that it does not clear the list before adding new items.
        <p>
        The optional <i>numStrPerItem</i> parameter allows loading data that
        occupies more than 1 element of the <i>RawData</i> string list.
        For example, if the value of this parameter is 2, then 2 sequential
        strings will be concatenated (using provided separator) before parsing
        the data for the next grid item.
    }
    procedure AppendRawData(RawData: TStrings; anIndexList: array of Integer;
      const Separator: String = '^'; const numStrPerItem: Integer = 1);
// ROR*1.5*19 2012-01-09 Adding Default
    procedure AppendRawDataDef(RawData: TStrings; anIndexList: array of Integer;
      const Separator: String; const numStrPerItem: Integer;
      const Default:String; const DefaultID: Integer);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Parses the raw data and initializes the item list.
      SeeAlso:      TCCRGridItem.AssignRawData; TCCRGridItems.AppendRawData;
                    TCCRGridItems.GetRawData
      Keywords:     AssignRawData,TCCRGridItems
      Description:
        Firstly, the AssignRawData clears the list. Then it does the following
        for each string of the <i>RawData</i>: adds a grid item and passes the
        string to the AssignRawData method of the newly added item. See the
        TCCRGridItem.AssignRawData for more details.
    }
    procedure AssignRawData(RawData: TStrings; anIndexList: array of Integer;
      const Separator: String = '^');

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Populates a string list with raw item data.
      SeeAlso:      TCCRGridItem.GetRawData; TCCRGridItems.AssignRawData
      Keywords:     GetRawData,TCCRGridItems
      Description:
        GetRawData calls GetRawData method of each grid item and adds resulting
        string to the <i>RawData</i> string list. See the TCCRGridItem.GetRawData
        for more details.
    }
    procedure GetRawData(RawData: TStrings; anIndexList: array of Integer;
      const Separator: String = '^');

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates a new grid item and inserts it into the grid view.
      SeeAlso:      TCCRGridItems.Add; TCCRGridItems.AddItem;
      Keywords:     Insert,TCCRGridItems
      Description:
        Call Insert to insert a new TCCRGridItem object into the grid view at
        the location specified by the <i>anIndex</i>. Insert returns the grid
        item that is inserted.
    }
    function Insert(anIndex: Integer): TCCRGridItem;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Lists all grid items managed by the TCCRGridItems object.
      SeeAlso:      TCCRGridItem; TListItems.Count
      Keywords:     Item,TCCRGridItems
      Description:
        Use Item to directly access a grid item, given its position in the
        grid view. The first item has an index of 0, the second an index
        of 1, and so on.
        <p>
        Item is the default property. So, if GridItems references an instance
        of the TCCRGridItems, then you can access the first item either as
        GridItems.Item[0] or as GridItems[0].
    }
    property Item[anIndex: Integer]: TCCRGridItem
      read GetItem  write SetItem;  default;

  end;

{$ENDREGION}

{$REGION ' TCCRGridView Class Definition '}
  {================================ TCCRGridView ===============================
    Overview:     CCR grid view.
    SeeAlso:      TCCRListView
    Description:
      Use TCCRGridView to display a table on a form. This control stores
      data in internal format and provides many conversion and formatting
      options.
  }
  TCCRGridView = class(TCCRCustomListView)
  private

    fFields:             TCCRGridFields;
    fOnFieldValueFormat: TCCRFormatFieldvalueEvent;
    fOnColumnResize:     TCCRColumnResizeEvent;

    function  GetItemFocused: TCCRGridItem;
    function  GetItems: TCCRGridItems;
    function  GetSelected: TCCRGridItem;
    procedure SetItemFocused(aValue: TCCRGridItem);
    procedure SetSelected(aValue: TCCRGridItem);

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnCompare event handler.
      SeeAlso:      TCustomListView.OnCompare;
                    TCCRCustomListView.CompareItems;
                    TCCRCustomListView.SortDescending;
                    TCCRCustomListView.SortField;
      Keywords:     CompareItems,TCCRGridView
      Description:
        The CompareItems method is used internally as the OnCompare event
        handler. It is called by the AlphaSort method (the <i>Data</i> parameter
        is 0), or when the CustomSort method is called without a SortProc
        parameter (the <i>Data</i> parameter is the value of the LParam
        parameter of CustomSort). The CompareItems method compares the field
        values of grid items passed as the <i>Item1</i> and <i>Item2</i>
        parameters. Index of the field is defined by the SortField property.

        If the field value in <i>Item1</i> is the same as that in <i>Item2</i>,
        set the <i>Compare</i> parameter to 0.
        <p>
        If the field value in <i>Item1</i> is less than that in <i>Item2</i>,
        set the <i>Compare</i> parameter to a value less than 0.
        <p>
        If the field value in <i>Item1</i> is greater than that in <i>Item2</i>,
        set the <i>Compare</i> parameter to a value greater than 0.
        <p>
        Then, if the SortDescending property is True, negate the sign of the
        <i>Compare</i> parameter.

        TCCRGridView overrides this method in order to implement appropriate
        sorting for each field type:
        <p>gftBoolean  - True follows False.
        <p>gftDateTime - Numeric comparison.
        <p>gftDouble   - Numeric comparison.
        <p>gftFMDate   - Numeric comparison.
        <p>gftInteger  - Numeric comparison.
        <p>gftMUMPS    - If both values have the same type, they are compared
                         accordingly (either as numbers or as strings).
                         Otherwise, a strings follows a number.
        <p>gftString   - CompareStr function (case sensitive).
    }
    procedure CompareItems(Sender: TObject; Item1, Item2: TListItem;
      Data: Integer; var Compare: Integer); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates a grid item for the collection specified by the
                    Items property.
      SeeAlso:      TCCRGridView.Items; TCustomListView.CreateListItem
      Keywords:     CreateListItem,TCCRGridView
      Description:
        Call CreateListItem to create a TCCRGridItem object that will be added
        to the Items collection. CreateListItem creates the grid item but does
        not insert it into the collection.
    }
    function CreateListItem: TListItem; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates the TCCRGridItems object that implements the
                    Items property.
      SeeAlso:      TCCRGridView.Items; TCustomListView.CreateListItems
      Keywords:     CreateListItems,TCCRGridView
      Description:
        TCCRGridView calls CreateListItems internally to create the object that
        implements the Items property. As implemented in TCCRGridView,
        CreateListItems creates and returns a TCCRGridItems instance.
    }
    function CreateListItems: TListItems; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates the window used by the grid view.
      SeeAlso:      TCustomListView.CreateParams; TCustomListView.CreateWnd
      Keywords:     CreateWnd,TCCRGridView
      Description:
        CreateWnd creates the grid views window, based on the parameter values
        provided by the CreateParams method. CreateWnd then sends messages to
        the newly created window so that it reflects the current property
        settings of the grid view object. TCCRGridView overrides this method
        in order to update the Columns collection according to the Fields.
    }
    procedure CreateWnd; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Generates an OnColumnResize event.
      SeeAlso:      TCCRGridView.OnColumnResize
      Description:
        DoOnColumnResize is the protected implementation of the OnColumnResize
        event. Override this method to perform class-specific actions in
        response to column resizing. The <i>aColumn</i> parameter references
        the grid column that was resized.
    }
    procedure DoOnColumnResize(aColumn: TListColumn); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the value of the FieldsCount property.
      SeeAlso:      TCCRCustomListView.FieldsCount;
                    TCCRCustomListView.GetFieldsCount; TCCRGridView.Fields
      Description:
        GetFieldCount returns number of fields defined in a grid view. As
        implemented in TCCRGridView, it returns the number of items in the
        Fields array.
    }
    function GetFieldsCount: Integer; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the value of the SortColumn property.
      SeeAlso:      TCCRCustomListView.GetSortColumn;
                    TCCRCustomListView.SortField
      Keywords:     GetSortColumn,TCCRGridView
      Description:
        The GetSortColumn method returns zero-based index of the column that
        the grid view is sorted by. If the control is not sorted, then this
        method return a negative value.
        <p>
        TCCRGridView overrides this method to adapt it to hidden fields. If a
        field is hidden, it does not have a corresponding column.
    }
    function GetSortColumn: Integer; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Replaces the Fields property implementation.
      SeeAlso:      TCustomListView.Columns; TCCRGridView.Fields
      Description:
         SetFields frees the current object that implements the Fields property,
         clears the Columns collection, and assigns the new implementation.
    }
    procedure SetFields(const aValue: TCCRGridFields); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Updates Columns according to Fields.
      SeeAlso:      TCustomListView.Columns; TCCRGridView.Fields
      Description:
        UpdateColumns updates items of the Columns collection so that they
        reflect corresponding field definitions stored in the Fields collection.
        If the <i>aClrFlg</i> parameter is True, the Columns collection is
        cleared beforehand. Use this parameter if some fields were hidden or
        deleted.
    }
    procedure UpdateColumns(const aClrFlg: Boolean); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Handler of the WM_NOTIFY message.
      SeeAlso:      TCCRGridView.OnColumnResize
      Description:
        The WM_NOTIFY message informs the parent window of a control that an
        event has occurred in the control or that the control requires some
        kind of information. TCCRGridView intercepts this message to track
        column resizing and generate the OnColumnResize event.
    }
    procedure WMNotify(var Msg: TWMNotify); message WM_NOTIFY;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property Columns;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRGridView.
      SeeAlso:      TComponent.Owner; TCCRCustomListView.Create
      Keywords:     Create,TCCRGridView
      Description:
        The Create constructor assigns default values to the properties and
        creates internal objects.
    }
    constructor Create(anOwner: TComponent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRCustomListView.
      SeeAlso:      TCCRCustomListView.Destroy; TObject.Free
      Keywords:     Destroy,TCCRGridView
      Description:
        Do not call Destroy directly in an application. Instead, call Free.
        Free verifies that the control is not nil, and only then calls Destroy.
        <p>
        Applications should only free controls explicitly when the constructor
        was called without assigning an owner to the control.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns column index for the data field.
      SeeAlso:      TCCRGridView.FieldIndex; TCCRGridView.Fields;
                    TCustomListView.Columns
      Keywords:     ColumnIndex,TCCRGridView
      Description:
        Use the ColumnIndex method to get zero-based index of the column that
        is associated with a field. <i>aFieldIndex</i> is the zero-based field
        index. If the parameter has an invalid value or the field does not
        have corresponding column (hidden field), a negative number is returned.
    }
    function ColumnIndex(const aFieldIndex: Integer): Integer; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns field index for the grid column.
      SeeAlso:      TCCRGridView.ColumnIndex; TCCRGridView.Fields;
                    TCustomListView.Columns
      Keywords:     FieldIndex,TCCRGridView
      Description:
        Use the FieldIndex method to get zero-based index of the data field that
        is associated with a column. <i>aColumnIndex</i> is the zero-based
        column index. If the parameter has an invalid value, a negative number
        is returned.
    }
    function FieldIndex(const aColumnIndex: Integer;
      const SortableOnly: Boolean = False): Integer; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the next grid item after the StartItem in the
                    specified direction.
      SeeAlso:      TCCRGridView.ItemFocused; TCCRGridView.Selected
      Keywords:     GetNextItem,TCCRGridView
      Description:
        Call GetNextItem to find the next grid item after <i>StartItem</i> in
        the direction given by the <i>Direction</i> parameter. Only items in
        the state indicated by the <i>States</i> parameter are considered.
    }
    function GetNextItem(StartItem: TCCRCustomListItem;
      Direction: TSearchDirection; States: TItemStates): TCCRGridItem;

    {$IFNDEF NOORPHEUS}
    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Loads a grid layout (column sizes).
      SeeAlso:      TCCRGridView.SaveLayout; TCCRVistAStore
      Keywords:     LoadLayout,TCCRGridView
      Description:
        Call LoadLayout to restore grid view layout (saved by the SaveLayout)
        from the data storage referenced by the <i>aStorage</i> parameter.
        <i>aSection</i> specifies the data section name (a key of the registry,
        an XML node, a section of an ini file, etc.) where the layout is stored.
        <p>
        As implemented in TCCRGridView, LoadLayout restores value of the Width
        property for each field (both visible and not). Override this method in
        descendant classes to load additional layout characteristics.
        <p>
        Descendant classes that override LoadLayout should always call the
        inherited method.
    }
    procedure LoadLayout(aStorage: TOvcAbstractStore;
      const aSection: String); override;
    {$ENDIF}

    {$IFNDEF NOORPHEUS}
    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Saves a grid layout (column sizes).
      SeeAlso:      TCCRGridView.LoadLayout; TCCRVistAStore
      Keywords:     SaveLayout,TCCRGridView
      Description:
        Call SaveLayout to save a grid view layout (it can be later restored by
        the LoadLayout) to the data storage referenced by the <i>aStorage</i>
        parameter. <i>aSection</i> specifies the data section name (a key of the
        registry, an XML node, a section of an ini file, etc.) where the layout
        is saved to.
        <p>
        As implemented in TCCRGridView, SaveLayout saves value of the Width
        property for each grid field (both visible and not). Override this
        method in descendant classes to save additional layout characteristics.
        <p>
        Descendant classes that override SaveLayout should always call the
        inherited method.
    }
    procedure SaveLayout(aStorage: TOvcAbstractStore;
      const aSection: String); override;
    {$ENDIF}

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates which item, if any, has focus.
      SeeAlso:      TCCRGridView.Selected
      Keywords:     ItemFocused,TCCRGridView
      Description:
        Read ItemFocused to determine which item, if any, can be edited by the
        user. Set ItemFocused to set focus to an item in the list. When an item
        has focus, it appears surrounded by a standard focus rectangle. If no
        item has focus, ItemFocused returns nil. Only one item can have focus
        at a time.
        <p>
        TCCRGridView redefines this property to typecast it to TCCRGridItem.
    }
    property ItemFocused: TCCRGridItem
      read GetItemFocused  write SetItemFocused;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Contains the list of items displayed by the grid view.
      SeeAlso:      TCCRCustomListView.Items; TCCRGridItems
      Keywords:     Items,TCCRGridView
      Description:
        Use Items to directly access the TCCRGridItems objects that represent
        the items in the grid view.
        <p>
        TCCRGridView redefines this property to typecast it to TCCRGridItems.
    }
    property Items: TCCRGridItems  read GetItems;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates the first selected item in the grid view.
      SeeAlso:      TCCRGridView.Focused; TCCRGridView.GetNextItem;
                    TCustomListView.MultiSelect; TCustomListView.Selected;
                    TCustomListView.SelCount
      Keywords:     Selected,TCCRGridView
      Description:
        Read Selected to access the properties of the first selected item in
        the grid. If SelCount is 0, Selected is nil. If SelCount is greater
        than 1, subsequent selected items can be located by checking the
        Selected property of the items found using the GetNextItem method.
        <p>
        Set the Selected property to select an item in the list. If MultiSelect
        is true, setting Selected adds an item to the selection. If MultiSelect
        is false, setting Selected changes the selection. Setting Selected to
        nil deselects all items in the grid.
        <p>
        When an item is selected, the OnChanging and OnChange events are
        generated.
        <p>
        TCCRGridView redefines this property to typecast it to TCCRGridItem.
    }
    property Selected: TCCRGridItem  read GetSelected  write SetSelected;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property SortColumn;

  published

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property Action;
    property Align;
    property AllocBy;
    property Anchors;
    property BevelEdges;
    property BevelInner;
    property BevelKind  default bkNone;
    property BevelOuter;
    property BevelWidth;
    property BiDiMode;
    property BorderStyle;
    property BorderWidth;
    property Checkboxes;
    property Color;
    property ColumnClick;
    //property Columns;
    property Constraints;
    property Ctl3D;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property FlatScrollBars;
    property Font;
    //property FullDrag;
    property GridLines  default True;
    property HideSelection;
    property HotTrack;
    property HotTrackStyles;
    property HoverTime;
    property IconOptions;
    property LargeImages;
    property MultiSelect;
    property OwnerData;
    property OwnerDraw;
    property ParentBiDiMode;
    property ParentColor  default False;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly  default False;
    property RowSelect  default True;
    property ShowColumnHeaders  default True;
    property ShowHint;
    property ShowWorkAreas;
    property SmallImages;
    property SortDescending;  // TCCRCustomListView
    property SortField;       // TCCRCustomListView
    //property SortType;
    property StateImages;
    property TabOrder;
    property TabStop  default True;
    //property ViewStyle;
    property Visible;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property OnAdvancedCustomDraw;
    property OnAdvancedCustomDrawItem;
    property OnAdvancedCustomDrawSubItem;
    property OnChange;
    property OnChanging;
    property OnClick;
    property OnColumnClick;
    //property OnColumnDragged;
    property OnColumnRightClick;
    //property OnCompare;
    property OnContextPopup;
    property OnCustomDraw;
    property OnCustomDrawItem;
    property OnCustomDrawSubItem;
    property OnData;
    property OnDataFind;
    property OnDataHint;
    property OnDataStateChange;
    property OnDblClick;
    property OnDeletion;
    property OnDragDrop;
    property OnDragOver;
    property OnDrawItem;
    property OnEdited;
    property OnEditing;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnGetImageIndex;
    property OnGetSubItemImage;
    property OnInfoTip;
    property OnInsert;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;
    property OnSelectItem;
    property OnStartDock;
    property OnStartDrag;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Describes the properties of the fields in the grid view.
      SeeAlso:      TCustomListView.Columns; TCCRGridView.ColumnIndex;
                    TCCRGridView.FieldIndex; TCCRGridField
      Description:
        Use Fields to add or delete data fields in the grid view, or to edit
        their properties. When setting Fields at design time a GridView Field
        Editor dialog appears. This dialog allows fields to be added or deleted,
        and the properties of individual fields to be changed. Changes to field
        definitions are automatically reflected by the Columns property.
        <p>
        <b>Note:</b> Hidden fields (Visible = False) do not have corresponding
        columns. As the result, field indexes are not the same as column
        indexes. Always use ColumnIndex and FieldIndex to get index of the
        column that is associated with a field and vice versa.
    }
    property Fields: TCCRGridFields  read fFields  write SetFields;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs when a column width is changed by the user or
                    programmatically.
      Description:
        Write an OnColumnResize event handler to respond to column width
        changes. The <i>aSender</i> parameter is the grid view whose event
        handler is called. The <i>aColumn</i> parameter references the
        TListColumn object from the Columns property that describes the column
        that was resized.
    }
    property OnColumnResize: TCCRColumnResizeEvent
      read fOnColumnResize  write fOnColumnResize;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs when the grid needs printable representation of a
                    cell internal value.
      SeeAlso:      TCCRGridField.Format; TCCRGridField.FMDTOptions
      Description:
        Write an OnFieldValueFormat event handler to implement custom conversion
        and/or formatting for grid cell values. The <i>aSender</i> parameter is
        the grid view whose event handler is called. The <i>anItem</i> and
        <i>aFieldIndex</i> reference the grid item and data field index
        corresponding to the grid cell that has to be formatted.
        <p>
        Assign the formatted value to the <i>aResult</i> parameter. You can also
        assign True to the <i>Default</i> parameter in order to instruct the
        grid to ignore the <i>aResult</i> value and apply default conversion
        and formatting rules to the cell value.
    }
    property OnFieldValueFormat: TCCRFormatFieldValueEvent
      read fOnFieldValueFormat  write fOnFieldValueFormat;

  end;
{$ENDREGION}

///////////////////////////////// Implementation \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

implementation

uses
  uROR_Resources, SysUtils;

{$REGION ' TCCRGridField Methods '}
//////////////////////////////// TCCRGridField \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRGridField.Create(aCollection: TCollection);
begin
  fAlignment   := taLeftJustify;
  fAllowResize := True;
  fAllowSort   := True;
  fColIndex    := -1;
  fDataType    := gftString;
  fFMDTOptions := fmdtShortDateTime;
  fVisible     := True;
  fWidth       := 50;
  inherited;
end;

procedure TCCRGridField.Assign(Source: TPersistent);
begin
  if Source is TCCRGridField then
    with TCCRGridField(Source) do
      begin
        Self.Alignment   := Alignment;
        Self.AllowResize := AllowResize;
        Self.AllowSort   := AllowSort;
        Self.AutoSize    := AutoSize;
        Self.Caption     := Caption;
        Self.ColIndex    := ColIndex;
        Self.DataType    := DataType;
        Self.FMDTOptions := FMDTOptions;
        Self.Format      := Format;
        Self.MaxWidth    := MaxWidth;
        Self.MinWidth    := MinWidth;
        Self.Tag         := Tag;
        Self.Visible     := Visible;
        Self.Width       := Width;
      end
  else if Source is TListColumn then
    with TListColumn(Source) do
      begin
        Self.Alignment   := Alignment;
        Self.AutoSize    := AutoSize;
        Self.MaxWidth    := MaxWidth;
        Self.MinWidth    := MinWidth;
        Self.Width       := Width;
        Self.Caption     := Caption;
      end
  else
    inherited;
end;

procedure TCCRGridField.AssignTo(aDest: TPersistent);
var
  wdc: Boolean;
begin
  if aDest is TListColumn then
    with TListColumn(aDest) do
      begin
        wdc := (Self.Width <> Width);

        Alignment := Self.Alignment;
        AutoSize  := Self.AutoSize;
        MaxWidth  := Self.MaxWidth;
        MinWidth  := Self.MinWidth;
        Tag       := Self.Index;
        Width     := Self.Width;
        Caption   := Self.Caption;

        if wdc and Assigned(GridView) then
          GridView.DoOnColumnResize(TListColumn(aDest));
      end
  else
    inherited;
end;

function TCCRGridField.GetDisplayName: String;
begin
  if fCaption <> '' then
    Result := Caption
  else
    Result := inherited GetDisplayName;
end;

function TCCRGridField.GetGridView: TCCRGridView;
begin
  if Assigned(Collection) then
    Result := TCCRGridFields(Collection).GridView
  else
    Result := nil;
end;

function TCCRGridField.IsValidColumn(const aColumnIndex: Integer): Boolean;
begin
  if Assigned(GridView) then
    Result := (aColumnIndex >= 0) and (aColumnIndex < GridView.Columns.Count)
  else
    Result := False;
end;

procedure TCCRGridField.SetAlignment(const aValue: TAlignment);
begin
  if aValue <> fAlignment then
    begin
      fAlignment := aValue;
      if IsValidColumn(ColIndex) then
        GridView.Columns[ColIndex].Alignment := aValue;
      Changed(False);
    end;
end;

procedure TCCRGridField.SetAutoSize(const aValue: Boolean);
begin
  if aValue <> fAutoSize then
    begin
      fAutoSize := aValue;
      if IsValidColumn(ColIndex) then
        GridView.Columns[ColIndex].AutoSize := aValue;
      Changed(False);
    end;
end;

procedure TCCRGridField.SetCaption(const aValue: String);
begin
  if aValue <> fCaption then
    begin
      fCaption := aValue;
      if IsValidColumn(ColIndex) then
        GridView.Columns[ColIndex].Caption := aValue;
      Changed(False);
    end;
end;

procedure TCCRGridField.SetDataType(const aValue: TCCRGridDataType);
begin
  if aValue <> fDataType then
    begin
      fDataType := aValue;
      Changed(False);
    end;
end;

procedure TCCRGridField.SetFMDTOptions(const aValue: TFMDateTimeMode);
begin
  if aValue <> fFMDTOptions then
    begin
      fFMDTOptions := aValue;
      Changed(False);
    end;
end;

procedure TCCRGridField.SetFormat(const aValue: String);
begin
  if aValue <> fFormat then
    begin
      fFormat := aValue;
      Changed(False);
    end;
end;

procedure TCCRGridField.SetMaxWidth(const aValue: Integer);
begin
  if aValue <> fmaxWidth then
    begin
      fMaxWidth := aValue;
      if IsValidColumn(ColIndex) then
        GridView.Columns[ColIndex].MaxWidth := aValue;
      Changed(False);
    end;
end;

procedure TCCRGridField.SetMinWidth(const aValue: Integer);
begin
  if aValue <> fMinWidth then
    begin
      fMinWidth := aValue;
      if IsValidColumn(ColIndex) then
        GridView.Columns[ColIndex].MinWidth := aValue;
      Changed(False);
    end;
end;

procedure TCCRGridField.SetTag(const aValue: Longint);
begin
  if aValue <> fTag then
    begin
      fTag := aValue;
      Changed(False);
    end;
end;

procedure TCCRGridField.SetVisible(const aValue: Boolean);
begin
  if aValue <> fVisible then
    begin
      fVisible := aValue;
      Changed(True);
    end;
end;

procedure TCCRGridField.SetWidth(const aValue: Integer);
begin
  if aValue <> fWidth then
    begin
      fWidth := aValue;
      if IsValidColumn(ColIndex) then
        with GridView do
          begin
            Columns[ColIndex].Width := aValue;
            DoOnColumnResize(Columns[ColIndex]);
          end;
      Changed(False);
    end;
end;
{$ENDREGION}

{$REGION ' TCCRGridFields Methods '}
///////////////////////////////// TCCRGridFields \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRGridFields.Create(anOwner: TCCRGridView;
  anItemClass: TCCRGridFieldClass);
begin
  if Assigned(anItemClass) then
    inherited Create(anOwner, anItemClass)
  else
    inherited Create(anOwner, TCCRGridField);
end;

function TCCRGridFields.GetColumn(aColumnIndex: Integer): TListColumn;
begin
  Result := nil;
  if Assigned(GridView) then
    if (aColumnIndex >= 0) and (aColumnIndex < GridView.Columns.Count) then
      Result := GridView.Columns[aColumnIndex];
end;

function TCCRGridFields.GetGridView: TCCRGridView;
begin
  Result := TCCRGridView(Owner);
end;

function TCCRGridFields.GetItem(anIndex: Integer): TCCRGridField;
begin
  Result := TCCRGridField(inherited GetItem(anIndex));
end;

procedure TCCRGridFields.Notify(anItem: TCollectionItem;
  anAction: TCollectionNotification);
var
  col: TListColumn;
  fld: TCCRGridField;
begin
  inherited;
  fld := TCCRGridField(anItem);
  case anAction of
    cnAdded:
      if fld.Visible and Assigned(GridView) then
        begin
          col := GridView.Columns.Add;
          col.Assign(fld);
          fld.ColIndex := col.Index;
        end;
    cnDeleting, cnExtracting:
      if Assigned(GridView) then
        begin
          col := GetColumn(fld.ColIndex);
          if Assigned(col) then
            GridView.Columns.Delete(col.Index);
          fld.ColIndex := -1;
        end;
  end;
end;

procedure TCCRGridFields.Update(anItem: TCollectionItem);
begin
  inherited;
  if (anItem = nil) and Assigned(GridView) then
    GridView.UpdateColumns(True);
end;
{$ENDREGION}

{$REGION ' TCCRGridItem Methods '}
////////////////////////////////// TCCRGridItem \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

destructor TCCRGridItem.Destroy;
var
  i: Integer;
begin
  for i:=0 to High(fFieldData) do
    fFieldData[i].VString := '';
  SetLength(fFieldData, 0);
  fFieldData := nil;
  inherited;
end;

procedure TCCRGridItem.Assign(Source: TPersistent);
var
  i: Integer;
  si: TCCRGridItem;
begin
  inherited;
  if Source is TCCRGridItem then
    begin
      si := TCCRGridItem(Source);
      SetLength(fFieldData, High(si.fFieldData)+1);
      for i:=0 to High(fFieldData) do
        fFieldData[i] := si.fFieldData[i];
    end
end;

procedure TCCRGridItem.AssignRawData(const RawData: String;
  anIndexList: array of Integer; const Separator: String);
// 2012-01-09 ROR*1.5*19 replaced by AssigneRawDataDef =======================
//var
//  i, ip, n: Integer;
begin
(*
  n := ListView.Fields.Count - 1;
  if n > High(anIndexList) then n := High(anIndexList);
  BeginUpdate;
  try
    for i:=0 to n do
      begin
        ip := anIndexList[i];
        if ip > 0 then
          AsString[i] := Piece(RawData, Separator, ip)
        //-- Section 508: Place a blank in the first column of each row so JAWS
        //   will read the headers and data starting with the second column.
        else if (ip = -99) and (i = 0) then
          AsString[0] := ' ';
      end;
  finally
    EndUpdate;
  end;
*)
  AssignRawDataDef(RawData,anIndexList,Separator,'',-999);
end;

procedure TCCRGridItem.AssignRawDataDef(const RawData: String;
  anIndexList: array of Integer; const Separator: String;
  const Default:String; const DefaultID: Integer =-99);
var
  i, ip, n: Integer;
begin
  n := ListView.Fields.Count - 1;
  if n > High(anIndexList) then n := High(anIndexList);
  BeginUpdate;
  try
    for i:=0 to n do
      begin
        ip := anIndexList[i];
        if ip > 0 then
          AsString[i] := Piece(RawData, Separator, ip)
        //-- Section 508: Place a blank in the first column of each row so JAWS
        //   will read the headers and data starting with the second column.
        else if (ip = -99) and (i = 0) then
          AsString[0] := ' '
        else if (ip = DefaultID) and (Default<>'')  then
          AsString[i] := Default;
      end;
  finally
    EndUpdate;
  end;
end;

function TCCRGridItem.FormatFieldValue(aFieldIndex: Integer): String;
var
  dt: TCCRGridDataType;
  defaultFormat: Boolean;
begin
  Result := '';
  dt := GetDataType(aFieldIndex);
  if (dt = gftUnknown) or (aFieldIndex > High(fFieldData)) then Exit;

  with ListView do
    begin
      //--- Get the custom-formatted external value (if possible)
      if Assigned(OnFieldValueFormat) then
        begin
          defaultFormat := False;
          OnFieldValueFormat(ListView, Self, aFieldIndex, Result, defaultFormat);
        end
      else
        defaultFormat := True;
      //--- Get the default external value of the field
      if defaultFormat then
        case dt of
          gftBoolean:
            Result := BooleanToString(
                          fFieldData[aFieldIndex].VBoolean,
                          Fields[aFieldIndex].Format);
          gftDateTime:
           if fFieldData[aFieldIndex].VDateTime > 0 then
              Result := FormatDateTime(
                          Fields[aFieldIndex].Format,
                          fFieldData[aFieldIndex].VDateTime);
          gftDouble:
            Result := FormatFloat(
                          Fields[aFieldIndex].Format,
                          fFieldData[aFieldIndex].VDouble);
          gftFMDate:
            if fFieldData[aFieldIndex].VDouble > 0 then
              Result := FMDateTimeStr(
                          FloatToStr(fFieldData[aFieldIndex].VDouble),
                          Fields[aFieldIndex].FMDTOptions);
          gftInteger:
            if Fields[aFieldIndex].Format <> '' then
              Result := Format(
                          Fields[aFieldIndex].Format,
                          [fFieldData[aFieldIndex].VInteger])
            else
              Result := IntToStr(fFieldData[aFieldIndex].VInteger);
          gftString, gftMUMPS:
            if Fields[aFieldIndex].Format <> '' then
              Result := Format(
                          Fields[aFieldIndex].Format,
                          [fFieldData[aFieldIndex].VString])
            else
              Result := fFieldData[aFieldIndex].VString;
        end;
    end;
end;

function TCCRGridItem.GetAsBoolean(aFieldIndex: Integer): Boolean;
var
  dt: TCCRGridDataType;
begin
  Result := False;
  if aFieldIndex <= High(fFieldData) then
    begin
      dt := GetDataType(aFieldIndex);
      if dt = gftMUMPS then
        dt := fFieldData[aFieldIndex].MType;
      case dt of
        gftBoolean:
          Result := fFieldData[aFieldIndex].VBoolean;
        gftDateTime:
          Result := (fFieldData[aFieldIndex].VDateTime > 0);
        gftDouble:
          Result := (fFieldData[aFieldIndex].VDouble <> 0);
        gftFMDate:
          Result := (fFieldData[aFieldIndex].VDouble > 0);
        gftInteger:
          Result := (fFieldData[aFieldIndex].VInteger <> 0);
        gftString:
           Result := StrToBoolDef(fFieldData[aFieldIndex].VString, False);
      end;
    end;
end;

function TCCRGridItem.GetAsDateTime(aFieldIndex: Integer): TDateTime;
var
  dt: TCCRGridDataType;
begin
  Result := 0;
  if aFieldIndex <= High(fFieldData) then
    begin
      dt := GetDataType(aFieldIndex);
      if dt = gftMUMPS then
        dt := fFieldData[aFieldIndex].MType;
      case dt of
        gftBoolean:
            Result := -1;
        gftDateTime:
          Result := fFieldData[aFieldIndex].VDateTime;
        gftDouble, gftFMDate:
          Result := fFieldData[aFieldIndex].VDouble;
        gftInteger:
          Result := fFieldData[aFieldIndex].VInteger;
        gftString:
          Result := StrToFloatDef(fFieldData[aFieldIndex].VString, 0);
      end;
    end;
  if Result < 0 then
    raise EConvertError.Create(RSC0020);
end;

function TCCRGridItem.GetAsDouble(aFieldIndex: Integer): Double;
var
  dt: TCCRGridDataType;
begin
  Result := 0;
  if aFieldIndex <= High(fFieldData) then
    begin
      dt := GetDataType(aFieldIndex);
      if dt = gftMUMPS then
        dt := fFieldData[aFieldIndex].MType;
      case dt of
        gftBoolean:
          if fFieldData[aFieldIndex].VBoolean then
            Result := 1
          else
            Result := 0;
        gftDateTime:
          Result := fFieldData[aFieldIndex].VDateTime;
        gftDouble, gftFMDate:
          Result := fFieldData[aFieldIndex].VDouble;
        gftInteger:
          Result := fFieldData[aFieldIndex].VInteger;
        gftString:
          Result := StrToFloatDef(fFieldData[aFieldIndex].VString, 0);
      end;
    end;
end;

function TCCRGridItem.GetAsInteger(aFieldIndex: Integer): Integer;
var
  dt: TCCRGridDataType;
begin
  Result := 0;
  if aFieldIndex <= High(fFieldData) then
    begin
      dt := GetDataType(aFieldIndex);
      if dt = gftMUMPS then
        dt := fFieldData[aFieldIndex].MType;
      case dt of
        gftBoolean:
          if fFieldData[aFieldIndex].VBoolean then
            Result := 1
          else
            Result := 0;
        gftDateTime:
          Result := Trunc(fFieldData[aFieldIndex].VDateTime);
        gftDouble, gftFMDate:
          Result := Trunc(fFieldData[aFieldIndex].VDouble);
        gftInteger:
          Result := fFieldData[aFieldIndex].VInteger;
        gftString:
          Result := StrToIntDef(fFieldData[aFieldIndex].VString, 0);
      end;
    end;
end;

function TCCRGridItem.GetAsString(aFieldIndex: Integer): String;
begin
  Result := '';
  if (aFieldIndex >= 0) and (aFieldIndex <= High(fFieldData)) then
    Result := fFieldData[aFieldIndex].VString;
end;

function TCCRGridItem.GetDataType(aFieldIndex: Integer): TCCRGridDataType;
begin
  Result := gftUnknown;
  if Assigned(ListView) then
    with ListView do
      if (aFieldIndex >= 0) and (aFieldIndex < Fields.Count) then
        Result := Fields[aFieldIndex].DataType;
end;

function TCCRGridItem.GetFieldValue(const aFieldIndex: Integer;
  var anInternalValue: Variant): String;
var
  dt: TCCRGridDataType;
begin
  Result := '';
  dt := GetDataType(aFieldIndex);
  if dt = gftMUMPS then
    dt := fFieldData[aFieldIndex].MType;
  case dt of
    gftBoolean:
      begin
        anInternalValue := fFieldData[aFieldIndex].VBoolean;
        Result := fFieldData[aFieldIndex].VString;
      end;
    gftDateTime:
      begin
        anInternalValue := fFieldData[aFieldIndex].VDateTime;
        Result := fFieldData[aFieldIndex].VString;
      end;
    gftDouble, gftFMDate:
      begin
        anInternalValue := fFieldData[aFieldIndex].VDouble;
        Result := fFieldData[aFieldIndex].VString;
      end;
    gftInteger:
      begin
        anInternalValue := fFieldData[aFieldIndex].VInteger;
        Result := fFieldData[aFieldIndex].VString;
      end;
    gftString:
      begin
        anInternalValue := fFieldData[aFieldIndex].VString;
        Result := anInternalValue;
      end;
  end;
end;

function TCCRGridItem.GetFormatted(aFieldIndex: Integer): String;
begin
  if (aFieldIndex < 0) or (aFieldIndex >= ListView.Fields.Count) then
    Result := ''
  else if ListView.Fields[aFieldIndex].Visible then
    Result := StringValues[ListView.ColumnIndex(aFieldIndex)]
  else
    Result := FormatFieldValue(aFieldIndex);
end;

function TCCRGridItem.GetListView: TCCRGridView;
begin
  Result := inherited ListView as TCCRGridView;
end;

function TCCRGridItem.GetRawData(anIndexList: array of Integer;
  const Separator: String): String;
var
  i, ifld, lastFld: Integer;
begin
  Result := '';
  lastFld := ListView.Fields.Count - 1;
  for i:=0 to High(anIndexList) do
    begin
      ifld := anIndexList[i];
      if (ifld >= 0) and (ifld <= lastFld) then
        Result := Result + AsString[ifld] + Separator
      else
        Result := Result + Separator;
    end;
end;

procedure TCCRGridItem.SetAsBoolean(aFieldIndex: Integer; const aValue: Boolean);
var
  dt: TCCRGridDataType;
begin
  dt := GetDataType(aFieldIndex);
  if dt = gftUnknown then Exit;

  with ListView do
    begin
      if aFieldIndex > High(fFieldData) then
        SetLength(fFieldData, aFieldIndex+1);
      if dt = gftMUMPS then
        begin
          fFieldData[aFieldIndex].MType := gftDouble;
          dt := gftDouble;
        end;
      case dt of
        gftBoolean:
          fFieldData[aFieldIndex].VBoolean := aValue;
        gftDateTime, gftFMDate:
          raise EConvertError.Create(RSC0020);
        gftDouble:
          if aValue then
            fFieldData[aFieldIndex].VDouble := 1
          else
            fFieldData[aFieldIndex].VDouble := 0;
        gftInteger:
          if aValue then
            fFieldData[aFieldIndex].VInteger := 1
          else
            fFieldData[aFieldIndex].VInteger := 0;
        gftString:
          fFieldData[aFieldIndex].VString := BoolToStr(aValue);
      end;
      UpdateStringValues(aFieldIndex);
    end;
end;

procedure TCCRGridItem.SetAsDateTime(aFieldIndex: Integer; const aValue: TDateTime);
begin
  SetAsDouble(aFieldIndex, aValue);
end;

procedure TCCRGridItem.SetAsDouble(aFieldIndex: Integer; const aValue: Double);
var
  dt: TCCRGridDataType;
begin
  dt := GetDataType(aFieldIndex);
  if dt = gftUnknown then Exit;

  with ListView do
    begin
      if aFieldIndex > High(fFieldData) then
        SetLength(fFieldData, aFieldIndex+1);
      if dt = gftMUMPS then
        begin
          fFieldData[aFieldIndex].MType := gftDouble;
          dt := gftDouble;
        end;
      case dt of
        gftBoolean:
          fFieldData[aFieldIndex].VBoolean := (aValue <> 0);
        gftDateTime:
          fFieldData[aFieldIndex].VDateTime := aValue;
        gftDouble, gftFMDate:
          fFieldData[aFieldIndex].VDouble := aValue;
        gftInteger:
          fFieldData[aFieldIndex].VInteger := Trunc(aValue);
        gftString:
          fFieldData[aFieldIndex].VString := FloatToStr(aValue);
      end;
      UpdateStringValues(aFieldIndex);
    end;
end;

procedure TCCRGridItem.SetAsInteger(aFieldIndex: Integer; const aValue: Integer);
var
  dt: TCCRGridDataType;
begin
  dt := GetDataType(aFieldIndex);
  if dt = gftUnknown then Exit;

  with ListView do
    begin
      if aFieldIndex > High(fFieldData) then
        SetLength(fFieldData, aFieldIndex+1);
      if dt = gftMUMPS then
        begin
          fFieldData[aFieldIndex].MType := gftDouble;
          dt := gftDouble;
        end;
      case dt of
        gftBoolean:
          fFieldData[aFieldIndex].VBoolean := (aValue <> 0);
        gftDateTime:
          fFieldData[aFieldIndex].VDateTime := aValue;
        gftDouble, gftFMDate:
          fFieldData[aFieldIndex].VDouble := aValue;
        gftInteger:
          fFieldData[aFieldIndex].VInteger := aValue;
        gftString:
          fFieldData[aFieldIndex].VString := IntToStr(aValue);
      end;
      UpdateStringValues(aFieldIndex);
    end;
end;

procedure TCCRGridItem.SetAsString(aFieldIndex: Integer; const aValue: String);
var
  dt: TCCRGridDataType;
begin
  dt := GetDataType(aFieldIndex);
  if dt = gftUnknown then Exit;

  with ListView do
    begin
      if aFieldIndex > High(fFieldData) then
        SetLength(fFieldData, aFieldIndex+1);
      case dt of
        gftBoolean:
          fFieldData[aFieldIndex].VBoolean := StrToBoolDef(aValue, False);
        gftDateTime:
          fFieldData[aFieldIndex].VDateTime := StrToDateTimeDef(aValue, 0);
        gftDouble, gftFMDate:
          fFieldData[aFieldIndex].VDouble := StrToFloatDef(aValue, 0);
        gftInteger:
          fFieldData[aFieldIndex].VInteger := StrToIntDef(aValue, 0);
        gftMUMPS:
          try
            fFieldData[aFieldIndex].VDouble := StrToFloat(aValue);
            fFieldData[aFieldIndex].MType   := gftDouble;
          except
            fFieldData[aFieldIndex].VString := aValue;
            fFieldData[aFieldIndex].MType   := gftString;
          end;
        gftString:
          fFieldData[aFieldIndex].VString := aValue;
      end;
      UpdateStringValues(aFieldIndex);
    end;
end;

procedure TCCRGridItem.UpdateStringValues(const aFieldIndex: Integer);
var
  i, n: Integer;

  procedure update_field(const aFieldIndex: Integer);
  var
    dt: TCCRGridDataType;
  begin
    dt := GetDataType(aFieldIndex);
    if (dt = gftUnknown) or (aFieldIndex > High(fFieldData)) then Exit;
    //--- Update the internal string value of the field
    with fFieldData[aFieldIndex] do
      begin
        if dt = gftMUMPS then
          dt := MType;
        case dt of
          gftBoolean:
            VString := BoolToStr(VBoolean);
          gftDateTime:
            VString := FloatToStr(VDateTime);
          gftDouble, gftFMDate:
            VString := FloatToStr(VDouble);
          gftInteger:
            VString := IntToStr(VInteger);
        end;
      end;
    //--- Update the Caption or SubItem if the field is visible
    with ListView do
      if Fields[aFieldIndex].Visible then
        StringValues[ColumnIndex(aFieldIndex)] := FormatFieldValue(aFieldIndex);
  end;

begin
  if UpdateLock = 0 then
    if aFieldIndex >= 0 then
      update_field(aFieldIndex)
    else
      begin
        SubItems.BeginUpdate;
        try
          n := ListView.Fields.Count - 1;
          for i:=0 to n do
            update_field(i);
        finally
          SubItems.EndUpdate;
        end;
      end;
end;
{$ENDREGION}

{$REGION ' TCCRGridItems Methods '}
////////////////////////////////// TCCRGridItems \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

function TCCRGridItems.Add: TCCRGridItem;
begin
  Result := TCCRGridItem(inherited Add);
end;

function TCCRGridItems.AddItem(anItem: TCCRGridItem; anIndex: Integer): TCCRGridItem;
begin
  Result := TCCRGridItem(inherited AddItem(anItem, anIndex));
end;

procedure TCCRGridItems.AppendRawData(RawData: TStrings; anIndexList: array of Integer;
      const Separator: String; const numStrPerItem: Integer);
var
  i, j: Integer;
  buf: String;
  gi : TCCRGridItem;
begin
  BeginUpdate;
  try
    i := 0;
    while i < RawData.Count do
      begin
        buf := RawData[i];
        Inc(i);
        for j:=2 to numStrPerItem do
          begin
            if i >= RawData.Count then
              Break;
            buf := buf + Separator + RawData[i];
            Inc(i);
          end;
        gi := Add;
        if Assigned(gi) then
          gi.AssignRawData(buf, anIndexList, Separator);
      end;
  finally
    EndUpdate;
  end;
end;

procedure TCCRGridItems.AppendRawDataDef(RawData: TStrings; anIndexList: array of Integer;
      const Separator: String; const numStrPerItem: Integer;
      const Default:String; const DefaultID: Integer);
var
  i, j: Integer;
  buf: String;
  gi : TCCRGridItem;
begin
  BeginUpdate;
  try
    i := 0;
    while i < RawData.Count do
      begin
        buf := RawData[i];
        Inc(i);
        for j:=2 to numStrPerItem do
          begin
            if i >= RawData.Count then
              Break;
            buf := buf + Separator + RawData[i];
            Inc(i);
          end;
        gi := Add;
        if Assigned(gi) then
// ROR*1.5*19 2012-01-09
//          gi.AssignRawData(buf, anIndexList, Separator);
          gi.AssignRawDataDef(buf, anIndexList, Separator, Default, DefaultID);
      end;
  finally
    EndUpdate;
  end;
end;

procedure TCCRGridItems.AssignRawData(RawData: TStrings;
  anIndexList: array of Integer; const Separator: String);
begin
  BeginUpdate;
  try
    Clear;
    AppendRawData(RawData, anIndexList, Separator);
  finally
    EndUpdate;
  end;
end;

function TCCRGridItems.GetItem(anIndex: Integer): TCCRGridItem;
begin
  Result := inherited Item[anIndex] as TCCRGridItem;
end;

procedure TCCRGridItems.GetRawData(RawData: TStrings; anIndexList: array of Integer;
  const Separator: String);
var
  i, numItems: Integer;
  buf: String;
begin
  RawData.BeginUpdate;
  try
    RawData.Clear;
    numItems := Count - 1;
    for i:=0 to numItems do
      begin
        buf := Item[i].GetRawData(anIndexList, Separator);
        RawData.Add(buf);
      end;
  finally
    RawData.EndUpdate;
  end;
end;

function TCCRGridItems.Insert(anIndex: Integer): TCCRGridItem;
begin
  Result := TCCRGridItem(inherited Insert(anIndex));
end;

procedure TCCRGridItems.SetItem(anIndex: Integer; const aValue: TCCRGridItem);
begin
  Item[anIndex].Assign(aValue);
end;
{$ENDREGION}

{$REGION ' TCCRGridView Methods '}
////////////////////////////////// TCCRGridView \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRGridView.Create(anOwner: TComponent);
begin
  inherited;

  BevelKind           := bkNone;
  GridLines           := True;
  ParentColor         := False;
  ReadOnly            := False;
  RowSelect           := True;
  ShowColumnHeaders   := True;
  TabStop             := True;
  ViewStyle           := vsReport;

  fFields             := TCCRGridFields.Create(Self);
  fOnColumnResize     := nil;
  fOnFieldValueFormat := nil;
end;

destructor TCCRGridView.Destroy;
begin
  FreeAndNil(fFields);
  inherited;
end;

function TCCRGridView.ColumnIndex(const aFieldIndex: Integer): Integer;
begin
  if (aFieldIndex >= 0) and (aFieldIndex < Fields.Count) then
    begin
      Result := Fields[aFieldIndex].ColIndex;
      if (Result < 0) or (Result >= Columns.Count) then
        Result := -1;
    end
  else
    Result := -1;
end;

procedure TCCRGridView.CompareItems(Sender: TObject; Item1, Item2: TListItem;
  Data: Integer; var Compare: Integer);
var
  dt1, dt2, dt: TCCRGridDataType;
  iv1, iv2: Integer;
  dv1, dv2: Double;
  bv1, bv2: Boolean;
begin
  if (SortField < 0) or (SortField >= Fields.Count) then
    Exit;
  Compare := 0;
  dt := Fields[SortField].DataType;

  if dt = gftMUMPS then
    begin
      //--- Get the actual data types
      dt1 := TCCRGridItem(Item1).fFieldData[SortField].MType;
      dt2 := TCCRGridItem(Item2).fFieldData[SortField].MType;
      //--- If they are different, perform MUMPS-like comparison.
      //--- Otherwise, skip to the regular comparison of values.
      if dt1 <> dt2 then
        begin
          //--- Item1
          if TCCRGridItem(Item1).GetAsString(SortField) = '' then
              Dec(Compare, 1)
          else if dt1 = gftDouble then
              Inc(Compare, 1)
          else if dt1 = gftString then
              Inc(Compare, 2);
          //--- Item2
          if TCCRGridItem(Item2).GetAsString(SortField) = '' then
              Inc(Compare, 1)
          else if dt2 = gftDouble then
              Dec(Compare, 1)
          else if dt2 = gftString then
              Dec(Compare, 2);
          {
          dv1\dv2 | empty | number | string
          --------+-------+--------+---------
          empty   |   0   |   -2   |   -3
          --------+-------+--------+---------
          number  |   2   |  cmp.  |   -1
          --------+-------+--------+---------
          string  |   3   |    1   |  cmp.
          }
          if Compare < -1 then
            Compare := -1;
          if Compare > 1 then
            Compare := 1;
        end
      else
        dt := dt1;
    end;
  //--- Regular comparison of field values
  case dt of
    gftBoolean:
      begin
        bv1 := TCCRGridItem(Item1).GetAsBoolean(SortField);
        bv2 := TCCRGridItem(Item2).GetAsBoolean(SortField);
        if      bv1 > bv2 then Compare :=  1
        else if bv1 < bv2 then Compare := -1;
      end;
    gftDateTime, gftDouble, gftFMDate:
      begin
        dv1 := TCCRGridItem(Item1).GetAsDouble(SortField);
        dv2 := TCCRGridItem(Item2).GetAsDouble(SortField);
        if      dv1 > dv2 then Compare :=  1
        else if dv1 < dv2 then Compare := -1;
      end;
    gftInteger:
      begin
        iv1 := TCCRGridItem(Item1).GetAsInteger(SortField);
        iv2 := TCCRGridItem(Item2).GetAsInteger(SortField);
        if      iv1 > iv2 then Compare :=  1
        else if iv1 < iv2 then Compare := -1;
      end;
    gftString:
      Compare := CompareStr(
        TCCRGridItem(Item1).GetAsString(SortField),
        TCCRGridItem(Item2).GetAsString(SortField));
  end;
  //--- Reverse the sort order if necessary
  if SortDescending then
    Compare := -Compare;
end;

function TCCRGridView.CreateListItem: TListItem;
var
  LClass: TListItemClass;
begin
  LClass := TCCRGridItem;
  if Assigned(OnCreateItemClass) then
    OnCreateItemClass(Self, LClass);
  Result := LClass.Create(Items);
end;

function TCCRGridView.CreateListItems: TListItems;
begin
  Result := TCCRGridItems.Create(self);
end;

procedure TCCRGridView.CreateWnd;
begin
  inherited;
  UpdateColumns(False);
end;

procedure TCCRGridView.DoOnColumnResize(aColumn: TListColumn);
begin
  if Assigned(OnColumnResize) then
    OnColumnResize(Self, aColumn);
end;

function TCCRGridView.FieldIndex(const aColumnIndex: Integer;
  const SortableOnly: Boolean): Integer;
begin
  if (aColumnIndex >= 0) and (aColumnIndex < Columns.Count) then
    begin
      Result := Columns[aColumnIndex].Tag;
      if (Result < 0) or (Result >= Fields.Count) then
        Result := -1
      else if not Fields[Result].AllowSort and SortableOnly then
        Result := -1;
    end
  else
    Result := -1;
end;

function TCCRGridView.GetFieldsCount: Integer;
begin
  Result := Fields.Count;
end;

function TCCRGridView.GetItemFocused: TCCRGridItem;
begin
  Result := inherited ItemFocused as TCCRGridItem;
end;

function TCCRGridView.GetItems: TCCRGridItems;
begin
  Result := inherited Items as TCCRGridItems;
end;

function TCCRGridView.GetNextItem(StartItem: TCCRCustomListItem;
  Direction: TSearchDirection; States: TItemStates): TCCRGridItem;
begin
  Result := TCCRGridItem(inherited GetNextItem(StartItem, Direction, States));
end;

function TCCRGridView.GetSelected: TCCRGridItem;
begin
  Result := inherited Selected as TCCRGridItem;
end;

function TCCRGridView.GetSortColumn: Integer;
begin
  Result := ColumnIndex(SortField);
end;

{$IFNDEF NOORPHEUS}
procedure TCCRGridView.LoadLayout(aStorage: TOvcAbstractStore; const aSection: String);
var
  i, wd: Integer;
begin
  try
    aStorage.Open;
    try
      for i:=0 to Fields.Count-1 do
        begin
          wd := aStorage.ReadInteger(aSection, Fields[i].GetNamePath, -1);
          if wd >= 0 then
            Fields[i].Width := wd;
        end;
    finally
      aStorage.Close;
    end;
  except
  end;
end;
{$ENDIF}

{$IFNDEF NOORPHEUS}
procedure TCCRGridView.SaveLayout(aStorage: TOvcAbstractStore; const aSection: String);
  var
    i: Integer;
begin
  try
    aStorage.Open;
    try
      for i:=0 to Fields.Count-1 do
        aStorage.WriteInteger(aSection, Fields[i].GetNamePath, Fields[i].Width);
    finally
      aStorage.Close;
    end;
  except
  end;
end;
{$ENDIF}

procedure TCCRGridView.SetFields(const aValue: TCCRGridFields);
begin
  try
    fFields.Free;
    Columns.Clear;
  except
  end;
  fFields := aValue;
end;

procedure TCCRGridView.SetItemFocused(aValue: TCCRGridItem);
begin
  inherited ItemFocused := aValue;
end;

procedure TCCRGridView.SetSelected(aValue: TCCRGridItem);
begin
  inherited Selected := aValue;
end;

procedure TCCRGridView.UpdateColumns(const aClrFlg: Boolean);
var
  i, n: Integer;
  col: TListColumn;
begin
  if aClrFlg then
    Clear;

  Columns.BeginUpdate;
  try
    Columns.Clear;
    n := Fields.Count - 1;
    for i:=0 to n do
      if Fields[i].Visible then
        begin
          col := Columns.Add;
          col.Assign(Fields[i]);
          Fields[i].ColIndex := col.Index;
        end
      else
        Fields[i].ColIndex := -1;
    UpdateSortField;
  finally
    Columns.EndUpdate;
  end;
end;

procedure TCCRGridView.WMNotify(var Msg: TWMNotify);
var
  colNdx, colWidth, fldNdx: Integer;
begin
  if csDesigning in	ComponentState then
    begin
      inherited;
      Exit;
    end;

  if (Msg.NMHdr^.code = HDN_BEGINTRACKA) or (Msg.NMHdr^.code = HDN_BEGINTRACKW) then
    begin
      colNdx := FindColumnIndex(Msg.NMHdr);
      fldNdx := FieldIndex(colNdx);
      if fldNdx >= 0 then
        if not Fields[fldNdx].AllowResize then
          begin
            Msg.Result := 1; // Disable resizing the column
            Exit;
          end;
    end;

  inherited;

  case Msg.NMHdr^.code of
    HDN_ENDTRACK:
      begin
        colNdx := FindColumnIndex(Msg.NMHdr);
        colWidth := FindColumnWidth(Msg.NMHdr);
        //--- Update the width of field
        fldNdx := FieldIndex(colNdx);
        if (fldNdx >= 0) and (colWidth >= 0) then
          Fields[fldNdx].fWidth := colWidth;
        //--- Call the event handler (if assigned)
        DoOnColumnResize(Columns[colNdx]);
      end;
    //HDN_BEGINTRACK:
    //HDN_TRACK: ;
  end;
end;
{$ENDREGION}

end.
