{******************************************************************************}
{ Package:      Clinical Case Registries Custom Components                     }
{ Date Created: November 18, 2004                                              }
{ Site Name:    Hines OIFO                                                     }
{ Developers:   Sergey Gavrilov                                                }
{ Description:  This unit defines classes for custom double-pane selector.     }
{ Note:                                                                        }
{******************************************************************************}

unit uROR_CustomSelector;

interface

uses
  Buttons, Classes, Controls, ExtCtrls,
  {$IFNDEF NOORPHEUS}OvcFiler,{$ENDIF}
  ComCtrls, uROR_GridView, uROR_Classes, ovcspeed;

type

  {============================= TCCRSelectorARMode ============================
    Overview:     Codes for item transfer modes used by a double-pane selector.
    SeeAlso:      TCCRCustomSelector.AddRemoveMode
  }
  TCCRSelectorARMode = (

    armDefault, { Move items in both directions (default). }

    armSpecial  { Copy items from the source pane to the resulting pane. Do not
                  copy items in the opposite direction; just delete them from
                  the destination pane. Use this mode if the destination pane
                  has complex structure (e.g. a tree) and items of different
                  types can be selected. Thus, you will avoid the situation
                  when items of different type and/or structure are intermixed
                  in the source pane. }

  );

  {================================= TSplitPos =================================
    Overview:     Position of a splitter inside a compound control (percentage
                  from the total control width).
    SeeAlso:      TCCRCustomSelector.SplitPos
    Description:
      This type describes relative position of internal splitter.
  }
  TSplitPos = 0..100;

  {======================= TCCRSelectorUpdateButtonsEvent ======================
    Overview:     Event type for updating buttons of a double-pane selector.
    SeeAlso:      TCCRCustomSelector.OnUpdateButtons
    Description:
      This event is generated when state of internal buttons of a double-pane
      selector should be updated.
  }
  TCCRSelectorUpdateButtonsEvent = procedure(aSender: TObject; var EnableAdd,
    EnableAddAll, EnableRemove, EnableRemoveAll: Boolean) of object;

  {============================= TCCRCustomSelector ============================
    Overview:     Base class for double-pane selectors.
    SeeAlso:      TCCRSelector
    Description:
      The TCCRCustomSelector class encapsulates basic functionality of
      a double-pane selector. Usually, panes contain simple grid views but more
      elaborate controls can be used as well (e.g. tree views).
      <p>
      Search results are presented to a user in the source pane. The user can
      select some or all records and move/copy them to and from the result pane.
      Items can be transferred between panes using buttons located on a panel
      between panes or keyboard shortcuts. They can also be dragged with a mouse.
      <p>
      Do not create instances of TCCRCustomSelector. To put a double-pane
      selector on a form, use a descendant of TCCRCustomSelector, such as
      TCCRSelector.
  }
  TCCRCustomSelector = class(TCustomPanel)
  private

    fAdd:              TOvcSpeedButton;
    fAddAll:           TOvcSpeedButton;
    fAdd508:           TOvcSpeedButton;
    fAddAll508:        TOvcSpeedButton;
    fAddRemoveMode:    TCCRSelectorARMode;
    fAutoSort:         Boolean;
    fCustomAdd:        Boolean;
    fCustomRemove:     Boolean;
    fDefaultResources: TCCRSingleton;
    fFocusedList:      TWinControl;
    fIDField:          Integer;
    fOnAdd:            TNotifyEvent;
    fOnAddAll:         TNotifyEvent;
    fOnRemove:         TNotifyEvent;
    fOnRemoveAll:      TNotifyEvent;
    fOnSplitterMoved:  TNotifyEvent;
    fOnUpdateButtons:  TCCRSelectorUpdateButtonsEvent;
    fRemove:           TOvcSpeedButton;
    fRemoveAll:        TOvcSpeedButton;
    fRemove508:        TOvcSpeedButton;
    fRemoveAll508:     TOvcSpeedButton;
    fResultChanged:    Boolean;
    fResultList:       TWinControl;
    fSourceList:       TWinControl;
    fSourcePanel:      TPanel;
    fSplitPos:         TSplitPos;
    fSplitter:         TSplitter;
    fSplitterPanel:    TPanel;

    procedure ActionUpdate(aSender: TObject);
    procedure buttonOnAdd(aSender: TObject);
    procedure buttonOnAddAll(aSender: TObject);
    procedure buttonOnRemove(aSender: TObject);
    procedure buttonOnRemoveAll(aSender: TObject);

    property ResultList: TWinControl  read fResultList;
    property SourceList: TWinControl  read fSourceList;

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Adjusts width of panes according to the splitter position.
      SeeAlso:      TCCRCustomSelector.SplitPos
      Description:
        AdjustControls resizes the source pane according to the value of the
        SplitPos property. Result pane fills the remaining space.
    }
    procedure AdjustControls; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates a result pane implementation.
      SeeAlso:      TCCRCustomSelector.Create;
                    TCCRCustomSelector.CreateSourceControl
      Keywords:     CreateResultControl,TCCRCustomSelector
      Description:
        Do not call CreateResultControl in your code; it is called internally
        by Create in order to create an implementation of the result pane.
        Override this method and return an appropriate windowed control.
    }
    function CreateResultControl: TWinControl; virtual; abstract;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates a source pane implementation.
      SeeAlso:      TCCRCustomSelector.Create;
                    TCCRCustomSelector.CreateResultControl
      Keywords:     CreateSourceControl,TCCRCustomSelector
      Description:
        Do not call CreateSourceControl in your code; it is called internally
        by Create in order to create an implementation of the source pane.
        Override this method and return an appropriate windowed control.
    }
    function CreateSourceControl: TWinControl; virtual; abstract;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates the window used by the selector.
      SeeAlso:      TCustomPanel.CreateParams; TWinControl.CreateWnd
      Keywords:     CreateWnd,TCCRCustomSelector
      Description:
        CreateWnd creates the selectors window, based on the parameter values
        provided by the CreateParams method. CreateWnd then sends messages to
        the newly created window so that it reflects the current property
        settings of the selector object. TCCRCustomSelector overrides this
        method in order to properly align internal controls.
    }
    procedure CreateWnd; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Respond to receiving input focus.
      SeeAlso:      TWinControl.DoEnter
      Keywords:     DoEnter,TCCRCustomSelector
      Description:
        DoEnter is called automatically when the control receives the input
        focus. As implemented in TCCRCustomSelector, DoEnter sets focus to the
        source pane if the result pane is not focused already.
        <p>
        Descendant classes that override DoEnter should always call the
        inherited method.
    }
    procedure DoEnter; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnUpdateButtons event handler dispatcher.
      SeeAlso:      TCCRCustomSelector.OnUpdateButtons;
                    TCCRCustomSelector.UpdateButtons
      Keywords:     DoUpdateButtons,TCCRCustomSelector
      Description:
        Do not call this method; OnUpdateButtons is called automatically when
        status of the buttons on a panel between source and result panes should
        be updated according to the content of the panes. As implemented in
        TCCRCustomSelector, DoUpdateButtons calls an OnUpdateButtons event
        handler, if defined.
        <p>
        Assign appropriate values to the <i>EnableAdd</i>, <i>EnableAddAll</i>,
        <i>EnableRemove</i>, and <i>EnableRemoveAll</i> parameters to enable or
        disable corresponding buttons.
        <p>
        Descendant classes that override DoUpdateButtons should always call the
        inherited method.
    }
    procedure DoUpdateButtons(var EnableAdd, EnableAddAll,
      EnableRemove, EnableRemoveAll: Boolean); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns generic reference to a result pane control.
      SeeAlso:      TCCRCustomSelector.CreateResultControl;
                    TCCRCustomSelector.GetSourceControl
      Description:
        GetResultControl returns a generic reference to the result pane
        control created by the CreateResultControl.
    }
    function GetResultControl: TWinControl;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns generic reference to a source pane control.
      SeeAlso:      TCCRCustomSelector.CreateSourceControl;
                    TCCRCustomSelector.GetResultControl
      Description:
        GetSourceControl returns a generic reference to the source pane
        control created by the CreateSourceControl.
    }
    function GetSourceControl: TWinControl;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Responds to notifications that components are being created
                    or destroyed.
      SeeAlso:      TControl.Notification
      Keywords:     Notification,TCCRCustomSelector
      Description:
        Do not call the Notification method in an application.  Notification is
        called automatically when the component specified by <i>AComponent</i>
        is about to be inserted or removed, as specified by <i>Operation</i>.
        <p>
        TCCRCustomSelector overrides this method in order to update its
        SourceList and ResultList properties when controls they refer to are
        destroyed.
    }
    procedure Notification(AComponent: TComponent;
      AOperation: TOperation); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Respond to control resize.
      SeeAlso:      TControl.Resize
      Keywords:     Resize,TCCRCustomSelector
      Description:
        Resize is called automatically immediately after the controls
        dimensions change. As implemented in TCCRCustomSelector, Resize adjusts
        splitter position before calling the inherited method.
    }
    procedure Resize; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the Enabled property.
      SeeAlso:      TCCRCustomSelector.ResultList; TCCRCustomSelector.SourceList;
                    TControl.SetEnabled
      Keywords:     SetEnabled,TCCRCustomSelector
      Description:
        SetEnabled is the protected write implementation of the Enabled property.
        TCCRCustomSelector overrides this method in order to enable/disable the
        source and result panes.
        <p>
        When overriding SetEnabled, be sure to call the inherited method.
    }
    procedure SetEnabled(aValue: Boolean); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the SplitPos property.
      SeeAlso:      TCCRCustomSelector.SplitPos
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
        <p>
        When overriding SetSplitPos, be sure to call the inherited method.
    }
    procedure SetSplitPos(const aValue: TSplitPos); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnMoved event handler for internal splitter.
      SeeAlso:      TCCRCustomSelector.OnSplitterMoved; TCCRCustomSelector.SplitPos;
                    TCCRCustomSelector.Splitter; TSplitter.OnMoved;
      Description:
        This method is called after the user has retiled the selector using the
        splitter. As implemented in TCCRCustomSelector, SplitterMoved updates
        value of the SplitPos property and calls the OnSplitterMoved event
        handler, if defined.
        <p>
        Descendant classes that override DoEnter should always call the
        inherited method.
    }
    procedure SplitterMoved(aSender: TObject); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies the item transfer mode.
      SeeAlso:      TCCRCustomSelector.Add; TCCRCustomSelector.AddAll;
                    TCCRCustomSelector.Remove; TCCRCustomSelector.RemoveAll
      Description:
        By default, when a user transfers items from the source pane to the
        destination one or vice versa, they are moved (added to the destination
        and removed from the source).
        <p>
        If the value of this property is armSpecial, items are copied from the
        source pane to the resulting pane (they are not deleted from the source).
        Items are not transferred in the opposite direction; they are just
        deleted from the destination pane.
        <p>
        Use the armSpecial mode if the destination pane has a complex structure
        (e.g. a tree) and items of different types can be loaded into the source
        pane. Thus, you will avoid a situation when items of different type
        and/or structure are intermixed in the source pane.
    }
    property AddRemoveMode: TCCRSelectorARMode
      read fAddRemoveMode  write fAddRemoveMode  default armDefault;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies if a destination pane is automatically sorted
                    after an item transfer.
      Description:
        Use AutoSort to control automatic sorting of a destination pane after
        an item transfer. If the value of this property is True, then the
        result pane is sorted after items are transferred there from the source
        pane and the source pane is sorted after a transfer in the opposite
        direction.
    }
    property AutoSort: Boolean  read fAutoSort  write fAutoSort  default True;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies if the internal transfer method should not be
                    called for adding items to the result pane.
      SeeAlso:      TCCRCustomSelector.Add; TCCRCustomSelector.AddAll
      Description:
        Descendant classes should implement their own item transfer methods and
        call them from overridden Add and AddAll methods. If the value of the
        CustomAdd property is True, then overridden Add and AddAll should not
        call the item transfer method; they should just call their inherited
        methods.
    }
    property CustomAdd: Boolean  read fCustomAdd  write fCustomAdd;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies if the internal transfer method should not be
                    called for removing items from the result pane.
      SeeAlso:      TCCRCustomSelector.Remove; TCCRCustomSelector.RemoveAll
      Description:
        Descendant classes should implement their own item transfer methods and
        call them from overridden Remove and RemoveAll methods. If the value of
        the CustomRemove property is True, then overridden Remove and RemoveAll
        should not call the item transfer method; they should just call their
        inherited methods.
    }
    property CustomRemove: Boolean  read fCustomRemove  write fCustomRemove;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies an index of the grid field that holds a unique
                    item identifier.
      SeeAlso:      TCCRSelectorList.Fields
      Description:
        Use IDField to get or set a zero-based index of the field that
        uniquely identifies an item (both in source and result pane lists).
    }
    property IDField: Integer  read fIDField  write fIDField  default 0;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs when selected items are transferred from the source
                    pane to the result one.
      SeeAlso:      TCCRCustomSelector.AddRemoveMode; TCCRCustomSelector.OnAddAll;
                    TCCRCustomSelector.OnRemove
      Description:
        Write an OnAdd event handler to add custom processing after selected
        items of the source pane are moved or copied (see the AddRemoveMode
        property) to the result one. The <i>aSender</i> is the double-pane
        selector object whose event handler is called.
        <p>
        If the value of the CustomAdd property is True, then this handler
        should actually transfer selected items itself.
    }
    property OnAdd: TNotifyEvent  read fOnAdd  write fOnAdd;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs when all items of the source pane are transferred to
                    the result one.
      SeeAlso:      TCCRCustomSelector.AddRemoveMode; TCCRCustomSelector.OnAdd;
                    TCCRCustomSelector.OnRemoveAll
      Description:
        Write an OnAddAll event handler to add custom processing after all
        source pane items are moved or copied (see the AddRemoveMode property)
        to the result pane. The <i>aSender</i> is the double-pane selector
        object whose event handler is called.
        <p>
        If the value of the CustomAdd property is True, then this handler
        should actually transfer all items itself.
    }
    property OnAddAll: TNotifyEvent  read fOnAddAll  write fOnAddAll;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs when selected items are moved from the result
                    pane to the source one.
      SeeAlso:      TCCRCustomSelector.AddRemoveMode; TCCRCustomSelector.OnAdd;
                    TCCRCustomSelector.OnRemoveAll
      Description:
        Write an OnRemove event handler to add custom processing after
        selected items are moved from the result pane to the source one or
        deleted (see the AddRemoveMode property). The <i>aSender</i>
        is the double-pane selector object whose event handler is called.
        <p>
        If the value of the CustomRemove property is True, then this handler
        should actually move or delete selected items itself.
    }
    property OnRemove: TNotifyEvent  read fOnRemove  write fOnRemove;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs when all items are moved from the result pane
                    to the source one.
      SeeAlso:      TCCRCustomSelector.AddRemoveMode; TCCRCustomSelector.OnAddAll;
                    TCCRCustomSelector.OnRemove
      Description:
        Write an OnRemoveAll event handler to add custom processing after
        all result pane items are moved to the source pane or deleted
        (see the AddRemoveMode property). The <i>aSender</i> is the
        double-pane selector object whose event handler is called.
        <p>
        If the value of the CustomRemove property is True, then this handler
        should actually move or delete all items itself.
    }
    property OnRemoveAll: TNotifyEvent  read fOnRemoveAll  write fOnRemoveAll;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs after a splitter between source and result panes
                    is moved to a new position.
      SeeAlso:      TCCRCustomSelector.SplitPos; TCCRCustomSelector.Splitter;
                    TCCRCustomSelector.SplitterMoved
      Description:
        Write an OnSplitterMoved event handler to take specific action(s) after
        user finishes dragging a splitter between source and result panes of
        a selector. The <i>aSender</i> is the double-pane selector object whose
        event handler is called.
    }
    property OnSplitterMoved: TNotifyEvent
      read fOnSplitterMoved  write fOnSplitterMoved;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs when status of the buttons on a panel between
                    selector panes should be updated.
      SeeAlso:      TCCRCustomSelector.DoUpdateButtons;
                    TCCRCustomSelector.UpdateButtons
      Description:
        Write OnUpdateButtons event handler to enable or disable buttons on
        a panel between source and result panes according to the content of the
        panes.
        <p>
        <i>aSender</i> is the double-pane selector object whose event handler
        is called. The DoUpdateButtons method, which generates this event,
        assigns default values to the <i>EnableAdd</i>, <i>EnableAddAll</i>,
        <i>EnableRemove</i>, and <i>EnableRemoveAll</i> parameters. Modify them
        to enable or disable corresponding buttons if you want to change default
        behavior.
    }
    property OnUpdateButtons: TCCRSelectorUpdateButtonsEvent
      read fOnUpdateButtons  write fOnUpdateButtons;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Parent panel for the source pane control.
      SeeAlso:      TCCRCustomSelector.GetSourceControl
      Description:
        Use SourcePanel to access properties of the parent panel of a source
        pane control.
    }
    property SourcePanel: TPanel  read fSourcePanel;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies position of a splitter between source and result
                    panels.
      SeeAlso:      TCCRCustomSelector.OnSplitterMoved;
                    TCCRCustomSelector.Splitter
      Description:
        Use SplitPos to get or set position of a splitter between source and
        result panes of a selector. Value of this property is the relative
        width of the source pane (percents) relative to the total width of
        the selector. Thus, when a selector is resized, relative pane sizes
        stay the same.
    }
    property SplitPos: TSplitPos  read fSplitPos  write SetSplitPos  default 50;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Internal splitter between source and result panes.
      SeeAlso:      TCCRCustomSelector.SplitPos;
                    TCCRCustomSelector.SplitterPanel; TSplitter
      Keywords:     Splitter,TCCRCustomSelector
      Description:
        Use Splitter to access properties of an internal splitter between
        source and result panes of a double-pane selector. 
    }
    property Splitter: TSplitter  read fSplitter;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Panel with buttons between the source and result panes.
      SeeAlso:      TCCRCustomSelector.Splitter
      Description:
        Use SplitterPanel to access properties of a panel located to the left
        from the internal splitter (between source and result panes). Add,
        Add All, Remove, and Remove All buttons are located on this panel.
    }
    property SplitterPanel: TPanel  read fSplitterPanel;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRCustomSelector.
      SeeAlso:      TComponent.Owner; TCustomPanel.Create
      Keywords:     Create,TCCRCustomSelector
      Description:
        Create initializes an instance of the TCCRCustomSelector. <i>anOwner</i>
        is the component, typically a form, that is responsible for freeing the
        double-pane selector control.
    }
    constructor Create(anOwner: TComponent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRCustomSelector.
      SeeAlso:      TCustomControl.Destroy; TObject.Free
      Keywords:     Destroy,TCCRCustomSelector
      Description:
        Do not call Destroy directly in an application. Instead, call Free.
        Free verifies that the control is not nil, and only then calls Destroy.
        <p>
        Applications should only free controls explicitly when the constructor
        was called without assigning an owner to the control.
    }
    destructor  Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnAdd event handler dispatcher.
      SeeAlso:      TCCRCustomSelector.AddAll; TCCRCustomSelector.AddRemoveMode;
                    TCCRCustomSelector.Remove; TCCRCustomSelector.OnAdd
      Keywords:     Add,TCCRCustomSelector
      Description:
        Override the Add method in descendant classes to implement moving or
        copying (see the AddRemoveMode property) of selected items from the
        source pane to the result one.
        <p>
        If the value of the CustomAdd property is True, then this method should
        not transfer the items itself; the OnAdd event handler is responsible
        for this. As implemented in TCCRCustomSelector, Add calls the OnAdd
        event handler, if defined.
    }
    procedure Add; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnAddAll event handler dispatcher.
      SeeAlso:      TCCRCustomSelector.Add; TCCRCustomSelector.AddRemoveMode;
                    TCCRCustomSelector.RemoveAll; TCCRCustomSelector.OnAddAll
      Keywords:     AddAll,TCCRCustomSelector
      Description:
        Override the AddAll method in descendant classes to implement moving or
        copying (see the AddRemoveMode property) of all source pane items to
        the result pane.
        <p>
        If the value of the CustomAdd property is True, then this method should
        not transfer the items itself; the OnAddAll event handler is responsible
        for this. As implemented in TCCRCustomSelector, AddAll calls the
        OnAddAll event handler, if defined.
    }
    procedure AddAll; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Clears source and result pane controls.
      Keywords:     Clear,TCCRCustomSelector
      Description:
        As implemented in TCCRCustomSelector, Clear does nothing. Override this
        method in descendant classes.
    }
    procedure Clear; virtual;

    {$IFNDEF NOORPHEUS}
    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Loads a double-pane selector layout.
      SeeAlso:      TCCRCustomSelector.SaveLayout; TCCRVistAStore
      Keywords:     LoadLayout,TCCRCustomSelector
      Description:
        Call LoadLayout to restore double-pane selector layout (saved by the
        SaveLayout) from the data storage referenced by the <i>aStorage</i>
        parameter. <i>aSection</i> specifies the data section name (a key of
        the registry, an XML node, a section of an ini file, etc.) where the
        layout is stored.
        <p>
        Override this method in descendant classes to implement appropriate
        functionality. As implemented in TCCRCustomSelector, LoadLayout does
        nothing.
    }
    procedure LoadLayout(aStorage: TOvcAbstractStore;
      const aSection: String = ''); virtual;
    {$ENDIF}


    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnRemove event handler dispatcher.
      SeeAlso:      TCCRCustomSelector.Add; TCCRCustomSelector.AddRemoveMode;
                    TCCRCustomSelector.RemoveAll; TCCRCustomSelector.OnRemove
      Keywords:     Remove,TCCRCustomSelector
      Description:
        Override the Remove method in descendant classes to implement moving
        of selected items from the result pane to the source one or deleting
        them (see the AddRemoveMode property).
        <p>
        If the value of the CustomRemove property is True, then this method
        should not move or delete the items itself; the OnRemove event handler
        is responsible this. As implemented in TCCRCustomSelector, Remove calls
        the OnRemove event handler, if defined.
    }
    procedure Remove; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnRemoveAll event handler dispatcher.
      SeeAlso:      TCCRCustomSelector.AddAll; TCCRCustomSelector.AddRemoveMode;
                    TCCRCustomSelector.Remove; TCCRCustomSelector.OnRemoveAll
      Keywords:     RemoveAll,TCCRCustomSelector
      Description:
        Override the RemoveAll method in descendant classes to implement moving
        of all result pane items to the source pane or deleting them (see the
        AddRemoveMode property).
        <p>
        If the value of the CustomRemove property is True, then this method
        should not move or delete the items itself; the OnRemoveAll event
        handler is responsible for this. As implemented in TCCRCustomSelector,
        RemoveAll calls the OnRemoveAll event handler, if defined.
    }
    procedure RemoveAll; virtual;

    {$IFNDEF NOORPHEUS}
    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Saves a double-pane selector layout.
      SeeAlso:      TCCRCustomSelector.LoadLayout; TCCRVistAStore
      Keywords:     SaveLayout,TCCRCustomSelector
      Description:
        Call SaveLayout to save a double-pane selector layout (it can be later
        restored by the LoadLayout) to the data storage referenced by the
        <i>aStorage</i> parameter. <i>aSection</i> specifies the data section
        name (a key of the registry, an XML node, a section of an ini file,
        etc.) where the layout is saved to.
        <p>
        Override this method in descendant classes to implement appropriate
        functionality. As implemented in TCCRCustomSelector, SaveLayout does
        nothing.
    }
    procedure SaveLayout(aStorage: TOvcAbstractStore;
      const aSection: String = ''); virtual;
    {$ENDIF}

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Updates state of buttons on a panel between panes according
                    to pane contents.
      SeeAlso:      TCCRCustomSelector.DoUpdateButtons;
                    TCCRCustomSelector.OnUpdateButtons;
                    TCCRCustomSelector.SplitterPanel
      Description:
        Call UpdateButtons to update state of the buttons located on a panel
        between source and result panes of a selector according to the content
        of the panes.
    }
    procedure UpdateButtons;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates if the result pane content was modified.
      SeeAlso:      TCCRCustomSelector.Add; TCCRCustomSelector.AddAll;
                    TCCRCustomSelector.Remove; TCCRCustomSelector.RemoveAll
      Description:
        Use ResultChanged to determine whether the content of the result pane
        was modified by the user or programmatically. As implemented in
        TCCRCustomSelector, the ResultChanged property is initialized with
        False value.
        <p>
        When override Add, AddAll, Clear, Remove, RemoveAll methods or write
        corresponding event handlers, assign True to this property when the
        content of the result pane is modified.
    }
    property ResultChanged: Boolean  read fResultChanged  write fResultChanged;

  end;

  {============================== TCCRSelectorList =============================
    Overview:     Base class for a grid view implementation of a selector pane.
    SeeAlso:      TCCRCustomSelector; TCCRResultList; TCCRSourceList
    Description:
      TCCRSelectorList encapsulates common functionality of a grid view
      implementation of a selector pane (both source and result).
  }
  TCCRSelectorList = class(TCCRGridView)
  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnDragOver event dispatcher.
      SeeAlso:      TControl.DragDrop; TControl.DragOver; TControl.OnDragOver
      Keywords:     DragOver,TCCRSelectorList
      Description:
        TCCRSelectorList overrides the DragOver in order to check if dragged
        item(s) originated from the same selector before the OnDragOver event
        handler is called.
        <p>
        DragOver sets the <i>Accept</i> parameter to true to indicate that the
        user can drop the dragged object on the control. It sets <i>Accept</i>
        to false to indicate that the user cannot drop the dragged object on
        the control.
        <p>
        The <i>Source</i> parameter is the object being dragged. The
        <i>State</i> parameter indicates how the dragged object is moving in
        relation to the control. <i>X</i> and <i>Y</i> indicate the current
        position of the mouse.
    }
    procedure DragOver(Source: TObject; X, Y: Integer; State: TDragState;
      var Accept: Boolean); override;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns an items that immediately follows the last selected
                    one.
      Description:
        GetItemFollowingSelection returns an item that immediately follows the
        last selected item in the grid. If the grid is empty, the last item of
        the grid is selected, or there are no selected items at all, this
        function returns nil.
    }
    function GetItemFollowingSelection: TCCRGridItem;

  end;

  {=============================== TCCRSourceList ==============================
    Overview:     Grid view implementation of a double-pane selector's source
                  pane.
    SeeAlso:      TCCRCustomSelector; TCCRResultList
    Description:
      TCCRSourceList implements a double-pane selector's source pane as a
      grid view.
  }
  TCCRSourceList = class(TCCRSelectorList)
  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnDblClick event dispatcher.
      SeeAlso:      TControl.DblClick; TControl.OnDblClick
      Keywords:     DblClick,TCCRSourceList
      Description:
        The DblClick method is triggered by left mouse-button double-click
        message (WM_LBUTTONDBLCLK) from Windows. It calls an event handler
        attached to the OnDblClick event, if defined.
        <p>
        TCCRSourceList overrides this method to allow transferring items from
        the source pane to the result pane by double-clicking them.
    }
    procedure DblClick; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Respond to key press events.
      SeeAlso:      TWinControl.DoKeyDown; TWinControl.OnKeyDown
      Keywords:     KeyDown,TCCRSourceList
      Description:
        When a windowed control receives a key-down message (WM_KEYDOWN) from
        Windows, its message handler calls the DoKeyDown method. If DoKeyDown
        determines that the control should, in fact, process the character, it
        calls KeyDown, passing the key code and shift-key state in the
        <i>Key</i> and <i>Shift</i> parameters, respectively. See the
        description of the TWinControl.KeyDown for more details.
        <p>
        TCCRSourceList overrides this method to allow transferring items from
        the source pane to the result pane by pressing Enter or Space keys.
    }
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnDragDrop event dispatcher.
      SeeAlso:      TControl.DragDrop; TControl.OnDragDrop
      Keywords:     DragDrop,TCCRSourceList
      Description:
        TCCRSourceList overrides the DragDrop method in order to check if
        dragged item(s) originated from the same selector before transferring
        them from the destination pane to the source one.
        <p>
        The <i>Source</i> parameter is the object that was dropped onto the
        control. The <i>X</i> and <i>Y</i> parameters are the mouse coordinates
        where the object was dropped.
    }
    procedure DragDrop(Source: TObject; X, Y: Integer); override;

  end;

///////////////////////////////// Implementation \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

implementation

uses
  Forms, SysUtils, StdCtrls, uROR_Resources, ActnList, Windows, Graphics;

const
  CCRSelSplitterWidth    = 32;
  CCRSelButtonSize       = 24;
  CCR508SelSplitterWidth = 82;
  CCR508SelButtonSize    = 74;

var
  defaultSplitterWidth, defaultButtonSize: integer;

type

  {======================== TCCRSelectorDefaultResources =======================
    Overview:     Singleton for default resources.
    Description:
      TCCRSelectorDefaultResources handles a single copy of default resources
      for double-pane selectors.
  }
  TCCRSelectorDefaultResources = class(TCCRSingleton)
  private

    fAdd:       TBitmap;
    fAddAll:    TBitmap;
    fRemove:    TBitmap;
    fRemoveAll: TBitmap;

  protected

    procedure Finalize; override;
    procedure Initialize; override;

  public

    property Add: TBitmap  read fAdd;
    property AddAll: TBitmap  read fAddAll;
    property Remove: TBitmap  read fRemove;
    property RemoveAll: TBitmap  read fRemoveAll;

  end;

/////////////////////////////// TCCRCustomSelector \\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRCustomSelector.Create(anOwner: TComponent);
var
  bl: Integer;
  dRes: TCCRSelectorDefaultResources;
begin
  inherited;
  dRes := TCCRSelectorDefaultResources.Create;
  fDefaultResources := dRes;

  AutoSize   := False;
  BevelInner := bvNone;
  BevelOuter := bvNone;
  Caption    := '';
  DockSite   := False;

  Constraints.MinHeight := CCRSelButtonSize * 4;
  Constraints.MinWidth  := CCRSelSplitterWidth * 3;

  fAddRemoveMode := armDefault;
  fAutoSort      := True;
  fResultChanged := False;
  fSourcePanel   := TPanel.Create(Self);
  fSplitter      := TSplitter.Create(Self);
  fSplitterPanel := TPanel.Create(Self);

  fSourceList    := CreateSourceControl;
  fResultList    := CreateResultControl;
  fFocusedList   := fSourceList;

  //-- Section 508: Set the default width for the splitter and buttons.
  if CCRScreenReaderActive and (not (csDesigning in ComponentState)) then
  begin
    fAdd508       := TOvcSpeedButton.Create(Self);
    fAddAll508    := TOvcSpeedButton.Create(Self);
    fRemove508    := TOvcSpeedButton.Create(Self);
    fRemoveAll508 := TOvcSpeedButton.Create(Self);
    defaultSplitterWidth := CCR508SelSplitterWidth;
    defaultButtonSize    := CCR508SelButtonSize;
  end
  else begin
    fAdd       := TOvcSpeedButton.Create(Self);
    fAddAll    := TOvcSpeedButton.Create(Self);
    fRemove    := TOvcSpeedButton.Create(Self);
    fRemoveAll := TOvcSpeedButton.Create(Self);
    defaultSplitterWidth := CCRSelSplitterWidth;
    defaultButtonSize    := CCRSelButtonSize;
  end;

  DisableAlign;
  try
    with SourcePanel do
      begin
        Parent := Self;

        Align       := alLeft;
        BevelInner  := bvNone;
        BevelOuter  := bvNone;
        BorderStyle := bsNone;
        Left        := 0;
        ParentColor := True;
        Top         := 0;
      end;

    with SourceList do
      begin
        SetSubComponent(True);
        Constraints.MinWidth := defaultSplitterWidth;
        Parent := SourcePanel;

        Align       := alClient;
        DragKind    := dkDrag;
        DragMode    := dmAutomatic;
        Left        := 0;
        Name        := 'SourceList';
        TabStop     := True;
        TabOrder    := 0;
        Top         := 0;
      end;

    with SplitterPanel do
      begin
        Parent := SourcePanel;

        Align       := alRight;
        BevelInner  := bvNone;
        BevelOuter  := bvNone;
        BorderStyle := bsNone;
        Left        := SourceList.Width;
        ParentColor := True;
        Top         := 0;
        Width       := defaultSplitterWidth;
      end;

    with ResultList do
      begin
        SetSubComponent(True);
        Constraints.MinWidth := defaultSplitterWidth;
        Parent := Self;

        Align       := alClient;
        DragKind    := dkDrag;
        DragMode    := dmAutomatic;
        Left        := Splitter.Left + Splitter.Width;
        Name        := 'ResultList';
        TabStop     := True;
        TabOrder    := 1;
        Top         := 0;
      end;


    with Splitter do
      begin
        Parent := Self;

        Align       := alLeft;
        Beveled     := True;
        Left        := SourcePanel.Width;
        OnMoved     := SplitterMoved;
        Top         := 0;
      end;

    bl := (defaultSplitterWidth - defaultButtonSize) div 2;

{$REGION ' Setup Section 508 related buttons '}
    //-- Section 508: Setup the button based upon whether a screen reader is in
    //   use or not.
    if CCRScreenReaderActive and (not (csDesigning in ComponentState)) then
    begin
      with fAdd508 do
      begin
        SetSubComponent(True);
        Parent := SplitterPanel;

        Action := TAction.Create(Self);
        Action.OnExecute := buttonOnAdd;
        Action.OnUpdate  := ActionUpdate;

        Caption   := 'Add';
        Height    := CCRSelButtonSize;
        Hint      := rscAddHint;
        Left      := bl;
        NumGlyphs := 1;
        ShowHint  := True;
        TabStop   := true;
        TabOrder  := SourceList.TabOrder + 1;
        Top       := 10;
        Width     := defaultButtonSize;
        Visible   := true;
      end;

      with fRemove508 do
      begin
        SetSubComponent(True);
        Parent := SplitterPanel;

        Action := TAction.Create(Self);
        Action.OnExecute := buttonOnRemove;
        Action.OnUpdate  := ActionUpdate;

        Caption   := 'Remove';
        Height    := CCRSelButtonSize;
        Hint      := rscRemoveHint;
        Left      := bl;
        NumGlyphs := 1;
        ShowHint  := True;
        TabStop   := true;
        TabOrder  := SourceList.TabOrder + 2;
        Top       := fAdd508.Top + CCRSelButtonSize;
        Width     := defaultButtonSize;
        Visible   := true;
      end;

      with fAddAll508 do
      begin
        SetSubComponent(True);
        Parent := SplitterPanel;

        Action := TAction.Create(Self);
        Action.OnExecute := buttonOnAddAll;
        Action.OnUpdate  := ActionUpdate;

        Caption   := 'Add All';
        Height    := CCRSelButtonSize;
        Hint      := rscAddAllHint;
        Left      := bl;
        NumGlyphs := 1;
        ShowHint  := True;
        TabStop   := true;
        TabOrder  := SourceList.TabOrder + 3;
        Top       := fRemove508.Top + CCRSelButtonSize;
        Width     := defaultButtonSize;
      end;

      with fRemoveAll508 do
      begin
        SetSubComponent(True);
        Parent := SplitterPanel;

        Action := TAction.Create(Self);
        Action.OnExecute := buttonOnRemoveAll;
        Action.OnUpdate  := ActionUpdate;

        Caption   := 'Remove All';
        Height    := CCRSelButtonSize;
        Hint      := rscRemoveAllHint;
        Left      := bl;
        NumGlyphs := 1;
        ShowHint  := True;
        TabStop   := true;
        TabOrder  := SourceList.TabOrder + 4;
        Top       := fAddAll508.Top + CCRSelButtonSize;
        Width     := defaultButtonSize;
      end;
    end
{$ENDREGION}
{$REGION ' Setup "standard" buttons '}
    else begin
      with fAdd do
      begin
        SetSubComponent(True);
        Parent := SplitterPanel;

        Action := TAction.Create(Self);
        Action.OnExecute := buttonOnAdd;
        Action.OnUpdate  := ActionUpdate;

        Flat      := True;
        Glyph     := dRes.Add;
        Height    := CCRSelButtonSize;
        Hint      := rscAddHint;
        Left      := bl;
        NumGlyphs := 2;
        ShowHint  := True;
        Top       := 10;
        Width     := defaultButtonSize;
        Visible   := false;
      end;

      with fRemove do
      begin
        SetSubComponent(True);
        Parent := SplitterPanel;

        Action := TAction.Create(Self);
        Action.OnExecute := buttonOnRemove;
        Action.OnUpdate  := ActionUpdate;

        Flat      := True;
        Glyph     := dRes.Remove;
        Height    := CCRSelButtonSize;
        Hint      := rscRemoveHint;
        Left      := bl;
        NumGlyphs := 2;
        ShowHint  := True;
        Top       := 10;
        Width     := defaultButtonSize;
        Visible   := false;
      end;

      with fAddAll do
      begin
        SetSubComponent(True);
        Parent := SplitterPanel;

        Action := TAction.Create(Self);
        Action.OnExecute := buttonOnAddAll;
        Action.OnUpdate  := ActionUpdate;

        Flat      := True;
        Glyph     := dRes.AddAll;
        Height    := CCRSelButtonSize;
        Hint      := rscAddAllHint;
        Left      := bl;
        NumGlyphs := 2;
        ShowHint  := True;
        Top       := fRemove.Top + CCRSelButtonSize;
        Width     := defaultButtonSize;
      end;

      with fRemoveAll do
      begin
        SetSubComponent(True);
        Parent := SplitterPanel;

        Action := TAction.Create(Self);
        Action.OnExecute := buttonOnRemoveAll;
        Action.OnUpdate  := ActionUpdate;

        Flat      := True;
        Glyph     := dRes.RemoveAll;
        Height    := CCRSelButtonSize;
        Hint      := rscRemoveAllHint;
        Left      := bl;
        NumGlyphs := 2;
        ShowHint  := True;
        Top       := fAddAll.Top + CCRSelButtonSize;
        Width     := defaultButtonSize;
      end;
    end;
{$ENDREGION}
  finally
    EnableAlign;
  end;

  SplitPos := 50;
  UpdateButtons;
end;

destructor TCCRCustomSelector.Destroy;
begin
  //-- Section 508: Release 508 related resources.
  if CCRScreenReaderActive and (not (csDesigning in ComponentState)) then
  begin
    fAdd508.Glyph        := nil;
    fAddAll508.Glyph     := nil;
    fRemove508.Glyph     := nil;
    fRemoveAll508.Glyph  := nil;

    fAdd508.Action.Free;
    fAddAll508.Action.Free;
    fRemove508.Action.Free;
    fRemoveAll508.Action.Free;

    fAdd508.Action       := nil;
    fAddAll508.Action    := nil;
    fRemove508.Action    := nil;
    fRemoveAll508.Action := nil;

    FreeAndNil(fAdd508);
    FreeAndNil(fAddAll508);
    FreeAndNil(fRemove508);
    FreeAndNil(fRemoveAll508);
  end
  else begin
    fAdd.Glyph        := nil;
    fAddAll.Glyph     := nil;
    fRemove.Glyph     := nil;
    fRemoveAll.Glyph  := nil;

    fAdd.Action.Free;
    fAddAll.Action.Free;
    fRemove.Action.Free;
    fRemoveAll.Action.Free;

    fAdd.Action       := nil;
    fAddAll.Action    := nil;
    fRemove.Action    := nil;
    fRemoveAll.Action := nil;

    FreeAndNil(fAdd);
    FreeAndNil(fAddAll);
    FreeAndNil(fRemove);
    FreeAndNil(fRemoveAll);
  end;

  FreeAndNil(fSplitterPanel);
  FreeAndNil(fSourcePanel);
  FreeAndNil(fSplitter);

  FreeAndNil(fSourceList);
  FreeAndNil(fResultList);

  FreeAndNil(fDefaultResources);
  inherited;
end;

procedure TCCRCustomSelector.ActionUpdate(aSender: TObject);
begin
  UpdateButtons;
end;

procedure TCCRCustomSelector.Add;
begin
  if Assigned(OnAdd) then
    OnAdd(Self);
end;

procedure TCCRCustomSelector.AddAll;
begin
  if Assigned(OnAddAll) then
    OnAddAll(Self);
end;

procedure TCCRCustomSelector.AdjustControls;
var
  wd: Integer;
begin
  wd := Trunc((Width - defaultSplitterWidth - Splitter.Width) * SplitPos / 100);
  SourcePanel.Width := wd + defaultSplitterWidth;
  SourceList.Width := wd;  // Just in case
end;

procedure TCCRCustomSelector.buttonOnAdd(aSender: TObject);
begin
  Add;
  //-- Section 508: Set the focus to the Source list.
  if CCRScreenReaderActive then
    SourceList.SetFocus;
end;

procedure TCCRCustomSelector.buttonOnAddAll(aSender: TObject);
begin
  AddAll;
  //-- Section 508: Set the focus to the Result list.
  if CCRScreenReaderActive and Assigned(ResultList) then
    ResultList.SetFocus;
end;

procedure TCCRCustomSelector.buttonOnRemove(aSender: TObject);
begin
  Remove;
  //-- Section 508: Set the focus to the Result list.
  if CCRScreenReaderActive then
    ResultList.SetFocus;
end;

procedure TCCRCustomSelector.buttonOnRemoveAll(aSender: TObject);
begin
  RemoveAll;
  //-- Section 508: Set the focus to the Source list.
  if CCRScreenReaderActive and Assigned(SourceList) then
    SourceList.SetFocus;
end;

procedure TCCRCustomSelector.Clear;
begin
end;

procedure TCCRCustomSelector.CreateWnd;
begin
  inherited;
  AdjustControls;
end;

procedure TCCRCustomSelector.DoEnter;
begin
  if Assigned(SourceList) then
    begin
      if SourceList.Focused then
        Exit;
      if Assigned(ResultList) and ResultList.Focused then
        Exit;
      SourceList.SetFocus;
    end;
  inherited;
end;

procedure TCCRCustomSelector.DoUpdateButtons(var EnableAdd, EnableAddAll,
  EnableRemove, EnableRemoveAll: Boolean);
begin
  if Assigned(OnUpdateButtons) then
    OnUpdateButtons(Self, EnableAdd, EnableAddAll, EnableRemove, EnableRemoveAll);
end;

function TCCRCustomSelector.GetResultControl: TWinControl;
begin
  Result := fResultList;
end;

function TCCRCustomSelector.GetSourceControl: TWinControl;
begin
  Result := fSourceList;
end;

{$IFNDEF NOORPHEUS}
procedure TCCRCustomSelector.LoadLayout(aStorage: TOvcAbstractStore;
  const aSection: String);
begin
end;
{$ENDIF}

procedure TCCRCustomSelector.Notification(AComponent: TComponent; AOperation: TOperation);
begin
  inherited;
  if AOperation = opRemove then
    if AComponent = SourceList then
      fSourceList := nil
    else if AComponent = ResultList then
      fResultList := nil
end;

procedure TCCRCustomSelector.Remove;
begin
  if Assigned(OnRemove) then
    OnRemove(Self);
end;

procedure TCCRCustomSelector.RemoveAll;
begin
  if Assigned(OnRemoveAll) then
    OnRemoveAll(Self);
end;

procedure TCCRCustomSelector.Resize;
begin
  SplitPos := SplitPos;
  inherited;
end;

{$IFNDEF NOORPHEUS}
procedure TCCRCustomSelector.SaveLayout(aStorage: TOvcAbstractStore;
  const aSection: String);
begin
end;
{$ENDIF}

procedure TCCRCustomSelector.SetEnabled(aValue: Boolean);
begin
  inherited;
  if Assigned(ResultList) then
    ResultList.Enabled := aValue;
  if Assigned(SourceList) then
    SourceList.Enabled := aValue;
end;

procedure TCCRCustomSelector.SetSplitPos(const aValue: TSplitPos);
begin
  if aValue < 0 then fSplitPos := 0
  else if aValue > 100 then fSplitPos := 100
  else fSplitPos := aValue;

  AdjustControls;

  if Assigned(OnSplitterMoved) then
    OnSplitterMoved(Self);
end;

procedure TCCRCustomSelector.SplitterMoved(aSender: TObject);
begin
  if (Width <> 0) and Assigned(SourceList) then
    begin
      fSplitPos := Trunc(SourceList.Width /
        (Width - defaultSplitterWidth - Splitter.Width) * 100);
      if Assigned(OnSplitterMoved) then
        OnSplitterMoved(Self);
    end;
end;

procedure TCCRCustomSelector.UpdateButtons;
var
  EnableAdd, EnableAddAll, EnableRemove, EnableRemoveAll: Boolean;
begin
  EnableAdd       := False;
  EnableAddAll    := False;
  EnableRemove    := False;
  EnableRemoveAll := False;

  if Enabled then
    DoUpdateButtons(EnableAdd, EnableAddAll, EnableRemove, EnableRemoveAll);

  //-- Section 508: Hide the Add/Remove button if a screen reader is not in use.
  if CCRScreenReaderActive and (not (csDesigning in ComponentState)) then
  begin
    fAdd508.Enabled        := EnableAdd;
    fAddAll508.Enabled     := EnableAddAll;
    fRemove508.Enabled     := EnableRemove;
    fRemoveAll508.Enabled  := EnableRemoveAll;
  end
  else begin
    if SourceList.ContainsControl(Screen.ActiveControl) then
    begin
      fAdd.Visible    := True;
      fRemove.Visible := False;
    end
    else if ResultList.ContainsControl(Screen.ActiveControl) then
    begin
      fAdd.Visible    := False;
      fRemove.Visible := True;
    end
    else begin
      fRemove.Visible := False;
      fAdd.Visible    := False;
    end;

    fAdd.Enabled        := EnableAdd;
    fAddAll.Enabled     := EnableAddAll;
    fRemove.Enabled     := EnableRemove;
    fRemoveAll.Enabled  := EnableRemoveAll;
  end;
end;

///////////////////////// TCCRSelectorDefaultResources \\\\\\\\\\\\\\\\\\\\\\\\\

procedure TCCRSelectorDefaultResources.Finalize;
begin
  FreeAndNil(fRemoveAll);
  FreeAndNil(fRemove);
  FreeAndNil(fAddAll);
  FreeAndNil(fAdd);
  inherited;
end;

procedure TCCRSelectorDefaultResources.Initialize;
begin
  inherited;

  fAdd       := TBitmap.Create;
  fAddAll    := TBitmap.Create;
  fRemove    := TBitmap.Create;
  fRemoveAll := TBitmap.Create;

  Add.LoadFromResourceName(HInstance, 'BTN_ADD');
  AddAll.LoadFromResourceName(HInstance, 'BTN_ADDALL');
  Remove.LoadFromResourceName(HInstance, 'BTN_REMOVE');
  RemoveAll.LoadFromResourceName(HInstance, 'BTN_REMOVEALL');
end;

//////////////////////////////// TCCRSelectortList \\\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TCCRSelectorList.DragOver(Source: TObject; X, Y: Integer;
  State: TDragState; var Accept: Boolean);
begin
  Accept := False;
  if Source <> Self then
    if (Source is TControl) and (TControl(Source).Owner = Owner) then
      begin
        Accept := True;
        if Assigned(OnDragOver) then
          inherited;
      end;
end;

function TCCRSelectorList.GetItemFollowingSelection: TCCRGridItem;
begin
  Result := nil;
  if SelCount > 0 then
    begin
      //--- Find the last selected item
      Result := Items[Items.Count-1];
      if not Result.Selected then
        Result := GetNextItem(Result, sdAbove, [isSelected]);
      //--- Find the next non-selected item
      if Assigned(Result) then
        begin
          Result := GetNextItem(Result, sdBelow, [isNone]);
          if Assigned(Result) and Result.Selected then
            Result := nil;
        end;
    end;
end;


///////////////////////////////// TCCRSourceList \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TCCRSourceList.DblClick;
begin
  inherited;
  if Assigned(Owner) and (Owner is TCCRCustomSelector) then
    TCCRCustomSelector(Owner).Add;
end;

procedure TCCRSourceList.DragDrop(Source: TObject; X, Y: Integer);
begin
  if Source <> Self then
    if (Source is TControl) and (TControl(Source).Owner = Owner) then
      if Assigned(Owner) and (Owner is TCCRCustomSelector) then
        TCCRCustomSelector(Owner).Remove;
end;

procedure TCCRSourceList.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited;
  if Shift = [] then
    if (Key = VK_RETURN) or (Key = Word(' ')) then
      if Assigned(Owner) and (Owner is TCCRCustomSelector) then
        TCCRCustomSelector(Owner).Add;
end;

end.
