{******************************************************************************}
{ Package:      Clinical Case Registries Custom Components                     }
{ Date Created: November 18, 2004                                              }
{ Site Name:    Hines OIFO                                                     }
{ Developers:   Sergey Gavrilov                                                }
{ Description:  This unit defines base classes for clinical context (CCOW)     }
{               related components.                                            }
{ Note:                                                                        }
{******************************************************************************}

unit uROR_CustomContextor;

{$I Components.inc}

interface

uses
  Dialogs, SysUtils, Classes, Controls, OleCtrls, VERGENCECONTEXTORLib_TLB, unicode,
   ExtCtrls, ImgList, uROR_CmdLineParams, uROR_Classes;

type
  TCCRCustomContextorIndicator = class;

  {============================= TContextLinkStatus ============================
    Overview:     Codes that describe extended status of a clinical context link.
    SeeAlso:      TCCRCustomContextorIndicator.LinkStatus

    Note: Do not change order of items in this list!
  }
  TCCRContextLinkStatus = (

    clsBroken,        { Link is broken (a contextor is suspended). The
                        application is not notified about context changes and
                        can update a local context without affecting other
                        applications. }

    clsChanging,      { Link is active and there is a context change in
                        progress. }

    clsLink,          { Link is active and the application participates in
                        context. }

    clsUnknown        { Status of a link cannot be determined or a contextor is
                        not running. }

  );

  {============================ TContextorResumeMode ===========================
    Overview:     Codes that specify how a contextor returns from a suspended
                  state.
    SeeAlso:      TCCRCustomContextor.Resume; TCCRCustomContextor.Suspend
  }
  TCCRContextorResumeMode = (

    crmAskUser,       { Ask a user if they want to synchronise the application
                        with the current clinical context or update the latter
                        with the application data. }

    crmUseGlobalData, { Synchronise the application with the current clinical
                        context. }

    crmUseAppData     { Update the current clinical context with the application
                        data. }

  );

  {======================== TCCRContextorControlPending ========================
    Overview:     Event type for an application survey.
    SeeAlso:      TCCRCustomContextor.OnPending
    Description:
      This event is generated when an application is surveyed about proposed
      context changes.
  }
  TCCRContextorControlPending = procedure(aSender: TObject;
    const aContextItemCollection: IDispatch) of object;

  {============================= TCCRContextSubject ============================
    Overview:     A context subject descriptor.
    Description:
      TCCRContextSubject instances are used internally (associated with string
      list items) to store state of context subjects.
  }
  TCCRContextSubject = class(TObject)
  private

    fName:      WideString;
    fLockCount: Integer;
    fPending:   Boolean;

    function GetLocked: Boolean;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRContextSubject.
      Keywords:     Create,TCCRContextSubject
      SeeAlso:      TCCRContextSubject.Name
      Description:
        Create initializes the Name property with the value of the <i>aName</i>
        parameter.
    }
    constructor Create(const aName: WideString);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Locks a subject before editing the subject related data.
      SeeAlso:      TCCRContextSubject.Locked; TCCRContextSubject.Unlock
      Keywords:     Lock,TCCRContextSubject
      Description:
        Use Lock to indicate that your application is going to edit data
        related to a subject. This method increments the value of the internal
        counter associated with the subject and returns True if the counter was
        not locked before calling this method.
    }
    function Lock: Boolean;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates if a subject is locked.
      SeeAlso:      TCCRContextSubject.Lock; TCCRContextSubject.Unlock
      Keywords:     Locked,TCCRContextSubject
      Description:
        Value of the Locked property indicates if the subject is locked and
        related data is edited by the application.
    }
    property Locked: Boolean  read GetLocked;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Subject name.
      SeeAlso:      TCCRContextSubject.Create
      Keywords:     Name,TCCRContextSubject
      Description:
        Use Name to get the subject name. This is a read-only property.
    }
    property Name: WideString  read fName;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Reserved
      SeeAlso:
      Keywords:     Pending,TCCRContextSubject
      Description:
    }
    property Pending: Boolean  read fPending;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Unlocks a subject after editing the subject related data.
      SeeAlso:      TCCRContextSubject.Lock; TCCRContextSubject.Locked
      Keywords:     Unlock,TCCRContextSubject
      Description:
        Use Unlock to indicate that your application finished editing data
        related to a subject. This method decrements the value of the internal
        counter associated with the subject and returns True if the counter
        becomes unlocked.
    }
    function Unlock: Boolean;

  end;

  {============================ TCCRCustomContextor ============================
    Overview:     Base class for a Vergence contextor wrapper.
    SeeAlso:      TCCRContextor; TCCRCustomContextorIndicator
    Description:
      The TCCRCustomContextor class encapsulates basic functionality of
      a custom wrapper for the Vergence contextor component developed by
      <a href="http://www.sentillion.com">Sentillion</a>. It works as a proxy
      and frees applications of necessity to check if the Vergence desktop
      components are installed before using them. It also introduces
      suspend/resume event handlers and enhances the contextor resume
      functionality.
      <p>
      Do not create instances of TCCRCustomContextor. To put a contextor on
      a form, use a descendant of TCCRCustomContextor, such as TCCRContextor.
  }
  TCCRCustomContextor = class(TComponent)
  private

    fApplicationName:       WideString;
    fCmdLineParams:         TCCRCustomCmdLineParams;
    fContextor:             TContextorControl;
    fContextSubjects:       TWideStringList;
    fEnabled:               Boolean;
    fForcedChange:          Boolean;
    fHasBeenRun:            Boolean;
    fIndicators:            TList;
    fLocalContext:          IContextItemCollection;
    fNotificationSubjects:  TWideStringList;
    fNotRunningReason:      String;
    fOnCanceled:            TNotifyEvent;
    fOnCommitted:           TNotifyEvent;
    fOnPending:             TCCRContextorControlPending;
    fOnResumed:             TNotifyEvent;
    fOnSuspended:           TNotifyEvent;
    fPassCode:              WideString;
    fPendingContext:        IContextItemCollection;
    fSurvey:                Boolean;
    fSuspensionCount:       Integer;
    fIsChanging:            Boolean;

    function  GetApplicationName: WideString;
    function  GetContextor: TContextorControl;
    function  GetNotificationFilter: WideString;
    function  GetPassCode: WideString;
    function  GetContextSubject(const aSubject: WideString): TCCRContextSubject;
    procedure ProcessContext(EnforceKnown: Boolean = False);
    procedure ProcessOnCanceled(aSender: TObject);
    procedure ProcessOnCommitted(aSender: TObject);
    procedure ProcessOnPending(aSender: TObject;
      const aContextItemCollection: IDispatch);
    procedure SetApplicationName(const aValue: WideString);
    procedure SetCmdLineParams(aValue: TCCRCustomCmdLineParams);
    procedure SetNotificationFilter(const aValue: WideString);
    procedure SetPassCode(const aValue: WideString);
    procedure SetPendingContext(const aValue: IContextItemCollection);
    procedure SetSurvey(const aValue: Boolean);

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Associates a status indicator with a contextor.
      SeeAlso:      TCCRCustomContextor.Indicators;
                    TCCRCustomContextor.RemoveIndicator
      Description:
        The AddIndicator method associates a clinical link status indicator
        referenced by the <i>anIndicator</i> parameter with a contextor.
        If the indicator is enabled, it starts reflecting contextor state.
        <p>
        Descendant classes that override AddIndicator should always call the
        inherited method.
    }
    procedure AddIndicator(anIndicator: TCCRCustomContextorIndicator); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Checks command-line parameters and modifies contextor
                    properties accordingly.
      SeeAlso:      TCCRCustomContextor.CmdLineParams;
                    TCCRCustomContextor.Enabled;
                    TCCRCustomContextor.NotificationFilter
      Description:
        If the CmdLineParams property references a descendant of
        TCCRCustomCmdLineParams, the CheckCmdLineParams method analyses
        the application's command-line parameters specified by a user and
        modifies the values of contextor properties accordingly.
        <p>
        NoCCOW parameter completely disables the contextor by assigning False
        to its Enabled property.
        <p>
        NoUserContext parameter disables the user context functionality by
        removing the 'user' subject from the value of the NotificationFilter
        property.
    }
    procedure CheckCmdLineParams; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates an instance of the Vergence contextor
                    control.
      SeeAlso:      TCCRCustomContextor.Contextor
      Description:
        CreateContextor destroys the current instance of the Vergence contextor
        control if it is defined, creates a new one, and assigns it to the
        Contextor property.
        <p>
        If the Enabled property is True and CreateContextor has not been called
        explicitly, then it is called automatically when the Contextor property
        is read for the first time.
    }
    procedure CreateContextor; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnCanceled event handler dispatcher.
      SeeAlso:      TCCRCustomContextor.DoCommitted;
                    TCCRCustomContextor.DoPending;
                    TCCRCustomContextor.OnCanceled
      Keywords:     DoCanceled,TCCRCustomContextor
      Description:
        Do not call this method; DoCanceled is called automatically when a
        context change proposed by another application is cancelled. As
        implemented in TCCRCustomContextor, DoCanceled calls the OnCanceled
        event handler, if defined.
        <p>
        Descendant classes that override DoCanceled should always call the
        inherited method.
    }
    procedure DoCanceled; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnCommitted event handler dispatcher.
      SeeAlso:      TCCRCustomContextor.DoCanceled;
                    TCCRCustomContextor.DoPending;
                    TCCRCustomContextor.OnCommitted
      Keywords:     DoCommitted,TCCRCustomContextor
      Description:
        Do not call this method; DoCommitted is called automatically when a
        context change proposed by another application is committed. A local
        context is updated and known subjects are processed beforehand. As
        implemented in TCCRCustomContextor, DoCommitted calls the OnCommitted
        event handler, if defined.
        <p>
        Descendant classes that override DoCommitted should always call the
        inherited method.
    }
    procedure DoCommitted;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnPending event handler dispatcher.
      SeeAlso:      TCCRCustomContextor.DoCanceled;
                    TCCRCustomContextor.DoCommitted;
                    TCCRCustomContextor.OnPending;
                    TCCRCustomContextor.SetSurveyResponse
      Keywords:     DoPending,TCCRCustomContextor
      Description:
        Do not call this method; DoPending is called automatically when another
        application proposes context changes. The <i>aContextItemCollection</i>
        parameter lists context items that are going to be updated and their
        proposed values. As implemented in TCCRCustomContextor, DoPending calls
        the OnPending event handler, if defined.
        <p>
        Use SetSurveyResponse to warn the user about unsaved data or another
        condition that might make the context change undesirable.
        <p>
        Descendant classes that override DoPending should always call the
        inherited method.
    }
    procedure DoPending(const aContextItemCollection: IDispatch); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnResumed event handler dispatcher.
      SeeAlso:      TCCRCustomContextor.DoSuspend;
                    TCCRCustomContextor.OnResumed;
                    TCCRCustomContextor.Resume
      Keywords:     DoResumed,TCCRCustomContextor
      Description:
        Do not call this method; DoResumed is called automatically after
        an application resumes its participation in a clinical context. As
        implemented in TCCRCustomContextor, DoResumed calls the OnResumed
        event handler, if defined.
        <p>
        Descendant classes that override DoResumed should always call the
        inherited method.
    }
    procedure DoResumed; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnSuspended event handler dispatcher.
      SeeAlso:      TCCRCustomContextor.DoResume;
                    TCCRCustomContextor.OnSuspended;
                    TCCRCustomContextor.Suspend
      Keywords:     DoSuspended,TCCRCustomContextor
      Description:
        Do not call this method; DoSuspended is called automatically after
        an application breaks a link and suspends its participation in a
        clinical context. As implemented in TCCRCustomContextor, DoSuspended
        calls the OnSuspended event handler, if defined.
        <p>
        Descendant classes that override DoSuspended should always call the
        inherited method.
    }
    procedure DoSuspended; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the current clinical context.
      SeeAlso:      TCCRCustomContextor.Contextor;
                    TCCRCustomContextor.LocalContext
      Description:
        If an application participates in a clinical context, GetCurrentContext
        returns a collection of global context items. If a context link is
        broken (the contextor is suspended) or the contextor has not been run,
        then this method returns a local context stored in the LocalContext
        property.
    }
    function GetCurrentContext: IContextItemCollection;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns a current state of an internal Vergence contextor.
      SeeAlso:      TCCRCustomContextor.Contextor; TCCRCustomContextor.Enabled;
                    TCCRCustomContextor.Run; TCCRCustomContextor.State
      Description:
        GetState returns a current state of a Vergence contextor:
        <p>csNotRunning - Contextor is disabled or has not been run and the
          application has not joined a context.
        <p>csParticipating - Contextor has been run and the application is
          participating in a context.
        <p>csSuspended - Contextor has been run, but is not participating in
          a context.
    }
    function GetState: TOleEnum; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Responds to notifications that components are being created
                    or destroyed.
      SeeAlso:      TControl.Notification
      Keywords:     Notification,TCCRCustomContextor
      Description:
        Do not call the Notification method in an application.  Notification is
        called automatically when the component specified by <i>aComponent</i>
        is about to be inserted or removed, as specified by <i>Operation</i>.
        <p>
        TCCRCustomContextor overrides this method in order to clear its
        CmdLineParams property when the control it refers to is destroyed.
    }
    procedure Notification(aComponent: TComponent;
      anOperation: TOperation); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Detaches a status indicator from a contextor.
      SeeAlso:      TCCRCustomContextor.AddIndicator;
                    TCCRCustomContextor.Indicators
      Description:
        The RemoveIndicator method detaches a status indicator referenced by
        the <i>anIndicator</i> parameter from a contextor. The indicator stops
        reflecting contextor state.
        <p>
        Descendant classes that override RemoveIndicator should always call the
        inherited method.
    }
    procedure RemoveIndicator(anIndicator: TCCRCustomContextorIndicator); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Removes a subject from a context.
      Description:
        RemoveSubject removes all items with a subject specified by the
        <i>aSubject</i> parameter from a context collection referenced by the
        <i>aContext</i>.
        <p>
        <b>Note:</b> If Vergence desktop components are not installed on the
        user's workstation, calling this method raises an exception.
    }
    procedure RemoveSubject(const aContext: IContextItemCollection;
      const aSubject: WideString);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the Enabled property.
      SeeAlso:      TCCRCustomContextor.CreateContextor;
                    TCCRCustomContextor.Enabled; TCCRCustomContextor.Run
      Keywords:     SetEnabled,TCCRCustomContextor
      Description:
        SetEnabled is the protected write implementation of the Enabled property.
        As implemented in TCCRCustomContextor, SetEnabled destroys an internal
        Vergence contextor control when a TCCRCustomContextor is disabled. When
        a TCCRCustomContextor is enabled, SetEnabled recreates the internal
        control and runs the contextor (only if the Run method was explicitly
        called before).
        <p>
        When overriding SetEnabled, be sure to call the inherited method.
    }
    procedure SetEnabled(aValue: Boolean); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Reference to an application command-line parameters object.
      SeeAlso:      TCCRCustomContextor.CheckCmdLineParams
      Description:
        If a reference to a command-line parameters object is assigned to this
        property, then contextor properties are automatically modified according
        to the command line parameters specified by the user. See the
        CheckCmdLineParams method for more details.
    }
    property CmdLineParams: TCCRCustomCmdLineParams
      read fCmdLineParams  write SetCmdLineParams;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Internal copy of the clinical context.
      SeeAlso:      TCCRCustomContextor.EndContextChange;
                    TCCRCustomContextor.NotificationFilter;
                    TCCRCustomContextor.Resume;
                    TCCRCustomContextor.UpdateLocalContext
      Description:
        When a contextor is participating in a clinical context, content of the
        LocalContent collection reflects that of the global clinical context
        filtered according to the value of the NotificationFilter property.
        <p>
        When an application breaks a link and suspends its participation in a
        clinical context, the EndContextChange method continues to apply
        application's context updates to the local copy of the context stored
        in this property. Then, the Resume method uses the LocalContext value
        to synchronise the global context with the application one if the
        crmUseAppData mode is specified.
        <p>
        <b>Note:</b> This property is nil if the contextor has not been run.
    }
    property LocalContext: IContextItemCollection  read fLocalContext;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     This is an internal string list representation of the
                    NotificationFilter property.
      SeeAlso:      TCCRCustomContextor.Contextor;
                    TCCRCustomContextor.NotificationFilter
      Description:
        The NotificationSubjects property holds a list of subjects included in
        a notification filter. The NotificationFilter property of the Vergence
        contextor is synchronized with this property when the contextor is
        running.
    }
    property NotificationSubjects: TWideStringList  read fNotificationSubjects;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRCustomContextor.
      SeeAlso:      TComponent.Owner; TComponent.Create
      Keywords:     Create,TCCRCustomContextor
      Description:
        Create initializes an instance of the TCCRCustomContextor. <i>anOwner</i>
        is the component, typically a form, that is responsible for freeing the
        contextor.
    }
    constructor Create(anOwner: TComponent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRCustomContextor.
      SeeAlso:      TComponent.Destroy; TObject.Free
      Keywords:     Destroy,TCCRCustomContextor
      Description:
        Do not call Destroy directly in an application. Instead, call Free.
        Free verifies that the control is not nil, and only then calls Destroy.
        <p>
        Applications should only free controls explicitly when the constructor
        was called without assigning an owner to the control.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Adds an item to a pending context.
      SeeAlso:      TCCRCustomContextor.EndContextChange;
                    TCCRCustomContextor.PendingContext;
                    TCCRCustomContextor.RemoveTXItem;
                    TCCRCustomContextor.StartContextChange
      Description:
        Use AddTXItem to add an item to a pending context collection (see the
        PendingContext property) during a clinical context change transaction.
        This is the most generic way to prepare transaction data (see the
        example). Pass the item name via the <i>aName</i> parameter. The
        <i>aValue</i> parameter specifies the value of the item. If an item with
        the same name already exists in the pending context, then its value is
        updated. Otherwise, a new item is added.
        <p>
        If added item is an identifier ('id' role) and no identifiers for
        this subject have been added yet during this transaction, then all items
        with this subject are automatically removed from the pending context
        before adding the item.
        <p>
        If the Vergence desktop components are not installed, this method does
        nothing.
    }
    function AddTXItem(const aName: WideString;
      const aValue: WideString): IContextItem;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Removes context subjects from the internal list and
                    destroys them.
      SeeAlso:      TCCRCustomContextor.ContextSubjects
      Description:
        ClearContextSubjects removes context subjects from the internal list and
        destroys them. If the <i>aSubject</i> parameter is specified and not
        empty, then only the subject with this name is deleted.
    }
    procedure ClearContextSubjects(const aSubject: WideString = ''); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Displays current status of a contextor.
      SeeAlso:      TCCRCustomContextor.NotRunningReason;
                    TCCRCustomContextor.State
      Description:
        DisplayStatus shows a message dialog that indicates the current state
        of the contextor. If the contextor is not running, the message also
        includes a short exaplanation.
    }
    procedure DisplayStatus;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Completes a context change transaction.
      SeeAlso:      TCCRCustomContextor.AddTXItem;
                    TCCRCustomContextor.LocalContext;
                    TCCRCustomContextor.PendingContext;
                    TCCRCustomContextor.StartContextChange;
                    TCCRCustomContextor.State
      Keywords:     EndContextChange,TCCRCustomContextor
      Description:
        Call EndContextChange to finish a context change transaction started by
        the StartContextChange. If the <i>Commit</i> parameter is True, then the
        contextor modifies the common context according to items specified by
        the <i>aContextItemCollection</i> parameter.
        <p>
        If the <i>aContextItemCollection</i> is not specified or nil, then the
        PendingContext property value is used. See the example for a recommended
        way to perform a transaction. The collection referenced by the
        PendingContext property is destroyed before exiting this method and nil
        is assigned to the PendingContext.
        <p>
        If all applications accept the transaction, EndContextChange returns
        urCommit. If at least one of the applications accepts conditionally or
        is busy, a contextor presents the user a dialog box enabling them to
        choose whether to proceed with the context change (urCommit), cancel
        the context change (urCancel), or break the application's link with the
        context (urBreak).
        <p>
        If a contextor is suspended (State = csSuspended), EndContextChange
        affects only a local copy of the context stored in the LocalContext
        property. If the function returns urBreak, the contextor is suspended
        automatically.
        <p>
        See the Vergence <i>Application SDK Developer's Guide</i> for more
        details.
    }
    function EndContextChange(Commit: Boolean;
      const aContextItemCollection: IContextItemCollection = nil): UserResponse;
      virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns application's acccess privileges for a subject.
      Description:
        GetPrivilege returns a code that indicates application's access
        privileges for a subject specified by the <i>aSubject</i> parameter.

        This function returns one of the following values:
        <p>apNone - Indicates that the application may not get or set the
                    context data for the subject.
        <p>apGet  - Indicates that the application may get the context data for
                    the subject.
        <p>apSet  - Indicates that the application may get and/or set the
                    context data for the subject.

        See the Vergence <i>Application SDK Developer's Guide</i> for more
        details.
    }
    function GetPrivilege(const aSubject: WideString): AccessPrivilege;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Processes known subjects and assign item values to the
                    corresponding properties.
      SeeAlso:      TCCRCustomContextor.CurrentContext
      Keywords:     ProcessKnownSubjects,TCCRCustomContextor
      Description:
        When a contextor is notified about changes in the clinical context, it
        calls ProcessKnownSubjects. This method should examine items of the
        CurrentContext collection, extract values of known items (i.e. patient
        ICN), and assign them to the corresponding properties.
        <p>
        If the <i>Enforce</i> parameter is True, the properties should be
        updated (cleared) even if there are no corresponding items in the
        current context. If this parameter is False (default), those properties
        should remain unchanged.
        <p>
        As implemented in TCCRCustomContextor, ProcessKnownSubjects does nothing.
        When overriding it, be sure to call the inherited method.
    }
    procedure ProcessKnownSubjects(Enforce: Boolean = False); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Removes an item from a pending context.
      SeeAlso:      TCCRCustomContextor.AddTXItem;
                    TCCRCustomContextor.EndContextChange;
                    TCCRCustomContextor.PendingContext;
                    TCCRCustomContextor.StartContextChange
      Description:
        Use RemoveTXItem to delete an item from a pending context collection
        (see the PendingContext property) during a clinical context change
        transaction. This is the most generic way to prepare transaction data
        (see the example). The <i>aName</i> parameter specifies the name of
        deleted item.
        <p>
        If the Vergence desktop components are not installed, this method does
        nothing.
    }
    procedure RemoveTXItem(const aName: WideString);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Re-establish a link with a clinical context.
      SeeAlso:      TCCRCustomContextor.State; TCCRCustomContextor.Suspend
      Keywords:     Resume,TCCRCustomContextor
      Description:
        Use Resume to re-establish a link with a clinical context (State =
        csParticipating) that was broken during a context change transaction or
        by the Suspend method. Each suspension of a contextor increments an
        internal counter. Each Resume call decrements it. When the counter
        value reaches 0, the link is re-established. Thus, nested calls to
        Suspend and Resume are processed correctly.

        Use the <i>aMode</i> parameter to specify whether a local or the common
        clinical context has priority:
        <p>crmUseGlobalData - Synchronize the application with the common
                              context.
        <p>crmUseAppData    - Try to synchronize the common context with a local
                              application's copy.
        <p>crmAskUser       - Display a dialog box and ask the user, which of
                              the aforementioned modes should be used.
    }
    procedure Resume(aMode: TCCRContextorResumeMode);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Runs a contextor.
      SeeAlso:
      Keywords:     Run,TCCRCustomContextor
      Description:
        Assign required values to contextor's properties, and then use Run to
        initialize it. After successful execution, the application has joined
        and is participating in the context (State = csParticipating).
        <p>
        When the Enabled property is False, the Run method does nothing. Most
        of contextor specific methods do nothing or return errors before the
        contextor is run.
        <p>
        See the Vergence <i>Application SDK Developer's Guide</i> for more
        details.
    }
    procedure Run;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Conditionally accepts proposed context changes.
      SeeAlso:      TCCRCustomContextor.DoPending; TCCRCustomContextor.OnPending;
                    TCCRCustomContextor.Survey
      Description:
        When surveyed, a contrextor normally indicates that an application
        accepts proposed context changes. Use SetSurveyResponse inside an
        OnPending event handler or a DoPending method to indicate that the
        application accepts the changes only conditionally. Pass a reason for
        this via the <i>aReason</i> parameter. This message will be presented
        to the user in the "Problem Changing Clinical Data" dialog box shown
        by the contextor.
    }
    procedure SetSurveyResponse(const aReason: WideString);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Starts a context change transaction.
      SeeAlso:      TCCRCustomContextor.AddTXItem;
                    TCCRCustomContextor.EndContextChange;
                    TCCRCustomContextor.PendingContext
      Keywords:     StartContextChange,TCCRCustomContextor
      Description:
        Use StartContextChange to initiate a context change transaction. It
        prevents other application from starting another transaction until this
        one is complete. Once the application starts the transaction, the
        contextor creates a collection for proposed context data items and
        assigns it to the PendingContext property. You can use AddTXItem and
        RemoveTXItem to modify this collection. When the data is ready, call
        the EndContextChange to complete the transaction.
        <p>
        See the Vergence <i>Application SDK Developer's Guide</i> for more
        details.
    }
    function StartContextChange: Boolean; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Break a clinical link and suspend a contextor.
      SeeAlso:      TCCRCustomContextor.Resume; TCCRCustomContextor.State
      Keywords:     Suspend,TCCRCustomContextor
      Description:
        Use Suspend to break a clinical link and suspend a contextor (State =
        csSuspended). Each call of this method increments an internal counter.
        Each Resume call decrements it. When the counter value reaches 0, the
        link is re-established. Thus, nested calls to Suspend and Resume are
        processed correctly.
    }
    procedure Suspend;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Updates indicators associated with a contextor.
      SeeAlso:      TCCRCustomContextor.Indicators;
                    TCCRCustomContextorIndicator.UpdateIndicator
      Description:
        A contextor calls UpdateIndicators to status indicators associated with
        a contextor according to its state. As implemented in TCCRCustomContextor,
        UpdateIndicators calls the UpdateIndicator method of each object
        referenced by the Indicators array.
        <p>
        Descendant classes that override UpdateIndicators should always call the
        inherited method.
    }
    procedure UpdateIndicators; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Synchronizes a local context copy with the common context.
      SeeAlso:      TCCRCustomContextor.LocalContext;
                    TCCRCustomContextor.NotificationFilter;
                    TCCRCustomContextor.NotificationSubjects
      Description:
        A contextor uses UpdateLocalContext to synchronize a local copy of the
        clinical context with the context referenced by the <i>aContext</i>
        parameter according to the value fo the NotificationFilter property.
        Only items with subjects listed in the notification filter are updated
        in the local context.
        <p>
        Descendant classes that override UpdateLocalContext should always call
        the inherited method.
    }
    procedure UpdateLocalContext(const aContext: IContextItemCollection); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Name by which an application wants to be known within the
                    context system.
      Description:
        This name may be presented to users, so it is important that it clearly
        represents the application. The name can consist of alphanumeric
        characters, underscores, periods, and blanks. However, leading and
        trailing blanks are ignored.
        <p>
        To enable multiple instances of an application to participate in a
        common context, the name should contain the '#' character followed by
        an identifier that is unique for each application instance. If the name
        ends with the '#' character, a contextor automatically generates a
        unique identifier.
    }
    property ApplicationName: WideString
      read GetApplicationName  write SetApplicationName;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Internal Vergence contextor control.
      SeeAlso:      TCCRCustomContextor.CreateContextor
      Keywords:     Contextor,TCCRCustomContextor
      Description:
        Contextor can be used to directly access methods and properties of the
        Vergence contextor control that does most of the context related work.
        This property is nil until the contextor is run or when a contextor is
        disabled (Enabled = False).
    }
    property Contextor: TContextorControl  read GetContextor;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Current clinical context.
      SeeAlso:      TCCRCustomContextor.LocalContext
      Description:
        CurrentContext references a collection that represents a current
        clinical context. If a contextor is suspended (State = csSuspended),
        this property references the same object as the LocalContext.
    }
    property CurrentContext: IContextItemCollection  read GetCurrentContext;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Current state of context subjects.
      SeeAlso:      TCCRCustomContextor.ClearContextSubjects
      Description:
        Elements of this array reflect current state of context subjects
        (locked, pending, etc.). If an application tries to access a
        non-exisiting element, it is automatically created.
    }
    property ContextSubjects[const aSubject: WideString]: TCCRContextSubject
      read GetContextSubject;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Controls whether the internal Vergence contextor is running
                    or not.
      SeeAlso:      TCCRCustomContextor.CmdLineParams;
                    TCCRCustomContextor.Contextor;
                    TCCRCustomContextor.CreateContextor;
                    TCCRCustomContextor.Run
      Keywords:     Enabled,TCCRCustomContextor
      Description:
        Enabled controls whether the internal Vergence contextor is run or not.
        Use it to completely disable CCOW functionality (e.g. if a command-line
        parameter is specified). If this property is False, the Run method does
        not initialize the internal contextor.
        <p>
        If this property changes its value from True to False after the
        contextor has been run, the internal contextor control is destroyed and
        nil is assigned to the Contextor property.
        <p>
        If this property changes its value from False to True after the
        contextor has been run at least once, the internal contextor control is
        recreated by the CreateContextor and then initialized by the Run method.
        <p>
        The State property always has the csNotRunning value while Enabled is
        False.
    }
    property Enabled: Boolean  read fEnabled  write SetEnabled  default True;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates a "forced" context change.
      SeeAlso:      TCCRCustomContextor.Resume;
                    TCCRCustomContextor.SetSurveyResponse;
                    TCCRCustomContextor.Survey
      Description:
        True value of this property indicates that the current context change
        is a forced" one. This means that a modal dialog box(es) should be
        closed if open) and unsaved data should be discarded.
        <p>
        If an application does not want to be surveyed (Survey = False), then
        all context changes are marked as "forced" by default.
        <p>
        Otherwise, the default value of the ForcedChange property is False. True
        is assigned to this property when a link is re-established by the Resume
        method with crmUseGlobalData parameter or SetSurveyResponse is used to
        conditionally accept proposed context changes.
    }
    property ForcedChange: Boolean  read fForcedChange;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     List of status indicators associated with a contextor.
      SeeAlso:      TCCRCustomContextor.AddIndicator;
                    TCCRCustomContextor.RemoveIndicator
      Description:
        When an indicator is associated with a contextor, it is added to this
        list. All these indicators reflect the current state of the contextor
        and clinical link.
    }
    property Indicators: TList  read fIndicators;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates that a context change transaction is in progress.
      SeeAlso:      TCCRCustomContextor.EndContextChange;
                    TCCRCustomContextor.StartContextChange
      Description:
        StartContextChange assigns True to the IsChanging property before
        starting a transactions and EndContextChange resets it to False after
        the transaction is complete.
    }
    property IsChanging: Boolean  read fIsChanging;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     List of subjects that an application wants to be notified
                    about.
      SeeAlso:      TCCRCustomContextor.CmdLineParams;
                    TCCRCustomContextor.NotificationSubjects
      Description:
        The value of the NotificationFilter filter property is semicolon
        separated list of subject names. The application will be notified about
        changes to items with these subjects.
        <p>
        If the application wants to be notified about all subjects, then the
        value of this property should be a string containing the single '*'
        character.
        <p>
        If the application does not want to be notified about any subjects,
        then the value of the NotificationFilter should be an empty string.
    }
    property NotificationFilter: WideString
      read GetNotificationFilter  write SetNotificationFilter;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates a reason for not running contextor.
      SeeAlso:      TCCRCustomContextor.DisplayStatus; TCCRCustomContextor.Run;
                    TCCRCustomContextor.State
      Description:
        If a contextor is not running (State = csNotRunning), NotRunningReason
        provides a short explanation for this.
    }
    property NotRunningReason: String
      read fNotRunningReason  write fNotRunningReason;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs when a context change transaction is cancelled.
      SeeAlso:      TCCRCustomContextor.DoCanceled;
                    TCCRCustomContextor.OnCommitted;
                    TCCRCustomContextor.OnPending
      Description:
        Write an OnCanceled event handler to add custom processing after a
        context change transaction instigated by an application is cancelled by
        a user. <i>aSender</i> is the contextor object whose event handler is
        called.
    }
    property OnCanceled: TNotifyEvent  read fOnCanceled  write fOnCanceled;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs when a context change transaction is committed.
      SeeAlso:      TCCRCustomContextor.DoCommitted;
                    TCCRCustomContextor.OnCanceled;
                    TCCRCustomContextor.OnPending
      Description:
        Write an OnCommitted event handler to add custom processing after a
        context change transaction instigated by an application is committed.
        <i>aSender</i> is the contextor object whose event handler is called.
    }
    property OnCommitted: TNotifyEvent  read fOnCommitted  write fOnCommitted;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs when an application is surveyed about proposed
                    context changes.
      SeeAlso:      TCCRCustomContextor.DoPending;
                    TCCRCustomContextor.OnCanceled;
                    TCCRCustomContextor.OnCommitted;
                    TCCRCustomContextor.SetSurveyResponse
      Description:
        Write an OnPending event handler to respond on surveys about proposed
        context changes. <i>aSender</i> is the contextor object whose event
        handler is called. The <i>aContextItemCollection</i> parameter lists
        context items that are going to be updated and their proposed values.
        <p>
        Use SetSurveyResponse to warn the user about unsaved data or another
        condition that might make the context change undesirable.
    }
    property OnPending: TCCRContextorControlPending
      read fOnPending  write fOnPending;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs when an application re-joins a context.
      SeeAlso:      TCCRCustomContextor.DoResumed;
                    TCCRCustomContextor.OnSuspended;
                    TCCRCustomContextor.Resume
      Description:
        Write an OnResumed event handler to add custom processing after a
        contextor is resumed and clinical link is re-established.
        <i>aSender</i> is the contextor object whose event handler is called.
    }
    property OnResumed: TNotifyEvent  read fOnResumed  write fOnResumed;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs when an application breaks a link to a clinical
                    context.
      SeeAlso:      TCCRCustomContextor.DoSuspended;
                    TCCRCustomContextor.OnResumed;
                    TCCRCustomContextor.Suspend
      Description:
        Write an OnSuspended event handler to add custom processing after a
        clinical link is broken and contextor is suspended. <i>aSender</i>
        is the contextor object whose event handler is called.
    }
    property OnSuspended: TNotifyEvent  read fOnSuspended  write fOnSuspended;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Application's password for secure subjects.
      Description:
        An application needs a password to access secure context subjects. If
        this property is empty, a contextor assumes that the application only
        needs to accesss context data for non-secure subjects.
        <p>
        See the Vergence <i>Application SDK Developer's Guide</i> for more
        details.
    }
    property PassCode: WideString  read GetPassCode  write SetPassCode;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     A buffer for proposed context changes.
      SeeAlso:      TCCRCustomContextor.AddTXItem;
                    TCCRCustomContextor.RemoveTXItem;
                    TCCRCustomContextor.StartContextChange;
                    TCCRCustomContextor.EndContextChange
      Description:
        During a context change transaction (between the StartContextChange and
        EndContextChange), the PendingContext property stores a collection of
        context items that are going to be updated and their proposed values.
        Use AddTXItem and RemoveTXItem to update content of this collection.
        The EndContextChange method uses value of this property if its second
        parameter is not specified.
        <p>
        Outside of a transaction, this property is nil.
    }
    property PendingContext: IContextItemCollection
      read fPendingContext  write SetPendingContext;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates a current state of a contextor.
      SeeAlso:      TCCRCustomContextor.Enabled; TCCRCustomContextor.GetState
      Keywords:     State,TCCRCustomContextor
      Description:
        <p>csNotRunning - Contextor is disabled or has not been run and the
          application has not joined a context.
        <p>csParticipating - Contextor has been run and the application is
          participating in a context.
        <p>csSuspended - Contextor has been run, but is not participating in
          a context.
    }
    property State: TOleEnum  read GetState;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies whether an application wants to be surveyed about
                    proposed context changes.
      SeeAlso:      TCCRCustomContextor.DoPending;
                    TCCRCustomContextor.NotificationFilter;
                    TCCRCustomContextor.OnPending;
      Description:
        Assign True to this property (default) if you want the application to be
        surveyed about proposed context changes. If the value of the Survey
        property is False, neither the DoPending method nor the OnPending event
        handler is called and application always uncoditionally accept context
        changes.
    }
    property Survey: Boolean  read fSurvey  write SetSurvey  default True;

  end;

  {======================== TCCRCustomContextorIndicator =======================
    Overview:     Base class for a contextor state indicator.
    SeeAlso:      TCCRContextorIndicator; TCCRCustomContextor.Indicators
    Description:
      TCCRCustomContextorIndicator encapsulates basic functionality of a status
      indicator that can be attached to a contextor object. The indicator
      reflects a current state of the contextor (status of a clinical link).
        <p>
      Do not create instances of TCCRCustomContextorIndicator. To put an
      indicator on a form, use a descendant of TCCRCustomContextorIndicator,
      such as TCCRContextorIndicator.
  }
  TCCRCustomContextorIndicator = class(TCustomPanel)
  private

    fContextor:        TCCRCustomContextor;
    fDefaultResources: TCCRSingleton;
    fImages:           TImageList;
    fLinkStatus:       TCCRContextLinkStatus;
    fOnUpdate:         TNotifyEvent;

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Initializes the control after it is loaded from a stream.
      SeeAlso:      TControl.Loaded
      Keywords:     Loaded,TCCRCustomContextorIndicator
      Description:
        The VCL streaming system calls Loaded automatically after the controls
        form is loaded into memory so that the control can complete any
        initializations that depend on other objects in the form.
        TCCRCustomContextorIndicator overrides this method in order to load
        default images for different link states.
    }
    procedure Loaded; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Responds to notifications that components are being created
                    or destroyed.
      SeeAlso:      TControl.Notification
      Keywords:     Notification,TCCRCustomContextorIndicator
      Description:
        Do not call the Notification method in an application.  Notification is
        called automatically when the component specified by <i>aComponent</i>
        is about to be inserted or removed, as specified by <i>Operation</i>.
        <p>
        TCCRCustomContextorIndicator overrides this method in order to update
        its Contextor property when the control it refers to is destroyed.
    }
    procedure Notification(aComponent: TComponent;
      Operation: TOperation); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the Contextor property.
      SeeAlso:      TCCRCustomContextorIndicator.Contextor
      Description:
        SetContextor is the protected write implementation of the Contextor
        property.
        <p>
        When overriding SetContextor, be sure to call the inherited method.
    }
    procedure SetContextor(const aValue: TCCRCustomContextor); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the Enabled property.
      SeeAlso:      TControl.Enabled; TControl.SetEnabled;
                    TCCRCustomContextorIndicator.UpdateIndicator
      Keywords:     SetEnabled,TCCRCustomContextorIndicator
      Description:
        SetEnabled is the protected write implementation of the Enabled property.
        TCCRCustomContextorIndicator overrides this method in order to update
        an indicator when a control state changes.
        <p>
        When overriding SetEnabled, be sure to call the inherited method.
    }
    procedure SetEnabled(aValue: Boolean); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Updates an indicator according to the contextor status.
      SeeAlso:      TCCRCustomContextorIndicator.Contextor;
                    TCCRCustomContextorIndicator.LinkStatus;
                    TCCRCustomContextorIndicator.OnUpdate;
                    TControl.Enabled
      Description:
        If an indicator is enabled and attached to a contextor, UpdateIndicator
        checks if the current contextor status code is different from that
        stored in the LinkStatus property. If this is so, it repaints itself
        with a new bitmap and updates the values of the LinkStatus.
        <p>
        UpdateIndicator also calls the OnUpdate event handler, if defined.
    }
    procedure UpdateIndicator; virtual;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of
                    TCCRCustomContextorIndicator.
      SeeAlso:      TComponent.Owner; TCustomPanel.Create
      Keywords:     Create,TCCRCustomContextorIndicator
      Description:
        The Create constructor initializes an instance of the
        TCCRCustomContextorIndicator. <i>anOwner</i> is the component,
        typically a form, that is responsible for freeing the contextor control.
    }
    constructor Create(anOwner: TComponent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRCustomContextorIndicator.
      SeeAlso:      TCustomControl.Destroy; TObject.Free
      Keywords:     Destroy,TCCRCustomContextorIndicator
      Description:
        Do not call Destroy directly. Instead, call Free, which checks that the
        TCCRCustomContextorIndicator reference is not nil before calling Destroy.
    }
    destructor  Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     A contextor object that an indicator is attached to.
      SeeAlso:      TCCRCustomContextor.AddIndicator;
                    TCCRCustomContextor.RemoveIndicator
      Keywords:     Contextor,TCCRCustomContextorIndicator
      Description:
        Use Contextor to access properties of a contextor object that an
        indicator is attahced to. If the indicator is enabled, it reflects
        status of this contextor.
    }
    property Contextor: TCCRCustomContextor  read fContextor  write SetContextor;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     A list of bitmaps that correspond to the contextor states.
      SeeAlso:      TCCRCustomContextorIndicator.LinkStatus;
                    TCCRCustomContextorIndicator.Loaded;
                    TCCRCustomContextorIndicator.UpdateIndicator
      Keywords:     Images,TCCRCustomContextorIndicator
      Description:
        Bitmaps stored in the Images list should be ordered as follow:
        <p>0 - Link is broken and contextor is suspended
               (LinkStatus = clsBroken).
        <p>1 - Transaction in progress (LinkStatus = clsChanging).
        <p>2 - Participating in context (LinkStatus = clsLink).
        <p>3 - Status is unknown or contextor is not running
               (LinkStatus = clsUnknown).
        <p>
        If this property is not defined during design time and not intialized
        in a constructor, standard images are loaded.
    }
    property Images: TImageList  read fImages  write fImages;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Status of a clinical link (state of a contextor).
      SeeAlso:      TCCRCustomContextor.State;
                    TCCRCustomContextorIndicator.Contextor;
                    TCCRCustomContextorIndicator.UpdateIndicator
      Description:
        value of the LinkStatus property reflects current status of a clinical
        link (state of a contextor that an indicator is attached to):
        <p>clsBroken   - The link is broken and the contextor is suspended.
        <p>clsChanging - A transaction is in progress.
        <p>clsLink     - Participating in context.
        <p>clsUnknown  - Status is unknown or the contextor is not running.
    }
    property LinkStatus: TCCRContextLinkStatus  read fLinkStatus;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs just before a control is repainted after status
                    change.
      SeeAlso:      TCCRCustomContextorIndicator.UpdateIndicator
      Keywords:     OnUpdate,TCCRCustomContextorIndicator
      Description:
        Write an OnUpdate event handler to add custom processing after the
        LinkStatus property is updated but before a control is repainted to
        reflect the new status.
    }
    property OnUpdate: TNotifyEvent  read fOnUpdate  write fOnUpdate;

  end;

///////////////////////////////// Implementation \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

implementation

uses
  Forms, StrUtils, uROR_Resources, uROR_Utilities, Graphics;

const
  HL7ORG = '[hl7.org]';

type

  {======================= TCCRContextorDefaultResources =======================
    Overview:     Singleton for default images.
    Description:
      TCCRContextorDefaultResources handles a single copy of default images for
      contextor indicators.
  }
  TCCRContextorDefaultResources = class(TCCRSingleton)
  private

    fImages: TImageList;

  protected

    procedure Finalize; override;
    procedure Initialize; override;

  public

    property Images: TImageList  read fImages;

  end;

///////////////////////// TCCRContextorDefaultResources \\\\\\\\\\\\\\\\\\\\\\\\

procedure TCCRContextorDefaultResources.Finalize;
begin
  FreeAndNil(fImages);
  inherited;
end;

procedure TCCRContextorDefaultResources.Initialize;
begin
  inherited;
  fImages := TImageList.Create(nil);
  with fImages do
    begin
      ResInstLoad(HInstance, rtBitmap, 'CCOW_BROKEN_16',     clFuchsia);
      ResInstLoad(HInstance, rtBitmap, 'CCOW_CHANGING_16',   clFuchsia);
      ResInstLoad(HInstance, rtBitmap, 'CCOW_LINK_16',       clFuchsia);
      ResInstLoad(HInstance, rtBitmap, 'CCOW_NOTRUNNING_16', clFuchsia);
    end;
end;

////////////////////////////// TCCRContextSubject \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRContextSubject.Create(const aName: WideString);
begin
  inherited Create;
  fName := aName;
end;

function TCCRContextSubject.GetLocked: Boolean;
begin
  Result := (fLockCount > 0);
end;

function TCCRContextSubject.Lock: Boolean;
begin
  Result := Locked; 
  if fLockCount = 0 then
    fPending := False;
  Inc(fLockCount);
end;

function TCCRContextSubject.Unlock: Boolean;
begin
  Dec(fLockCount);
  Result := Locked; 
end;

////////////////////////////// TCCRCustomContextor \\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRCustomContextor.Create(anOwner: TComponent);
begin
  inherited;
  fNotRunningReason := RSC0007;

  fContextSubjects := TWideStringList.Create;
  fContextSubjects.Sorted := True;
  fContextSubjects.Duplicates := dupError;

  fNotificationSubjects := TWideStringList.Create;
  fNotificationSubjects.Delimiter := ';';
  fNotificationSubjects.Sorted    := True;
  NotificationFilter  := 'Patient;User';

  fEnabled            := True;
  fForcedChange       := False;
  fIndicators         := TList.Create;
  fIsChanging         := False;
  fLocalContext       := nil;
  fPendingContext     := nil;
  fSurvey             := True;
  fSuspensionCount    := 0;
end;

destructor TCCRCustomContextor.Destroy;
begin
  FreeAndNil(fContextor);
  FreeAndNil(fIndicators);
  FreeAndNil(fNotificationSubjects);

  ClearContextSubjects;
  FreeAndNil(fContextSubjects);

  inherited;
end;

procedure TCCRCustomContextor.AddIndicator(
  anIndicator: TCCRCustomContextorIndicator);
begin
  if Assigned(Indicators) and Assigned(anIndicator) then
    begin
      Indicators.Add(anIndicator);
      FreeNotification(anIndicator);
    end;
end;

function TCCRCustomContextor.AddTXItem(const aName: WideString;
  const aValue: WideString): IContextItem;
begin
  if Assigned(PendingContext) then
    begin
      // Check if the item with the provided name already
      // exists in the transaction context
      Result := PendingContext.Present(aName);
      // Otherwise, create a new item
      if not Assigned(Result) then
        Result := CoContextItem.Create;
      // Initialize/Update the item and add it to the transaction context
      Result.Name  := aName;
      Result.Value := aValue;
      PendingContext.Add(Result);
    end
  else
    Result := nil;
end;

procedure TCCRCustomContextor.CheckCmdLineParams;
begin
  if Assigned(CmdLineParams) then
    begin
      SetEnabled(Enabled);
      SetNotificationFilter(NotificationFilter);
    end;
end;

procedure TCCRCustomContextor.ClearContextSubjects(const aSubject: WideString);
var
  i: Integer;
begin
  with fContextSubjects do
    if aSubject <> '' then
      begin
        i := IndexOf(aSubject);
        if i >= 0 then
          begin
            Objects[i].Free;
            Objects[i] := nil;
            Delete(i);
          end;
      end
    else
      begin
        for i:=0 to Count-1 do
          begin
            Objects[i].Free;
            Objects[i] := nil;
          end;
        Clear;
      end;
end;

procedure TCCRCustomContextor.CreateContextor;
begin
  //--- Destroy the current contextor
  FreeAndNil(fContextor);
  //--- Create a new one
  try
    fContextor := TContextorControl.Create(Self);
    fContextor.SetSubComponent(True);
  except
    on e: Exception do
      begin
        Enabled := False;
        fNotRunningReason := e.Message;
        FreeAndNil(fContextor);
      end;
  end;
end;


procedure TCCRCustomContextor.DisplayStatus;
var
  msg: String;
begin
  case State of
    csNotRunning:
      msg := RSC0003 + #13#13 + NotRunningReason;
    csParticipating:
      msg := RSC0004;
    csSuspended:
      msg := RSC0005;
    else
      msg := RSC0006;
  end;
  MessageDlg508('', msg, mtInformation, [mbOK], 0);
end;

procedure TCCRCustomContextor.DoCanceled;
begin
  if Assigned(OnCanceled) then
    OnCanceled(Self);
end;

procedure TCCRCustomContextor.DoCommitted;
begin
  if Assigned(OnCommitted) then
    OnCommitted(Self);
end;

procedure TCCRCustomContextor.DoPending(const aContextItemCollection: IDispatch);
begin
  if Assigned(OnPending) then
    OnPending(Self, aContextItemCollection);
end;

procedure TCCRCustomContextor.DoResumed;
begin
  if Assigned(OnResumed) then
    OnResumed(Self);
  UpdateIndicators;
end;

procedure TCCRCustomContextor.DoSuspended;
begin
  if Assigned(OnSuspended) then
    OnSuspended(Self);
  UpdateIndicators;
end;

function TCCRCustomContextor.EndContextChange(Commit: Boolean;
  const aContextItemCollection: IContextItemCollection): UserResponse;
begin
  if State = csParticipating then
    begin
      Result := urCancel;
      if Assigned(aContextItemCollection) then
        PendingContext := aContextItemCollection;
      // Cancel the transaction if there is nothing to send
      if PendingContext.Count <= 0 then
        Commit := False;
      try
        try
          Result := Contextor.EndContextChange(Commit, PendingContext);
        finally
          fIsChanging := False;
          // Reflect the suspension as a result of user selection
          if State = csSuspended then
            begin
              Inc(fSuspensionCount);
              DoSuspended;
            end
          else
            UpdateIndicators;
        end;
      except
        if Commit and (PendingContext.Count > 0) then
          Raise;
      end;
    end
  else
    Result := urCommit;

  // Update the local context
  if Commit and (State <> csNotRunning) and (Result <> urCancel) then
    UpdateLocalContext(PendingContext);

  PendingContext := nil;
end;

function TCCRCustomContextor.GetApplicationName: WideString;
begin
  if State <> csNotRunning then
    Result := Contextor.Name
  else
    Result := fApplicationName;
end;

function TCCRCustomContextor.GetContextor: TContextorControl;
begin
  if (fContextor = nil) and Enabled then
    CreateContextor;
  Result := fContextor;
end;

function TCCRCustomContextor.GetCurrentContext: IContextItemCollection;
begin
  if State = csParticipating then
    Result := Contextor.CurrentContext
  else
    Result := LocalContext;
end;

function TCCRCustomContextor.GetNotificationFilter: WideString;
begin
  Result := NotificationSubjects.DelimitedText;
end;

function TCCRCustomContextor.GetPassCode: WideString;
var
  i, n: Integer;
begin
  Result := fPassCode;
  if csWriting in ComponentState then
    begin
      n := Length(Result);
      for i:=1 to n do
        Result[i] := WideChar(Ord(Result[i]) xor $A5);
    end;
end;

function TCCRCustomContextor.GetPrivilege(
  const aSubject: WideString): AccessPrivilege;
begin
  if State <> csNotRunning then
    Result := Contextor.GetPrivilege(aSubject)
  else
    Result := apSet;
end;

function TCCRCustomContextor.GetState: TOleEnum;
begin
  if Assigned(fContextor) then
    Result := Contextor.State
  else
    Result := csNotRunning;
end;

function TCCRCustomContextor.GetContextSubject(
  const aSubject: WideString): TCCRContextSubject;
var
  i: Integer;
begin
  with fContextSubjects do
    begin
      i := IndexOf(aSubject);
      if i < 0 then
        i := AddObject(aSubject, TCCRContextSubject.Create(aSubject));
      Result := TCCRContextSubject(Objects[i])
    end;
end;

procedure TCCRCustomContextor.Notification(aComponent: TComponent;
  anOperation: TOperation);
begin
  inherited;
  if (aComponent = CmdLineParams) and (anOperation = opRemove) then
    CmdLineParams := nil;
end;

procedure TCCRCustomContextor.ProcessContext(EnforceKnown: Boolean);
var
  i, ifs: Integer;
  nf: WideString;
begin
  if State <> csNotRunning then
    begin
      LocalContext.RemoveAll;
      { If the application do not want to be surveyed then all
        context changes are mandatory and should be enforced. }
      if not Survey then
        fForcedChange := True;

      nf := NotificationFilter;
      if nf = '*' then
        begin
          { Copy all items to local context. }
          for i:=1 to CurrentContext.Count do
            LocalContext.Add(CurrentContext.Item(i).Clone);
        end
      else if nf <> '' then
        begin
          { Copy items that have subjects defined by the
            NotificationFilter property to the local context. }
          for i:=1 to CurrentContext.Count do
            with CurrentContext.Item(i) do
              begin
                if NotificationSubjects.Find(Subject, ifs) then
                  LocalContext.Add(Clone);
              end;
        end;

      ProcessKnownSubjects(EnforceKnown);
      DoCommitted;
    end;
end;

procedure TCCRCustomContextor.ProcessKnownSubjects(Enforce: Boolean);
begin
end;

procedure TCCRCustomContextor.ProcessOnCanceled(aSender: TObject);
begin
  DoCanceled;
end;

procedure TCCRCustomContextor.ProcessOnCommitted(aSender: TObject);
begin
  ProcessContext;
end;

procedure TCCRCustomContextor.ProcessOnPending(aSender: TObject;
  const aContextItemCollection: IDispatch);
begin
  if State <> csNotRunning then
    begin
      fForcedChange := False;
      DoPending(aContextItemCollection);
    end;
end;

procedure TCCRCustomContextor.RemoveIndicator(
  anIndicator: TCCRCustomContextorIndicator);
begin
  if Assigned(Indicators) and Assigned(anIndicator) then
    begin
      Indicators.Remove(anIndicator);
      RemoveFreeNotification(anIndicator);
    end;
end;

procedure TCCRCustomContextor.RemoveSubject(
  const aContext: IContextItemCollection;  const aSubject: WideString);
var
  ci: IContextItem;
  i: Integer;
  subj: WideString;
begin
  subj := AnsiReplaceText(aSubject, HL7ORG, '');
  for i:=aContext.Count downto 1 do
    begin
      ci := aContext.Item(i);
      if WideSameText(ci.Subject, subj) then
        aContext.Remove(ci.Name);
    end;
end;

procedure TCCRCustomContextor.RemoveTXItem(const aName: WideString);
begin
  if Assigned(PendingContext) then
    try
      PendingContext.Remove(aName);
    except
    end;
end;

procedure TCCRCustomContextor.Resume(aMode: TCCRContextorResumeMode);
var
  i, n: Integer;
  ci: IContextItem;
  secured, commit: Boolean;
begin
  if fSuspensionCount > 0 then
    begin
      if (fSuspensionCount = 1) and (State = csSuspended) then
        begin
          if aMode = crmAskUser then
            if MessageDlg508(RSC0001, RSC0002,
              mtConfirmation, [mbYes,mbNo], 0) = mrYes then
              aMode := crmUseAppData
            else
              aMode := crmUseGlobalData;

          Contextor.Resume;
          Dec(fSuspensionCount);
          DoResumed;

          case aMode of
            crmUseGlobalData:
              begin
                fForcedChange := True;
                ProcessContext(True);
              end;
            crmUseAppData:
              begin
                commit := False;
                n := LocalContext.Count;
                if (n > 0) and StartContextChange then
                  try
                    secured := (PassCode <> '');
                    for i:=1 to n do
                      begin
                        ci := LocalContext.Item(i);
                        try
                          // Unfortunately, the GetPrivilege method does not work
                          // in nonsecured applications (it raises an exception) :-(
                          if secured then
                            begin
                              if Contextor.GetPrivilege(ci.Subject) = apSet then
                                AddTXItem(ci.Name, ci.Value);
                            end
                          else if not WideSameText(ci.Subject, 'User') then
                            AddTXItem(ci.Name, ci.Value);
                        except
                        end;
                      end;
                    commit := True;
                  finally
                    if EndContextChange(commit) = urCancel then
                      Suspend;
                  end;
              end;
          end;
        end
      else
        Dec(fSuspensionCount);
    end;
end;

procedure TCCRCustomContextor.Run;
begin
  if not Enabled then
    fNotRunningReason := RSC0008
  else if Assigned(Contextor) and (State = csNotRunning) then
    begin
      try
        Contextor.Run(ApplicationName, PassCode, Survey, NotificationFilter);
        fHasBeenRun := True;
      except
        on e: Exception do
          begin
            Enabled := False;
            fNotRunningReason := e.Message;
            FreeAndNil(fContextor);
          end;
      end;
      if State = csParticipating then
        begin
          fLocalContext := CoContextItemCollection.Create;
          with Contextor do
            begin
              OnCanceled  := ProcessOnCanceled;
              OnCommitted := ProcessOnCommitted;
              OnPending   := ProcessOnPending;
            end;
          ProcessContext(True);
        end;
      UpdateIndicators;
    end;
end;

procedure TCCRCustomContextor.SetApplicationName(const aValue: WideString);
begin
  if State = csNotRunning then
    fApplicationName := aValue;
end;

procedure TCCRCustomContextor.SetCmdLineParams(aValue: TCCRCustomCmdLineParams);
begin
  if aValue <> fCmdLineParams then
    begin
      if Assigned(fCmdLineParams) then
        fCmdLineParams.RemoveFreeNotification(Self);

      fCmdLineParams := aValue;
      
      if Assigned(fCmdLineParams) then
        fCmdLineParams.FreeNotification(Self);

      CheckCmdLineParams;
    end;
end;

procedure TCCRCustomContextor.SetEnabled(aValue: Boolean);
begin
  if Assigned(CmdLineParams) and CmdLineParams.NoCCOW and
    not (csDesigning in ComponentState) then
      aValue := False;

  if aValue <> fEnabled then
    begin
      fEnabled := aValue;
      if fHasBeenRun then
        if aValue then
          begin
            CreateContextor;
            Run;
          end
        else
          begin
            FreeAndNil(fContextor);
            UpdateIndicators;
          end;
    end;
end;

procedure TCCRCustomContextor.SetNotificationFilter(const aValue: WideString);
var
  i: Integer;
begin
  NotificationSubjects.DelimitedText := AnsiReplaceStr(aValue, HL7ORG, '');

  if Assigned(CmdLineParams) and CmdLineParams.NoUserContext and
    not (csDesigning in ComponentState) and NotificationSubjects.Find('User', i) then
      NotificationSubjects.Delete(i);

  if State <> csNotRunning then
    Contextor.NotificationFilter := NotificationSubjects.DelimitedText;
end;

procedure TCCRCustomContextor.SetPassCode(const aValue: WideString);
var
  i, n: Integer;
begin
  fPassCode := aValue;
  if csReading in ComponentState then
    begin
      n := Length(fPassCode);
      for i:=1 to n do
        fPassCode[i] := WideChar(Ord(fPassCode[i]) xor $A5);
    end;
end;

procedure TCCRCustomContextor.SetPendingContext(
  const aValue: IContextItemCollection);
begin
  if aValue <> nil then
    fPendingContext := nil;
  fPendingContext := aValue;
end;

procedure TCCRCustomContextor.SetSurvey(const aValue: Boolean);
begin
  if State = csNotRunning then
    fSurvey := aValue;
end;

procedure TCCRCustomContextor.SetSurveyResponse(const aReason: WideString);
begin
  if (State = csParticipating) and (aReason <> '') then
    begin
      Contextor.SetSurveyResponse(aReason);
      fForcedChange := True;
    end;
end;

function TCCRCustomContextor.StartContextChange: Boolean;
begin
  Result := False;
  case State of
    csParticipating:
      begin
        fIsChanging := True;
        UpdateIndicators;
        Contextor.StartContextChange;
        PendingContext := CoContextItemCollection.Create;
        Result := True;
      end;
    csSuspended:
      begin
        PendingContext := CoContextItemCollection.Create;
        Result := True;
      end;
  end;
end;

procedure TCCRCustomContextor.Suspend;
begin
  if (fSuspensionCount = 0) and (State = csParticipating) then
    begin
      Contextor.Suspend;
      Inc(fSuspensionCount);
      DoSuspended;
    end
  else
    Inc(fSuspensionCount);
end;

procedure TCCRCustomContextor.UpdateIndicators;
var
  i, n: Integer;
begin
  n := Indicators.Count - 1;
  for i:=0 to n do
    if Assigned(Indicators[i]) then
      TCCRCustomContextorIndicator(Indicators[i]).UpdateIndicator;
  Application.ProcessMessages;
end;

procedure TCCRCustomContextor.UpdateLocalContext(
  const aContext: IContextItemCollection);
var
  lci: IContextItem;
  i: Integer;
  subjlst: TWideStringList;
begin
  if Assigned(aContext) then
    begin
      subjlst := TWideStringList.Create;
      subjlst.Sorted := True;
      subjlst.Duplicates := dupError;
      try
        for i:=1 to aContext.Count do
          begin
            lci := aContext.Item(i).Clone;
            // Remove the standard domain name (it is redundant)
            lci.Subject := AnsiReplaceStr(lci.Subject, HL7ORG, '');
            // When a new identifier is added to the local context,
            // clear the old items with the same subject
            if WideSameText(lci.Role, 'id') then
              try
                subjlst.Add(lci.Subject);
                // If the subject is already in the list then the
                // EStringListError exception is generated and
                // the following operator is not executed
                RemoveSubject(LocalContext, lci.Subject);
              except
              end;
            // Copy the item to the local context
            LocalContext.Add(lci);
            lci := nil;
          end;
      finally
        subjlst.Free;
      end;
    end;
end;

///////////////////////// TCCRCustomContextorIndicator \\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRCustomContextorIndicator.Create(anOwner: TComponent);
begin
  inherited;

  AutoSize    := False;
  BevelInner  := bvNone;
  BevelOuter  := bvNone;
  Caption     := '';
  DockSite    := False;
  Hint        := '';

  if csDesigning in ComponentState then
    fLinkStatus := clsBroken
  else
    fLinkStatus := clsUnknown;

  Constraints.MinHeight := 16;
  Constraints.MinWidth  := 16;
end;

destructor TCCRCustomContextorIndicator.Destroy;
begin
  Contextor := nil;
  FreeAndNil(fDefaultResources);
  inherited;
end;

procedure TCCRCustomContextorIndicator.Loaded;
begin
  inherited;
  if not Assigned(Images) then
    begin
      fDefaultResources := TCCRContextorDefaultResources.Create;
      Images := TCCRContextorDefaultResources(fDefaultResources).Images;
    end;
end;

procedure TCCRCustomContextorIndicator.Notification(aComponent: TComponent;
  Operation: TOperation);
begin
  inherited;
  if Operation = opRemove then
    begin
      if aComponent = Contextor then
        Contextor := nil
      else if aComponent = Images then
        Images := nil;
    end;
end;

procedure TCCRCustomContextorIndicator.SetContextor(
  const aValue: TCCRCustomContextor);
begin
  if aValue <> fContextor then
    begin
      if Assigned(fContextor) then
        fContextor.RemoveIndicator(Self);
      fContextor := aValue;
      if Assigned(fContextor) then
        fContextor.AddIndicator(Self);
    end;
end;

procedure TCCRCustomContextorIndicator.SetEnabled(aValue: Boolean);
begin
  inherited;
  UpdateIndicator;
end;

procedure TCCRCustomContextorIndicator.UpdateIndicator;
var
  newState: TCCRContextLinkStatus;
begin
  if Assigned(Contextor) and (Enabled or (fLinkStatus <> clsUnknown)) then
    begin
      newState := clsUnknown;
      if Enabled then
        case Contextor.State of
          csParticipating:
            if Contextor.IsChanging then
              newState := clsChanging
            else
              newState := clsLink;
          csSuspended:
            newState := clsBroken;
        end;
      if newState <> fLinkStatus then
        begin
          fLinkStatus := newState;
          if Assigned(OnUpdate) then
            OnUpdate(Self);
          RePaint;
        end;
    end;
end;

end.
