{******************************************************************************}
{ Package:      Clinical Case Registries Custom Components                     }
{ Date Created: January 26, 2007                                               }
{ Site Name:    Hines OIFO                                                     }
{ Developers:   Sergey Gavrilov                                                }
{ Description:  Base classes for procedure broker wrappers.                    }
{ Note:                                                                        }
{******************************************************************************}

unit uROR_CustomBroker;

{$I Components.inc}

interface

uses
  Classes, Contnrs, Controls, SysUtils, uROR_CmdLineParams, Windows;

const

  { Overview:     Remote procedure did not return any results.
    SeeAlso:      TCCRBrokerCallInfo.ErrorCode; TCCRBrokerCallInfo.ErrorType
    Description:
      This error code is returned by a procedure broker wrapper if a remote
      procedure did not return any results (Results.Count = 0). A procedure must
      return at least Results[0] to indicate success or failure.
      <p>
      At the same time, the rpeBroker code is assigned to the ErrorType.
  }
  CCRBEC_NORPCRESULTS = -1;

  { Overview:     VistA RPC Broker error.
    SeeAlso:      TCCRBrokerCallInfo.ErrorCode; TCCRBrokerCallInfo.ErrorType
    Description:
      An error occured during execution of a remote procedure. The execution
      was interrupted.
      <p>
      At the same time, the rpeBroker code is assigned to the ErrorType.
  }
  CCRBEC_RPCEXCEPTION = -2;

type
  TCCRCustomBroker = class;
  TCCRBrokerCallInfo = class;
  TCCRProcedureError = class;

  {============================ TCCRBrokerErrorType ============================
    Overview:     Types of errors that can be returned by a procedure broker.
    SeeAlso:      TCCRBrokerCallInfo.ErrorType
  }
  TCCRBrokerErrorType = (

    rpeNone = 0,      { No error(s). }

    rpeBroker,        { Internal broker error during a remote procedure call;
                        the call was not completed. }

    rpeProcedure      { A remote procedure call was completed but the procedure
                        returned error(s). }

  );



  {============================= TCCRBrokerProcMode ============================
    Overview:     Flags that control execution of a procedure call.
    SeeAlso:      TCCRCustomBroker.CallProc; TCCRCustomBroker.DefaultProcMode
  }
  TCCRBrokerProcModes = (
    rpcDefault,       { Use flags stored in the DefaultProcMode property. }

    rpcNoErrorCheck,  { Do not check for errors returned by a remote procedure.
                        This flag should be used for remote procedures that do
                        not conform to the error reporting format(s) supported
                        by the CheckProcError function. }

    rpcSilent         { Do not show any error messages (just return error
                        codes). }

  );

  TCCRBrokerProcMode = set of TCCRBrokerProcModes;
  TCCRBrokerDefaultProcModes = rpcNoErrorCheck..rpcSilent;
  TCCRBrokerDefaultProcMode = set of TCCRBrokerDefaultProcModes;

  {========================= TCCRBrokerCheckErrorEvent =========================
    Overview:     Event type for checking if a procedure returned error(s).
    SeeAlso:      TCCRCustomBroker.OnCheckProcError
    Description:
      This event is generated by a procedure broker when it checks if a
      procedure returned error(s).
  }
  TCCRBrokerCheckErrorEvent = procedure(aSender: TObject;
    CallInfo: TCCRBrokerCallInfo; var Handled: Boolean) of object;

  {============================= TCCRBrokerCallInfo ============================
    Overview:     Internal object for passing procedure information and results
                  to error processing methods.
    SeeAlso:      TCCRCustomBroker.CallProc; TCCRCustomBroker.CheckProcError;
                  TCCRCustomBroker.DoCheckProcError;
                  TCCRCustomBroker.OnCheckProcError
    Description:
      TCCRBrokerCallInfo is used internally by procedure brokers to pass
      information about a procedure call (procedure name, mode, etc.) and
      errors returned by the procedure into the error processing code.
  }
  TCCRBrokerCallInfo = class(TPersistent)
  private

    fErrorCode:     Integer;
    fErrorType:     TCCRBrokerErrorType;
    fErrors:        TObjectList;
    fProcedureName: String;
    fProcMode:      TCCRBrokerProcMode;
    fResults:       TStrings;

    function  GetError(const anIndex: Integer): TCCRProcedureError;
    function  GetErrorCount: Integer;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRBrokerCallInfo.
      SeeAlso:      TObject.Create
      Keywords:     Create,TCCRBrokerCallInfo
      Description:
        In addition to calling the inherited constructor, Create constructs and
        initializes internal objects.
    }
    constructor Create;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRBrokerCallInfo.
      SeeAlso:      TObject.Free; TPersistent.Destroy
      Keywords:     Destroy,TCCRBrokerCallInfo
      Description:
        Do not call Destroy directly in an application. Instead, call Free.
        Free verifies that the object is not nil, and only then calls Destroy.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Stores information about an error in the internal list.
      SeeAlso:      TCCRBrokerCallInfo.ErrorCount; TCCRBrokerCallInfo.Errors
      Description:
        Use AddError to create an instance of an error descriptor
        (TCCRProcedureError) and add it to the internal list. Pass the error
        code and message via the <i>Code</i> and <i>aMessage</i> parameters.
        <i>aPlace</i> parameter indicates location of the error in the remote
        procedure code. Optional description and/or additional information
        about the error can be passed via the <i>aDetails</i> parameter.
    }
    procedure AddError(const aMessage: String; const aCode: Integer = 0;
      const aPlace: String = ''; const aDetails: TStrings = nil);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Clears the internal list of errors.
      SeeAlso:      TCCRBrokerCallInfo.Errors
      Keywords:     ClearErrors,TCCRBrokerCallInfo
      Description:
        Use ClearErrors to remove all error descriptors from the list and
        destroy them.
    }
    procedure ClearErrors;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Primary error code.
      SeeAlso:      CCRBEC_NORPCRESULTS; CCRBEC_RPCEXCEPTION;
                    TCCRBrokerCallInfo.ErrorType
      Keywords:     ErrorCode,TCCRBrokerCallInfo
      Description:
        Use ErrorCode to get or set code of a primary error that occured during
        execution of a remote procedure or was returned by it. The ErrorType
        property specifies the type of the error.
    }
    property ErrorCode: Integer  read fErrorCode  write fErrorCode;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Number of errors in the internal list.
      SeeAlso:      TCCRBrokerCallInfo.Errors
      Keywords:     ErrorCount,TCCRBrokerCallInfo
      Description:
        Use ErrorCount to determine number of error descriptors stored in the
        Errors list.
    }
    property ErrorCount: Integer  read GetErrorCount;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Provides direct access to error descriptors.
      SeeAlso:      TCCRBrokerCallInfo.ErrorCount
      Keywords:     Errors,TCCRBrokerCallInfo
      Description:
        Use Errors to directly access an error descriptor in the internal list.
        <i>anIndex</i> identifies a descriptor by its position in the list,
        where 0 is the first item, 1 is the second item, and so on.
    }
    property Errors[const anIndex: Integer]: TCCRProcedureError  read GetError;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Primary error type.
      SeeAlso:      TCCRBrokerCallInfo.ErrorCode
      Keywords:     ErrorCode,TCCRBrokerCallInfo
      Description:
        Use ErrorType to get or set type of a primary error that occured during
        execution of a remote procedure or was returned by it. The ErrorCode
        property specifies the code of the error.
    }
    property ErrorType: TCCRBrokerErrorType  read fErrorType  write fErrorType;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Procedure name.
      SeeAlso:      TCCRBrokerCallInfo.ProcMode; TCCRBrokerCallInfo.Results
      Keywords:     ProcedureName,TCCRBrokerCallInfo
      Description:
        Use ProcedureName to get or set name of the remote procedure, results
        of which are being checked.
    }
    property ProcedureName: String  read fProcedureName  write fProcedureName;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Control flags that were used for the remote procedure call.
      SeeAlso:      TCCRBrokerCallInfo.ProcedureName; TCCRBrokerCallInfo.Results;
                    TCCRCustomBroker.CallProc; TCCRCustomBroker.DefaultProcMode
      Keywords:     ProcMode,TCCRBrokerCallInfo
      Description:
        Use ProcMode to get or set the combination of control flags that were
        used for the remote procedure call whose results are being checked.
    }
    property ProcMode: TCCRBrokerProcMode
      read fProcMode  write fProcMode  default [rpcDefault];

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Results of a remote procedure call.
      SeeAlso:      TCCRBrokerCallInfo.ProcedureName;
                    TCCRBrokerCallInfo.ProcMode
      Keywords:     Results,TCCRBrokerCallInfo
      Description:
        Use Results to get or set a reference to the buffer that contain results
        returned by the remote procedure call.
    }
    property Results: TStrings  read fResults  write fResults;

  end;

  {============================== TCCRBrokerLogMode ============================
    Overview:     Parameters of remote procedure calls logging.
    SeeAlso:      TCCRCustomBroker.CallProc; TCCRCustomBroker.DebugLog
    Description:
      TCCRBrokerLogMode properties specify if a CCR remote procedure broker
      records procedure calls in a debug log and what data is recorded.
  }
  TCCRBrokerLogMode = class(TPersistent)
  private

    fBrokerErrors:  Boolean;
    fCmdLineParams: TCCRCustomCmdLineParams;
    fEnabled:       Boolean;
    fLimitParams:   Word;
    fLimitResults:  Word;
    fOwner:         TCCRCustomBroker;
    fParameters:    Boolean;
    fResults:       Boolean;

    procedure SetCmdLineParams(aValue: TCCRCustomCmdLineParams);
    procedure SetEnabled(aValue: Boolean);

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Checks application command-line parameters and modifies
                    properties accordingly.
      SeeAlso:      TCCRBrokerLogMode.CmdLineParams; TCCRBrokerLogMode.Enabled
      Description:
        If the CmdLineParams property references a descendant of
        TCCRCustomCmdLineParams, the CheckCmdLineParams method analyses
        the application's command-line parameters specified by a user and
        modifies the property values accordingly.
        <p>
        Debug parameter enables logging of procedure calls by assigning True to
        the Enabled property.
    }
    procedure CheckCmdLineParams;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Reference to an application command-line parameters object.
      SeeAlso:      TCCRBrokerLogMode.CheckCmdLineParams
      Description:
        If a reference to a command-line parameters object is assigned to this
        property, then properties are automatically modified according to the
        parameters specified by the user. See the CheckCmdLineParams method for
        more details.
    }
    property CmdLineParams: TCCRCustomCmdLineParams
      read fCmdLineParams  write SetCmdLineParams;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRBrokerLogMode.
      SeeAlso:      TCCRBrokerLogMode.Owner; TObject.Create
      Keywords:     Create,TCCRBrokerLogMode
      Description:
        In addition to calling the inherited constructor, Create assigns default
        values to the properties. The <i>anOwner</i> parameter specifies the
        CCR procedure broker that created this TCCRBrokerLogMode instance.
    }
    constructor Create(anOwner: TCCRCustomBroker);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Copies procedure calls logging parameters.
      SeeAlso:      TPersistent.Assign
      Keywords:     Assign,TCCRBrokerLogMode
      Description:
        Call Assign to copy the remote procedure calls logging parameters. If
        the <i>aSource</i> object is another TCCRBrokerLogMode object, Assign
        copies the values of its Enabled, LimitParameters, LimitResults,
        Parameters, and Results properties. Otherwise, Assign calls the
        inherited method.
    }
    procedure Assign(aSource: TPersistent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     A CCR remote procedure broker that owns a TCCRBrokerLogMode
                    instance.
      SeeAlso:      TCCRBrokerLogMode.Create
      Keywords:     Owner,TCCRBrokerLogMode
      Description:
        Owner references a broker that created this instance of logging
        parameters. The owner is also responsible for destroying this object.
    }
    property Owner: TCCRCustomBroker  read fOwner;

  published

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies whether Broker errors/exceptions should be written
                    to a log.
      SeeAlso:      TCCRBrokerLogMode.Enabled; TCCRBrokerLogMode.Parameters;
                    TCCRBrokerLogMode.Results
      Keywords:     BrokerErrors,TCCRBrokerLogMode
      Description:
        Assign True to the BrokerErrors property to enable recording Broker
        errors/exceptions.
    }
    property BrokerErrors: Boolean
      read fBrokerErrors  write fBrokerErrors  default True;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies if procedure logging should be enabled.
      SeeAlso:      TCCRBrokerLogMode.Parameters; TCCRBrokerLogMode.Results
      Keywords:     Enabled,TCCRBrokerLogMode
      Description:
        If True is assigned to Enabled, then procedure calls are recorded in the
        current debug log. Amount of recorded information is determined by other
        properties of the TCCRBrokerLogMode.
    }
    property Enabled: Boolean read fEnabled write SetEnabled;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Limits number of parameter lines written to a log for a
                    single procedure.
      SeeAlso:      TCCRBrokerLogMode.Enabled; TCCRBrokerLogMode.LimitResults;
                    TCCRBrokerLogMode.Params
      Description:
        If LimitParams is greater than 0, then its value limits number of lines
        that are added to the log during recording a multi-line parameter of a
        single procedure call. Otherwise, all parameter lines are recorded.
    }
    property LimitParams: Word read fLimitParams write fLimitParams default 0;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Limits number of result lines written to a log for a single
                    procedure.
      SeeAlso:      TCCRBrokerLogMode.Enabled; TCCRBrokerLogMode.LimitParams;
                    TCCRBrokerLogMode.Results
      Description:
        If LimitResults is greater than 0, then its value limits number of
        lines that are added to the log during recording results returned
        from a single procedure call. Otherwise, all results are recorded.
    }
    property LimitResults: Word read fLimitResults write fLimitResults default 0;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies whether procedure parameters should be written to
                    a log.
      SeeAlso:      TCCRBrokerLogMode.Enabled; TCCRBrokerLogMode.LimitParams;
                    TCCRBrokerLogMode.Results
      Keywords:     Parameters,TCCRBrokerLogMode
      Description:
        Assign True to the Parameters property to enable recording parameters
        passed into procedure calls. You can use LimitParameters to limit number
        of multi-line parameter lines for each call.
    }
    property Parameters: Boolean read fParameters write fParameters default True;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies whether procedure results should be written to
                    a log.
      SeeAlso:      TCCRBrokerLogMode.Enabled; TCCRBrokerLogMode.LimitResults;
                    TCCRBrokerLogMode.Parameters
      Keywords:     Results,TCCRBrokerLogMode
      Description:
        Assign True to the Results property to enable recording results returned
        from procedure calls. You can use LimitResults to limit number of result
        lines for each call.
    }
    property Results: Boolean read fResults write fResults default True;

  end;

  {============================== TCCRCustomBroker =============================
    Overview:     Base class for a remote procedure broker wrapper.
    SeeAlso:      TCCRBroker
    Description:
      TCCRCustomBroker is a generic wrapper for a remote procedure broker. It
      encapsulates implemetation details, which allows greater flexibility
      (SQL requests and/or remote procedures, demo mode brokers, etc.). It also
      provides mechanisms for error processing and reporting, debug logging,
      and application command-line parameters processing.
      <p>
      Do not create instances of TCCRCustomBroker. To put a broker on a form,
      use a descendant of TCCRCustomBroker, such as TCCRBroker.
  }
  TCCRCustomBroker = class(TComponent)
  private

    fCmdLineParams:    TCCRCustomCmdLineParams;
    fConnected:        Boolean;
    fDebugLog:         TCCRBrokerLogMode;
    fDefaultProcMode:  TCCRBrokerDefaultProcMode;
    fOnCheckProcError: TCCRBrokerCheckErrorEvent;
    fResultBuffers:    TObjectList;

    procedure SetCmdLineParams(aValue: TCCRCustomCmdLineParams);
    procedure SetDebugLog(const aValue: TCCRBrokerLogMode);

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Checks command-line parameters and modifies broker
                    properties accordingly.
      SeeAlso:      TCCRCustomBroker.CmdLineParams
      Keywords:     CheckCmdLineParams,TCCRCustomBroker
      Description:
        If the CmdLineParams property references a descendant of
        TCCRCustomCmdLineParams, the CheckCmdLineParams method analyses
        the application's command-line parameters specified by a user and
        modifies values of broker properties accordingly.
        <p>
        As implemented in TCCRCustomBroker, CheckCmdLineParams does nothing.
    }
    procedure CheckCmdLineParams; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Protected implementation of the CheckProcError method.
      SeeAlso:      TCCRCustomBroker.CheckProcError;
                    TCCRCustomBroker.OnCheckProcError
      Keywords:     DoCheckProcError,TCCRCustomBroker
      Description:
        Do not call DoCheckProcError directly; override it to implement a
        different format of error reporting used by remote procedures.
        Information about the remote procedure call and its results are passed
        via the <i>CallInfo</i> parameter.
        <p>
        This method should check the procedure results in the CallInfo.Results
        and either populate the CallInfo.Errors list using the AddError method
        of the <i>CallInfo</i> or display error messages itself if the rpcSilent
        is not a member of the CallInfo.ProcMode set.
        <p>
        If the ErrorCount property of the <i>CallInfo</i> is greater than 0
        after returning from this method, then errors stored in the Errors list
        are automatically formatted and shown to the user in a modal dialog
        box. If a descendant class wants to override this behavior (e.g. to
        display the errors itself), it should not populate the error list or
        should clear it before exiting this method.
        <p>
        If an OnCheckProcError event handler is defined, then this method is
        called only if the handler does not complete the error reporting itself
        (the value of the Handled parameter is False).
        <p>
        As implemented in TCCRCustomBroker, DoCheckProcError assigns 0 to the
        ErrorCode and rpeNone to the ErrorType properties of the <i>CallInfo</i>.
    }
    procedure DoCheckProcError(CallInfo: TCCRBrokerCallInfo); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Protected implementation of the Connect method.
      SeeAlso:      TCCRCustomBroker.Connect
      Keywords:     DoConnect,TCCRCustomBroker
      Description:
        Do not call DoConnect directly; override it to establish a connection
        to a main application server. If the connection is succesfully open,
        this method should return True. Otherwise, False should be returned.
        <p>
        As implemented in TCCRCustomBroker, DoConnect does nothing and returns
        True.
    }
    function DoConnect: Boolean; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Protected implementation of the Disconnect method.
      SeeAlso:      TCCRCustomBroker.Disconnect
      Keywords:     DoDisconnect,TCCRCustomBroker
      Description:
        Do not call DoDisconnect directly; override it to close a connection to
        a main application server. As implemented in TCCRCustomBroker,
        DoDisconnect does nothing.
    }
    procedure DoDisconnect; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the value of the Connected property.
      SeeAlso:      TCCRCustomBroker.Connected
      Keywords:     GetConnected,TCCRCustomBroker
      Description:
        GetConnected returns True if a broker is connected to a server.
    }
    function GetConnected: Boolean; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the value of the Results property.
      SeeAlso:      TCCRCustomBroker.CreateResults; TCCRCustomBroker.Results
      Keywords:     GetResults,TCCRCustomBroker
      Description:
        As implemented in TCCRCustomBroker, GetResults returns a top value from
        the internal stack of result buffers created by the CreateResults. If
        the stack is empty, nil is returned.
    }
    function GetResults: TStrings; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Responds to notifications that components are being created
                    or destroyed.
      SeeAlso:      TComponent.Notification
      Keywords:     Notification,TCCRCustomBroker
      Description:
        Do not call the Notification method in an application.  Notification is
        called automatically when the component specified by <i>aComponent</i>
        is about to be inserted or removed, as specified by <i>Operation</i>.
        <p>
        TCCRCustomBroker overrides this method in order to update its
        CmdLineParams property when a control it refers to is destroyed.
    }
    procedure Notification(aComponent: TComponent;
      Operation: TOperation); override;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRCustomBroker.
      SeeAlso:      TComponent.Create
      Keywords:     Create,TCCRCustomBroker
      Description:
        Create initializes an instance of the TCCRCustomBroker. <i>anOwner</i>
        is the component, typically a form, that is responsible for freeing the
        broker.
    }
    constructor Create(anOwner: TComponent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRCustomBroker.
      SeeAlso:      TComponent.Destroy; TObject.Free
      Keywords:     Destroy,TCCRCustomBroker
      Description:
        Do not call Destroy directly in an application. Instead, call Free.
        Free verifies that the component is not nil, and only then calls Destroy.
        <p>
        Applications should only free components explicitly when the constructor
        was called without assigning an owner to the component.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Performs any necessary actions before the first destructor
                    is called.
      SeeAlso:      TCCRCustomBroker.Destroy; TCCRCustomBroker.Disconnect;
                    TComponent.BeforeDestruction
      Keywords:     BeforeDestruction,TCCRCustomBroker
      Description:
        BeforeDestruction is called automatically immediately before the
        components first destructor executes. Do not call it explicitly in
        your applications.
        <p>
        As implemented in TCCRCustomBroker, BeforeDestruction calls the
        Disconnect method.
        <p>
        Descendants that override this method to perform other actions before a
        component is destroyed should call the inherited method first.
    }
    procedure BeforeDestruction; override;

    // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    function CallProc(CallInfo: TCCRBrokerCallInfo;
      const Parameters: array of String;
      MultList: TStringList = nil): Boolean; overload; virtual; abstract;
    {
      Overview:     Calls a remote procedure.
      SeeAlso:      TCCRCustomBroker.DefaultProcMode; TCCRCustomBroker.Results
      Keywords:     CallProc,TCCRCustomBroker
      Description:
        First overloaded version of CallProc executes a remote procedure
        specified by the <i>ProcedureName</i> with string values passed in the
        <i>Parameters</i> array and content of the optional <i>MultList</i>
        string list as parameters. The optional <i>ProcMode</i> parameter
        specifies flags that control the execution and error processing. If it
        is omitted or includes rpcDefault, then the value of the DefaultProcMode
        property is used.
        <p>
        If the <i>ProcResults</i> parameter is not nil, then the procedure
        results are returned into the referenced TStrings instance (usually a
        TStringList). Otherwise, the results are stored into the default buffer
        referenced by the Results property.
        <p>
        This method returns True in case of a successful procedure call.
        Otherwise, False is returned. Use the other overloaded version of the
        CallProc if you need to analyze the error(s).
        <p>
        As implemented in TCCRCustomBroker, this overloaded version of the
        CallProc creates a temporary instance of the TCCRBrokerCallInfo,
        populates its properties with parameter values, and calls the other
        overloaded version of the CallProc.

        Second overloaded version of CallProc executes a remote procedure
        specified by the ProcedureName property of the <i>CallInfo</i> with
        string values passed in the <i>Parameters</i> array and content of the
        optional <i>MultList</i> string list as parameters.
        <p>
        The ProcMode property of the <i>CallInfo</i> specifies flags that
        control the execution and error processing. If it includes rpcDefault,
        then the value of the DefaultProcMode property of the broker is assigned
        to it before calling the procedure.
        <p>
        Procedure results are returned into a TStrings instance (usually a
        TStringList) referenced by the Results property of the <i>CallInfo</i>.
        If it is nil, a buffer referenced by the Results property of the broker
        is assigned to the it before calling the procedure.
        <p>
        CallProc is an abstract method and must be implemented by descendant
        classes. This method should also call the CheckProcError method (to
        check results returned by the procedure) if ProcMode does not include
        rpcNoErrorCheck and return True in case of successful procedure call.
        Otherwise, False should be returned and a primary error code should be
        assigned to the ErrorCode property of the <i>CallInfo</i>.
    }
    function CallProc(const ProcedureName: String;
      const Parameters: array of String; MultList: TStringList = nil;
      ProcResults: TStrings = nil;
      ProcMode: TCCRBrokerProcMode = [rpcDefault]): Boolean; overload; virtual;

    // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    function CheckProcError(CallInfo: TCCRBrokerCallInfo): Boolean; overload;
    {
      Overview:     Checks for errors returned by a remote procedure.
      SeeAlso:      TCCRCustomBroker.DoCheckProcError;
                    TCCRCustomBroker.OnCheckProcError
      Description:
        First overloaded version of CheckProcError creates a temporary instance
        of the TCCRBrokerCallInfo, populates its properties with parameter
        values and calls the other overloaded version of the CheckProcError.
        <p>
        <i>ProcedureName</i> specifies the name of a remote procedure whose
        results should be analyzed.
        <p>
        Optional <i>ProcResults</i> parameter references the buffer that
        contians results returned by the remote procedure. If it is omitted or
        nil, then the value of the broker's Results property is assgined to the
        CallInfo.Results.
        <p>
        Optional <i>ProcMode</i> parameter specifies execution and error
        processing flags that were used to call the rempte procedure. If it is
        omitted or includes rcDefault, then the value of the broker's
        DefaultProcMode property is assgined to the CallInfo.ProcMode.

        Second overloaded version of CheckProcError checks results of a remote
        procedure call specified by the <i>CallInfo</i> parameter.
        <p>
        CheckProcError delegates the error processing to the OnCheckProcError
        event handler if it is defined. Then the DoCheckProcError method is
        called, unless the OnCheckProcError event handler returns True in the
        Handled parameter.
        <p>
        If errors were returned by the procedure, they have been handled neither
        by the event handler nor by the DoCheckProcError (CallInfo.ErrorCount is
        greater than 0), and rpcSilent is not present in the CallInfo.ProcMode
        set, then error messages are presented to the user in a modal dialog box.
        <p>
        Use this method to analyze a TCCRBrokerCallInfo instance populated by
        the CallProc method. Otherwise, use the other overloaded version of the
        CheckProcError.
    }
    function CheckProcError(const ProcedureName: String;
      ProcResults: TStrings = nil;
      const ProcMode: TCCRBrokerProcMode = [rpcDefault]): Boolean; overload;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Connects to a server.
      SeeAlso:      TCCRCustomBroker.Connected; TCCRCustomBroker.Disconnect;
                    TCCRCustomBroker.DoConnect
      Keywords:     Connect,TCCRCustomBroker
      Description:
        If a procedure broker is not already connected to a server (the value of
        the Connected property is False), Connect calls the DoConnect method and
        returns the new value of the Connected property. Otherwise, Connect does
        nothing and returns True.
    }
    function Connect: Boolean;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates a new buffer for procedure results.
      SeeAlso:      TCCRCustomBroker.CallProc; TCCRCustomBroker.Results
      Description:
        Use CreateResults to create a new temporary buffer that will be used by
        the CallProc method to store procedure results if its ProcResults
        parameter is omitted or nil. The Results property will reference the
        new buffer after calling the CreateResults.
        <p>
        Since the CreateResults returns an interface reference, the new buffer
        is automatically destroyed when it goes out of scope at the end of the
        begin-end block where it was created.
        <p>
        When a buffer is destroyed, the previous value of the Results property
        is restored. In other words, buffers created by the CreateResults form
        a stack.
    }
    function CreateResults: IInterface;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Closes a connection to a server.
      SeeAlso:      TCCRCustomBroker.Connect; TCCRCustomBroker.Connected;
                    TCCRCustomBroker.DoDisconnect
      Keywords:     Disconnect,TCCRCustomBroker
      Description:
        If a procedure broker is connected to a server (the value of the
        Connected property is True) and the connection was explicitly
        established by the Connect method, Disconnect calls the DoDisconnect
        method. Otherwise, Disconnect does nothing.
    }
    procedure Disconnect;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Reference to a command-line parameters object.
      SeeAlso:      TCCRCustomBroker.CheckCmdLineParams
      Description:
        If a reference to a command-line parameters object is assigned to this
        property, then broker properties are automatically modified according
        to the command-line parameters specified by the user. See the
        CheckCmdLineParams method for more details.
    }
    property CmdLineParams: TCCRCustomCmdLineParams
      read fCmdLineParams  write SetCmdLineParams;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates if a broker is connected to a server.
      SeeAlso:      TCCRCustomBroker.Connect; TCCRCustomBroker.Disconnect
      Keywords:     Connected,TCCRCustomBroker
      Description:
        Use Connected to determine if a broker is connected to a server. This is
        a read-only property.
    }
    property Connected: Boolean  read GetConnected;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Procedure logging parameters.
      Keywords:     DebugLog,TCCRCustomBroker
      Description:
        DebugLog specifies if the broker records procedure calls in a debug log
        and what data is recorded.
    }
    property DebugLog: TCCRBrokerLogMode read fDebugLog write SetDebugLog;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Default procedure execution and error processing flags.
      SeeAlso:      TCCRCustomBroker.CallProc
      Description:
        The CallProc and CheckProcError methods use value of this property if
        their optional ProcMode parameter is omitted or includes rpcDefault.
        These flags customize procedure calls and error processing.
    }
    property DefaultProcMode: TCCRBrokerDefaultProcMode
      read fDefaultProcMode  write fDefaultProcMode;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs when a broker checks for errros returned by a
                    procedure.
      SeeAlso:      TCCRBrokerCallInfo.AddError;
                    TCCRCustomBroker.CheckProcError;
                    TCCRCustomBroker.DoCheckProcError
      Description:
        Write an OnCheckProcError event handler to examine results returned by
        your remote procedures if they use an error reporting format different
        from the default one that is implemented by the DoCheckProcError.
        <p>
        <i>aSender</i> is the procedure broker object whose event handler is
        called. Information about the procedure call and its results are passed
        via the <i>CallInfo</i> parameter.
        <p>
        If the procedure call returned errors, add their descriptors to the
        CallInfo.Errors list (using the AddError method of the <i>CallInfo</i>)
        and assign the primary error code and type to the ErrorCode and
        ErrorType properties of the <i>CallInfo</i>. Otherwise, assign 0 to the
        CallInfo.ErrorCode and rpeNone to the CallInfo.ErrorType.
        <p>
        The DoCheckProcError method will be called after exiting this handler,
        unless the latter assigns True to the <i>Handled</i> parameter.
        <p>
        After that, the errors will be presented to the user in a modal dialog
        box. If you want to display them yourself (not recommended), do so and
        clear the list of errors (by calling the ClearErrors method of the
        <i>CallInfo</i>) before exiting the handler or do not populate the list
        at all.
    }
    property OnCheckProcError: TCCRBrokerCheckErrorEvent
      read fOnCheckProcError  write fOnCheckProcError;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Default buffer for procedure results.
      SeeAlso:      TCCRBrokerCallInfo.Results; TCCRCustomBroker.CallProc;
                    TCCRCustomBroker.CreateResults; TCCRCustomBroker.GetResults
      Keywords:     Results,TCCRCustomBroker
      Description:
        Use Results to access the results if a procedure was called without
        specifying the result buffer (see the ProcResults parameter of the
        CallProc method).
        <p>
        You can use CreateResults to preserve the current buffer and create a
        new one. See the TCCRCustomBroker.CreateResults topic for more details.
    }
    property Results: TStrings  read GetResults;

  end;

  {============================= TCCRProcedureError ============================
    Overview:     Descriptor of an error returned by a remote procedure.
    SeeAlso:      TCCRBrokerCallInfo.Errors; TCCRCustomBroker.DoCheckProcError;
                  TCCRCustomBroker.OnCheckProcError
    Description:
      These error descriptors are used internally in the error processing code.
  }
  TCCRProcedureError = class(TPersistent)
  private

    fCode:    Integer;
    fDetails: TStrings;
    fMessage: String;
    fPlace:   String;

    procedure SetDetails(aValue: TStrings);

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRProcedureError.
      SeeAlso:      TObject.Free; TPersistent.Destroy
      Keywords:     Destroy,TCCRProcedureError
      Description:
        Do not call Destroy directly in an application. Instead, call Free.
        Free verifies that the object is not nil, and only then calls Destroy.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Error code.
      SeeAlso:      TCCRProcedureError.Details; TCCRProcedureError.Message;
                    TCCRProcedureError.Place
      Keywords:     Code,TCCRProcedureError
      Description:
        Use Code to get or set the error code.
    }
    property Code: Integer  read fCode  write fCode;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Error description and/or additional details.
      SeeAlso:      TCCRProcedureError.Code; TCCRProcedureError.Message;
                    TCCRProcedureError.Place
      Keywords:     Details,TCCRProcedureError
      Description:
        Use Details to get or set the error description and/or additional
        details about the error.
    }
    property Details: TStrings  read fDetails  write SetDetails;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Error message.
      SeeAlso:      TCCRProcedureError.Code; TCCRProcedureError.Details;
                    TCCRProcedureError.Place
      Keywords:     Message,TCCRProcedureError
      Description:
        Use Message to get or set the error message.
    }
    property Message: String  read fMessage  write fMessage;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Location of the error.
      SeeAlso:      TCCRProcedureError.Code; TCCRProcedureError.Details;
                    TCCRProcedureError.Message
      Keywords:     Place,TCCRProcedureError
      Description:
        Use Place to get or set the location of the error in a code that
        implements a remote procedure.
    }
    property Place: String  read fPlace  write fPlace;

  end;

///////////////////////////////// Implementation \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

implementation

uses
  Dialogs, uROR_Resources, uROR_Utilities, fROR_BrokerErrors;

type

  {============================ TCCRProcResultsBuffer ==========================
    Overview:     Buffer for procedure results.
    SeeAlso:      TCCRCustomBroker.CreateResults
    Description:
      TCCRProcResultsBuffer is a string list that automatically maintains its
      relations with the parent TObjectList. It also implements a generic
      interface that allows automatic destruction when a reference goes out of
      scope.
  }
  TCCRProcResultsBuffer = class(TStringList,IInterface)
  private

    fRefCount: Integer;
    fParent:   TObjectList;

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Implements the IInterface _AddRef method.
      SeeAlso:      IInterface._AddRef; TCCRProcResultsBuffer._Release;
                    TCCRProcResultsBuffer.RefCount
      Keywords:     _AddRef,TCCRProcResultsBuffer
      Description:
        Do not call the protected _AddRef method directly. _AddRef is called
        through the interface to indicate that another object is using the
        interfaced objects interface pointer. _AddRef increments the RefCount
        property.
    }
    function _AddRef: Integer; stdcall;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Implements the IInterface _Release method.
      SeeAlso:      IInterface._Release; TCCRProcResultsBuffer._AddRef;
                    TCCRProcResultsBuffer.RefCount
      Keywords:     _Release,TCCRProcResultsBuffer
      Description:
        Do not call the protected _Release method directly. _Release is called
        through the IInterface interface to indicate that another object is
        through with the interfaced objects interface pointer. _Release
        decrements the RefCount property.
    }
    function _Release: Integer; stdcall;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Implements the IInterface QueryInterface method.
      SeeAlso:      IInterface.QueryInterface
      Keywords:     QueryInterface,TCCRProcResultsBuffer
      Description:
        Do not call the protected QueryInterface method directly. QueryInterface
        is called through the IInterface interface to obtain an interface
        pointer for the interface identified by the IID parameter. If the object
        supports the requested interface, it is returned in the Obj parameter
        and QueryInterface returns 0. If the object does not support the
        interface, QueryInterface returns E_NOINTERFACE.
    }
    function QueryInterface(const IID: TGUID; out Obj): HResult; stdcall;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRProcResultsBuffer.
      SeeAlso:      TObject.Create
      Keywords:     Create,TCCRProcResultsBuffer
      Description:
        Create calls the inherited constructor and adds the object to a parent
        list specified by the <i>aParent</i> parameter.
    }
    constructor Create(aParent: TObjectList);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys the TCCRProcResultsBuffer instance and frees its
                    memory.
      SeeAlso:      TObject.Free; TStringList.Destroy
      Keywords:     Destroy,TCCRProcResultsBuffer
      Description:
        Do not call Destroy directly in an application. Instead, call Free.
        Free verifies that the TCCRProcResultsBuffer reference is not nil and
        only then calls Destroy.
        <p>
        Destroy removes a reference to the object from the parent list before
        calling the inherited destructor.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Allocates memory for an instance of an object type and
                    returns a pointer to that new instance.
      SeeAlso:      TObject.NewInstance
      Keywords:     NewInstance,TCCRProcResultsBuffer
      Description:
        All constructors call NewInstance automatically. NewInstance calls
        InstanceSize to determine how much memory to allocate from the heap to
        contain a particular instance. Do not call NewInstance directly.
    }
    class function NewInstance: TObject; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Responds after the last constructor has executed.
      SeeAlso:      TCCRProcResultsBuffer.Create; TObject.AfterConstruction
      Keywords:     AfterConstruction,TCCRProcResultsBuffer
      Description:
        AfterConstruction is called automatically after the objects last
        constructor has executed. Do not call it explicitly in your applications.
    }
    procedure AfterConstruction; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Responds before the first destructor executes.
      SeeAlso:      TCCRProcResultsBuffer.Destroy; TObject.BeforeDestruction
      Keywords:     BeforeDestruction,TCCRProcResultsBuffer
      Description:
        BeforeDestruction is called automatically before the objects first
        destructor executes. Do not call it explicitly in your applications.
    }
    procedure BeforeDestruction; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates the number of interface pointers currently
                    dependent upon the object.
      SeeAlso:      TCCRProcResultsBuffer._AddRef; TCCRProcResultsBuffer._Release
      Keywords:     RefCount,TCCRProcResultsBuffer
      Description:
        RefCount defines the lifetime of the object. When RefCount is zero, the
        object is destroyed. RefCount is incremented by calls to the _AddRef
        method and decremented by the _Release method.
    }
    property RefCount: Integer  read fRefCount;

  end;

////////////////////////////// TCCRBrokerCallInfo \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRBrokerCallInfo.Create;
begin
  inherited;
  fErrors := TObjectList.Create;
  fErrors.OwnsObjects := True;
  ProcMode := [rpcDefault];
  ErrorType := rpeNone;
end;

destructor TCCRBrokerCallInfo.Destroy;
begin
  FreeAndNil(fErrors);
  inherited;
end;

procedure TCCRBrokerCallInfo.AddError(const aMessage: String;
  const aCode: Integer; const aPlace: String; const aDetails: TStrings);
var
  err: TCCRProcedureError;
begin
  err := TCCRProcedureError.Create;
  with err do
    begin
      Code    := aCode;
      Details := aDetails;
      Message := aMessage;
      Place   := aPlace;
    end;
  fErrors.Add(err)
end;

procedure TCCRBrokerCallInfo.ClearErrors;
begin
  if Assigned(fErrors) then
    fErrors.Clear;
end;

function TCCRBrokerCallInfo.GetError(const anIndex: Integer): TCCRProcedureError;
begin
  if Assigned(fErrors) then
    Result := TCCRProcedureError(fErrors.Items[anIndex])
  else
    Result := nil;
end;

function TCCRBrokerCallInfo.GetErrorCount: Integer;
begin
  if Assigned(fErrors) then
    Result := fErrors.Count
  else
    Result := 0;
end;

/////////////////////////////// TCCRBrokerLogMode \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRBrokerLogMode.Create(anOwner: TCCRCustomBroker);
begin
  inherited Create;
  fBrokerErrors := True;
  fOwner        := anOwner;
  fParameters   := True;
  fResults      := True;
end;

procedure TCCRBrokerLogMode.Assign(aSource: TPersistent);
begin
  if aSource is TCCRBrokerLogMode then
    with TCCRBrokerLogMode(aSource) do
      begin
        Self.fBrokerErrors := fBrokerErrors;   
        Self.fEnabled      := fEnabled;
        Self.fLimitParams  := fLimitParams;
        Self.fLimitResults := fLimitResults;
        Self.fParameters   := fParameters;
        Self.fResults      := fResults;
      end
  else
    inherited;
end;

procedure TCCRBrokerLogMode.CheckCmdLineParams;
begin
  if Assigned(CmdLineParams) then
    begin
      SetEnabled(Enabled);
    end;
end;

procedure TCCRBrokerLogMode.SetCmdLineParams(aValue: TCCRCustomCmdLineParams);
begin
  if aValue <> fCmdLineParams then
    begin
      fCmdLineParams := aValue;
      CheckCmdLineParams;
    end;
end;

procedure TCCRBrokerLogMode.SetEnabled(aValue: Boolean);
begin
  if Assigned(CmdLineParams) and CmdLineParams.DebugMode and
    not (csDesigning in Owner.ComponentState) then
      aValue := True;

  fEnabled := aValue;
end;

/////////////////////////////// TCCRCustomBroker \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRCustomBroker.Create(anOwner: TComponent);
begin
  inherited;
  fDebugLog := TCCRBrokerLogMode.Create(Self);
  fResultBuffers := TObjectList.Create;
  fResultBuffers.OwnsObjects := False;
end;

destructor TCCRCustomBroker.Destroy;
begin
  FreeAndNil(fResultBuffers);
  FreeAndNil(fDebugLog);
  inherited;
end;

procedure TCCRCustomBroker.BeforeDestruction;
begin
  inherited;
  Disconnect;
end;

function TCCRCustomBroker.CallProc(const ProcedureName: String;
  const Parameters: array of String; MultList: TStringList;
  ProcResults: TStrings; ProcMode: TCCRBrokerProcMode): Boolean;
var
  CallInfo: TCCRBrokerCallInfo;
begin
  CallInfo := TCCRBrokerCallInfo.Create;
  try
    CallInfo.ProcedureName := ProcedureName;

    if not (rpcDefault in ProcMode) then
      CallInfo.ProcMode := ProcMode
    else
      CallInfo.ProcMode := DefaultProcMode;

    if Assigned(ProcResults) then
      CallInfo.Results := ProcResults
    else
      CallInfo.Results := Results;

    Result := CallProc(CallInfo, Parameters, MultList);
  finally
    CallInfo.Free;
  end;
end;

procedure TCCRCustomBroker.CheckCmdLineParams;
begin
end;

function TCCRCustomBroker.CheckProcError(CallInfo: TCCRBrokerCallInfo): Boolean;
var
  Handled: Boolean;
begin
  if CallInfo.Results.Count > 0 then
    begin
      Handled := False;
      if Assigned(OnCheckProcError) then
        OnCheckProcError(Self, CallInfo, Handled);

      if not Handled then
        DoCheckProcError(CallInfo);

      if not (rpcSilent in CallInfo.ProcMode) and (CallInfo.ErrorCount > 0) then
        TCCRFormProcedureErrors.Show(CallInfo);

      Result := (CallInfo.ErrorCode <> 0);
    end
  else
    with CallInfo do
      begin
        ErrorType := rpeBroker;
        ErrorCode := CCRBEC_NORPCRESULTS;
        if not (rpcSilent in ProcMode) then
          MessageDlg508('RPC Error', Format(RSC0040, [ProcedureName]),
            mtError, [mbOK], 0);
        Result := True;
      end;
end;

function TCCRCustomBroker.CheckProcError(const ProcedureName: String;
  ProcResults: TStrings; const ProcMode: TCCRBrokerProcMode): Boolean;
var
  CallInfo: TCCRBrokerCallInfo;
begin
  CallInfo := TCCRBrokerCallInfo.Create;
  try
    CallInfo.ProcedureName := ProcedureName;

    if Assigned(ProcResults) then
      CallInfo.Results := ProcResults
    else
      CallInfo.Results := Results;

    if not (rpcDefault in ProcMode) then
      CallInfo.ProcMode := ProcMode
    else
      CallInfo.ProcMode := DefaultProcMode;

    Result := CheckProcError(CallInfo);
  finally
    CallInfo.Free;
  end;
end;

function TCCRCustomBroker.Connect: Boolean;
begin
  if not Connected then
    fConnected := DoConnect;
  Result := Connected;
end;

function TCCRCustomBroker.CreateResults: IInterface;
begin
  Result := TCCRProcResultsBuffer.Create(fResultBuffers);
end;

procedure TCCRCustomBroker.Disconnect;
begin
  if Connected and fConnected then
    begin
      DoDisconnect;
      fConnected := False;
    end;
end;

procedure TCCRCustomBroker.DoCheckProcError(CallInfo: TCCRBrokerCallInfo);
begin
  CallInfo.ErrorCode := 0;
  CallInfo.ErrorType := rpeNone;
end;

function TCCRCustomBroker.DoConnect: Boolean;
begin
  Result := True;
end;

procedure TCCRCustomBroker.DoDisconnect;
begin
end;

function TCCRCustomBroker.GetConnected: Boolean;
begin
  Result := fConnected;
end;

function TCCRCustomBroker.GetResults: TStrings;
begin
  with fResultBuffers do
    if Count > 0 then
      Result := TStrings(Last)
    else
      Result := nil;
end;

procedure TCCRCustomBroker.Notification(aComponent: TComponent;
  Operation: TOperation);
begin
  inherited;
  if Operation = opRemove then
    begin
      if aComponent = CmdLineParams then
        CmdLineParams := nil;
    end;
end;

procedure TCCRCustomBroker.SetCmdLineParams(aValue: TCCRCustomCmdLineParams);
begin
  if aValue <> fCmdLineParams then
    begin
      if Assigned(fCmdLineParams) then
        fCmdLineParams.RemoveFreeNotification(Self);

      fCmdLineParams := aValue;
      DebugLog.CmdLineParams := aValue;

      if Assigned(fCmdLineParams) then
        fCmdLineParams.FreeNotification(Self);

      CheckCmdLineParams;
    end;
end;

procedure TCCRCustomBroker.SetDebugLog(const aValue: TCCRBrokerLogMode);
begin
  fDebugLog.Assign(aValue);
end;

////////////////////////////// TCCRProcedureError \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

destructor TCCRProcedureError.Destroy;
begin
  FreeAndNil(fDetails);
  inherited;
end;

procedure TCCRProcedureError.SetDetails(aValue: TStrings);
begin
  if aValue <> fDetails then
    begin
      try
        FreeAndNil(fDetails);
      except
      end;
      fDetails := aValue;
    end;
end;

///////////////////////////// TCCRProcResultsBuffer \\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRProcResultsBuffer.Create(aParent: TObjectList);
begin
  inherited Create;
  if Assigned(aParent) then
    begin
      fParent := aParent;
      aParent.Add(Self);
    end;
end;

destructor TCCRProcResultsBuffer.Destroy;
begin
  if Assigned(fParent) then
    fParent.Extract(Self);
  inherited;
end;

function TCCRProcResultsBuffer._AddRef: Integer;
begin
  Result := InterlockedIncrement(fRefCount)
end;

function TCCRProcResultsBuffer._Release: Integer;
begin
  Result := InterlockedDecrement(fRefCount);
  if Result = 0 then
    Destroy;
end;

procedure TCCRProcResultsBuffer.AfterConstruction;
begin
  inherited;
  InterlockedDecrement(fRefCount);
end;

procedure TCCRProcResultsBuffer.BeforeDestruction;
begin
  if RefCount <> 0 then
    System.Error(reInvalidPtr);
  inherited;
end;

class function TCCRProcResultsBuffer.NewInstance: TObject;
begin
  Result := inherited NewInstance;
  TCCRProcResultsBuffer(Result).fRefCount := 1;
end;

function TCCRProcResultsBuffer.QueryInterface(const IID: TGUID; out Obj): HResult;
begin
  if GetInterface(IID, Obj) then
    Result := 0
  else
    Result := E_NOINTERFACE;
end;

end.
