{******************************************************************************}
{ Package:      Clinical Case Registries Custom Components                     }
{ Date Created: January 26, 2007                                               }
{ Site Name:    Hines OIFO                                                     }
{ Developers:   Sergey Gavrilov                                                }
{ Description:  Command-line parameters.                                       }
{ Note:                                                                        }
{******************************************************************************}

unit uROR_CmdLineParams;

{$I Components.inc}

interface

uses
  Classes, ComCtrls, Graphics, SysUtils, uROR_Utilities;

type

  {============================ TCCRCmdLineHelpItems ===========================
    Overview:     List of help items that describe command-line parameters.
    SeeAlso:      TCCRCustomCmdLineParams.QueryHelpItems;
                  TCCRCustomCmdLineParams.ShowHelp
    Description:
      TCCRCmdLineHelpItems is used to collect help items and sort them in
      alphabetical order. Each item describes a single command-line parameter.
  }
  TCCRCmdLineHelpItems = class(TPersistent)
  private

    fList: TStringList;

    function GetCount: Integer;

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Copies the properties of an object to a destination object.
      SeeAlso:      TPersistent.Assign; TPersistent.AssignTo
      Keywords:     AssignTo,TCCRCmdLineHelpItems
      Description:
        Override the AssignTo method to extend the functionality of the Assign
        method of destination objects so that they handle newly created object
        classes.
        <p>
        As implemented in TCCRCmdLineHelpItems, AssignTo populates a rich text
        editor according to the help items stored in the internal list.
    }
    procedure AssignTo(Dest: TPersistent); override;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Constructs an object and initializes its data before the
                    object is first used.
      SeeAlso:      TCCRCmdLineHelpItems.Destroy; TObject.Free
      Keywords:     Create,TCCRCmdLineHelpItems
      Description:
        Create calls the inherited constructor and creates the internal string
        list.
    }
    constructor Create;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys the TCCRCmdLineHelpItems instance and frees its
                    memory.
      SeeAlso:      TObject.Destroy; TObject.Free
      Keywords:     Destroy,TCCRCmdLineHelpItems
      Description:
        Do not call Destroy directly. Call Free instead. Free checks that the
        object reference is not nil before calling Destroy.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Adds a help item to the list.
      SeeAlso:      TCCRCustomCmdLineParams.QueryHelpItems
      Keywords:     Add,TCCRCmdLineHelpItems
      Description:
        The Add method associates description passed via the <i>aDescr</i> with
        command-line parameter(s) specified by the <i>aParam</i> and adds the
        new help item to the internal list.
    }
    procedure Add(const aParam, aDescr: String);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Clears a list of help items.
      Keywords:     Clear,TCCRCmdLineHelpItems
      Description:
        Clear deletes all strings from the internal list and destroys the
        corresponding objects.
    }
    procedure Clear;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns a help items.
      Keywords:     GetItem,TCCRCmdLineHelpItems
      SeeAlso:      TCCRCmdLineHelpItems.Count
      Description:
        GetItem returns command-line parameter name(s) (<i>aParam</i>) and its
        description (<i>aDescr</i>) from a help item specified by a zero-based
        index passed in the <i>Index</i> parameter.
    }
    procedure GetItem(const Index: Integer; var aParam, aDescr: String);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies number of help items in a list.
      Keywords:     Count,TCCRCmdLineHelpItems
      Description:
        Use Count to get number of help items stored in a list.
    }
    property Count: Integer  read GetCount;

  end;

  {========================== TCCRCustomCmdLineParams ==========================
    Overview:     Base class for application command-line parameters.
    SeeAlso:      TCCRCmdLineParams
    Description:
      Use TCCRCustomCmdLineParams to access parsed values of the command-line
      parameters of an application.
      <p>
      Do not create instances of TCCRCustomCmdLineParams; use a descendant class
      instead, such as TCCRCmdLineParams.
  }
  TCCRCustomCmdLineParams = class(TComponent)
  private

    fDebugMode:     Boolean;
    fHelp:          Boolean;
    fNoCCOW:        Boolean;
    fNoUserContext: Boolean;

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Builds a list of help items.
      SeeAlso:      TCCRCmdLineHelpItems.Add; TCCRCustomCmdLineParams.ShowHelp
      Description:
        The ShowHelp method calls the QueryHelpItems to build a list of help
        items. Use the Add method of the list specified by the <i>HelpItems</i>
        to add new help items corresponding to the command-line parameters
        supported by the class.
    }
    procedure QueryHelpItems(HelpItems: TCCRCmdLineHelpItems); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the DebugMode property.
      SeeAlso:      TCCRCustomCmdLineParams.DebugMode
      Description:
        Use SetDebugMode in descendant classes to modify the value of the
        DebugMode property.
    }
    procedure SetDebugMode(const Value: Boolean);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the NoCCOW property.
      SeeAlso:      TCCRCustomCmdLineParams.NoCCOW
      Description:
        Use SetNoCCOW in descendant classes to modify the value of the NoCCOW
        property.
    }
    procedure SetNoCCOW(const Value: Boolean);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the NoUserContext property.
      SeeAlso:      TCCRCustomCmdLineParams.NoUserContext
      Description:
        Use SetNoUserContext in descendant classes to modify the value of the
        NoUserContext property.
    }
    procedure SetNoUserContext(const Value: Boolean);

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Responds after the last constructor has executed.
      SeeAlso:      TCCRCustomCmdLineParams.Parse; TObject.AfterConstruction
      Keywords:     AfterConstruction,TCCRCustomCmdLineParams
      Description:
        AfterConstruction is called automatically after the objects last
        constructor has executed. Do not call it explicitly in your applications.
        <p>
        The AfterConstruction method implemented in TCCRCustomCmdLineParams
        calls the Parse method.
    }
    procedure AfterConstruction; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Parses command-line parameters.
      Keywords:     Parse,TCCRCustomCmdLineParams
      Description:
        Override this method to parse additional command-line parameters and
        assign their values to the corresponding properties. Descendant classes
        that override Parse should always call the inherited method.
    }
    procedure Parse; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Shows a list of command-line parameters and their
                    descriptions.
      SeeAlso:      TCCRCustomCmdLineParams.Help;
                    TCCRCustomCmdLineParams.QueryHelpItems
      Keywords:     ShowHelp,TCCRCustomCmdLineParams
      Description:
        Call ShowHelp to display a modal dialog box with a list of supported
        command line parameters and their descriptions.
    }
    procedure ShowHelp;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies whether an application should be run in debug mode.
      Keywords:     DebugMode,TCCRCustomCmdLineParams
      Description:
        DebugMode is True if the '/debug' or '-debug' parameter is specified.
        An application should switch to the debug mode (enable debug log, debug
        trace, etc.).
    }
    property DebugMode: Boolean  read fDebugMode  default False;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies whether an application should display a list of
                    supported command-line parameters.
      Keywords:     Help,TCCRCustomCmdLineParams
      Description:
        Help is True if one the following parameters is specified: '/?', '-?',
        '/h', or '-h'. An application should display a message window with a
        list of command line parameters and their short descriptions.
    }
    property Help: Boolean  read fHelp  default False;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies whether the CCOW functionality of an application
                    should be completely disabled.
      SeeAlso:      TCCRCustomCmdLineParams.NoUserContext
      Keywords:     NoCCOW,TCCRCustomCmdLineParams
      Description:
        NoCCOW is True if one of the following parameters is specified:
        '/noccow','-noccow', '/ccow=off', or '-ccow=off'. An application should
        completely disable CCOW functionality (do not run the contextor).
    }
    property NoCCOW: Boolean  read fNoCCOW  default False;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies whether the user context functionality of an
                    application should be disabled.
      SeeAlso:      TCCRCustomCmdLineParams.NoCCOW
      Keywords:     NoUserContext,TCCRCustomCmdLineParams
      Description:
        NoUserContext is True if one of the following parameters is specified:
        '/patientonly', '-patientonly', '/ccow=patientonly', or
        '-ccow=patientonly'. An application should disable the user context
        functionality (assign nil to the Contextor property of the RPC broker
        component, exclude user' from the contextor's notification filter, etc.).
    }
    property NoUserContext: Boolean  read fNoUserContext  default False;

  end;

  {============================= TCCRCmdLineParams =============================
    Overview:     Application command-line parameters.
    Description:
      Properties of the TCCRCmdLineParams represent command-line parameters
      specified by a user.
  }
  TCCRCmdLineParams = class(TCCRCustomCmdLineParams)
  private

    fAVCodes:       String;
    fBrokerPort:    Integer;
    fBrokerServer:  String;

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Builds a list of help items.
      SeeAlso:      TCCRCmdLineHelpItems.Add;
                    TCCRCustomCmdLineParams.QueryHelpItems
      Description:
        Calls the Add method of the list specified by the <i>HelpItems</i>
        parameter to add new help items corresponding to the command-line
        parameters supported by the class.
    }
    procedure QueryHelpItems(HelpItems: TCCRCmdLineHelpItems); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the BrokerPort property.
      SeeAlso:      TCCRCmdLineParams.BrokerPort
      Description:
        Use SetBrokerPort in descendant classes to modify the value of the
        BrokerPort property.
    }
    procedure SetBrokerPort(const Value: Integer);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the BrokerServer property.
      SeeAlso:      TCCRCmdLineParams.BrokerServer
      Description:
        Use SetBrokerServer in descendant classes to modify the value of the
        BrokerServer property.
    }
    procedure SetBrokerServer(const Value: String);

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Parses command-line parameters.
      SeeAlso:      TCCRCustomCmdLineParams.Parse
      Keywords:     Parse,TCCRCmdLineParams
      Description:
        Override this method to parse additional command-line parameters and
        assign their values to the corresponding properties. Descendant classes
        that override Parse should always call the inherited method.
    }
    procedure Parse; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Access and verify codes separated by semicolon.
      SeeAlso:      TCCRBroker.Connect; TCCRCmdLineParams.BrokerPort;
                    TCCRCmdLineParams.BrokerServer
      Description:
        If the CCR components are compiled with the CCRDEBUG symbol defined,
        this property stores access and verify codes specified by a user in the
        application command line ('/av=' or '-av='). TCCRBroker uses a value of
        this property to automatically connect to the server.  
        <p>
        This feature is intended only for debugging/development purposes and
        must not be included in released application.
    }
    property AVCodes: String  read fAVCodes;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Port number of an RPC broker listener on a VistA M server.
      SeeAlso:      TCCRCmdLineParams.BrokerServer
      Keywords:     BrokerPort,TCCRCmdLineParams
      Description:
        Use BrokerPort to get a port number of an RPC broker listener specified
        by the user in the application command line ('/p=', '-p=', 'p=',
        '/port=', or '-port=').
    }
    property BrokerPort: Integer  read fBrokerPort;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Name or internet protocol (IP) address of a VistA M server.
      SeeAlso:      TCCRCmdLineParams.BrokerPort
      Keywords:     BrokerServer,TCCRCmdLineParams
      Description:
        Use BrokerServer to get a name or an IP address of an RPC broker
        listener specified by the user in the application command line ('/s=',
        '-s=', 's=', '/server=', '-server=').
    }
    property BrokerServer: String  read fBrokerServer;

  end;

///////////////////////////////// Implementation \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

implementation

uses
  fROR_CmdLineParams, uROR_Classes;

type

  {============================ TCCRCmdLineHelpItem ============================
    Overview:     Help item.
    SeeAlso:      TCCRCmdLineHelpItems
    Description:
      TCCRCmdLineHelpItem is used internally to associate command-line
      parameters with their descriptions.
  }
  TCCRCmdLineHelpItem = class(TObject)
  private

    fDescr: String;
    fParam: String;

  public

    property Descr: String  read fDescr  write fDescr;
    property Param: String  read fParam  write fParam;

  end;

///////////////////////////// TCCRCmdLineHelpItems \\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRCmdLineHelpItems.Create;
begin
  inherited;
  fList := TStringList.Create;
  fList.Sorted := True;
  fList.Duplicates := dupIgnore;
end;

destructor TCCRCmdLineHelpItems.Destroy;
begin
  Clear;
  FreeAndNil(fList);
  inherited;
end;

procedure TCCRCmdLineHelpItems.Add(const aParam, aDescr: String);
var
  hi: TCCRCmdLineHelpItem;
begin
  hi := TCCRCmdLineHelpItem.Create;
  hi.Param := aParam;
  hi.Descr := aDescr;
  fList.AddObject(aParam, hi);
end;

procedure TCCRCmdLineHelpItems.AssignTo(Dest: TPersistent);
var
  i, n: Integer;
  hi: TCCRCmdLineHelpItem;
begin
  if Dest is TRichEdit then
    with TRichEdit(Dest) do
      begin
        Clear;
        n := fList.Count - 1;
        for i:=0 to n do
          begin
            hi := TCCRCmdLineHelpItem(fList.Objects[i]);
            //--- Parameter(s)
            Paragraph.FirstIndent := 0;
            Paragraph.RightIndent := 50;
            SelAttributes.Style := [fsBold];
            Lines.Add('');
            Lines.Add(hi.Param);
            //--- Description
            Paragraph.FirstIndent := 50;
            Paragraph.RightIndent := 0;
            SelAttributes.Style := [];
            Lines.Add(hi.Descr);
          end;
      end
  else
    inherited;
end;

procedure TCCRCmdLineHelpItems.Clear;
var
  i: Integer;
begin
  with fList do
    begin
      for i:=Count-1 downto 0 do
        begin
          Objects[i].Free;
          Objects[i] := nil;
        end;
      Clear;
    end;
end;

function TCCRCmdLineHelpItems.GetCount: Integer;
begin
  Result := fList.Count;
end;

procedure TCCRCmdLineHelpItems.GetItem(const Index: Integer;
  var aParam, aDescr: String);
begin
  with TCCRCmdLineHelpItem(fList.Objects[Index]) do
    begin
      aParam := Param;
      aDescr := Descr;
    end;
end;

////////////////////////////// TCCRCmdLineParams \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TCCRCmdLineParams.Parse;
var
  i: Integer;
begin
  inherited;
  for i := 1 to ParamCount do
    begin
      if StartString(ParamStr(i),
        ['/p=', '-p=', 'p=', '/port=', '-port='], False) then
        fBrokerPort := StrToIntDef(Piece(ParamStr(i), '=', 2), 9200)
      else if StartString(ParamStr(i),
        ['/s=', '-s=', 's=', '/server=', '-server='], False) then
        fBrokerServer := Piece(ParamStr(i), '=', 2);
    end;
  {$IFDEF CCRDEBUG}
  CmdLineSwitch(['-av=', '/av='], fAVCodes);
  {$ENDIF}
end;

procedure TCCRCmdLineParams.QueryHelpItems(HelpItems: TCCRCmdLineHelpItems);
begin
  inherited;
  HelpItems.Add(
    '/port=, /p=, P=,'#13'-port=, -p=',
    'Port number of the RPC Broker listener.');
  HelpItems.Add(
    '/server=, /s=, S=,'#13'-server=, -s=',
    'Server name or IP address of the RPC Broker listener.');
end;

procedure TCCRCmdLineParams.SetBrokerPort(const Value: Integer);
begin
  fBrokerPort := Value;
end;

procedure TCCRCmdLineParams.SetBrokerServer(const Value: String);
begin
  fBrokerServer := Value;
end;

/////////////////////////// TCCRCustomCmdLineParams \\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TCCRCustomCmdLineParams.AfterConstruction;
begin
  inherited;
  if not (csDesigning in ComponentState) then
    Parse;
end;

procedure TCCRCustomCmdLineParams.Parse;
var
  i: Integer;
  val: String;
begin
  for i := 1 to ParamCount do
    begin
      if StartString(ParamStr(i), ['/?','-?','/h','-h'], False) then
        fHelp := True
      else if StartString(ParamStr(i), ['/debug', '-debug'], False) then
        fDebugMode := True
      else if StartString(ParamStr(i), ['/at', '-at'], False) then
        SetCCRScreenReaderOn
      else if StartString(ParamStr(i), ['/noccow','-noccow'], False) then
        fNoCCOW := True
      else if StartString(ParamStr(i),
        ['/patientonly','-patientonly'], False) then
        fNoUserContext := True
      else if StartString(ParamStr(i), ['/ccow=','-ccow='], False) then
        begin
          val := Piece(ParamStr(i), '=', 2);
          if AnsiSameText(val, 'off') then
            fNoCCOW := True
          else if AnsiSameText(val, 'patientonly') then
            fNoUserContext := True;
        end;
    end;
end;

procedure TCCRCustomCmdLineParams.QueryHelpItems(HelpItems: TCCRCmdLineHelpItems);
begin
  HelpItems.Add(
    '/?, -?, /h, -h',
    'Show a list of command-line parameters.');
  HelpItems.Add(
    '/debug, -debug',
    'Run the application in debug mode.');
  HelpItems.Add(
    '/noccow, /ccow=off,'#13'-noccow, -ccow=off',
    'Completely disable CCOW functionality.');
  HelpItems.Add(
    '/at, -at',
    'Turn assistive technology mode ON for non-JAWS users.');
  HelpItems.Add(
    '/patientonly, /ccow=patientonly,'#13'-patientonly, -ccow=patientonly',
    'Disable user context functionality.');
end;

procedure TCCRCustomCmdLineParams.SetDebugMode(const Value: Boolean);
begin
  fDebugMode := Value;
end;

procedure TCCRCustomCmdLineParams.SetNoCCOW(const Value: Boolean);
begin
  fNoCCOW := Value;
end;

procedure TCCRCustomCmdLineParams.SetNoUserContext(const Value: Boolean);
begin
  fNoUserContext := Value;
end;

procedure TCCRCustomCmdLineParams.ShowHelp;
var
  HelpItems: TCCRCmdLineHelpItems;
begin
  HelpItems := TCCRCmdLineHelpItems.Create;
  try
    QueryHelpItems(HelpItems);
    TFormCCRCmdLineParams.Show(HelpItems);
  finally
    HelpItems.Free;
  end;
end;

end.
