{******************************************************************************}
{ Package:      Clinical Case Registries Custom Components                     }
{ Date Created: November 18, 2004                                              }
{ Site Name:    Hines OIFO                                                     }
{ Developers:   Sergey Gavrilov                                                }
{ Description:  Design pattern classes.                                        }
{ Note:                                                                        }
{******************************************************************************}

unit uROR_Classes;

{$I Components.inc}

interface

uses
  Classes, SysUtils, Windows ;

type

{$REGION ' TCCRSingleton Class Definition '}
  {=============================== TCCRSingleton ===============================
    Overview:     Thread-safe singleton base class.
    Description:
      Use TCCRSingleton as a base class for singleton classes. Do not use
      Create and Destroy to initialize/finalize a singleton instance; override
      Initialize and Finalize instead.
  }
  TCCRSingleton = class(TPersistent)
  private

    fRefCount: Integer;

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Finalizes a singleton instance.
      SeeAlso:      TCCRSingleton.Initialize
      Keywords:     Finalize,TCCRSingleton
      Description:
        Override this method to finalize a singleton instance before it is
        destroyed. Unlike Destroy, the Finalize method is called only once.
        Do not call this method directly.
    }
    procedure Finalize; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Initializes a singleton instance.
      SeeAlso:      TCCRSingleton.Finalize
      Keywords:     Initialize,TCCRSingleton
      Description:
        Override this method to initialize a singleton instance after it is
        created. Unlike Create, the Initialize method is called only once.
        Do not call this method directly.
    }
    procedure Initialize; virtual;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys all singletons.
      Keywords:     DeleteAll,TCCRSingleton
      Description:
        Call DeleteAll to destroy all singletons. It is called automatically
        during the uROR_Classes unit finalization.
    }
    class procedure DeleteAll;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Deallocates memory allocated by a previous call to the
                    NewInstance method.
      SeeAlso:      TCCRSingleton.NewInstance; TObject.FreeInstance
      Keywords:     FreeInstance,TCCRSingleton
      Description:
        All destructors call FreeInstance automatically to deallocate memory
        that was allocated by overriding NewInstance.
    }
    procedure FreeInstance; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Prevents other threads from modifying singleton properties.
      SeeAlso:      TCCRSingleton.Unlock
      Keywords:     Lock,TCCRSingleton
      Description:
        Place any code that modifies singleton properties between Lock and
        Unlock in multi-threaded applications.
    }
    class procedure Lock;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Allocates memory for an instance of an object type and
                    returns a pointer to that new instance.
      SeeAlso:      TCCRSingleton.FreeInstance; TObject.NewInstance
      Keywords:     NewInstance,TCCRSingleton
      Description:
        All constructors call NewInstance automatically. NewInstance calls
        InstanceSize to determine how much memory to allocate from the heap to
        contain a particular instance. Do not call NewInstance directly.
    }
    class function NewInstance: TObject; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Allows changing singleton properties in other threads.
      SeeAlso:      TCCRSingleton.Lock
      Keywords:     Unlock,TCCRSingleton
      Description:
        Place any code that modifies singleton properties between Lock and
        Unlock in multi-threaded applications.
    }
    class procedure Unlock;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Reference counter for a singleton.
      SeeAlso:      TObject.Create; TObject.Free
      Keywords:     RefCount,TCCRSingleton
      Description:
        RefCount is incremented every time a new singleton reference is created
        using Create. It is decremented every time a singleton reference is
        destroyed by Free. When RefCount value reaches 0, a singleton instance
        is destroyed.
    }
    property RefCount: Integer  read fRefCOunt;

  end;
{$ENDREGION}

{$REGION ' TCCRScreenReader Class Definition '}
  {============================== TCCRScreenReader =============================
    Overview:     Singleton for checking if screen reader is being used.
    Description:
      TCCRScreenReader checks (one time) whether a screen reader is in use using
      the ScreenReaderSystemActive function from the VA508AccessibilityRouter.
  }
  TCCRScreenReader = class(TCCRSingleton)
  private

    fScreenReaderInUse:       boolean;

  protected

    procedure Finalize; override;
    procedure Initialize; override;
    procedure SetOverride;

  public

    property ScreenReaderInUse: boolean  read fScreenReaderInUse;

  end;
{$ENDREGION}

{- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Overview:     Returns true if a screen reader is in use.
  Description:
    CCRScreenReaderActive returns true if a screen reader is in use using the
    TCCRScreenReader object.
}

function CCRScreenReaderActive(): boolean;

{- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Overview:     Turns the CCRScreenReaderActive flag ON.
  Description:
    SetCCRScreenReaderOn sets the flag so that CCRScreenReaderActive will
    return true.  This is only needed to override the default behavior when a
    screen reader other than JAWS is being used.
}
procedure SetCCRScreenReaderOn;

///////////////////////////////// Implementation \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

implementation

uses VA508AccessibilityRouter;

var
  _SingletonList: TStringList = nil;
  _FLock: TRTLCriticalSection;
  _ScreenReader: TCCRScreenReader = nil;




{$REGION ' TCCRSingleton Methods '}
///////////////////////////////// TCCRSingleton \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TCCRSingleton.Initialize;
begin
end;

procedure TCCRSingleton.Finalize;
begin
end;

class procedure TCCRSingleton.DeleteAll;
var
  i: Integer;
begin
  Lock;

  for i:=_SingletonList.Count-1 downto 0 do
    _SingletonList.Objects[i].Free;

  Unlock;
end;

procedure TCCRSingleton.FreeInstance;
var
  ndx: integer;
begin
  Lock;

  Dec(fRefCount);
  if fRefCount = 0 then
    begin
      if _SingletonList.Find(ClassName, ndx) then
        _SingletonList.Delete(ndx);
      Finalize;
      inherited FreeInstance;
    end;

  Unlock;
end;

class procedure TCCRSingleton.Lock;
begin
  EnterCriticalSection(_FLock);
end;

class function TCCRSingleton.NewInstance: TObject;
var
  ndx: integer;
begin
  Lock;

  Result := nil;

  if _SingletonList.Find(ClassName, ndx) then
    Result := _SingletonList.Objects[ndx];

  if not Assigned(Result) then
    begin
      Result := inherited NewInstance;
      TCCRSingleton(Result).Initialize;
      _SingletonList.AddObject(ClassName, Result);
    end;

  Inc(TCCRSingleton(Result).fRefCount);

  Unlock;
end;

class procedure TCCRSingleton.Unlock;
begin
  LeaveCriticalSection(_FLock);
end;
{$ENDREGION}

{$REGION ' TCCRScreenReader Methods '}
/////////////////////////////// TCCRScreenReader \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TCCRScreenReader.Finalize;
begin
  inherited;
end;

procedure TCCRScreenReader.Initialize;
begin
  inherited;

  fScreenReaderInUse := ScreenReaderSystemActive;
end;

procedure TCCRScreenReader.SetOverride;
begin
  fScreenReaderInUse := true;
end;
{$ENDREGION}

/////////////////////////////// Public Functions \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

function CCRScreenReaderActive(): boolean;
begin
  Result := False;
  if Assigned(_ScreenReader) then
  begin
    Result := _ScreenReader.ScreenReaderInUse;
  end;
end;


procedure SetCCRScreenReaderOn;
begin
  if Assigned(_ScreenReader) then
  begin
    _ScreenReader.SetOverride;
  end;
end;

////////////////////////// Initialization/Finalization \\\\\\\\\\\\\\\\\\\\\\\\\

initialization
  InitializeCriticalSection(_FLock);
  _SingletonList := TStringList.Create;
  _SingletonList.Sorted := true;
  _ScreenReader := TCCRScreenReader.Create;

finalization
  if _SingletonList.Count = 0 then
    begin
      FreeAndNil(_ScreenReader);
      FreeAndNil(_SingletonList);
      DeleteCriticalSection(_Flock);
    end;

end.
