{******************************************************************************}
{ Package:      Clinical Case Registries Custom Components                     }
{ Date Created: January 26, 2007                                               }
{ Site Name:    Hines OIFO                                                     }
{ Developers:   Sergey Gavrilov                                                }
{ Description:  Wrapper for a VistA RPC Broker.                                }
{ Note:                                                                        }
{******************************************************************************}

unit uROR_Broker;

{$I Components.inc}

interface

uses
  SysUtils, Classes, Forms, Dialogs,  Contnrs, Controls, TRPCB,CCOWRPCBroker,
  uROR_CustomContextor, uROR_CmdLineParams, uROR_CustomBroker;

type
  TCCRVistABrokerState = class;

  {================================= TCCRBroker ================================
    Overview:     Wrapper for the VistA RPC Broker.
    Description:
      TCCRBroker is a wrapper for the VistA RPC Broker. It encapsulates broker
      implemetation details and provides mechanisms for error processing and
      reporting, debug logging, and application command-line parameters
      processing.
  }
  TCCRBroker = class(TCCRCustomBroker)
  private

    fContextor:      TCCRCustomContextor;
    fContextStack:   TStack;
    fDefaultResults: TStringList;
    fListenerPort:   Integer;
    fRPCBroker:      TRPCBroker;
    fRPContext:      String;
    fSavedState:     TCCRVistABrokerState; 
    fServer:         String;

    function  GetCmdLineParams: TCCRCmdLineParams;
    procedure SetContextor(aValue: TCCRCustomContextor);
    procedure SetCmdLineParams(aValue: TCCRCmdLineParams);
    procedure SetListenerPort(const aValue: Integer);
    procedure SetRPCBroker(aValue: TRPCBroker);
    procedure SetRPContext(const aName: String);
    procedure SetServer(const aValue: String);

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Checks command-line parameters and modifies broker
                    properties accordingly.
      SeeAlso:      TCCRBroker.CmdLineParams
      Keywords:     CheckCmdLineParams,TCCRBroker
      Description:
        If the CmdLineParams property references a descendant of
        TCCRCustomCmdLineParams, the CheckCmdLineParams method analyses
        the application's command-line parameters specified by a user and
        modifies values of broker properties accordingly.
        <p>
        As implemented in TCCRBroker, CheckCmdLineParams updates the
        ListenerPort and Server properties.
    }
    procedure CheckCmdLineParams; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Protected implementation of the CheckProcError method.
      SeeAlso:      TCCRCustomBroker.CheckProcError;
                    TCCRCustomBroker.DoCheckProcError;
                    TCCRCustomBroker.OnCheckProcError
      Keywords:     DoCheckProcError,TCCRBroker
      Description:
        TCCRBroker overrides DoCheckProcError to implement the error processing
        and reporting used by the CCR remote procedures. Information about the
        remote procedure call and its results are passed via the <i>CallInfo</i>
        parameter.
        <P>
        A negative value of the first "^"-piece of the Results[0] indicates that
        an error occurred during the execution of the remote procedure. In this
        case, the second piece of the Results[0] contains number of error
        descriptors returned in the subsequent elements of the Results array.
        <p>
        Subsequent Results elements store error code, error message, and error
        location separated by "^". They are returned in reverse chronological
        order (most recent error first).
        <p>
        As implemented in TCCRBroker, DoCheckProcError analyzes the results and
        adds error descriptors to the internal errros list (Errros property of
        the <i>CallInfo</i>). It also assigns the primary error code (value of
        the first "^"-piece of the Resuls[0]) to the ErrorCode property of the
        <i>CallInfo</i>. At the same time, rpeProcedure is assigned to the
        ErrorType.
    }
    procedure DoCheckProcError(CallInfo: TCCRBrokerCallInfo); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Protected implementation of the Connect method.
      SeeAlso:      TCCRCustomBroker.CmdLineParams; TCCRCustomBroker.Connect;
                    TCCRCustomBroker.DoConnect; TCCRBroker.RPCBroker
      Keywords:     DoConnect,TCCRBroker
      Description:
        Do not call DoConnect directly; override it to establish a connection
        to a main application server. If the connection is succesfully open,
        this method should return True. Otherwise, False should be returned.
        <p>
        As implemented in TCCRBroker, If the connection point (the ListenerPort
        and Server properties) is not specified, the DoConnect calls the
        GetServerInfo procedure to get the server name/IP address and port
        number or let the user select them.
        <p>
        If an existing VistA RPC broker has not been assigned to the RPCBroker
        property, the wrapper creates a new instance of the TCCOWRPCBroker and
        initializes its properties.
        <p>
        If the components are compiled with the CCRDEBUG symbol, then an
        additional '-av' command-line parameters is supported. It allows to
        specify access and verify codes separated by semicolon. <u>This
        parameter is indended for debugging only (specify the parameter on the
        Run Parameters dialog box); you must not distribute applications
        compiled like this!</u>
        <p>
        Then, DoConnect tries to connect the RPC broker to the server. If an
        error occurs, this method displays an approriate error message and
        returns False. Otherwise, True is returned.
    }
    function DoConnect: Boolean; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Protected implementation of the Disconnect method.
      SeeAlso:      TCCRCustomBroker.Disconnect; TCCRCustomBroker.DoDisconnect;
                    TCCRBroker.RPCBroker
      Keywords:     DoDisconnect,TCCRBroker
      Description:
        Do not call DoDisconnect directly; override it to close a connection to
        a main application server. As implemented in TCCRBroker, DoDisconnect
        disconnects the VistA RPC broker referenced by the RPCBroker property
        from the server.
    }
    procedure DoDisconnect; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the value of the Connected property.
      SeeAlso:      TCCRCustomBroker.Connected; TCCRCustomBroker.GetConnected;
                    TCCRBroker.RPCBroker
      Keywords:     GetConnected,TCCRBroker
      Description:
        If the RPCBroker property is not nil, GetConnected returns the value of
        the Connected property of the referenced VistA RPC broker. Otherwise,
        False is returned.
    }
    function GetConnected: Boolean; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the value of the Results property.
      SeeAlso:      TCCRCustomBroker.CreateResults; TCCRCustomBroker.GetResults;
                    TCCRCustomBroker.Results
      Keywords:     GetResults,TCCRBroker
      Description:
        As implemented in TCCRBroker, GetResults calls the inherited method and
        returns its resul if it is not nil (a buffer from the top of the buffer
        stack). Otherwise, the default internal buffer is returned.
    }
    function GetResults: TStrings; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Responds to notifications that components are being created
                    or destroyed.
      SeeAlso:      TCCRCustomBroker.Notification
      Keywords:     Notification,TCCRBroker
      Description:
        Do not call the Notification method in an application.  Notification is
        called automatically when the component specified by <i>aComponent</i>
        is about to be inserted or removed, as specified by <i>Operation</i>.
        <p>
        TCCRBroker overrides this method in order to update its Contextor and
        RPCBroker properties when controls they refer to are destroyed.
    }
    procedure Notification(aComponent: TComponent;
      Operation: TOperation); override;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRBroker.
      SeeAlso:      TCCRCustomBroker.Create
      Keywords:     Create,TCCRBroker
      Description:
        Create initializes an instance of the TCCRBroker. <i>anOwner</i> is the
        component, typically a form, that is responsible for freeing the broker.
    }
    constructor Create(anOwner: TComponent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRBroker.
      SeeAlso:      TCCRCustomBroker.Destroy; TObject.Free
      Keywords:     Destroy,TCCRBroker
      Description:
        Do not call Destroy directly in an application. Instead, call Free.
        Free verifies that the component is not nil, and only then calls Destroy.
        <p>
        Applications should only free components explicitly when the constructor
        was called without assigning an owner to the component.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Calls a remote procedure.
      SeeAlso:      TCCRCustomBroker.CallProc; TCCRCustomBroker.DebugLog;
                    TCCRCustomBroker.DefaultProcMode; TCCRBroker.Results
      Keywords:     CallProc,TCCRBroker
      Description:
        This overloaded version of CallProc executes a remote procedure specified
        by the ProcedureName property of the <i>CallInfo</i> with string values
        passed in the <i>Parameters</i> array and content of the optional
        <i>MultList</i> string list as parameters.
        <p>
        The ProcMode property of the <i>CallInfo</i> specifies flags that
        control the execution and error processing. If it includes rpcDefault,
        then the value of the DefaultProcMode property of the broker is assigned
        to it before calling the procedure.
        <p>
        Procedure results are returned into a TStrings instance (usually a
        TStringList) referenced by the Results property of the <i>CallInfo</i>.
        If it is nil, a buffer referenced by the Results property of the broker
        is assigned to the it before calling the procedure.
        <p>
        As implemented in TCCRBroker, CallProc translates parameters into VistA
        RPC broker's format, delegates the procedure call to the VistA RPC
        broker referenced by the RPCBroker property, and processes the errors if
        they occur.
        <p>
        String values from the <i>Parameters</i> array are added to the Param
        array of the VistA RPOC broker (starting from 0 and in the same order).
        If a value starts from "@", it is considered as varable name and added
        to the Param array not as 'Literal' but as 'Reference'.
        <p>
        If an optional <i>MultList</i> parameter is specified and not empty,
        then its values are added to the 'List' parameter that is automatically
        added to the Param array. String indexes in the list starts from '1'
        (i.e. MultList[0] -> Param[i].Mult['1'], MultList[1] -> Param[i].Mult['2'],
        and so on). Indexes in the RPC broker Mult parameter starts from 1 to
        allow pass the array on the server side into FileMan APIs, which often
        ignore the 0 subscript or treat it in a special way.
        <p>
        If logging is enabled (see the DebugLog property), then procedure
        parameters and/or results are recorded into a debug log (see the
        TCCRCustomDebugLog).
    }
    function CallProc(CallInfo: TCCRBrokerCallInfo;
      const Parameters: array of String;
      MultList: TStringList = nil): Boolean; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Restores a saved VistA RPC broker context.
      SeeAlso:      TCCRBroker.PushRPContext; TCCRBroker.RPContext
      Description:
        Restores the topmost VistA RPC broker context previously saved by the
        PushRPContext in the internal stack. If the stack is empty, this method
        does nothing.
    }
    procedure PopRPContext;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Saves the current VistA RPC broker context and creates a
                    new one.
      SeeAlso:      TCCRBroker.PopRPContext; TCCRBroker.RPContext
      Description:
        PushRPContext tries to create a new context specified by the <i>aName</i>
        parameter. If the attempt is successful, then the previous VsitA RPC
        broker context (value of the RPContext property) is saved in the
        internal stack. It can be later restored by PopContext.
    }
    function PushRPContext(const aName: String): Boolean;

  published

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property DebugLog;         // TCCRCustomBroker
    property DefaultProcMode;  // TCCRCustomBroker
    property OnCheckProcError; // TCCRCustomBroker

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Reference to a command-line parameters object.
      SeeAlso:      TCCRBroker.CheckCmdLineParams
      Description:
        If a reference to a command-line parameters object is assigned to this
        property, then broker properties are automatically modified according
        to the command line parameters specified by the user. See the
        CheckCmdLineParams method for more details.
    }
    property CmdLineParams: TCCRCmdLineParams
      read GetCmdLineParams  write SetCmdLineParams;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Clinical contextor (CCOW).
      SeeAlso:      TCCRCustomBroker.CmdLineParams
      Keywords:     Contextor,TCCRBroker
      Description:
        Assign an instance of a CCR clinical contextor wrapper to the Contextor
        property before connecting to a server if you you want the broker
        handle user context changes. If this property is nil, or CmdLineParams
        is not nil and its NoUserContext property is True, then the user context
        functionality is disabled.
    }
    property Contextor: TCCRCustomContextor
      read fContextor  write SetContextor;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Port number used by a VistA RPC broker listener.
      SeeAlso:      TCCRCustomBroker.CmdLineParams; TCCRCustomBroker.Connect;
                    TRPCBroker.ListenerPort
      Keywords:     ListenerPort,TCCRBroker
      Description:
        The ListenerPort design-time property gives the developer the ability to
        select the Listener port on the VistA M Server.
        <p>
        If CmdLineParams is not nil and a user specified the 'port' parameter,
        then the value of this parameter is automatically assigned to the
        ListenerPort property.
        <p>
        If the value of this property is 0 at the moment when a connection
        attempt is made, the GetServerInfo method of the VistA RPC Broker is
        called (it may display a modal dialog box for server selection).
        <p>
        Changing the port number while the TCCRBroker component is connected
        has no effect (the property keeps its old value).
    }
    property ListenerPort: Integer
      read fListenerPort  write SetListenerPort  default 0;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     VistA RPC broker that is used to perform procedure call.
      SeeAlso:      TComponent.Owner
      Keywords:     RPCBroker,TCCRBroker
      Description:
        RPCBroker references a VistA RPC broker that actually performs remote
        procedure calls.
        <p>
        You can either assign an instance of a descendant of the TRPCBroker to
        this property or leave the nil value. In the latter case, an isntance
        of the TCCOWRPCBroker will be created automatically (just before
        connecting to the server) and assigned to the RPCBroker. The new object
        will have the same owner as the TCCRBroker wrapper.
        <p>
        If a broker that is being assigned to this property is already
        connected to a server, the DoConnect method is not called and some
        properties of the VsitA RPC broker are saved in an internally created
        TCCRVistABrokerInstance. When the value of the RPCBroker property is
        modified or the wrapper is destroyed, those properties of the VistA RPC
        broker are restored. See the TCCRVistABrokerInstance.Assign for the
        affected list of properties.
    }
    property RPCBroker: TRPCBroker  read fRPCBroker  write SetRPCBroker;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Current context for the VistA RPC broker.
      SeeAlso:      TCCRBroker.PopRPContext; TCCRBroker.PushRPContext
      Description:
        Use RPContext to get or set the current context for the VistA RPC
        broker. You can also use the PushRPContext and PopRPContext to
        save and restore the current context.
    }
    property RPContext: String  read fRPContext  write SetRPContext;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Server name or IP address.
      Keywords:     Server,TCCRBroker
      Description:
        The Server design-time property contains the name or Internet Protocol
        (IP) address of the VistA M Server system. If the name is used instead
        of the IP address, Microsoft Windows Winsock should be able to resolve
        it. Winsock can resolve a name to an IP address either through the
        Domain Name Service (DNS) or by looking it up in the HOSTS file on the
        client workstation. In the case where the same name exists in the DNS
        and in the HOSTS file, the HOSTS file entry takes precedence.
        <p>
        If CmdLineParams is not nil and a user specified the 'server' parameter,
        then the value of this parameter is automatically assigned to the
        Server property.
        <p>
        If this property is empty at the moment when a connection attempt is
        made, the GetServerInfo method of the VistA RPC Broker is called (it
        may display a modal dialog box for server selection).
        <p>
        Changing the name of the server while the TCCRBroker component is
        connected has no effect (the property keeps its old value).
    }
    property Server: String  read fServer  write SetServer;

  end;

  {============================ TCCRVistABrokerState ===========================
    Overview:     Storage for a RPC Broker state.
    Description:
      Use TCCRVistABrokerState to save and restore a current state of a
      TRPCBroker instance.
  }
  TCCRVistABrokerState = class(TPersistent)
  private

    fClearParameters: Boolean;
    fClearResults:    Boolean;
    fCurrentContext:  String;
    fOnRPCBFailure:   TOnRPCBFailure;
    fParam:           TParams;
    fRemoteProcedure: String;
    fResults:         TStrings;
    fRPCVersion:      String;
    fShowErrorMsgs:   TShowErrorMsgs;

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Copies the properties of an object to a destination object.
      SeeAlso:      TCCRVistABrokerState.Assign; TPersistent.AssignTo
      Keywords:     AssignTo,TCCRVistABrokerState
      Description:
        Override the AssignTo method to extend the functionality of the Assign
        method of destination objects so that they handle newly created object
        classes.
        <p>
        If the <i>Dest</i> is a TRPCBroker instance, Assign copies values of the
        ClearParameters, ClearResults, CurrentContext, Param, RemoteProcedure,
        Results, RPCVersion, and ShowErrorMsgs properties.
    }
    procedure AssignTo(Dest: TPersistent); override;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Constructs an object and initializes its data before the
                    object is first used.
      SeeAlso:      TObject.Create
      Keywords:     Create,TCCRVistABrokerState
      Description:
        Create constructs an object.
    }
    constructor Create;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys the TCCRVistABrokerState instance and frees its
                    memory.
      SeeAlso:      TObject.Free; TPersistent.Destroy
      Keywords:     Destroy,TCCRVistABrokerState
      Description:
        Do not call Destroy directly. Call Free instead. Free checks that the
        object reference is not nil before calling Destroy.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Copies the contents of another, similar object.
      SeeAlso:      TCCRVistABrokerState.AssignTo; TPersistent.Assign
      Keywords:     Assign,TCCRVistABrokerState
      Description:
        Call Assign to copy the properties or other attributes of one object
        from another.
        <p>
        If the <i>Source</i> is a TRPCBroker instance, Assign copies values of
        the ClearParameters, ClearResults, CurrentContext, Param, RemoteProcedure,
        Results, RPCVersion, and ShowErrorMsgs properties.
    }
    procedure Assign(Source: TPersistent); override;

  end;

///////////////////////////////// Implementation \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

implementation

uses
  Math, RpcConf1, RpcbErr, uROR_Utilities, uROR_Debug, uROR_Resources
{$IFDEF RPCLOG}
  , fZZ_EventLog, uZZ_RPCEvent
{$ENDIF}  
  ;

////////////////////////////////// TCCRBroker \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRBroker.Create(anOwner: TComponent);
begin
  inherited;
  fDefaultResults := TStringList.Create;
  fContextStack := TStack.Create;
  fListenerPort := 0;
  fServer := '';
end;

destructor TCCRBroker.Destroy;
begin
  RPCBroker := nil;
  FreeAndNil(fContextStack);
  FreeAndNil(fDefaultResults);
  inherited;
end;

function TCCRBroker.CallProc(CallInfo: TCCRBrokerCallInfo;
  const Parameters: array of String; MultList: TStringList = nil): Boolean;
var
  ip, j, n: Integer;
  idt: ICCRDebugTrace;
{$IFDEF RPCLOG}
  aStart,aStop: TDateTime;
  anEvent: TRPCEventItem;
{$ENDIF}

  procedure broker_error(const ErrMsg: String);
  begin
    if DebugLog.Enabled and DebugLog.BrokerErrors then
      idt.Write('ERROR: ''%s''', [ErrMsg]);

    if not (rpcSilent in CallInfo.ProcMode) then
      MessageDlg508(RSC0051,
        Format(RSC0050, [Server, ListenerPort, CallInfo.ProcedureName, ErrMsg]),
        mtError, [mbOK], 0);

    with CallInfo do
      begin
        Results.Clear;
        ErrorCode := CCRBEC_RPCEXCEPTION;
        ErrorType := rpeProcedure;
      end;
  end;

begin
  if not Assigned(CallInfo.Results) then
    CallInfo.Results := Self.Results;
  CallInfo.Results.Clear;

  with RPCBroker do
    if Connected then
      begin
        if DebugLog.Enabled then
          idt := TCCRDebugTrace.Create('RPC: ' + CallInfo.ProcedureName);

        Param.Clear;
        RemoteProcedure := CallInfo.ProcedureName;

        if rpcDefault in CallInfo.ProcMode then
          CallInfo.ProcMode := DefaultProcMode;
        n := High(Parameters);

        if DebugLog.Enabled and DebugLog.Parameters then
          for j:=0 to n do
            idt.Write('PRM[%03d]: ''%s''', [j,Parameters[j]]);

        ip := 0;
        while ip <= n do
          begin
            if (Copy(Parameters[ip], 1, 1) = '@') and (Parameters[ip] <> '@') then
              begin
                Param[ip].Value := Copy(Parameters[ip], 2, Length(Parameters[ip]));
                Param[ip].PType := Reference;
              end
            else
              begin
                Param[ip].Value := Parameters[ip];
                Param[ip].PType := Literal;
              end;
            Inc(ip);
          end;

        if Assigned(MultList) and (MultList.Count > 0) then
          begin
            if DebugLog.Enabled and DebugLog.Parameters then
              begin
                if DebugLog.LimitParams > 0 then
                  n := Min(MultList.Count, DebugLog.LimitParams) - 1
                else
                  n := MultList.Count - 1;
                for j:=0 to n do
                  idt.Write('MLT[%03d]: ''%s''', [j,MultList[j]]);
                if n < (MultList.Count-1) then
                  idt.Write('...');
              end;
            for j:=1 to MultList.Count do
              Param[ip].Mult[IntToStr(j)] := MultList[j-1];
            Param[ip].PType := List;
          end;

        try
          Result := True;
{$IFDEF RPCLOG}
  aStart := Now;
  anEvent := getTRPCBEventItem(RPCBroker);
{$ENDIF}
          lstCall(CallInfo.Results);
{$IFDEF RPCLOG}
  aStop := Now;
  anEvent.AppendResults(CallInfo.Results,aStart,aStop);
  addRPCEvent(anEvent);
{$ENDIF}
          if (ShowErrorMsgs = semQuiet) and (RPCBError <> '') then
            begin
              broker_error(RPCBError);
              Result := False;
            end
          else if not (rpcNoErrorCheck in CallInfo.ProcMode) then
            Result := not CheckProcError(CallInfo);
        except
          on e: EBrokerError do
            begin
              broker_error(e.Message);
              Result := False;
            end;
          else
            raise;
        end;

        if DebugLog.Enabled and DebugLog.Results then
          begin
            if DebugLog.LimitResults > 0 then
              n := Min(CallInfo.Results.Count, DebugLog.LimitResults) - 1
            else
              n := CallInfo.Results.Count - 1;
            for j:=0 to n do
              idt.Write('RES[%03d]: ''%s''',
                [j,CallInfo.Results[j]]);
            if n < (CallInfo.Results.Count-1) then
              idt.Write('...');
          end;
      end
    else
      Result := False;
end;

procedure TCCRBroker.CheckCmdLineParams;
begin
  inherited;
  if Assigned(CmdLineParams) then
    begin
      SetListenerPort(ListenerPort);
      SetServer(Server);
    end;
end;

procedure TCCRBroker.DoCheckProcError(CallInfo: TCCRBrokerCallInfo);
var
  iErr, ir, numErr: Integer;
  buf, errLoc, errMsg: String;
  errCode: Integer;
  errDetails: TStringList;
begin
  with CallInfo do
    begin
      ErrorCode := StrToIntDef(Piece(Results[0], '^'), 0);

      if ErrorCode >= 0 then
        begin
          ErrorCode := 0;
          ErrorType := rpeNone;
        end
      else if not (rpcSilent in ProcMode) then
        begin
          ErrorType := rpeProcedure;
          numErr := StrToIntDef(Piece(Results[0], '^', 2), 0);
          buf := Format(RSC0041, [ErrorCode, ProcedureName]);
          if numErr > 0 then
            begin
              buf := buf + RSC0042;
              if numErr > 1 then
                buf := buf + RSC0043
              else
                buf := buf + RSC0044;
              AddError(buf);

              errDetails := nil;
              ir := 1;
              for iErr := 1 to numErr do
                begin
                  if ir >= Results.Count then
                    Break;

                  buf := Results[ir];
                  errCode := StrToIntDef(Piece(buf, '^'), 0);
                  errMsg := Piece(buf, '^', 2);
                  errLoc := StringReplace(Piece(buf, '^', 3), '~', '^', []);

                  while True do
                    begin
                      Inc(ir);
                      if ir >= Results.Count then
                        Break;
                      buf := Results[ir];
                      if Piece(buf, '^') <> '' then
                        Break;
                      if not Assigned(errDetails) then
                        errDetails := TStringList.Create;
                      errDetails.Add(Piece(buf, '^', 2));
                    end;

                  AddError(errMsg, errCode, errLoc, errDetails);
                  errDetails := nil;
                end;
            end
          else
            AddError(buf);
        end;
    end;
end;

function TCCRBroker.DoConnect: Boolean;
var
  srv, port: String;

  procedure SignOnError(RPCB: TRPCBroker; const errMsg: String);
  begin
    MessageDlg508('', Format(RSC0049, [RPCB.Server, RPCB.ListenerPort, errMsg]),
      mtError, [mbOK], 0);
  end;

begin
  Result := False;

  if (ListenerPort = 0) or (Server = '') then
    if GetServerInfo(srv, port) = mrOK then
      begin
        ListenerPort := StrToIntDef(port, PORT);
        Server := srv;
      end
    else
      begin
        MessageDlg508('', RSC0047, mtInformation, [mbok], 0);
        Exit;
      end;

  if not Assigned(RPCBroker) then
    RPCBroker := TCCOWRPCBroker.Create(Owner);

  if RPCBroker is TCCOWRPCBroker then
    if not Assigned(Contextor) or
      (Assigned(CmdLineParams) and CmdLineParams.NoUserContext) then
        TCCOWRPCBroker(RPCBroker).Contextor := nil
    else
      TCCOWRPCBroker(RPCBroker).Contextor := Contextor.Contextor;

  RPCBroker.ListenerPort := ListenerPort;
  RPCBroker.Server := Server;
  {$IFDEF CCRDEBUG}
  if Assigned(CmdLineParams) and (CmdLineParams.AVCodes <> '') then
    RPCBroker.AccessVerifyCodes := CmdLineParams.AVCodes;
  {$ENDIF}

  try
    RPCBroker.Connected := True;

    // Workaround for a Broker bug (it always shows the main form)
    if Assigned(Application.MainForm) and not Application.ShowMainForm then
      with Application do
        begin
          BringToFront;
          MainForm.Visible := False;
        end;

    if RPCBroker.Connected then
      begin
        Application.ProcessMessages;
        if (RPContext <> '') and not RPCBroker.CreateContext(RPContext) then
          begin
            SignOnError(RPCBroker, Format(RSC0048, [RPContext]));
            RPCBroker.Connected := False;
          end
        else if not inherited DoConnect then
          RPCBroker.Connected := False;
      end;
  except
    on E: EBrokerError do
      begin
        if E.Code <> XWB_BadSignOn then
          SignOnError(RPCBroker, E.Message);
      end;
    else
      raise;
  end;

  Result := RPCBroker.Connected;
end;

procedure TCCRBroker.DoDisconnect;
begin
  if Assigned(RPCBroker) then
    RPCBroker.Connected := False;
  inherited;
end;

function TCCRBroker.GetCmdLineParams: TCCRCmdLineParams;
begin
  Result := TCCRCmdLineParams(inherited CmdLineParams);
end;

function TCCRBroker.GetConnected: Boolean;
begin
  if Assigned(RPCBroker) then
    Result := RPCBroker.Connected
  else
    Result := False;
end;

function TCCRBroker.GetResults: TStrings;
begin
  Result := inherited GetResults;
  if not Assigned(Result) then
    Result := fDefaultResults;
end;

procedure TCCRBroker.Notification(aComponent: TComponent;
  Operation: TOperation);
begin
  inherited;
  if Operation = opRemove then
    begin
      if aComponent = Contextor then
        Contextor := nil
      else if aComponent = RPCBroker then
        RPCBroker := nil;
    end;
end;

procedure TCCRBroker.PopRPContext;
begin
  if fContextStack.Count > 0 then
    RPContext := String(fContextStack.Pop);
end;

function TCCRBroker.PushRPContext(const aName: String): Boolean;
var
  ctx: String;
begin
  Result := False;
  ctx := RPContext;
  RPContext := aName;
  if RPContext = aName then
    begin
      fContextStack.Push(Pointer(ctx));
      Result := True;
    end;
end;

procedure TCCRBroker.SetCmdLineParams(aValue: TCCRCmdLineParams);
begin
  inherited CmdLineParams := aValue;
end;

procedure TCCRBroker.SetContextor(aValue: TCCRCustomContextor);
begin
  if aValue <> fContextor then
    begin
      if Assigned(fContextor) then
        fContextor.RemoveFreeNotification(Self);

      fContextor := aValue;

      if Assigned(fContextor) then
        fContextor.FreeNotification(Self);
    end;
end;

procedure TCCRBroker.SetListenerPort(const aValue: Integer);
begin
  if Assigned(CmdLineParams) and (CmdLineParams.BrokerPort <> 0) and
    not (csDesigning in ComponentState) then
      fListenerPort := CmdLineParams.BrokerPort
  else if not Connected or (csDesigning in ComponentState) then
    fListenerPort := aValue;
end;

procedure TCCRBroker.SetRPCBroker(aValue: TRPCBroker);
begin
  if aValue <> fRPCBroker then
    begin
      if Assigned(fRPCBroker) then
        begin
          fRPCBroker.RemoveFreeNotification(Self);
          //--- Restore original broker state (if it was saved)
          if not (csDesigning in ComponentState) and Assigned(fSavedState) then
            begin
              fRPCBroker.Assign(fSavedState);
              FreeAndNil(fSavedState);
            end;
        end;

      fRPCBroker := aValue;

      if Assigned(fRPCBroker) then
        begin
          fRPCBroker.FreeNotification(Self);
          //--- Save current broker state
          if not (csDesigning in ComponentState) and fRPCBroker.Connected then
            begin
              fSavedState := TCCRVistABrokerState.Create;
              fSavedState.Assign(fRPCBroker);
            end;
        end;
    end;
end;

procedure TCCRBroker.SetRPContext(const aName: String);
begin
  if aName <> fRPContext then
    begin
      if not (Assigned(RPCBroker) and RPCBroker.Connected) then
        fRPContext := aName
      else if RPCBroker.CreateContext(aName) then
        fRPContext := aName;
    end;
end;

procedure TCCRBroker.SetServer(const aValue: String);
begin
  if Assigned(CmdLineParams) and (CmdLineParams.BrokerServer <> '') and
    not (csDesigning in ComponentState) then
      fServer := CmdLineParams.BrokerServer
  else if not Connected or (csDesigning in ComponentState) then
    fServer := aValue;
end;

///////////////////////////// TCCRVistABrokerState \\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRVistABrokerState.Create;
begin
  inherited;
  fParam   := TParams.Create(nil);
  fResults := TStringList.Create;
end;

destructor TCCRVistABrokerState.Destroy;
begin
  FreeAndNil(fResults);
  FreeAndNil(fParam);
  inherited;
end;

procedure TCCRVistABrokerState.AssignTo(Dest: TPersistent);
begin
  if Dest is TRPCBroker then
    with TRPCBroker(Dest) do
      begin
        ClearParameters := fClearParameters;
        ClearResults    := fClearResults;
        OnRPCBFailure   := fOnRPCBFailure;
        RemoteProcedure := fRemoteProcedure;
        RPCVersion      := fRPCVersion;
        ShowErrorMsgs   := fShowErrorMsgs;

        CreateContext(fCurrentContext);

        Param.Assign(fParam);
        Results.Assign(fResults);
      end
  else
    inherited;
end;

procedure TCCRVistABrokerState.Assign(Source: TPersistent);
begin
  if Source is TRPCBroker then
    with TRPCBroker(Source) do
      begin
        fClearParameters := ClearParameters;
        fClearResults    := ClearResults;
        fCurrentContext  := CurrentContext;
        fOnRPCBFailure   := OnRPCBFailure;
        fRemoteProcedure := RemoteProcedure;
        fRPCVersion      := RPCVersion;
        fShowErrorMsgs   := ShowErrorMsgs;

        fParam.Assign(Param);
        fResults.Assign(Results);
      end
  else
    inherited;
end;

end.
