(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Example: Globals.CmdLineSwitch
  <p>
  This example shows how to recognize different versions of the same
  command-line parameter and get a parameter value.
  <p>
  CodeExample:
*)

  if CmdLineSwitch(['/?','-?','/h','-h']) then
    // Display the list of application command-line parameters

  ccowSwitch := CmdLineSwitch(['/ccow=','-ccow='], ccowVal);
  if CmdLineSwitch(['/noccow','-noccow']) or
    (ccowSwitch and AnsiSameText(ccowVal, 'off')) then
    begin
      // Disable CCOW
    end;

(* EndExample: *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Example: TCCRCBGroup.CBHAlign;TCCRCBGroup.CBHMargin;TCCRCBGroup.CBStep
  <p>
  If CBHMargin = 20 and CBStep = 60, then distances on the picture below will
  be as follow: A = 20, B = 60, and C = 80.
  <p>
  \{bml bmp\\CBAlign.bmp\}
  <p>
*)

(* EndExample: *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Example: TCCRSearchEdit.OnSearch; TCCRSearchEdit.OnSearchEnd; TCCRSearchEdit.OnSearchStart
  <p>
  This example shows how to write event handlers for simple and advanced search
  modes.
  <p>
  CodeExample:
*)
  <u><b>Simple Mode Search</b></u>

  ...
  DataGrid:   TCCRGridView;    // Grid view for search results
  SearchEdit: TCCRSearchEdit;  // Search pattern edit control
  ...

  TFormSimple.OnSearchStart(Sender: TObject; var aCommand: TCCRSearchCommand);
  var
    rawData: TStringList;
  begin
    rawData := TStringList.Create;
    try
      // Call the RPC that searches for data
      if RPCBroker.CallProc('ZZZZ SIMPLE SEARCH',
        [SearchEdit.Text], nil, [], rawData) then
        begin
          // Assign the data to the grid view
          DataGrid.Items.AssignRawData(rawData, [1,2]);
          // Activate the grid view
          DataGrid.Activate;
        end;
    finally
      rawData.Free;
    end;
    aCommand := cscFinish
  end;

  <u><b>Advanced Mode Search</b></u>

  ...
  DataGrid:   TCCRGridView;    // Grid view for search results
  MaxNumRec:  Integer;         // Maximum number of records loaded at once
  SearchEdit: TCCRSearchEdit;  // Search pattern edit control
  ...

  TFormAdvanced.OnSearchStart(Sender: TObject; var aCommand: TCCRSearchCommand);
  begin
    // Clear the grid view
    DataGrid.Items.Clear;
  end;

  TFormAdvanced.OnSearch(Sender: TObject; var aCommand: TCCRSearchCommand);
  var
    rawData: TStringList;
  begin
      aCommand := cscCancel;
      rawData := TStringList.Create;
      try
        // Call the RPC that searches for data
        if RPCBroker.CallProc('ZZZZ ADVANCED SEARCH',
          [SearchEdit.Text, MaxNumRec], nil, [], rawData) then
          if (rawData.Count > 0) then
            begin
              // Append the data to the grid view
              DataGrid.Items.AppendRawData(rawData, [1,2]);
              // Indicate that the search should be continued
              aCommand := cscContinue;
            end
          else
            // No more data: finish the search
            aCommand := cscFinish;
      finally
        rawData.Free;
      end;
  end;

  TFormAdvanced.OnSearchEnd(Sender: TObject; var aCommand: TCCRSearchCommand);
  begin
    // Activate the grid view
    if aCommand = cscFinish then
      DataGrid.Activate;
  end;

(* EndExample: *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Example: TCCRCustomSelector.Add; TCCRCustomSelector.AddAll; TCCRCustomSelector.CustomAdd
  <p>
  This an actual piece of code from the uROR_Selector.pas. It shows how to
  correctly override the Add and AddAll methods in a double-pane selector
  implementation.
  <p>
  CodeExample:
*)

procedure TCCRSelector.Add;
begin
  if Assigned(SourceList.Selected) then
    begin
      //--- Default processing
      if not CustomAdd then
        TransferSelectedItems(SourceList, ResultList);

      //--- Call the event handler
      inherited;

      //--- Common post-processing
      ResultChanged := True;
    end;
end;

procedure TCCRSelector.AddAll;
begin
  //--- Default processing
  if not CustomAdd then
    begin
      SourceList.SelectAll;
      TransferSelectedItems(SourceList, ResultList);
    end;

  //--- Call the event handler
  inherited;

  //--- Common post-processing
  ResultChanged := True;
end;

(* EndExample: *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Example:  TCCRCustomSelector.CustomRemove; TCCRCustomSelector.Remove; TCCRCustomSelector.RemoveAll
  <p>
  This an actual piece of code from the uROR_Selector.pas. It shows how to
  correctly override the Remove and RemoveAll methods in a double-pane selector
  implementation.
  <p>
  CodeExample:
*)

procedure TCCRSelector.Remove;
begin
  if Assigned(ResultList.Selected) then
    begin
      //--- Default processing
      if not CustomRemove then
        TransferSelectedItems(ResultList, SourceList);

      //--- Call the event handler
      inherited;

      //--- Common post-processing
      ResultChanged := True;
    end;
end;

procedure TCCRSelector.RemoveAll;
begin
  //--- Default processing
  if not CustomRemove then
    begin
      ResultList.SelectAll;
      TransferSelectedItems(ResultList, SourceList);
    end;

  //--- Call the event handler
  inherited;

  //--- Common post-processing
  ResultChanged := True;
end;

(* EndExample: *)


(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Example:  TCCRGridView.LoadLayout; TCCRGridView.SaveLayout
  <p>
  This is an example of a grid view layout saved in the VistA PARAMETERS file
  (#8989.5). "lsvPatientList" is the value of the <i>aSection</i> parameter
  of the LoadLayout and SaveLayout.
  <p>
  CodeExample:
*)

  <lsvPatientList>
    <Fields[0] value="150"/>
    <Fields[1] value="0"/>
    <Fields[2] value="40"/>
    <Fields[3] value="75"/>
    <Fields[4] value="75"/>
    <Fields[5] value="85"/>
    <Fields[6] value="40"/>
    <Fields[7] value="100"/>
    <Fields[8] value="50"/>
    <Fields[9] value="50"/>
    <Fields[10] value="82"/>
    <Fields[11] value="92"/>
    <Fields[12] value="50"/>
    <Fields[13] value="100"/>
    <Fields[14] value="163"/>
  </lsvPatientList>

(* EndExample: *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Example:  TCCRSelector.LoadLayout; TCCRSelector.SaveLayout
  <p>
  This is an example of a grid view selector layout saved in the VistA
  PARAMETERS file (#8989.5). "CPTSelector" is the value of the <i>aSection</i>
  parameter of the LoadLayout and SaveLayout.
  <p>
  CodeExample:
*)

  <CPTSelector>
    <SourceList>
      <Fields[0] value="60"/>
      <Fields[1] value="150"/>
    </SourceList>
    <ResultList>
      <Fields[0] value="60"/>
      <Fields[1] value="150"/>
    </ResultList>
  </CPTSelector>

(* EndExample: *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Example: TCCRCustomContextor.AddTXItem; TCCRCustomContextor.EndContextChange; TCCRCustomContextor.RemoveTXItem; TCCRCustomContextor.StartContextChange
  <p>
  Below is an example of a generic clinical contex change transaction. If a
  contextor is suspended, the transaction will affect only a local context copy.
  If the Vergence Desktop components are not installed, then the code will do
  nothing.
  <p>
  CodeExample:
*)

  commit := False;
  if StartContextChange then
    try
      AddTXItem('patient.id.mrn.nationalidnumber', Piece(anICN, 'V'));
      AddTXItem('patient.id.mrn.dfn_640', aDFN);
      commit := True;
    finally
      Result := (EndContextChange(commit) <> urCancel);
    end;

(* EndExample: *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Example: TCCRCustomDebugLog.BeginUpdate; TCCRCustomDebugLog.EndUpdate; TCCRStrDebugLog.BeginUpdate; TCCRStrDebugLog.EndUpdate
  <p>
  Code below demonstrates a recommended way to perform several consecutive log
  updates:
  <p>
  CodeExample:
*)

  var
    i: Integer;
    sl: TStringList

  ...

  with CCRDebugLog do
    begin
      BeginUpdate;
      try
        Append('Log Item #%d', [i]);
        Inc(i);

        Append(sl);
        Inc(i. sl.Count);

        Append('Log Item #' + IntToStr(i));
        Inc(i);
      finally
        EndUpdate;
      end;
    end;

(* EndExample: *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Example: TCCRCustomDebugLog.Lock; TCCRCustomDebugLog.Unlock
  <p>
  Below is a piece of code that demonstrates a correct usage of Lock and Unlock
  methods:
  <p>
  CodeExample:
*)

procedure TCCRCustomDebugLog.SetMaxCount(const aValue: Integer);
begin
  Lock;
  try
    if aValue <> fMaxCount then
      begin
        fMaxCount := aValue;
        Truncate;
      end;
  finally
    Unlock;
  end;
end;

(* EndExample: *)


(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Example: ICCRDebugTrace; ICCRDebugTrace.Error; ICCRDebugTrace.Write; TCCRDebugTrace; TCCRDebugTrace.Create
  <p>
  Below is a piece of code that demonstrates usage of trace objects. If the
  DEBUG_TRACE symbol is defined, this example produces the following output in
  the Event Log window:
  <p>
  <p>ODS: >>> TFormMain.FormCreate
  <p>ODS: <<< TFormMain.FormCreate
  <p>ODS: >>> TFormMain.Activate
  <p>ODS:    >>> TFormMain.FormActivate
  <p>ODS:       Example of a message.
  <p>ODS:       ! ! !  Example of an error message.
  <p>ODS:    <<< TFormMain.FormActivate
  <p>ODS: <<< TFormMain.Activate
  <p>
  CodeExample:
*)

program Project1;

uses
  Forms,
  {$IFDEF DEBUG_TRACE}uROR_Debug,{$ENDIF}
  Unit1 in 'Unit1.pas' {FormMain};

{$R *.res}

begin
  Application.Initialize;
  {$IFDEF DEBUG_TRACE}
  with TCCRDebugTrace.Create do
    begin
      { Uncomment the following lines to change initial trace object properties.
        For example, you can place this code in a thread initialization
        section to assign different prefixes and/or indentation level. }
      //EntryPrefix := '==> ';
      //ExitPrefix  := '<== ';
      //Indent      := 10;
    end;
  {$ENDIF}
  Application.CreateForm(TFormMain, FormMain);
  Application.Run;
end.


unit Unit1;

interface

uses
  Forms,
  {$IFDEF DEBUG_TRACE}uROR_Debug,{$ENDIF}
  uROR_Classes;
  
...

implementation

procedure TFormMain.Activate;
begin
  {$IFDEF DEBUG_TRACE}
  TCCRDebugTrace.Create('TFormMain.Activate');
  {$ENDIF}
  inherited;
end;

procedure TFormMain.FormActivate(Sender: TObject);
{$IFDEF DEBUG_TRACE}
var
  dt: ICCRDebugTrace;
{$ENDIF}
begin
  {$IFDEF DEBUG_TRACE}
  dt := TCCRDebugTrace.Create('TFormMain.FormActivate');
  {$ENDIF}
  // ...
  {$IFDEF DEBUG_TRACE}
  dt.Write('Example of a message.');
  {$ENDIF}
  // ...
  {$IFDEF DEBUG_TRACE}
  dt.Error('Example of an error message.');
  {$ENDIF}
end;

procedure TFormMain.FormCreate(Sender: TObject);
begin
  {$IFDEF DEBUG_TRACE}
  TCCRDebugLog.Create;
  TCCRDebugTrace.Create('TFormMain.FormCreate');
  {$ENDIF}
end;

procedure TFormMain.FormDestroy(Sender: TObject);
begin
  CCRDebugLog.Free;
end;

(* EndExample: *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Example: TCCRCustomCmdLineParams.Parse
  <p>
  Below is a piece of code that demonstrates processing of additional
  command-line parameters.
  <p>
  CodeExample:
*)

interface

uses
  uROR_CmdLineParams;

type

  TAppCmdLineParams = class(TCCRCmdLineParams)
  private

    fRegistry: String;
    ...

  public

    procedure Parse; override;

    property Registry: String  read fRegistry;
    ...

  end;

implementation

procedure TAppCmdLineParams.Parse;
begin
  inherited;

  //--- Default values
  fRegistry := 'VA HEPC';
  ...

  //--- Parse command-line parameters
  for i := 1 to ParamCount do
    begin
      if StartString(ParamStr(i),
        ['/registry=', '-registry='], False) then
        fRegistry := Piece(ParamStr(i), '=', 2);
      else if ...
    end;
end;

(* EndExample: *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Example: TCCRCustomCmdLineParams.ShowHelp
  <p>
  Below is a piece of code that demonstrates displaying help for command-line
  parameters when a help command-line parameter (/?, -h, etc.) is specified.
  It is assumed that a TCCRCmdLineParams component is placed on the main form.
  <p>
  CodeExample:
*)

program Project1;

uses
  Forms,
  Unit1 in 'Unit1.pas' {FormMain};

{$R *.res}

begin
  Application.Initialize;
  Application.Title := 'Test';
  Application.CreateForm(TFormMain, FormMain);

  if FormMain.CmdLineParams.Help then
    begin
      FormMain.CmdLineParams.ShowHelp;    // Show the help modal dialog
      Application.ShowMainForm := False;  // Do not show the main window
      Application.Terminate;              // Terminate the application
    end;

  Application.Run;
end.

(* EndExample: *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Example: TCCRCustomBroker.DoCheckProcError; TCCRBroker.DoCheckProcError
  <p>
  Below is an example of a simplified remote procedure error processing.
  A negative value of the first '^' piece of the first result line indicates
  an error. The second piece contains the corresponding error message.
  <p>
  CodeExample:
*)

procedure TMyBroker.DoCheckProcError(CallInfo: TCCRBrokerCallInfo);
var
  msg: String;
begin
  with CallInfo do
    begin
      if Results.Count = 0 then
        begin
          Result := 1;
          if Not (rpcSilent in ProcMode) then
            begin
              msg := Format('The ''%s'' remote procedure returned nothing!',
                [ProcedureName]);
              MessageDialog('RPC Error', msg, mtError, [mbOK], mrOK, 0);
            end;
          Exit;
        end;

      ErrroCode := StrToIntDef(Piece(Results[0], '^'), 0);

      if Result >= 0 then
        begin
          ErrroCode := 0;
          ErrorType := rpeProcedure;
        end
      else if not (rpcSilent in ProcMode) then
        begin
          msg := Piece(Results[0], '^', 2);
          MessageDialog('RPC Error', buf, mtError, [mbOK], mrOK, 0);
        end;
    end;
end;

(* EndExample: *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Example: TCCRCustomBroker.CreateResults; TCCRCustomBroker.Results
  <p>
  Below is an example of using the CreateResults method that allows using a
  default procedure results buffer without affecting other methods that do the
  same. After returning from the LoadPatientData, the Results property of the
  procedure broker will reference the same buffer that it did before the call
  and the content of the buffer will be the same.
  <p>
  CodeExample:
*)

procedure TMyObject.LoadPatientData(const DFN: String);
var
  i, n: Integer;
begin
  Broker.CreateResults;
  ...

  if Broker.CallProc('ZZSG RPC1', [DFN]) then
    with Broker do
      begin
        n := Results.Count - 1;
        for i:=0 to n do
          // Process the Results[i]

      end;
  ...

  if Broker.CallProc('ZZSG RPC2', []) then
    with Broker do
      begin
        n := Results.Count - 1;
        for i:=0 to n do
          // Process the Results[i]

      end;
  ...
  
end;  // Temporary buffer is automatically destroyed here

(* EndExample: *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  Example: TCCRBroker.PopRPContext; TCCRBroker.PushRPContext; TCCRBroker.RPContext
  <p>
  If you need to call a remote procedure that is associated with a context that
  is different from the default application context, you can use the following
  construction: 
  <p>
  CodeExample:
*)

  CCRBroker.PushRPContext('ZZ TEMP CONTEXT');
  try
    ...
    CCRBroker.CallProc( ... );
    ...
  finally
    CCRBroker.PopRPContext;
  end;

(* EndExample: *)

