{******************************************************************************}
{ Package:      Clinical Case Registries Custom Components                     }
{ Date Created: November 18, 2004                                              }
{ Site Name:    Hines OIFO                                                     }
{ Developers:   Sergey Gavrilov                                                }
{ Description:  This unit defines classes for a double-pane selector           } 
{               implementation.                                                }
{ Note:                                                                        }
{******************************************************************************}

unit uROR_Selector;
{$I Components.inc}

interface

uses
  ComCtrls, Forms, Buttons, Classes, Controls, Windows,
  {$IFNDEF NOORPHEUS}OvcFiler,{$ENDIF}
  uROR_GridView, uROR_CustomSelector;

type

  {=============================== TCCRResultList ==============================
    Overview:     Grid view implementation of a double-pane selector's result
                  pane.
    SeeAlso:      TCCRSelector; TCCRSourceList
    Description:
      TCCRResultList implements a double-pane selector's result pane as a
      grid view.
  }
  TCCRResultList = class(TCCRSelectorList)
  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnDblClick event dispatcher.
      SeeAlso:      TControl.DblClick; TControl.OnDblClick
      Keywords:     DblClick,TCCRResultList
      Description:
        The DblClick method is triggered by left mouse-button double-click
        message (WM_LBUTTONDBLCLK) from Windows. It calls an event handler
        attached to the OnDblClick event, if defined.
        <p>
        TCCRResultList overrides this method to allow removing items from
        the result pane by double-clicking them.
    }
    procedure DblClick; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Respond to key press events.
      SeeAlso:      TWinControl.DoKeyDown; TWinControl.OnKeyDown
      Keywords:     KeyDown,TCCRResultList
      Description:
        When a windowed control receives a key-down message (WM_KEYDOWN) from
        Windows, its message handler calls the DoKeyDown method. If DoKeyDown
        determines that the control should, in fact, process the character, it
        calls KeyDown, passing the key code and shift-key state in the
        <i>Key</i> and <i>Shift</i> parameters, respectively. See the
        description of the TWinControl.KeyDown for more details.
        <p>
        TCCRResultList overrides this method to allow removing items from
        the result pane by pressing Enter or Space keys.
    }
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnDragDrop event dispatcher.
      SeeAlso:      TControl.DragDrop; TControl.OnDragDrop
      Keywords:     DragDrop,TCCRResultList
      Description:
        TCCRResultList overrides the DragDrop method in order to check if
        dragged item(s) originated from the same selector before transferring
        them from the source pane to the result one.
        <p>
        The <i>Source</i> parameter is the object that was dropped onto the
        control. The <i>X</i> and <i>Y</i> parameters are the mouse coordinates
        where the object was dropped.
    }
    procedure DragDrop(Source: TObject; X, Y: Integer); override;

  end;

  {================================ TCCRSelector ===============================
    Overview:     Selector with two grid view panes.
    Description:
      The TCCRSelector is a grid view implementation of a double-pane selector.
      <p>
      Search results are presented to a user in the source grid view. The user
      can select some or all records and move/copy them to and from the result
      grid view. Items can be transferred between panes using buttons located
      on a panel between panes or keyboard shortcuts. They can also be dragged
      with a mouse.
  }
  TCCRSelector = class(TCCRCustomSelector)
  private

    function GetResultList: TCCRResultList;
    function GetSourceList: TCCRSourceList;

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Copies selected grid view items.
      SeeAlso:      TCCRCustomSelector.IDField; TCCRSelector.MoveSelectedItems;
                    TCCRSelector.TransferSelectedItems; TCCRGridView.Fields
      Description:
        CopySelectedItems copies selected items of the <i>SrcLst</i> grid view
        to the <i>DstLst</i> grid view.
        <p>
        Before copying an item, this method checks its identifier (a value of
        the data field with index specified by the IDField property). If it is
        not empty and the destination grid view already has an item with the
        same identifier, then the item is not copied.
    }
    procedure CopySelectedItems(SrcLst, DstLst: TCCRSelectorList); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates a result pane implementation.
                    TCCRCustomSelector.Create; TCCRSelector.CreateSourceControl
      Keywords:     CreateResultControl,TCCRSelector
      Description:
        Do not call CreateResultControl in your code; it is called internally
        in order to create an implementation of the result pane. As implemented
        in TCCRSelector, CreateResultControl returns a TCCRResultList instance.
    }
    function CreateResultControl: TWinControl; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates a source pane implementation.
      SeeAlso:      TCCRCustomSelector.Create; TCCRSelector.CreateResultControl
      Keywords:     CreateSourceControl,TCCRSelector
      Description:
        Do not call CreateSourceControl in your code; it is called internally
        in order to create an implementation of the source pane. As implemented
        in TCCRSelector, CreateSourceControl returns a TCCRSourceList instance.
    }
    function CreateSourceControl: TWinControl; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnUpdateButtons event handler dispatcher.
      SeeAlso:      TCCRCustomSelector.OnUpdateButtons;
                    TCCRCustomSelector.UpdateButtons
      Keywords:     DoUpdateButtons,TCCRSelector
      Description:
        OnUpdateButtons updates status of buttons located on a panel between
        the source and result panes according to the content of the panes.
        <p>
        Add and Remove buttons are enabled if at least one item of the source
        and result grid views respectively is selected. AddAll and RemoveAll
        buttons are enabled if the source and result grid views respectively
        are not empty.
    }
    procedure DoUpdateButtons(var EnableAdd, EnableAddAll,
      EnableRemove, EnableRemoveAll: Boolean); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Moves selected grid view items.
      SeeAlso:      TCCRSelector.CopySelectedItems;
                    TCCRSelector.TransferSelectedItems
      Description:
        MoveSelectedItems copies selected items of the <i>SrcLst</i> grid view
        to the <i>DstLst</i> grid view and then deletes copied items from the
        <i>SrcLst</i> grid view. See the CopySelectedItems for more details.
    }
    procedure MoveSelectedItems(SrcLst, DstLst: TCCRSelectorList); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Transfers selected grid view items between panes of the
                    selector.
      SeeAlso:      TCCRCustomSelector.IDField; TCCRSelector.MoveSelectedItems;
                    TCCRSelector.CopySelectedItems; TCCRGridView.Fields
      Description:
        Depending on the value of the AddRemoveMode property (see the
        corresponding topic for more details), TransferSelectedItems moves or
        copies selected items of the <i>SrcLst</i> grid view to the i>DstLst</i>
        grid view, or deletes the items.
        <p>
        Before copying an item, this method checks its identifier (a value of
        the data field with index specified by the IDField property). If it is
        not empty and the destination grid view already has an item with the
        same identifier, then the item is not copied.
        <p>
        After moving or deleting selected items, TransferSelectedItems selects
        an item that follows the previous selection. If there is no such item
        and the grid view is not empty, then the last item is selected.
    }
    procedure TransferSelectedItems(SrcLst, DstLst: TCCRSelectorList); virtual;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRSelector.
      SeeAlso:      TCCRCustomSelector.Create; TComponent.Owner
      Keywords:     Create,TCCRSelector
      Description:
        Call Create to instantiate a double-pane selector control at runtime.
        Controls placed in forms at design time do not need to be explicitly
        created because they are created automatically.
        <p>
        <i>anOwner</i> is the component, typically a form, that is responsible
        for freeing the double-pane selector control.
    }
    constructor Create(anOwner: TComponent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnAdd event handler dispatcher.
      SeeAlso:      TCCRCustomSelector.Add; TCCRCustomSelector.AddRemoveMode;
                    TCCRCustomSelector.OnAdd; TCCRSelector.TransferSelectedItems
      Keywords:     Add,TCCRSelector
      Description:
        Add moves or copies (see the AddRemoveMode property) selected items from
        the source grid view to the result one. As implemented in TCCRSelector,
        Add calls the TransferSelectedItems if the value of the CustomAdd
        property is False, and then it calls inherited method.
    }
    procedure Add; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnAddAll event handler dispatcher.
      SeeAlso:      TCCRCustomSelector.AddAll; TCCRCustomSelector.AddRemoveMode;
                    TCCRCustomSelector.OnAddAll;
                    TCCRSelector.TransferSelectedItems
      Keywords:     AddAll,TCCRSelector
      Description:
        AddAll moves or copies (see the AddRemoveMode property) all items from
        the source grid view to the result one. As implemented in TCCRSelector,
        AddAll calls the TransferSelectedItems if the value of the CustomAdd
        property is False, and then it calls inherited method.
    }
    procedure AddAll; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Clears source and result pane controls.
      SeeAlso:      TCCRSelector.ResultList; TCCRSelector.SourceList;
                    TCustomListView.Clear
      Keywords:     Clear,TCCRSelector
      Description:
        As implemented in TCCRSelector, Clear calls inherited method, and then
        clears both grid view controls by calling their Clear methods.
    }
    procedure Clear; override;

    {$IFNDEF NOORPHEUS}
    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Loads a grid view selector layout.
      SeeAlso:      TCCRGridView.LoadLayout; TCCRSelector.SaveLayout;
                    TCCRSelector.SourceList; TCCRSelector.ResultList;
                    TCCRVistAStore
      Keywords:     LoadLayout,TCCRSelector
      Description:
        Call LoadLayout to restore grid view layout (saved by the SaveLayout)
        from the data storage referenced by the <i>aStorage</i> parameter.
        <i>aSection</i> specifies the data section name (a key of the registry,
        an XML node, a section of an ini file, etc.) where the layout is stored.
        <p>
        As implemented in TCCRSelector, LoadLayout restores layouts of the
        source and destination grid views by calling their LoadLayout methods.
        Override LoadLayout in descendant classes to load additional layout
        characteristics.
        <p>
        Descendant classes that override LoadLayout should always call the
        inherited method.
    }
    procedure LoadLayout(aStorage: TOvcAbstractStore;
      const aSection: String = ''); override;
    {$ENDIF}

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnRemove event handler dispatcher.
      SeeAlso:      TCCRCustomSelector.AddRemoveMode;
                    TCCRCustomSelector.OnRemove; TCCRCustomSelector.Remove;
                    TCCRSelector.TransferSelectedItems
      Keywords:     Remove,TCCRSelector
      Description:
        Remove moves selected items from the result grid view to the source one
        or deletes them (see the AddRemoveMode property). As implemented in
        TCCRSelector, Remove calls the TransferSelectedItems if the value of
        the CustomAdd property is False, and then it calls inherited method.
    }
    procedure Remove; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnRemoveAll event handler dispatcher.
      SeeAlso:      TCCRCustomSelector.AddRemoveMode;
                    TCCRCustomSelector.OnRemoveAll; TCCRCustomSelector.RemoveAll;
                    TCCRSelector.TransferSelectedItems
      Keywords:     RemoveAll,TCCRSelector
      Description:
        RemoveAll moves all items from the result grid view to the source one
        or deletes them (see the AddRemoveMode property). As implemented in
        TCCRSelector, RemoveAll calls the TransferSelectedItems if the value of
        the CustomAdd property is False, and then it calls inherited method.
    }
    procedure RemoveAll; override;

    {$IFNDEF NOORPHEUS}
    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Saves a grid view selector layout.
      SeeAlso:      TCCRGridView.SaveLayout; TCCRSelector.LoadLayout;
                    TCCRSelector.SourceList; TCCRSelector.ResultList;
                    TCCRVistAStore
      Keywords:     SaveLayout,TCCRSelector
      Description:
        Call SaveLayout to save a grid view selector layout (it can be later
        restored by the LoadLayout) to the data storage referenced by the
        <i>aStorage</i> parameter. <i>aSection</i> specifies the data section
        name (a key of the registry, an XML node, a section of an ini file,
        etc.) where the layout is saved to.
        <p>
        As implemented in TCCRSelector, SaveLayout saves layouts of the source
        and destination grid views by calling their SaveLayout methods. Override
        this method in descendant classes to save additional layout
        characteristics.
        <p>
        Descendant classes that override SaveLayout should always call the
        inherited method.
    }
    procedure SaveLayout(aStorage: TOvcAbstractStore;
      const aSection: String = ''); override;
    {$ENDIF}

  published

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property AddRemoveMode;  // TCCRCustomSelector
    property Align;
    //property Alignment;
    property Anchors;
    //property AutoSize;
    property AutoSort;  // TCCRCustomSelector
    property BevelInner default bvNone;
    property BevelOuter default bvNone;
    property BevelWidth;
    property BiDiMode;
    property BorderWidth;
    property BorderStyle;
    //property Caption;
    property Color;
    property Constraints;
    property Ctl3D;
    property CustomAdd;     // TCCRCustomSelector
    property CustomRemove;  // TCCRCustomSelector
    //property DockSite;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property FullRepaint;
    property Font;
    property IDField;  // TCCRCustomSelector
    //property Locked;
    property ParentBiDiMode;
    {$IFDEF VERSION7}
    property ParentBackground;
    {$ENDIF}
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property SplitPos;  // TCCRCustomSelector
    property TabOrder;
    property TabStop;
    //property UseDockManager default True;
    property Visible;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property OnAdd;     // TCCRCustomSelector
    property OnAddAll;  // TCCRCustomSelector
    property OnCanResize;
    property OnClick;
    property OnConstrainedResize;
    property OnContextPopup;
    //property OnDockDrop;
    //property OnDockOver;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    //property OnGetSiteInfo;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnRemove;     // TCCRCustomSelector
    property OnRemoveAll;  // TCCRCustomSelector
    property OnResize;
    property OnSplitterMoved;  // TCCRCustomSelector
    property OnStartDock;
    property OnStartDrag;
    //property OnUnDock;
    property OnUpdateButtons;  // TCCRCustomSelector

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Result grid view control.
      SeeAlso:      TCCRSelector.SourceList; TCCRSourceList
      Description:
        Use ResultList to access properties of a grid view control in the
        result pane.
    }
    property ResultList: TCCRResultList  read GetResultList;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Source grid view control.
      SeeAlso:      TCCRSelector.ResultList; TCCRResultList
      Description:
        Use SourceList to access properties of a grid view control in the
        source pane.
    }
    property SourceList: TCCRSourceList  read GetSourceList;

  end;

///////////////////////////////// Implementation \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

implementation

///////////////////////////////// TCCRResultList \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TCCRResultList.DblClick;
begin
  inherited;
  if Assigned(Owner) and (Owner is TCCRCustomSelector) then
    TCCRCustomSelector(Owner).Remove;
end;

procedure TCCRresultList.DragDrop(Source: TObject; X, Y: Integer);
begin
  if Source <> Self then
    if (Source is TControl) and (TControl(Source).Owner = Owner) then
      if Assigned(Owner) and (Owner is TCCRCustomSelector) then
        TCCRCustomSelector(Owner).Add;
end;

procedure TCCRResultList.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited;
  if Shift = [] then
    if (Key = VK_RETURN) or (Key = Word(' ')) then
      if Assigned(Owner) and (Owner is TCCRCustomSelector) then
        TCCRCustomSelector(Owner).Remove;
end;

////////////////////////////////// TCCRSelector \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRSelector.Create(anOwner: TComponent);
begin
  inherited;

  with SourceList do
    begin
      MultiSelect := True;
      ReadOnly    := True;
    end;

  with ResultList do
    begin
      MultiSelect := True;
      ReadOnly    := True;
    end;
end;

procedure TCCRSelector.Add;
begin
  if Assigned(SourceList.Selected) then
    begin
      //--- Default processing
      if not CustomAdd then
        TransferSelectedItems(SourceList, ResultList);

      //--- Call the event handler
      inherited;

      //--- Common post-processing
      ResultChanged := True;
    end;
end;

procedure TCCRSelector.AddAll;
begin
  //--- Default processing
  if not CustomAdd then
    begin
      SourceList.SelectAll;
      TransferSelectedItems(SourceList, ResultList);
    end;

  //--- Call the event handler
  inherited;

  //--- Common post-processing
  ResultChanged := True;
end;

procedure TCCRSelector.Clear;
begin
  inherited;
  SourceList.Clear;
  ResultList.Clear;
end;

procedure TCCRSelector.CopySelectedItems(SrcLst, DstLst: TCCRSelectorList);
var
  dsti, srci: TCCRGridItem;
  oldCursor: TCursor;
  restoreCursor, ok2Add: Boolean;
  itemId: String;
  i: integer;

begin
  oldCursor := Screen.Cursor;
  if SrcLst.SelCount > 30 then
    begin
      Screen.Cursor := crHourGlass;
      restoreCursor := True;
    end
  else
    restoreCursor := False;

  DstLst.Items.BeginUpdate;
  try
    srci := SrcLst.Selected;
    while Assigned(srci) do
      begin
        itemId := srci.AsString[IDField];
        //--Section 508: The following code was added because of changes to the
        //  listviews to move the ID out of column 0 which caused the search
        //  using the DstLst.FindCaption to stop working.
        ok2Add := true;
        if (itemId <> '') then
        begin
          i := 0;
          while (i < DstLst.Items.Count) and (ok2Add) do
          begin
            if itemId = DstLst.Items[i].AsString[IDField] then
              ok2Add := false;
            inc(i);
          end;
        end;

        //-- Section 508: Changed to use ok2Add boolean instead of the original
        //   DstLst.FindCaption(0, itemId, False, True, False) = nil) line
        if (itemId = '') or ok2Add then
          begin
            dsti := DstLst.Items.Add;
            dsti.Assign(srci);
          end;
        srci := SrcLst.GetNextItem(srci, sdAll, [isSelected]);
      end;
  finally
    DstLst.Items.EndUpdate;
    if restoreCursor then
      Screen.Cursor := oldCursor;
  end;
end;

function TCCRSelector.CreateResultControl: TWinControl;
begin
  Result := TCCRResultList.Create(Self);
end;

function TCCRSelector.CreateSourceControl: TWinControl;
begin
  Result := TCCRSourceList.Create(Self);
end;

procedure TCCRSelector.DoUpdateButtons(var EnableAdd, EnableAddAll,
  EnableRemove, EnableRemoveAll: Boolean);
begin
  if SourceList.Items.Count > 0 then
    begin
      EnableAdd    := Assigned(SourceList.Selected);
      EnableRemove := Assigned(ResultList.Selected);
      EnableAddAll := True;
    end
  else
    begin
      EnableAdd    := False;
      EnableAddAll := False;
    end;
  if ResultList.Items.Count > 0 then
    begin
      EnableAdd       := Assigned(SourceList.Selected);
      EnableRemove    := Assigned(ResultList.Selected);
      EnableRemoveAll := True;
    end
  else
    begin
      EnableRemove    := False;
      EnableRemoveAll := False;
    end;
  inherited;
end;

function TCCRSelector.GetResultList: TCCRResultList;
begin
  Result := TCCRResultList(GetResultControl);
end;

function TCCRSelector.GetSourceList: TCCRSourceList;
begin
  Result := TCCRSourceList(GetSourceControl);
end;

{$IFNDEF NOORPHEUS}
procedure TCCRSelector.LoadLayout(aStorage: TOvcAbstractStore;
  const aSection: String);
var
  sn: String;
begin
  if aSection <> '' then
    sn := aSection
  else
    sn := Name;

  try
    aStorage.Open;
    try
      SourceList.LoadLayout(aStorage, sn);
      ResultList.LoadLayout(aStorage, sn);
    finally
      aStorage.Close;
    end;
  except
  end;
end;
{$ENDIF}

procedure TCCRSelector.MoveSelectedItems(SrcLst, DstLst: TCCRSelectorList);
begin
  SrcLst.Items.BeginUpdate;
  try
    CopySelectedItems(SrcLst, DstLst);
    SrcLst.DeleteSelected;
  finally
    SrcLst.Items.EndUpdate;
  end;
end;

procedure TCCRSelector.Remove;
begin
  if Assigned(ResultList.Selected) then
    begin
      //--- Default processing
      if not CustomRemove then
        TransferSelectedItems(ResultList, SourceList);

      //--- Call the event handler
      inherited;

      //--- Common post-processing
      ResultChanged := True;
    end;
end;

procedure TCCRSelector.RemoveAll;
begin
  //--- Default processing
  if not CustomRemove then
    begin
      ResultList.SelectAll;
      TransferSelectedItems(ResultList, SourceList);
    end;

  //--- Call the event handler
  inherited;

  //--- Common post-processing
  ResultChanged := True;
end;

{$IFNDEF NOORPHEUS}
procedure TCCRSelector.SaveLayout(aStorage: TOvcAbstractStore;
  const aSection: String);
var
  sn: String;
begin
  if aSection <> '' then
    sn := aSection
  else
    sn := Name;

  try
    aStorage.Open;
    try
      SourceList.SaveLayout(aStorage, sn);
      ResultList.SaveLayout(aStorage, sn);
    finally
      aStorage.Close;
    end;
  except
  end;
end;
{$ENDIF}

procedure TCCRSelector.TransferSelectedItems(SrcLst, DstLst: TCCRSelectorList);
var
  next: TCCRGridItem;
begin
  SrcLst.Items.BeginUpdate;
  try
    if (SrcLst = SourceList) or (AddRemoveMode = armDefault) then
      CopySelectedItems(SrcLst, DstLst);
    if AutoSort then
      DstLst.AlphaSort;
    if SrcLst.SelCount > 0 then
      begin
        //--- Find the first item after selection
        next := SrcLst.GetItemFollowingSelection;
        //--- Remove the selected items from the list or unselect them
        if (SrcLst = ResultList) or (AddRemoveMode = armDefault) then
          SrcLst.DeleteSelected
        else
          SrcLst.ClearSelection;
        //--- Select the item
        with SrcLst do
          if Assigned(next) then
            begin
              Selected := next;
              ItemFocused := Selected;
            end
          else if Items.Count > 0 then
            begin
              Selected := Items[Items.Count-1];
              ItemFocused := Selected;
            end;
      end;
  finally
    SrcLst.Items.EndUpdate;
  end;
end;

end.
