{******************************************************************************}
{ Package:      Clinical Case Registries Custom Components                     }
{ Date Created: November 18, 2004                                              }
{ Site Name:    Hines OIFO                                                     }
{ Developers:   Sergey Gavrilov                                                }
{ Description:  This unit defines classes for the custom CCR list view.        }
{ Note:                                                                        }
{******************************************************************************}

unit uROR_ListView;

{$I Components.inc}

interface

uses
  ComCtrls, Controls, Classes, Variants, uROR_Utilities, uROR_CustomListView;

type
  TCCRListView = class;

  {========================== TCCRGetFieldValueEvent ===========================
    Overview:     Event type for the list view data retrieval events.
    SeeAlso:      TCCRListView.OnFieldValueGet
    Description:
      This event is generated when a list view needs field data values. See
      the OnFieldValueGet property of the TCCRListView for more details.
  }
  TCCRGetFieldValueEvent = procedure(Sender: TCCRCustomListView;
    anItem: TCCRCustomListItem; const aFieldIndex: Integer;
    var anExternalValue: String; var anInternalValue: Variant) of object;

  {=============================== TCCRListItem ================================
    Overview:     TCCRListItem represents an item of the custom list view.
    SeeAlso:      TCCRListItems; TCCRListView
    Description:
      The TCCRListItem encapsulates functionality of a list view item. The item
      has no storage for internal field values and uses the OnFieldValueGet
      event handler of the list view to get them.
  }
  TCCRListItem = class(TCCRCustomListItem)
  private

    function GetListView: TCCRListView;

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns internal field value and its default printable
                    representation.
      SeeAlso:      TCCRCustomListView.GetFieldValue;
                    TCCRCustomListView.StringValues
      Keywords:     GetFieldValue,TCCRListItem
      Description:
        The GetFieldValue is called by a list view to get internal field values
        of an item and their default printable counterparts. TCCRListItem
        overrides this method to call the OnFieldValueGet event handler of the
        list view. If the OnFieldValueGet handler is not defined, then the item
        object assumes that internal values are the same as the printable ones
        and gets them from the StringValues property.
        <p>
        Zero-based index of the field is passed via the <i>aFieldIndex</i>
        input parameter. Internal value of the field is returned via the
        <i>anInternalValue</i> output parameter. Printable field value is
        returned as the function result.
    }
    function GetFieldValue(const aFieldIndex: Integer;
      var anInternalValue: Variant): String; override;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates the list view control that displays the item.
      SeeAlso:      TCCRListView
      Keywords:     ListView,TCCRListItem
      Description:
        Do not confuse the ListView that displays the item with the Owner of
        the item. The Owner is the TCCRListViewItems object that manages the
        collection of all list view items. The list view is the Owner of that
        TCCRListViewItems object.
        <p>
        TCCRListItem redefines this property in order to typecast it to
        TCCRListView.
    }
    property ListView: TCCRListView  read GetListView;

  end;

  {=============================== TCCRListItems ===============================
    Overview:     TCCRListItems maintains the collection of items that appear
                  in a list view control.
    SeeAlso:      TCCRListItem; TCCRListView
    Description:
      Use the properties and methods of TCCRListItems to manipulate the list of
      items displayed by a list view control.
  }
  TCCRListItems = class(TCCRCustomListItems)
  private

    function  GetItem(anIndex: Integer): TCCRListItem;
    procedure SetItem(anIndex: Integer; const aValue: TCCRListItem);

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates a new list item and adds it to the list view control.
      SeeAlso:      TCCRListItems.AddItem; TCCRListItems.AddObject;
                    TCCRListItems.Insert
      Keywords:     Add,TCCRListItems
      Description:
        Call Add to add a new list item to the end of the list. Add returns
        the newly created TCCRListItem object.
        <p>
        To create an item and insert it into the beginning or middle of the
        list, use the Insert method instead.
    }
    function Add: TCCRListItem;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Adds an item in a specified location.
      SeeAlso:      TCCRListItems.Add; TCCRListItems.Insert
      Keywords:     AddItem,TCCRListItems
      Description:
        Call AddItem to add grid item referenced by the <i>anItem</i> parameter
        at any place in the list. If <i>anItem</i> is not nil, its properties
        are duplicated. <i>anIndex</i> is location of the added item; if the
        index is negative, the TCCRListItem is appended to the end of the list.
        <p>
        AddItem returns the TCCRListItem that was added.
    }
    function AddItem(anItem: TCCRListItem; anIndex: Integer = -1): TCCRListItem;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Adds a new list item associated with an object to the list
                    view control.
      SeeAlso:      TCCRListItems.Add; TCCRListItems.AddObject
      Keywords:     AddObject,TCCRListItems
      Description:
        AddObject adds a new list item to the end of the list and assigns the
        <i>aCaption</i> parameter value to the Caption property of the item. It
        also assigns the <i>anObject</i> reference to the Data property of the
        item. AddObject returns the newly created TCCRListItem object.
    }
    function AddObject(const aCaption: String; anObject: TObject): TCCRListItem;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates a new grid item and inserts it into the list view.
      SeeAlso:      TCCRListItems.Add; TCCRListItems.AddObject;
                    TCCRListItems.AddItem;
      Keywords:     Insert,TCCRListItems
      Description:
        Call Insert to insert a new TCCRListItem object into the list view at
        the location specified by the <i>anIndex</i>. Insert returns the list
        item that is inserted.
    }
    function Insert(anIndex: Integer): TCCRListItem;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Lists all list items managed by the TCCRListItems object.
      SeeAlso:      TCCRListItem; TListItems.Count
      Keywords:     Item,TCCRListItems
      Description:
        Use Item to directly access a list item, given its position in the
        list view. The first item has an index of 0, the second an index
        of 1, and so on.
        <p>
        Item is the default property. So, if ListItems references an instance
        of the TCCRListItems, then you can access the first item either as
        ListItems.Item[0] or as ListItems[0].
    }
    property Item[anIndex: Integer]: TCCRListItem
      read GetItem  write SetItem;  default;

  end;

  {=============================== TCCRListView ================================
    Overview:     CCR list view.
    SeeAlso:      TCCRGridView
    Description:
      Use TCCRListView to display a simple table on a form. This control does
      not store data in internal format; it uses the OnFieldValueGet event
      handler to get and format the data.
  }
  TCCRListView = class(TCCRCustomListView)
  private

    fOnFieldValueGet: TCCRGetFieldValueEvent;

    function  GetItemFocused: TCCRListItem;
    function  GetItems: TCCRListItems;
    function  GetSelected: TCCRListItem;
    procedure SetItemFocused(aValue: TCCRListItem);
    procedure SetSelected(aValue: TCCRListItem);

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates a list item for the collection specified by the
                    Items property.
      SeeAlso:      TCCRListView.Items; TCustomListView.CreateListItem
      Keywords:     CreateListItem,TCCRListView
      Description:
        Call CreateListItem to create a TCCRListItem object that will be added
        to the Items collection. CreateListItem creates the list item but does
        not insert it into the collection.
    }
    function CreateListItem: TListItem; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates the TCCRListItems object that implements the
                    Items property.
      SeeAlso:      TCCRListView.Items; TCustomListView.CreateListItems
      Keywords:     CreateListItems,TCCRListView
      Description:
        TCCRListView calls CreateListItems internally to create the object that
        implements the Items property. As implemented in TCCRListView,
        CreateListItems creates and returns a TCCRListItems instance.
    }
    function CreateListItems: TListItems; override;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRListView.
      SeeAlso:      TComponent.Owner; TCCRCustomListView.Create
      Keywords:     Create,TCCRListView
      Description:
        The Create constructor assigns default values to the properties.
    }
    constructor Create(anOwner: TComponent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Adds a new list item to the list view control and associates
                    an object with it.
      SeeAlso:      TCCRListItems.AddObject
      Keywords:     AddItem,TCCRListView
      Description:
        AddItem adds a new list item to the end of the list and assigns the
        <i>aCaption</i> parameter value to the Caption property of the item. It
        also assigns the <i>anObject</i> reference to the Data property of the
        item.
    }
    procedure AddItem(aCaption: String; anObject: TObject); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the next list item after the StartItem in the
                    specified direction.
      SeeAlso:      TCCRListView.ItemFocused; TCCRListView.Selected
      Keywords:     GetNextItem,TCCRListView
      Description:
        Call GetNextItem to find the next list item after <i>StartItem</i> in
        the direction given by the <i>Direction</i> parameter. Only items in
        the state indicated by the <i>States</i> parameter are considered.
    }
    function GetNextItem(StartItem: TCCRCustomListItem;
      Direction: TSearchDirection; States: TItemStates): TCCRListItem;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates which item, if any, has focus.
      SeeAlso:      TCCRListView.Selected
      Keywords:     ItemFocused,TCCRListView
      Description:
        Read ItemFocused to determine which item, if any, can be edited by the
        user. Set ItemFocused to set focus to an item in the list. When an item
        has focus, it appears surrounded by a standard focus rectangle. If no
        item has focus, ItemFocused returns nil. Only one item can have focus
        at a time.
        <p>
        TCCRListView redefines this property to typecast it to TCCRListItem.
    }
    property ItemFocused: TCCRListItem  read GetItemFocused  write SetItemFocused;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Contains the list of items displayed by the list view.
      SeeAlso:      TCCRCustomListView.Items; TCCRListItems
      Keywords:     Items,TCCRListView
      Description:
        Use Items to directly access the TCCRListItems objects that represent
        the items in the list view.
        <p>
        TCCRListView redefines this property to typecast it to TCCRListItems.
    }
    property Items: TCCRListItems  read GetItems;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates the first selected item in the list view.
      SeeAlso:      TCCRListView.Focused; TCCRListView.GetNextItem;
                    TCustomListView.MultiSelect; TCustomListView.Selected;
                    TCustomListView.SelCount
      Keywords:     Selected,TCCRListView
      Description:
        Read Selected to access the properties of the first selected item in
        the list. If SelCount is 0, Selected is nil. If SelCount is greater
        than 1, subsequent selected items can be located by checking the
        Selected property of the items found using the GetNextItem method.
        <p>
        Set the Selected property to select an item in the list. If MultiSelect
        is true, setting Selected adds an item to the selection. If MultiSelect
        is false, setting Selected changes the selection. Setting Selected to
        nil deselects all items in the list.
        <p>
        When an item is selected, the OnChanging and OnChange events are
        generated.
        <p>
        TCCRListView redefines this property to typecast it to TCCRListItem.
    }
    property Selected: TCCRListItem  read GetSelected  write SetSelected;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property SortColumn;

  published

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property Action;
    property Align;
    property AllocBy;
    property Anchors;
    property BevelEdges;
    property BevelInner;
    property BevelKind  default bkNone;
    property BevelOuter;
    property BevelWidth;
    property BiDiMode;
    property BorderStyle;
    property BorderWidth;
    property Checkboxes;
    property Color;
    property ColumnClick;
    property Columns;
    property Constraints;
    property Ctl3D;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property FlatScrollBars;
    property Font;
    //property FullDrag;
    property GridLines  default False;
    property HideSelection;
    property HotTrack;
    property HotTrackStyles;
    property HoverTime;
    property IconOptions;
    property LargeImages;
    property MultiSelect;
    property OwnerData;
    property OwnerDraw;
    property ParentBiDiMode;
    property ParentColor  default False;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly  default False;
    property RowSelect;
    property ShowColumnHeaders  default True;
    property ShowHint;
    property ShowWorkAreas;
    property SmallImages;
    property SortDescending default False;  // TCCRCustomListView
    property SortField default -1;          // TCCRCustomListView
    //property SortType;
    property StateImages;
    property TabOrder;
    property TabStop  default True;
    property ViewStyle;
    property Visible;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property OnAdvancedCustomDraw;
    property OnAdvancedCustomDrawItem;
    property OnAdvancedCustomDrawSubItem;
    property OnChange;
    property OnChanging;
    property OnClick;
    property OnColumnClick;
    //property OnColumnDragged;
    property OnColumnRightClick;
    //property OnCompare;
    property OnContextPopup;
    property OnCustomDraw;
    property OnCustomDrawItem;
    property OnCustomDrawSubItem;
    property OnData;
    property OnDataFind;
    property OnDataHint;
    property OnDataStateChange;
    property OnDblClick;
    property OnDeletion;
    property OnDragDrop;
    property OnDragOver;
    property OnDrawItem;
    property OnEdited;
    property OnEditing;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnGetImageIndex;
    property OnGetSubItemImage;
    property OnInfoTip;
    property OnInsert;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;
    property OnSelectItem;
    property OnStartDock;
    property OnStartDrag;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs when the list view needs internal and/or external
                    cell values.
      SeeAlso:      TCCRListViewItem.GetFieldValue                    
      Description:
        Write an OnFieldValueGet event handler to supply the list view with the
        data. The <i>aSender</i> parameter is the list view whose event handler
        is called. The <i>anItem</i> and <i>aFieldIndex</i> reference the list
        item and data field index corresponding to the cell that has to be
        retrieved and/or formatted. Assign internal and printable values to the
        output parameters <i>anInternalValue</i> and <i>anExternalValue</i>
        accordingly.
        <p>
        If the OnFieldValueGet handler is not defined, then the TCCRListViewItem
        object assumes that internal values are the same as the printable ones
        and gets them from the StringValues property.
    }
    property OnFieldValueGet: TCCRGetFieldValueEvent
      read fOnFieldValueGet  write fOnFieldValueGet;

  end;

///////////////////////////////// Implementation \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

implementation

///////////////////////////////// TCCRListItem \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

function TCCRListItem.GetFieldValue(const aFieldIndex: Integer;
  var anInternalValue: Variant): String;
begin
  Result := '';
  anInternalValue := Unassigned;
  if (aFieldIndex >= 0) and (aFieldIndex < ListView.Columns.Count) then
    if Assigned(ListView.OnFieldValueGet) then
      ListView.OnFieldValueGet(ListView, Self, aFieldIndex, Result, anInternalValue)
    else
      begin
        Result := StringValues[aFieldIndex];
        //--- By default, internal value = external value
        anInternalValue := Result;
      end;
end;

function TCCRListItem.GetListView: TCCRListView;
begin
  Result := inherited ListView as TCCRListView;
end;

////////////////////////////////// TCCRListItems \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

function TCCRListItems.Add: TCCRListItem;
begin
  Result := TCCRListItem(inherited Add);
end;

function TCCRListItems.AddItem(anItem: TCCRListItem; anIndex: Integer): TCCRListItem;
begin
  Result := TCCRListItem(inherited AddItem(anItem, anIndex));
end;

function TCCRListItems.AddObject(const aCaption: String; anObject: TObject): TCCRListItem;
begin
  Result := Add;
  with Result do
    begin
      Caption := aCaption;
      Data := anObject;
      UpdateStringValues;
    end;
end;

function TCCRListItems.GetItem(anIndex: Integer): TCCRListItem;
begin
  Result := inherited Item[anIndex] as TCCRListItem;
end;

function TCCRListItems.Insert(anIndex: Integer): TCCRListItem;
begin
  Result := TCCRListItem(inherited Insert(anIndex));
end;

procedure TCCRListItems.SetItem(anIndex: Integer; const aValue: TCCRListItem);
begin
  Item[anIndex].Assign(aValue);
end;

////////////////////////////////// TCCRListView \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRListView.Create(anOwner: TComponent);
begin
  inherited;

  BevelKind         := bkNone;
  GridLines         := False;
  ParentColor       := False;
  ReadOnly          := False;
  ShowColumnHeaders := True;
  TabStop           := True;
end;

procedure TCCRListView.AddItem(aCaption: String; anObject: TObject);
begin
  Items.AddObject(aCaption, anObject);
end;

function TCCRListView.CreateListItem: TListItem;
var
  LClass: TListItemClass;
begin
  LClass := TCCRListItem;
  if Assigned(OnCreateItemClass) then
    OnCreateItemClass(Self, LClass);
  Result := LClass.Create(Items);
end;

function TCCRListView.CreateListItems: TListItems;
begin
  Result := TCCRListItems.Create(self);
end;

function TCCRListView.GetItemFocused: TCCRListItem;
begin
  Result := TCCRListItem(inherited ItemFocused);
end;

function TCCRListView.GetItems: TCCRListItems;
begin
  Result := inherited Items as TCCRListItems;
end;

function TCCRListView.GetNextItem(StartItem: TCCRCustomListItem;
  Direction: TSearchDirection; States: TItemStates): TCCRListItem;
begin
  Result := TCCRListItem(inherited GetNextItem(StartItem, Direction, States));
end;

function TCCRListView.GetSelected: TCCRListItem;
begin
  Result := TCCRListItem(inherited Selected);
end;

procedure TCCRListView.SetItemFocused(aValue: TCCRListItem);
begin
  inherited ItemFocused := aValue;
end;

procedure TCCRListView.SetSelected(aValue: TCCRListItem);
begin
  inherited Selected := aValue;
end;

end.
