{******************************************************************************}
{ Package:      Clinical Case Registries Custom Components                     }
{ Date Created: November 18, 2004                                              }
{ Site Name:    Hines OIFO                                                     }
{ Developers:   Sergey Gavrilov                                                }
{ Description:  Debug and troubleshooting classes.                             }
{ Note:                                                                        }
{******************************************************************************}

unit uROR_Debug;

{$I Components.inc}

interface

uses
  Classes, StrUtils, SysUtils, Windows, StdCtrls, Messages;

type

  {=============================== ICCRDebugTrace ==============================
    Overview:     Interface for debug trace objects.
    SeeAlso:      TCCRDebugLog; TCCRDebugTrace; TCCRStrDebugLog 
    Description:
      In order to implement automatic destruction, debug trace objects are
      referenced via the ICCRDebugTrace interface. Use the TCCRDebugTrace.Create
      to create a trace object and get its interface.
      <p>
      If a name is assigned to a trace object (usually a name of the function or
      procedure where the object is created), the object automatically writes
      entry (when created) and exit (when destroyed) messages to a current
      debug log.
      <p>
      You can also store the interface in a local variable and use the Error
      and Write methods to write additional messages to the log between entry
      and exit messages.
      <p>
      See the TCCRDebugTrace class for implemetation details.
  }
  ICCRDebugTrace = interface(IInterface)
    ['{C3A5E15C-2099-4916-801F-4D10339C81EE}']

    // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    procedure Error(const aFormat: String; const Args: array of const); overload;
    {
      Overview:     Writes an error message to a debug log.
      SeeAlso:      ICCRDebugTrace.ErrorPrefix; ICCRDebugTrace.Write
      Keywords:     Error,ICCRDebugTrace
      Description:
        There are 2 overloaded versions of the Error method.

        Use Error(const aFormat: String; const Args: array of const) to format
        a list of paramaters specified by the <i>Args</i> according to the
        <i>aFormat</i> format string and write the resulting error message to
        a debug log. All error messages are prepended with a string stored in
        the ErrorPrefix property.
        
        Use Error(const aMsg: String) to write an error message specified by
        the <i>aMsg</i> parameter to a debug log. All error messages are
        prepended with a string stored in the ErrorPrefix property.
    }
    procedure Error(const aMsg: String); overload;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the value of the EntryPrefix property.
      SeeAlso:      ICCRDebugTrace.EntryPrefix; ICCRDebugTrace.SetEntryPrefix
      Description:
        GetEntryPrefix is the read implementation of the EntryPrefix property.
    }
    function GetEntryPrefix: String;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the value of the ErrorPrefix property.
      SeeAlso:      ICCRDebugTrace.ErrorPrefix; ICCRDebugTrace.SetErrorPrefix
      Description:
        GetErrorPrefix is the read implementation of the ErrorPrefix property.
    }
    function GetErrorPrefix: String;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the value of the ExitPrefix property.
      SeeAlso:      ICCRDebugTrace.ExitPrefix; ICCRDebugTrace.SetExitPrefix
      Description:
        GetExitPrefix is the read implementation of the ExitPrefix property.
    }
    function GetExitPrefix: String;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the value of the Indent property.
      SeeAlso:      ICCRDebugTrace.Indent; ICCRDebugTrace.SetIndent
      Description:
        GetIndent is the read implementation of the Indent property.
    }
    function GetIndent: Integer;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Restores default property values.
      SeeAlso:      ICCRDebugTrace.EntryPrefix; ICCRDebugTrace.ErrorPrefix;
                    ICCRDebugTrace.ExitPrefix; ICCRDebugTrace.Indent;
                    TCCRDebugTrace.RestoreDefaults
      Keywords:     RestoreDefaults,ICCRDebugTrace
      Description:
        Use RestoreDefaults to restore initial values of the trace prefixes
        (EntryPrefix, ErrorPrefix, and ExitPrefix) and indentation level
        (Indent).
        <p>
        See the TCCRDebugTrace class for implemetation details.
    }
    procedure RestoreDefaults;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the EntryPrefix property.
      SeeAlso:      ICCRDebugTrace.EntryPrefix; ICCRDebugTrace.GetEntryPrefix
      Description:
        SetEntryPrefix is the write implementation of the EntryPrefix property.
    }
    procedure SetEntryPrefix(const aValue: String);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the ErrorPrefix property.
      SeeAlso:      ICCRDebugTrace.ErrorPrefix; ICCRDebugTrace.GetErrorPrefix
      Description:
        SetErrorPrefix is the write implementation of the ErrorPrefix property.
    }
    procedure SetErrorPrefix(const aValue: String);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the ExitPrefix property.
      SeeAlso:      ICCRDebugTrace.ExitPrefix; ICCRDebugTrace.GetExitPrefix
      Description:
        SetExitPrefix is the write implementation of the ExitPrefix property.
    }
    procedure SetExitPrefix(const aValue: String);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the Indent property.
      SeeAlso:      ICCRDebugTrace.Indent; ICCRDebugTrace.GetIndent
      Description:
        SetIndent is the write implementation of the Indent property.
    }
    procedure SetIndent(const aValue: Integer);

    // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    procedure Write(const aFormat: String; const Args: array of const); overload;
    {
      Overview:     Writes a message to a debug log.
      SeeAlso:      ICCRDebugTrace.Error
      Keywords:     Write,ICCRDebugTrace
      Description:
        There are 2 overloaded versions of the Write method.

        Use Write(const aMsg: String) to write a message specified by the
        <i>aMsg</i> parameter to a debug log.

        Use Write(const aFormat: String; const Args: array of const) to format
        a list of paramaters specified by the <i>Args</i> according to the
        <i>aFormat</i> format string and write the resulting message to a debug
        log.
    }
    procedure Write(const aMsg: String); overload;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     A prefix that indicates creation of a trace object.
      SeeAlso:      ICCRDebugTrace.ExitPrefix; TCCRDebugTrace.EntryPrefix
      Keywords:     EntryPrefix,ICCRDebugTrace
      Description:
        Use EntryPrefix to get or set the prefix of the entry message that is
        written to a debug log when a named trace object is created.
        <p>
        See the TCCRDebugTrace class for implemetation details.
    }
    property EntryPrefix: String  read GetEntryPrefix  write SetEntryPrefix;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     A prefix that marks an error message.
      SeeAlso:      ICCRDebugTrace.Error; TCCRDebugTrace.ErrorPrefix
      Keywords:     ErrorPrefix,ICCRDebugTrace
      Description:
        Use ErrorPrefix to get or set the prefix for the error messages.
        <p>
        See the TCCRDebugTrace class for implemetation details.
    }
    property ErrorPrefix: String  read GetErrorPrefix  write SetErrorPrefix;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     A prefix that indicates destruction of a trace object.
      SeeAlso:      ICCRDebugTrace.EntryPrefix; TCCRDebugTrace.ExitPrefix
      Keywords:     ExitPrefix,ICCRDebugTrace
      Description:
        Use ExitPrefix to get or set the prefix of the exit message that is
        written to a debug log when a named trace object is destroyed. 
        <p>
        See the TCCRDebugTrace class for implemetation details.
    }
    property ExitPrefix: String  read GetExitPrefix  write SetExitPrefix;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Number of spaces that are added before a message written
                    to a debug log.
      SeeAlso:      ICCRDebugTrace.Error; ICCRDebugTrace.Write;
                    TCCRDebugTrace.Indent
      Keywords:     Indent,ICCRDebugTrace
      Description:
        Indent specifies the number of spaces added before each message written
        to a debug log by the Error and Write methods.
        <p>
        See the TCCRDebugTrace class for implemetation details.
    }
    property Indent: Integer  read GetIndent  write SetIndent;

  end;

  {============================= TCCRCustomDebugLog ============================
    Overview:     Base class for a debug log.
    SeeAlso:      TCCRDebugLog; TCCRStrDebugLog
    Description:
      TCCRCustomDebugLog encapsulates a basic functionality of a debug log.
      Descendants of this classs are thread-safe.
      <p>
      Do not create instances of TCCRCustomDebugLog. To open a log, use a
      descendant of TCCRCustomDebugLog, such as TCCRDebugLog or TCCRStrDebugLog.
      No more than one debug log can exists inside an application.
  }
  TCCRCustomDebugLog = class(TComponent)
  private

    fIndent:   Integer;
    fMaxCount: Integer;
    fMaxSize:  Integer;

    procedure SetIndent(const aValue: Integer);
    procedure SetMaxCount(const aValue: Integer);
    procedure SetMaxSize(const aValue: Integer);

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Internal method that appends a string to a log.
      SeeAlso:      TCCRCustomDebugLog.Append
      Keywords:     AppendString,TCCRCustomDebugLog
      Description:
        Override AppendString in a descendant class to implement addition of a
        string to a debug log. <i>aText</i> specifies the string.
    }
    procedure AppendString(const aText: String); virtual; abstract;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Truncates a debug log.
      SeeAlso:      TCCRCustomDebugLog.MaxCount; TCCRCustomDebugLog.MaxSize
      Keywords:     Truncate,TCCRCustomDebugLog
      Description:
        This method truncates a debug log according to values of the MaxCount
        and/or MaxSize properties. As implemented in TCCRCustomDebugLog,
        Truncate does nothing.
    }
    procedure Truncate; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indentation level for new log lines.
      SeeAlso:      TCCRCustomDebugLog.AppendString
      Keywords:     Indent,TCCRCustomDebugLog
      Description:
        Use Indent to get or set indentation level (in characters) for new log
        lines.
    }
    property Indent: Integer  read fIndent  write SetIndent;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Maximum number of lines in a log.
      SeeAlso:      TCCRCustomDebugLog.MaxSize; TCCRCustomDebugLog.Truncate
      Keywords:     MaxCount,TCCRCustomDebugLog
      Description:
        Use MaxCount to get or set maximum number of lines that can be stored
        in a debug log. When number of lines in a log becomes greater than the
        value of this parameter, the oldest lines are automatically deleted
        from the beginning of the log.
        <p>
        <b>Note:</b> Not all descendant classes support this kind of truncation.
    }
    property MaxCount: Integer  read fMaxCount  write SetMaxCount;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Maximum log size (in bytes).
      SeeAlso:      TCCRCustomDebugLog.MaxCount; TCCRCustomDebugLog.Truncate
      Keywords:     MaxSize,TCCRCustomDebugLog
      Description:
        Use MaxSize to get or set maximum size of a debug log in bytes. When
        the log size becomes greater than the value of this parameter, the
        oldest lines are automatically deleted from the beginning of the log.
        <p>
        <b>Note:</b> Not all descendant classes support this kind of truncation.
    }
    property MaxSize: Integer  read fMaxSize  write SetMaxSize;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRCustomDebugLog.
      SeeAlso:      CCRDebugLog; TComponent.Owner; TObject.Create
      Keywords:     Create,TCCRCustomDebugLog
      Description:
        As implemented in TCCRCustomDebugLog, Create assigns a reference to a
        new instance to an internal variable. If another log object has been
        created before, it is destroyed.
    }
    constructor Create(aDummyParameter: TComponent = nil); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRCustomDebugLog.
      SeeAlso:      TObject.Free; TComponent.Destroy
      Keywords:     Destroy,TCCRCustomDebugLog
      Description:
        Do not call Destroy directly. Instead, call Free, which checks that the
        TCCRCustomDebugLog reference is not nil before calling Destroy.
    }
    destructor Destroy; override;

    // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    procedure Append(const aStr: String); overload;
    procedure Append(const aFormat: String; const Args: array of const); overload;
    {
      Overview:     Appends content of a string list to a debug log.
      SeeAlso:      TCCRCustomDebugLog.AppendString
      Keywords:     Append,TCCRCustomDebugLog
      Description:
        There are 3 overloaded version of the Append procedure.

        Use Append(const aFormat: String; const Args: array of const) to format
        a list of paramaters specified by the <i>Args</i> according to the
        <i>aFormat</i> format string and add the result to the end of a debug
        log.

        Use Append(const aStr: String) to add a new line specified by the
        <i>aStr</i> parameter to the end of a debug log.

        Use Append(const aText: TStringList) to add items of the <i>aText</i>
        string list to the end of a debug log.
    }
    procedure Append(const aText: TStringList); overload;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Starts a group of updates to a debug log.
      SeeAlso:      TCCRCustomDebugLog.EndUpdate; TCCRCustomDebugLog.Lock
      Keywords:     BeginUpdate,TCCRCustomDebugLog
      Description:
        Call BeginUpdate before adding multiple items to a log. In addition to
        improving the peroformance, this method also protects the log from
        concurrent updates from multiple threads.
        <p>
        Descendant classes that override BeginUpdate should always call the
        inherited method right after entering the procedure.
    }
    procedure BeginUpdate; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Clears a log.
      Keywords:     Clear,TCCRCustomDebugLog
      Description:
        Use clear to delete all entries from a log. As implemented in
        TCCRCustomDebugLog, Clear does nothing.
        <p>
        <b>Note:</b> Not all descendant classes support this functionality.
    }
    procedure Clear; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Ends a group of updates to a debug log.
      SeeAlso:      TCCRCustomDebugLog.BeginUpdate; TCCRCustomDebugLog.Unlock
      Keywords:     EndUpdate,TCCRCustomDebugLog
      Description:
        Call EndUpdate after adding multiple items to a log. In addition to
        improving the peroformance, this method also re-enables log updates
        from from other threads.
        <p>
        Descendant classes that override EndUpdate should always call the
        inherited method right before returning from the procedure.
    }
    procedure EndUpdate; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Prevents other threads from modifying a log.
      SeeAlso:      TCCRCustomDebugLog.Unlock
      Keywords:     Lock,TCCRCustomDebugLog
      Description:
        Place any code that modifies a log object between Lock and Unlock in
        multi-threaded applications.
    }
    class procedure Lock;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Saves a debug log to a file.
      Keywords:     SaveToFile,TCCRCustomDebugLog
      Description:
        Use SaveToFile to write content of a log to a file. <i>aFileName</i>
        specifies a full file name.
        <p>
        <b>Note:</b> Not all descendant classes support this functionality.
    }
    procedure SaveToFile(const aFileName: String); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Allows unpdating a log in other threads.
      SeeAlso:      TCCRCustomDebugLog.Lock
      Keywords:     Unlock,TCCRCustomDebugLog
      Description:
        Place any code that modifies a log object between Lock and Unlock in
        multi-threaded applications.
    }
    class procedure Unlock;

  end;

  {================================ TCCRDebugLog ===============================
    Overview:     Debug log in a debugger window.
    SeeAlso:      TCCRMemoDebugLog; TCCRStrDebugLog
    Description:
      TCCRDebugLog implements a debug log that is shown in a debugger window
      and in the Event Log window of the Delphi IDE. It can also be monitored
      using applications like the freeware DebugView developed by
      <a href="http://www.sysinternals.com">Sysinternals</a>.
  }
  TCCRDebugLog = class(TCCRCustomDebugLog)
  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Internal method that appends a string to a log.
      SeeAlso:      TCCRCustomDebugLog.AppendString
      Keywords:     AppendString,TCCRDebugLog
      Description:
        AppendString uses the OutputDebugString procedure (Win32 API) to send a
        string specified by the <i>aText</i> parameter to the debugger for the
        current application.
    }
    procedure AppendString(const aText: String); override;

  end;

  {=============================== TCCRDebugTrace ==============================
    Overview:     Thread-safe implemetation of a trace object.
    SeeAlso:      TCCRDebugLog; TCCRDebugTrace.Create; TCCRStrDebugLog
    Description:
      TCCRDebugTrace is a debug trace object. Usually an instance of this object
      is created in the very beginning of a function/procedure. The object
      writes an entry message to a current debug log when it is created, and an
      exit message when it is automatically destroyed right before exiting from
      the function/procedure.
      <p>
      TCCRDebugTrace implements the ICCRDebugTrace interface. Use the Create
      class method to create a trace object and get its interface.
  }
  TCCRDebugTrace = class(TInterfacedObject,ICCRDebugTrace)
  private

    fEntryPrefix:   String;
    fErrorPrefix:   String;
    fExitPrefix:    String;
    fIndent:        Integer;
    fName:          String;
    fParent:        TCCRDebugTrace;

    function  GetEntryPrefix: String;
    function  GetErrorPrefix: String;
    function  GetExitPrefix: String;
    function  GetIndent: Integer;
    procedure SetEntryPrefix(const aValue: String);
    procedure SetErrorPrefix(const aValue: String);
    procedure SetExitPrefix(const aValue: String);
    procedure SetIndent(const aValue: Integer);

  protected

    // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    procedure Error(const aFormat: String; const Args: array of const); overload;
    {
      Overview:     Implementations of Error methods.
      SeeAlso:      ICCRDebugTrace.Error
      Keywords:     Error,TCCRDebugTrace
      Description:
        See the ICCRDebugTrace interface for details.
    }
    procedure Error(const aMsg: String); overload;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Restores default property values.
      SeeAlso:      TCCRDebugTrace.EntryPrefix; TCCRDebugTrace.ErrorPrefix;
                    TCCRDebugTrace.ExitPrefix; TCCRDebugTrace.Indent;
                    ICCRDebugTrace.RestoreDefaults
      Keywords:     RestoreDefaults,TCCRDebugTrace
      Description:
        RestoreDefaults restores initial values of the trace prefixes
        (EntryPrefix, ErrorPrefix, and ExitPrefix) and indentation level
        (Indent). If a trace object has a parent, corresponding parent's
        values are assigned to object properties. Otherwise, default values
        are assigned as follow:
        <p>
        <p>  EntryPrefix := '>>> ';
        <p>  ErrorPrefix := '! ! !  ';
        <p>  ExitPrefix := '<<< ';
        <p>  Indent := 0;
    }
    procedure RestoreDefaults; virtual;

    // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    procedure Write(const aFormat: String; const Args: array of const); overload;
    {
      Overview:     Implementations of Write methods.
      SeeAlso:      ICCRDebugTrace.Write
      Keywords:     Write,TCCRDebugTrace
      Description:
        See the ICCRDebugTrace interface for details.
    }
    procedure Write(const aMsg: String); overload;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     A prefix that indicates creation of a trace object.
      SeeAlso:      ICCRDebugTrace.EntryPrefix
      Keywords:     EntryPrefix,TCCRDebugTrace
      Description:
        EntryPrefix stores a prefix of the entry message that is written to a
        debug log when a named trace object is created. This message is
        formatted as follow:
        <p>
          <Indent><EntryPrefix><Name>
        <p>
        where <Indent> is amount of space characters specified by the Indent
        property, <EntryPrefix> is the values of the EntryPrefix property, and
        <Name> is the name assigned to the trace object.
        <p>
        The property value is used as default prefix for all subsequent trace
        objects until one of them changes it again. The change does not affect
        trace objects that have been created earlier.
    }
    property EntryPrefix: String  read GetEntryPrefix  write SetEntryPrefix;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     A prefix that marks an error message.
      SeeAlso:      ICCRDebugTrace.ErrorPrefix
      Keywords:     ErrorPrefix,TCCRDebugTrace
      Description:
        Use ErrorPrefix to get or set the prefix for the error messages. The
        value is used as default prefix for all subsequent calls of the Error
        method. The change does not affect trace objects that have been created
        earlier.
    }
    property ErrorPrefix: String  read GetErrorPrefix  write SetErrorPrefix;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     A prefix that indicates destruction of a trace object.
      SeeAlso:      ICCRDebugTrace.ExitPrefix
      Keywords:     ExitPrefix,TCCRDebugTrace
      Description:
        Use ExitPrefix to get or set the prefix of the exit message that is
        written to a debug log when a named trace object is destroyed. This
        message is formatted as follow:
        <p>
          <Indent><ExitPrefix><Name>
        <p>
        where <Indent> is amount of space characters specified by the Indent
        property, <ExitPrefix> is the values of the ExitPrefix property, and
        <Name> is the name assigned to the trace object.
        <p>
        A new property value will be used as default prefix for all subsequent
        trace objects until one of them changes it again.  The change does not
        affect trace objects that have been created earlier.
    }
    property ExitPrefix: String  read GetExitPrefix  write SetExitPrefix;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Number of spaces that are added before a message written
                    to a debug log.
      SeeAlso:      ICCRDebugTrace.Eror; ICCRDebugTrace.Indent;
                    ICCRDebugTrace.Write
      Keywords:     Indent,TCCRDebugTrace
      Description:
        Indent specifies the number of spaces added before each message written
        to a debug log by the Error and Write methods.
        <p>
        When a new trace object is created, it inherits property values from
        its parent (the previous trace object). If a name parameter was
        specified during creation of a trtace object, the entry message is
        written to a log using the parent indentation level. After that, the
        Indent property is automatically incremented by 3. Initial value of the
        Indent property is automatically restored before writing the exit
        message.
        <p>
        If no name is assigned to a trace object, then the object does not
        change the indentation level inherited from its parent.
    }
    property Indent: Integer  read GetIndent  write SetIndent;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRDebugTrace.
      SeeAlso:      TObject.Destroy
      Keywords:     Destroy,TCCRDebugTrace
      Description:
        Do not call Destroy directly.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates a trace object and returns its interface.
      Keywords:     Create,TCCRDebugTrace
      Description:
        Use this method to create a trace object and get its interface. A class
        function is used instead of a constructor in order to create an
        interface reference instead of an object reference. This guarantees that
        the object will be automatically destroyed when the reference goes out
        of scope (in the end of the block where the object was created). Use
        <i>aName</i> to specify a name of a trace object (usually a name of the
        function or procedure where the object is created).
        <p>
        If a trace object has a name, it automatically writes an entry message
        when it is created and an exit message when destroyed. It also
        increments the indentation level by 3.
        <p>
        Unnamed trace objects do not display entry/exit messages and do not
        change the indentation level. For example, you can use such object in a
        thread initialization section to modify the default prefixes and/or
        indentation level (to distinguish log entries from different threads).
        <p>
        It is recommended to create no more than one trace object per
        function/procedure (in the very beginning of it).
    }
    class function Create(const aName: String = ''): ICCRDebugTrace;

  end;

  {============================= TCCRDummyDebugLog =============================
    Overview:     Pseudo debug log.
    SeeAlso:      CCRDebugLog
    Description:
      TCCRDummyDebugLog implements a debug log that discards all entries added
      to it. An instance of this class is returned by the CCRDebugLog if no
      other log object has been explicitly created.
  }
  TCCRDummyDebugLog = class(TCCRCustomDebugLog)
  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Internal method that appends a string to a log.
      SeeAlso:      TCCRCustomDebugLog.AppendString
      Keywords:     AppendString,TCCRDummyDebugLog
      Description:
        As implementned in TCCRDummyDebugLog, AppendString does nothing.
    }
    procedure AppendString(const aText: String); override;

  end;

  {============================== TCCRStrDebugLog ==============================
    Overview:     String list implementation of a debug log.
    SeeAlso:      TCCRDebugLog; TCCRMemoDebugLog
    Description:
      TCCRStrDebugLog implements a debug log that is stored in an internal
      string list. Maximum size of the log can be limited by the MaxCount
      property.
  }
  TCCRStrDebugLog = class(TCCRCustomDebugLog)
  private

    fStrLst: TStrings;

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Internal method that appends a string to a log.
      SeeAlso:      TCCRCustomDebugLog.AppendString
      Keywords:     AppendString,TCCRStrDebugLog
      Description:
        AppendString adds a string specified by the <i>aText</i> parameter to
        the end of the internal string list and calls Truncate to check its
        size.
    }
    procedure AppendString(const aText: String); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specialized AssignTo.
      SeeAlso:      TPersistent.AssignTo
      Keywords:     AssignTo,TCCRStrDebugLog
      Description:
        Do not call protected AssignTo method. The Assign method of a persistent
        object calls AssignTo if it is passed the TCCRStrDebugLog as a Source
        and the persistent object does not know how to copy the properties of
        the object. The <i>aDest</i> parameter is the persistent object that
        should have its properties copied from the debug log.
    }
    procedure AssignTo(Dest: TPersistent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Truncates a debug log.
      SeeAlso:      TCCRCustomDebugLog.MaxCount; TCCRCustomDebugLog.Truncate
      Keywords:     Truncate,TCCRStrDebugLog
      Description:
        This method deletes the oldest entries from the beginning of the log
        when the number of entries becomes greater than the value of the
        MaxCount property.
    }
    procedure Truncate; override;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRStrDebugLog.
      SeeAlso:      TCCRCustomDebugLog.Create; TComponent.Owner
      Keywords:     Create,TCCRStrDebugLog
      Description:
        In addition to the inherited functionality, this method creates an
        internal string list object that stores the log entries.
    }
    constructor Create(aDummyParameter: TComponent = nil); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRStrDebugLog.
      SeeAlso:      TCCRCustomDebugLog.Destroy; TObject.Free
      Keywords:     Destroy,TCCRStrDebugLog
      Description:
        Do not call Destroy directly. Instead, call Free, which checks that the
        TCCRStrDebugLog reference is not nil before calling Destroy. In addition
        to the inherited functionality, this method destroys an internal string
        list object that stores the log entries.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Starts a group of updates to a debug log.
      SeeAlso:      TCCRCustomDebugLog.BeginUpdate; TCCRStrDebugLog.EndUpdate
      Keywords:     BeginUpdate,TCCRStrDebugLog
      Description:
        Call BeginUpdate before adding multiple items to a log. In addition to
        the inherited functionality, this method calls BeginUpdate of the
        internal string list that stores the log entries.
        <p>
        Descendant classes that override BeginUpdate should always call the
        inherited method right after entering the procedure.
    }
    procedure BeginUpdate; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Clears a log.
      Keywords:     Clear,TCCRStrDebugLog
      Description:
        Use Clear to delete all entries from a log. As implemented in
        TCCRStrDebugLog, Clear deletes all items from the internal string
        list.
    }
    procedure Clear; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Ends a group of updates to a debug log.
      SeeAlso:      TCCRCustomDebugLog.EndUpdate; TCCRStrDebugLog.BeginUpdate
      Keywords:     EndUpdate,TCCRStrDebugLog
      Description:
        Call EndUpdate after adding multiple items to a log. In addition to
        the inherited functionality, this method calls EndUpdate of the internal
        string list that stores the log entries.
        <p>
        Descendant classes that override EndUpdate should always call the
        inherited method.
    }
    procedure EndUpdate; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Saves a debug log to a file.
      Keywords:     SaveToFile,TCCRStrDebugLog
      Description:
        Use SaveToFile to write content of a log to a file. <i>aFileName</i>
        specifies a full file name. As implemented in TCCRStrDebugLog, this
        method just calls the SaveToFile method of the internal string list
        that stores log entries.
    }
    procedure SaveToFile(const aFileName: String); override;

  published
  
    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property MaxSize;

  end;

  {============================== TCCRMemoDebugLog =============================
    Overview:     Visual implementation of a debug log (TMemo).
    SeeAlso:      TCCRDebugLog; TCCRStrDebugLog
    Description:
      TCCRMemoDebugLog implements a debug log that is stored in a TMemo
      instance referenced by the Memo property. Maximum size of the log can be
      limited by the MaxCount property.
  }
  TCCRMemoDebugLog = class(TCCRStrDebugLog)
  private

    fMemo: TMemo;

    procedure SetMemo(aValue: TMemo);

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Internal method that appends a string to a log.
      SeeAlso:      TCCRStrDebugLog.AppendString
      Keywords:     AppendString,TCCRMemoDebugLog
      Description:
        AppendString adds a string specified by the <i>aText</i> parameter to
        the end of the log. It calls the inherited method and scrolls the memo
        control to the end of the log.
    }
    procedure AppendString(const aText: String); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Responds to notifications that components are being created
                    or destroyed.
      SeeAlso:      TComponent.Notification
      Keywords:     Notification,TCCRMemoDebugLog
      Description:
        Do not call the Notification method in an application.  Notification is
        called automatically when the component specified by <i>aComponent</i>
        is about to be inserted or removed, as specified by <i>Operation</i>.
        <p>
        TCCRMemoDebugLog overrides this method in order to update its Memo
        property when the control it refers to is destroyed.
    }
    procedure Notification(aComponent: TComponent;
      Operation: TOperation); override;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRMemoDebugLog.
      SeeAlso:      TCCRStrDebugLog.Create; TComponent.Owner
      Keywords:     Create,TCCRMemoDebugLog
      Description:
        In addition to the inherited functionality, this method destroys the
        internal string list (string list from the linked TMemo control will
        be used instead) and assigns nil to the corresponding field.
    }
    constructor Create(aDummyParameter: TComponent = nil); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRMemoDebugLog.
      SeeAlso:      TCCRStrDebugLog.Destroy; TObject.Free
      Keywords:     Destroy,TCCRMemoDebugLog
      Description:
        Do not call Destroy directly. Instead, call Free, which checks that the
        TCCRMemoDebugLog reference is not nil before calling Destroy. This
        method assigns nil to the field that references the internal string list
        before calling the inherited destructor. This is necessary becuase this
        field actually references the internal string list of the linked memo
        control.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Clears a log.
      SeeAlso:      TCCRStrDebugLog.Clear
      Keywords:     Clear,TCCRMemoDebugLog
      Description:
        Use Clear to delete all entries from a log. As implemented in
        TCCRMemoDebugLog, this method calls the Clear method of the linked memo
        control.
    }
    procedure Clear; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Ends a group of updates to a debug log.
      SeeAlso:      TCCRStrDebugLog.EndUpdate; TCCRStrDebugLog.BeginUpdate
      Keywords:     EndUpdate,TCCRMemoDebugLog
      Description:
        Call EndUpdate after adding multiple items to a log. In addition to
        the inherited functionality, this method scrolls the linked memo control
        to the end of its content.
        <p>
        Descendant classes that override EndUpdate should always call the
        inherited method.
    }
    procedure EndUpdate; override;

  published

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Reference to a linked TMemo instance.
      Keywords:     Memo,TCCRMemoDebugLog
      Description:
        Assign an instance of the TMemo control to the Memo property before
        using the debug log. The log will be shown in that control. If the value
        of this property is nil, all data added to the log is discarded.
    }
    property Memo: TMemo  read fMemo  write SetMemo;

  end;


{============================= Functions/Procedures ===========================}

  {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    Overview:     Returns a current debug log instance.
    SeeAlso:      TCCRDebugLog; TCCRStrDebugLog
    Description:
      Use CCRDebugLog to access a current debug log instance. If a log has not
      been created before calling this function for the first time, it returns
      a "dummy" log that discards all data added to it.
  }
  function CCRDebugLog: TCCRCustomDebugLog;

///////////////////////////////// Implementation \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

implementation

var
  _DebugLog: TCCRCustomDebugLog = nil;
  _Lock: TRTLCriticalSection;

threadvar
  _DebugTrace: TCCRDebugTrace;
  _DebugTraceId: DWORD;

////////////////////////////// TCCRCustomDebugLog \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRCustomDebugLog.Create(aDummyParameter: TComponent);
begin
  inherited Create(nil);
  if Assigned(_DebugLog) then
    _DebugLog.Free;
  _DebugLog := Self;
end;

destructor TCCRCustomDebugLog.Destroy;
begin
  if _DebugLog = Self then
    _DebugLog := nil;
  inherited;
end;

procedure TCCRCustomDebugLog.Append(const aStr: String);
begin
  BeginUpdate;
  try
    AppendString(aStr);
    Truncate;
  finally
    EndUpdate;
  end;
end;

procedure TCCRCustomDebugLog.Append(const aFormat: String;
  const Args: array of const);
begin
  BeginUpdate;
  try
    AppendString(Format(aFormat, Args));
    Truncate;
  finally
    EndUpdate;
  end;
end;

procedure TCCRCustomDebugLog.Append(const aText: TStringList);
var
  i, n: Integer;
begin
  Assert(aText <> nil, 'Invalid string list reference.');
  BeginUpdate;
  try
    n := aText.Count - 1;
    for i:=0  to n do
      AppendString(aText[i]);
    Truncate;
  finally
    EndUpdate;
  end;
end;

procedure TCCRCustomDebugLog.BeginUpdate;
begin
  Lock;
end;

procedure TCCRCustomDebugLog.Clear;
begin
end;

procedure TCCRCustomDebugLog.EndUpdate;
begin
  Unlock;
end;

class procedure TCCRCustomDebugLog.Lock;
begin
  EnterCriticalSection(_Lock);
end;

procedure TCCRCustomDebugLog.SaveToFile(const aFileName: String);
begin
end;

procedure TCCRCustomDebugLog.SetIndent(const aValue: Integer);
begin
  Lock;
  try
    fIndent := aValue;
  finally
    Unlock;
  end;
end;

procedure TCCRCustomDebugLog.SetMaxCount(const aValue: Integer);
begin
  Lock;
  try
    if aValue <> fMaxCount then
      begin
        fMaxCount := aValue;
        Truncate;
      end;
  finally
    Unlock;
  end;
end;

procedure TCCRCustomDebugLog.SetMaxSize(const aValue: Integer);
begin
  Lock;
  try
    if aValue <> fMaxSize then
      begin
        fMaxSize := aValue;
        Truncate;
      end;
  finally
    Unlock;
  end;
end;

procedure TCCRCustomDebugLog.Truncate;
begin
end;

class procedure TCCRCustomDebugLog.Unlock;
begin
  LeaveCriticalSection(_Lock);
end;

///////////////////////////////// TCCRDebugLog \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TCCRDebugLog.AppendString(const aText: String);
begin
  OutputDebugString(PChar(DupeString(' ', Indent) + aText));
end;

//////////////////////////////// TCCRDebugTrace \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

destructor TCCRDebugTrace.Destroy;
begin
  if fName <> '' then
    begin
      RestoreDefaults;
      Write(ExitPrefix + fName);
    end;
  _DebugTrace := fParent;
  inherited;
end;

class function TCCRDebugTrace.Create(const aName: String): ICCRDebugTrace;
var
  dt: TCCRDebugTrace;
begin
  if _DebugTraceId <> GetCurrentThreadId then
    begin
      _DebugTrace := nil;
      _DebugTraceId := GetCurrentThreadId;
    end;

  dt := inherited Create;
  with dt do
    begin
      fParent := _DebugTrace;
      RestoreDefaults;

      fName := aName;
      if fName <> '' then
        begin
          Write(EntryPrefix + fName);
          Indent := Indent + 3;
        end;
    end;

  _DebugTrace := dt;
  Result := dt;
end;

procedure TCCRDebugTrace.Error(const aFormat: String; const Args: array of const);
begin
  Write(ErrorPrefix + Format(aFormat, Args));
end;

procedure TCCRDebugTrace.Error(const aMsg: String);
begin
  Write(ErrorPrefix + aMsg);
end;

function TCCRDebugTrace.GetEntryPrefix: String;
begin
  Result := fEntryPrefix;
end;

function TCCRDebugTrace.GetErrorPrefix: String;
begin
  Result := fErrorPrefix;
end;

function TCCRDebugTrace.GetExitPrefix: String;
begin
  Result := fExitPrefix;
end;

function TCCRDebugTrace.GetIndent: Integer;
begin
  Result := fIndent;
end;

procedure TCCRDebugTrace.RestoreDefaults;
begin
  if Assigned(fParent) then
    begin
      fEntryPrefix := fParent.fEntryPrefix;
      fErrorPrefix := fParent.fErrorPrefix;
      fExitPrefix  := fParent.fExitPrefix;
      fIndent      := fParent.fIndent;
    end
  else
    begin
      fEntryPrefix := '>>> ';
      fErrorPrefix := '! ! !  ';
      fExitPrefix  := '<<< ';
      fIndent      := 0;
    end;
end;

procedure TCCRDebugTrace.SetEntryPrefix(const aValue: String);
begin
  fEntryPrefix := aValue;
end;

procedure TCCRDebugTrace.SetErrorPrefix(const aValue: String);
begin
  fErrorPrefix := aValue;
end;

procedure TCCRDebugTrace.SetExitPrefix(const aValue: String);
begin
  fExitPrefix := aValue;
end;

procedure TCCRDebugTrace.SetIndent(const aValue: Integer);
begin
  if aValue >= 0 then
    begin
      fIndent := aValue;
      CCRDebugLog.Indent := aValue;
    end;
end;

procedure TCCRDebugTrace.Write(const aFormat: String; const Args: array of const);
begin
  CCRDebugLog.Indent := Indent;
  CCRDebugLog.Append(aFormat, Args);
end;

procedure TCCRDebugTrace.Write(const aMsg: String);
begin
  CCRDebugLog.Indent := Indent;
  CCRDebugLog.Append(aMsg);
end;

/////////////////////////////// TCCRDummyDebugLog \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TCCRDummyDebugLog.AppendString(const aText: String);
begin
end;

/////////////////////////////// TCCRMemoDebugLog \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRMemoDebugLog.Create(aDummyParameter: TComponent);
begin
  inherited;
  FreeAndNil(fStrLst);
end;

destructor TCCRMemoDebugLog.Destroy;
begin
  fStrLst := nil;
  fMemo   := nil;
  inherited;
end;

procedure TCCRMemoDebugLog.AppendString(const aText: String);
begin
  if Assigned(Memo) then
    begin
      inherited;
      Memo.Perform(EM_SCROLLCARET, 0, 0);
    end;
end;

procedure TCCRMemoDebugLog.Clear;
begin
  if Assigned(Memo) then
    begin
      Lock;
      try
        Memo.Clear;
      finally
        Unlock;
      end;
    end;
end;

procedure TCCRMemoDebugLog.EndUpdate;
begin
  inherited;
  if Assigned(Memo) then
    Memo.Perform(EM_SCROLLCARET, 0, 0);
end;

procedure TCCRMemoDebugLog.Notification(aComponent: TComponent;
  Operation: TOperation);
begin
  if Operation = opRemove then
    begin
      if aComponent = Memo then
        Memo := nil;
    end;
  inherited;
end;

procedure TCCRMemoDebugLog.SetMemo(aValue: TMemo);
begin
  if aValue <> fMemo then
    begin
      if Assigned(fMemo) then
        fMemo.RemoveFreeNotification(Self);

      fMemo := aValue;

      if Assigned(fMemo) then
        begin
          fMemo.FreeNotification(Self);
          fStrLst := fMemo.Lines;
        end
      else
        fStrLst := nil;
    end;
end;

//////////////////////////////// TCCRStrDebugLog \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRStrDebugLog.Create(aDummyParameter: TComponent);
begin
  inherited;
  fStrLst := TStringList.Create;
end;

destructor TCCRStrDebugLog.Destroy;
begin
  FreeAndNil(fStrLst);
  inherited;
end;

procedure TCCRStrDebugLog.AppendString(const aText: String);
begin
  if Assigned(fStrLst) then
    fStrLst.Add(DupeString(' ', Indent) + aText);
end;

procedure TCCRStrDebugLog.AssignTo(Dest: TPersistent);
begin
  if (Dest is TStrings) then
    if Assigned(fStrLst) then
      TStrings(Dest).Assign(fStrLst)
    else
      TStrings(Dest).Clear;
end;

procedure TCCRStrDebugLog.BeginUpdate;
begin
  inherited;
  if Assigned(fStrLst) then
    fStrLst.BeginUpdate;
end;

procedure TCCRStrDebugLog.Clear;
begin
  if Assigned(fStrLst) then
    begin
      Lock;
      try
        fStrLst.Clear;
      finally
        Unlock;
      end;
    end;
end;

procedure TCCRStrDebugLog.EndUpdate;
begin
  if Assigned(fStrLst) then
    fStrLst.EndUpdate;
  inherited;
end;

procedure TCCRStrDebugLog.SaveToFile(const aFileName: String);
begin
  if Assigned(fStrLst) then
    begin
      Lock;
      try
        fStrLst.SaveToFile(aFileName);
      finally
        Unlock;
      end;
    end;
end;

procedure TCCRStrDebugLog.Truncate;
var
  i, n: Integer;
begin
  if (MaxCount > 0) and (fStrLst.Count > MaxCount) and Assigned(fStrLst) then
    begin
      fStrLst.BeginUpdate;
      try
        n := fStrLst.Count - MaxCount - 1;
        for i:=0 to n do
          fStrLst.Delete(i);
      finally
        fStrLst.EndUpdate;
      end;
    end;
end;

///////////////////////////// Functions/Procedures \\\\\\\\\\\\\\\\\\\\\\\\\\\\\

function CCRDebugLog: TCCRCustomDebugLog;
begin
  if not Assigned(_DebugLog) then
    TCCRDummyDebugLog.Create;
  Result := _DebugLog;
end;

////////////////////////// Initialization/Finalization \\\\\\\\\\\\\\\\\\\\\\\\\

initialization
  InitializeCriticalSection(_Lock);

finalization
  DeleteCriticalSection(_Lock);

end.
