{******************************************************************************}
{ Package:      Clinical Case Registries Custom Components                     }
{ Date Created: November 18, 2004                                              }
{ Site Name:    Hines OIFO                                                     }
{ Developers:   Sergey Gavrilov                                                }
{ Description:  Miscellaneous custom controls.                                 }
{ Note:                                                                        }
{******************************************************************************}

unit uROR_CustomControls;

{$I Components.inc}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, ComCtrls, Buttons, ActnList;

type
  TCCRCBGroup = class;

  {================================= TCBHAlign =================================
    Overview:     Horizontal alignment of a check-box group.
    SeeAlso:      TCCRCBGroup.CBHAlign; TCCRCBGroup.CBHMargin
  }
  TCBHAlign = (

    cbaLeft,    { Align group of check-boxes to the left side of the panel. }

    cbaCenter,  { Center group of check-boxes inside the panel.}

    cbaRight    { Align group of check-boxes to the right side of the panel. }

  );

  {================================= TCBVAlign =================================
    Overview:     Vertical alignment of a check-box group.
    SeeAlso:      TCCRCBGroup.CBVAlign; TCCRCBGroup.CBVMargin
  }
  TCBVAlign = (

    cbaTop,     { Align group of check-boxes to the top side of the panel. }

    cbaMiddle,  { Center group of check-boxes inside the panel.}

    cbaBottom   { Align group of check-boxes to the bottom side of the panel. }

  );

  {================================= TCCRCBDef =================================
    Overview:     Definition of a check-box included in a group.
    SeeAlso:      TCCRCBDefs; TCCRCBGroup
    Description:
      TCCRCBDef represents a check-box that is a part of the TCCRCBGroup
      check-box group.
  }
  TCCRCBDef = class(TCollectionItem)
  private

    fCaption: String;
    fCode:    String;
    fControl: TCheckBox;

    procedure SetCaption(const aValue: String);
    procedure SetCode(const aValue: String);

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the name of the collection item as it appears in
                    the collection editor.
      SeeAlso:      TCollectionItem.DisplayName; TCollectionItem.GetDisplayName
      Keywords:     GetDisplayName,TCCRCBDef
      Description:
        GetDisplayName is the protected read implementation of the DisplayName
        property. TCCRCBDef overrides this method to return the value of
        Caption property if it is not empty. 
    }
    function GetDisplayName: String; override;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes a TCCRCBDef instance.
      SeeAlso:      TCCRCBDefs.Add
      Keywords:     Create,TCCRCBDef
      Description:
        Do not call Create to instantiate a check-box definition. Instead, call
        the Add method of the TCCRCBDefs collection to which the item should
        belong. <i>aCollection</i> indicates a TCCRCBDefs instance to which the
        new item belongs.
    }
    constructor Create(aCollection: TCollection); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys the TCCRCBDef instance and frees its memory.
      SeeAlso:      TCollection.Clear; TCollection.Delete; TObject.Free
      Keywords:     Destroy,TCCRCBDef
      Description:
        Destroy is called indirectly by collections Clear or Delete method.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     References a check-box that corresponds to a definition.
      Keywords:     Control,TCCRCBDef
      Description:
        Control associates a check-box definition with a check-box at run time.
    }
    property Control: TCheckBox  read fControl  write fControl;

  published

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Caption for a check-box.
      SeeAlso:      TControl.Caption
      Keywords:     Caption,TCCRCBDef
      Description:
        Use Caption to get or set a caption of a check-box. It will be shown
        to the right of the control.
    }
    property Caption: String  read fCaption  write SetCaption;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Code associated with a check-box.
      SeeAlso:      TCCRCBGroup.Code
      Keywords:     Code,TCCRCBDef
      Description:
        Use Code to get or set a code associated with the check-box. When the
        check-box is checked, this code is assigned to the Code property of
        the check-box group. 
    }
    property Code: String  read fCode  write SetCode;

  end;

  {================================= TCCRCBDefs ================================
    Overview:     Collection of check-box definitions.
    SeeAlso:      TCCRCBDef; TCCRCBGroup
    Description:
      TCCRCBDefs represents a collection of check-box definitions in a check-box
      group. At design time, use the check-box group controls CheckBoxes editor
      to add, remove, or modify definitions.
  }
  TCCRCBDefs = class(TOwnedCollection)
  private

    function GetCBDef(anIndex: Integer): TCCRCBDef;

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Signals the owner of the collection that the latter was
                    changed.
      SeeAlso:      TCCRCBDef; TCCRCBGroup.UpdateCheckBoxes
      Keywords:     UpdateOwner,TCCRCBDefs
      Description:
        As implemented in the TCCRCBDefs, UpdateOwner checks if the collection is
        owned by a check-box group, then it calls the owner's UpdateCheckBoxes
        method.
    }
    procedure UpdateOwner; virtual;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes a TCCRCBDefs instance.
      SeeAlso:      TCCRCBDefs.Add
      Keywords:     Create,TCCRCBDefs
      Description:
        The <i>anOwner</i> parameter is the check-box group that owns the
        check-box definitions.
    }
    constructor Create(anOwner: TCCRCBGroup);

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates a new TCCRCBDef instance and adds it to the CBDefs
                    array.
      SeeAlso:      TCCRCBDefs.CBDefs; TCollection.Delete; TCollection.Insert
      Keywords:     Add,TCCRCBDefs
      Description:
        Call Add to create a check-box definition in the collection. The new
        definition is placed at the end of the CBDefs array. Add returns the
        new check-box definition.
    }
    function Add: TCCRCBDef;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Updates the collection to reflect changes to its items.
      SeeAlso:      TCCRCBDefs.UpdateOwner; TCollection.Changed;
                    TCollection.Update
      Keywords:     Update,TCCRCBDefs
      Description:
        Override Update in a descendant class to make any necessary changes
        when the items in the collection change. This method is called
        automatically when an update is complete. As implemented in TCCRCBDefs,
        Update updates check-boxes according to their definitions by calling
        UpdateOwner.
        <p>
        <i>Item</i> identifies the check-box definition that changed. If the
        <i>Item</i> parameter is nil, then the change affects more than one
        item in the collection. <i>Item</i> can be safely typecasted to
        TCCRCBDef.
    }
    procedure Update(Item: TCollectionItem); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Lists the check-box definitions in the collection.
      SeeAlso:      TCCRCBDef; TCollection.Count 
      Description:
        Use CBDefs to access individual check-box definitions in the collection.
        The value of the <i>Index</i> parameter corresponds to the Index
        property of TCCRCBDef. It represents the position of the definition in
        the collection.
    }
    property CBDefs[Index: Integer]: TCCRCBDef  read GetCBDef;  default;

  end;

  {================================ TCCRCBGroup ================================
    Overview:     Panel with a group of check-boxes.
    SeeAlso:      TCCRCBDef; TCCRCBDefs; TCheckBox; TRadioButton
    Description:
      TCCRCBGroup represents a panel with a row of check-boxes. It is a hybrid
      of a radio-button group and a set of check-boxes. No more than one
      check-box can be checked.
  }
  TCCRCBGroup = class(TCustomPanel)
  private

    fCBHAlign:   TCBHAlign;
    fCBHMargin:  Integer;
    fCBStep:     Integer;
    fCBVAlign:   TCBVAlign;
    fCBVMargin:  Integer;
    fCheckBoxes: TCCRCBDefs;
    fCode:       String;
    fOnChange:   TNotifyEvent;
    fUpdating:   Boolean;

    procedure CheckBoxClick(Sender: TObject);

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the Enabled property.
      SeeAlso:      TCCRCBGroup.Enabled; TControl.SetEnabled
      Keywords:     SetEnabled,TCCRCBGroup
      Description:
        As implemented in the TCCRCBGroup, this method sets Enable property
        of all check-boxes included in the group (after calling the inherited
        method). Descendant classes can override this method in order to
        implement additional post-processing.
    }
    procedure SetEnabled(aValue: Boolean); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the CBHAlign property.
      SeeAlso:      TCCRCBGroup.CBHAlign
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
    }
    procedure SetCBHAlign(aValue: TCBHAlign); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the CBHMargin property.
      SeeAlso:      TCCRCBGroup.CBHMargin
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
    }
    procedure SetCBHMargin(aValue: Integer); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the CBStep property.
      SeeAlso:      TCCRCBGroup.CBStep
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
    }
    procedure SetCBStep(aValue: Integer); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the CBVAlign property.
      SeeAlso:      TCCRCBGroup.CBVAlign
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
    }
    procedure SetCBVAlign(aValue: TCBVAlign); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the CBVMargin property.
      SeeAlso:      TCCRCBGroup.CBVMargin
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
    }
    procedure SetCBVMargin(aValue: Integer); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets value of the Code property.
      SeeAlso:      TCCRCBGroup.Code
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
    }
    procedure SetCode(aValue: String); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Updates check-boxes according to their definitions.
      SeeAlso:      TCCRCBGroup.CheckBoxes
      Keywords:     UpdateCheckBoxes,TCCRCBGroup
      Description:
        UpdateCheckBoxes deletes existing check-boxes and creates new controls
        according to the check-box definitions stored in the CheckBoxes array.
    }
    procedure UpdateCheckBoxes; virtual;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRCBGroup.
      SeeAlso:      TComponent.Owner
      Keywords:     Create,TCCRCBGroup
      Description:
        Call Create to instantiate a check-box group at runtime. Groups placed
        in forms at design time do not need to be explicitly created because
        they are created automatically.
        <p>
        <i>anOwner</i> is the component, typically a form, that is responsible
        for freeing the check-box group.
    }
    constructor Create(anOwner: TComponent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRCBGroup.
      SeeAlso:      TObject.Free
      Keywords:     Destroy,TCCRCBGroup
      Description:
        Do not call Destroy directly in an application. Instead, call Free.
        Free verifies that the control is not nil, and only then calls Destroy.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Aligns any controls for which the control is the parent
                    within a specified area of the control.
      SeeAlso:      TCCRCBGroup.CBHAlign; TCCRCBGroup.CBVAlign;
                    TCCRCBGroup.CBHMargin; TCCRCBGroup.CBVMargin
                    TCCRCBGroup.CBStep; TWinControl.AlignControls
      Keywords:     AlignControls,TCCRCBGroup
      Description:
        TCCRCBGroup overrides AlignControls to align check-boxes according to
        values of the CBHAlign, CBHMargin, CBVAlign, CBVMargin, and CBStep
        properties. The <i>Rect</i> parameter specifies the area in which to
        align the controls.
    }
    procedure AlignControls(AControl: TControl; var Rect: TRect); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Initializes the control after it is loaded from a stream.
      SeeAlso:      TCCRCBGroup.CheckBoxes; TControl.Loaded
      Keywords:     Loaded,TCCRCBGroup
      Description:
        Do not call Loaded. The VCL streaming system calls Loaded automatically
        after the controls form is loaded into memory so that the control can
        complete any initializations that depend on other objects in the form.
        <p>
        TCCRCBGroup overrides Loaded to create check-box controls according to
        their definitions stored in the CheckBoxes array.
    }
    procedure Loaded; override;

  published

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property Align;
    //property Alignment;
    property Anchors;
    property AutoSize;
    property BevelInner   default bvNone;
    property BevelOuter   default bvNone;
    property BevelWidth;
    //property BiDiMode;
    property BorderWidth;
    property BorderStyle  default bsNone;
    //property Caption;
    property Color;
    property Constraints;
    property Ctl3D;
    //property UseDockManager default True;
    //property DockSite;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property FullRepaint;
    property Font;
    //property Locked;
    //property ParentBiDiMode;
    {$IFDEF VERSION7}
    property ParentBackground;
    {$ENDIF}
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property OnCanResize;
    property OnClick;
    property OnConstrainedResize;
    property OnContextPopup;
    //property OnDockDrop;
    //property OnDockOver;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    //property OnGetSiteInfo;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;
    property OnStartDock;
    property OnStartDrag;
    property OnUnDock;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies horizontal alignment for check-boxes.
      SeeAlso:      TCCRCBGroup.CBVAlign
      Description:
        If check-boxes are aligned to the left (cbaLeft), the first check-box is
        placed CBHMargin pixels to the right from the left border, the next one
        is placed CBStep pixels to the right from the first, and so on.
        <p>
        If check-boxes are aligned to the right (cbaRight), the last check-box
        is placed CBHMargin+CBStep pixels to the left from the right border,
        the previous one is placed CBStep pixels to the left from the last,
        and so on.
        <p>
        If check-boxes are centered (cbaCenter), the group of check-boxes is
        centered inside the control. Distance between check-boxes is CBStep
        pixels.
    }
    property CBHAlign: TCBHAlign
      read fCBHAlign  write SetCBHAlign  default cbaRight;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies minimum distance from the check-boxes to the
                    left or right side of the control.
      SeeAlso:      TCCRCBGroup.CBVMargin
      Description:
        If check-boxes are aligned to the left, this property specifies the
        distance from the first check-box to the left side of the control.
        <p>
        If check-boxes are aligned to the right, this property specifies the
        distance from the area occupied by the last check-box (including the
        caption) to the right side of the control.
    }
    property CBHMargin: Integer  read fCBHMargin  write SetCBHMargin  default 5;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies distance between check-boxes.
      SeeAlso:      TCCRCBGroup.CBHMargin
      Description:
        All check-boxes in the group are placed horizontally with the same
        distance between them. If a check-box caption does not fit into this
        space it is truncated.
    }
    property CBStep: Integer  read fCBStep  write SetCBStep  default 45;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies vertical alignment for check-boxes.
      SeeAlso:      TCCRCBGroup.CBHAlign
      Description:
        If check-boxes are aligned to the top (cbaTop), the check-boxes are
        placed CBHVargin pixels from the top border.
        <p>
        If check-boxes are aligned to the right (cbaBottom), the check-boxes are
        placed CBHVargin pixels from the bottom border.
        <p>
        If check-boxes are centered (cbaMiddle), the group of check-boxes is
        centered inside the control.
    }
    property CBVAlign: TCBVAlign
      read fCBVAlign  write SetCBVAlign  default cbaMiddle;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies minimum distance from the check-boxes to the
                    top or bottom side of the control.
      SeeAlso:      TCCRCBGroup.CBHMargin
      Description:
        If check-boxes are aligned to the top, this property specifies the
        distance from the top of check-boxes to the top border.
        <p>
        If check-boxes are aligned to the bottom, this property specifies the
        distance from the bottom of check-boxes to the bottom border.
    }
    property CBVMargin: Integer
      read fCBVMargin  write SetCBVMargin  default 10;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Contains definitions of check-boxes included in the group.
      SeeAlso:      TCCRCBDef
      Keywords:     CheckBoxes,TCCRCBGroup
      Description:
        Use CheckBoxes to add or delete check-box definitions in the group, or
        to edit their properties. When setting CheckBoxes at design time a
        CheckBoxes Editor dialog appears. This dialog allows definitions to be
        added or deleted, and the properties of individual check-boxes to be
        changed.
    }
    property CheckBoxes: TCCRCBDefs  read fCheckBoxes  write fCheckBoxes;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates the code of the checked check-box.
      SeeAlso:      TCCRCBDef.Code; TCCRCBDef.OnChange
      Keywords:     Code,TCCRCBGroup
      Description:
        Use Code to get or set the result code of the group. Value of this
        property corresponds to that of the checked check-box. If nothing is
        checked, then Code returns empty string. When you assign a code to this
        property, the corresponding check-box is checked automatically.
    }
    property Code: String  read fCode  write SetCode;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs when value of Code changes.
      SeeAlso:      TCCRCBGroup.Code
      Keywords:     OnChange,TCCRCBGroup
      Description:
        Write OnCode handler to react on changes of the Code property value.
        This event generated when user checks a check-box in a group or a value
        is assigned to the Code property. The <i>aSender</i> parameter is the
        check-box group whose event handler is called.
    }
    property OnChange: TNotifyEvent  read fOnChange  write fOnChange;

  end;

  {================================ TCCRToolBar ================================
    Overview:     TCCRToolBar is a custom single-row version of a TToolBar that
                  aligns its child controls to the right.
    SeeAlso:      TToolBar
    Description:
      TCCRToolBar manages tool buttons and other controls, arranging them in
      a single row and automatically aligning them to the right.
  }
  TCCRToolBar = class(TToolBar)
  private

    fSpacer: Tpanel;

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Respond to control resize.
      SeeAlso:      TControl.Resize
      Keywords:     Resize,TCCRToolBar
      Description:
        Resize is called automatically immediately after the controls
        dimensions change. TCCRToolBar overrides this method to align all
        child controls to the right.
    }
    procedure Resize; override;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes a TCCRToolBar instance.
      SeeAlso:      TToolWindow.Create
      Keywords:     Create,TCCRToolBar
      Description:
        Create generates a TCCRToolBar and initializes its properties.
    }
    constructor Create(anOwner: TComponent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys the TToolBar instance and frees its memory.
      SeeAlso:      TObject.Free; TToolBar.Destroy
      Keywords:     Destroy,TCCRToolBar
      Description:
        Do not call Destroy directly in an application. Instead, call Free.
        Free verifies that the TCCRToolBar reference is not nil, and only then
        calls Destroy.
    }
    destructor Destroy; override;

  end;

///////////////////////////////// Implementation \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

implementation

uses
  uROR_Resources, uROR_Utilities;

/////////////////////////////////// TCCRCBDef \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRCBDef.Create(aCollection: TCollection);
begin
  inherited;
  fCaption := '';
  fCode    := '';
  fControl := nil;
end;

destructor TCCRCBDef.Destroy;
begin
  FreeAndNil(fControl);
  inherited;
end;

function TCCRCBDef.GetDisplayName: String;
begin
  if fCaption <> '' then
    Result := Caption
  else
    Result := inherited GetDisplayName;
end;

procedure TCCRCBDef.SetCaption(const aValue: String);
begin
  if aValue <> fCaption then
    begin
      fCaption := aValue;
      Changed(False);
    end;
end;

procedure TCCRCBDef.SetCode(const aValue: String);
begin
  if aValue <> fCode then
    begin
      fCode := aValue;
      Changed(False);
    end;
end;

/////////////////////////////////// TCCRCBDefs \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRCBDefs.Create(anOwner: TCCRCBGroup);
begin
  inherited Create(anOwner, TCCRCBDef);
end;

function TCCRCBDefs.Add: TCCRCBDef;
begin
  Result := TCCRCBDef(inherited Add);
end;

function TCCRCBDefs.GetCBDef(anIndex: Integer): TCCRCBDef;
begin
  Result := TCCRCBDef(Items[anIndex]);
end;

procedure TCCRCBDefs.Update(Item: TCollectionItem);
begin
  inherited;
  UpdateOwner;
end;

procedure TCCRCBDefs.UpdateOwner;
begin
  if Owner <> nil then
    TCCRCBGroup(Owner).UpdateCheckBoxes;
end;

////////////////////////////////// TCCRCBGroup \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRCBGroup.Create(anOwner: TComponent);
begin
  inherited;
  fCheckBoxes := TCCRCBDefs.Create(self);

  BevelInner     := bvNone;
  BevelOuter     := bvNone;
  BorderStyle    := bsNone;
  Caption        := '';
  DockSite       := False;
  Height         := 33;
  ParentBiDiMode := True;

  fCBHAlign      := cbaRight;
  fCBHMargin     := 5;
  fCBStep        := 45;
  fCBVAlign      := cbaMiddle;
  fCBVMargin     := 10;

  fCode          := '';
  fUpdating      := False;
end;

destructor TCCRCBGroup.Destroy;
begin
  FreeAndNil(fCheckBoxes);
  inherited;
end;

procedure TCCRCBGroup.AlignControls(AControl: TControl; var Rect: TRect);
var
  cbx, i, n, rh, rw: Integer;
begin
  if CheckBoxes.Count > 0 then
    begin
      n := CheckBoxes.Count - 1;
      rh := Rect.Bottom - Rect.Top + 1;
      rw := Rect.Right - Rect.Left + 1;
      cbx := (rw - CBStep * CheckBoxes.Count) div 2 + 2;
      for i:=0 to n do
        if Assigned(CheckBoxes[i].Control) then
          with CheckBoxes[i].Control do
            begin
              Width := CBStep;
              case CBHAlign of
                cbaLeft:
                  Left := i * CBStep + CBHMargin;
                cbaCenter:
                  Left :=  i * CBStep + cbx;
                cbaRight:
                  Left := rw - (n - i + 1) * CBStep - CBHMargin;
              end;
              case CBVAlign of
                cbaTop:
                  Top := CBVMargin;
                cbaMiddle:
                  Top := (rh - Height) div 2;
                cbaBottom:
                  Top := rh - Height - CBVMargin;
              end;
            end;
    end;
  inherited;
end;

procedure TCCRCBGroup.CheckBoxClick(Sender: TObject);
var
  cbt: Integer;
begin
  if not fUpdating and (Sender is TCheckBox) then
    try
      fUpdating := True;
      cbt := TCheckBox(Sender).Tag;
      if (cbt >= 0) and (cbt < CheckBoxes.Count) then
        if TCheckBox(Sender).Checked then
          Code := CheckBoxes[cbt].Code
        else
          Code := '';
    finally
      fUpdating := False;
    end;
end;

procedure TCCRCBGroup.Loaded;
begin
  inherited;
  UpdateCheckBoxes;
end;

procedure TCCRCBGroup.SetCBHAlign(aValue: TCBHAlign);
begin
  fCBHAlign := aValue;
  Realign;
end;

procedure TCCRCBGroup.SetCBHMargin(aValue: Integer);
begin
  fCBHMargin := aValue;
  Realign;
end;

procedure TCCRCBGroup.SetCBStep(aValue: Integer);
begin
  fCBStep := aValue;
  Realign;
end;

procedure TCCRCBGroup.SetCBVAlign(aValue: TCBVAlign);
begin
  fCBVAlign := aValue;
  Realign;
end;

procedure TCCRCBGroup.SetCBVMargin(aValue: Integer);
begin
  fCBVMargin := aValue;
  Realign;
end;

procedure TCCRCBGroup.SetCode(aValue: String);
var
  i, n: Integer;
begin
  fCode := aValue;
  n := CheckBoxes.Count - 1;
  for i:=0 to n do
    if Assigned(CheckBoxes[i].Control) and (CheckBoxes[i].Code <> '') then
      CheckBoxes[i].Control.Checked := (CheckBoxes[i].Code = aValue);
  if Assigned(OnChange) then OnChange(self);
end;

procedure TCCRCBGroup.SetEnabled(aValue: Boolean);
var
  i, n: Integer;
begin
  inherited SetEnabled(aValue);
  n := CheckBoxes.Count - 1;
  for i:=0 to n do
    if Assigned(CheckBoxes[i].Control) then
      CheckBoxes[i].Control.Enabled := aValue;
end;

procedure TCCRCBGroup.UpdateCheckBoxes;
var
  i, n: Integer;
begin
  n := CheckBoxes.Count - 1;
  try
    DisableAlign;
    for i:=0 to n do
      with CheckBoxes[i] do
        begin
          Control.Free;
          Control := TCheckBox.Create(self);
          Control.Caption := CheckBoxes[i].Caption;
          Control.Enabled := self.Enabled;
          Control.Name := Format('cb%d', [ID]);
          Control.TabStop := True;
          Control.Tag := i;
          Control.OnClick := CheckBoxClick;

          Control.Parent := self;
          //-- Section 508: Temporary fix until patch 12 until component can be
          //   re-designed.  Add 1 to tab order - DEBUG(ETM)
          Control.TabOrder := i+1;
        end;
  finally
    EnableAlign;
  end;
end;

////////////////////////////////// TCCRToolBar \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRToolBar.Create(anOwner: TComponent);
begin
  inherited;
  if not (csDesigning in ComponentState) then
    begin
      fSpacer := TPanel.Create(Self);
      with fSpacer do
        begin
          Align       := alLeft;
          BevelInner  := bvNone;
          BevelOuter  := bvNone;
          Name        := 'pnlSpacer';
          Caption     := '';
          ParentColor := True;

          SetSubComponent(True);
          Parent := Self;
        end;
    end;
end;

destructor TCCRToolBar.Destroy;
begin
  FreeAndNil(fSpacer);
  inherited;
end;

procedure TCCRToolBar.Resize;
var
  i, n, wd: Integer;
begin
  if Assigned(fSpacer) then
    begin
      wd := ClientWidth;
      n := ControlCount - 1;
      for i:=0  to n do
        if (Controls[i] <> fSpacer) and Controls[i].Visible then
          Dec(wd, Controls[i].Width);
      fSpacer.Width := wd;
    end;
  inherited;
end;

end.
