{******************************************************************************}
{ Package:      Clinical Case Registries Custom Components                     }
{ Date Created: November 18, 2004                                              }
{ Site Name:    Hines OIFO                                                     }
{ Developers:   Sergey Gavrilov                                                }
{ Description:  This unit defines clinical context (CCOW) related components.  }
{ Note:                                                                        }
{******************************************************************************}

unit uROR_Contextor;

{$I Components.inc}

interface

uses
  SysUtils, Classes, Controls, VERGENCECONTEXTORLib_TLB, Menus,
  ImgList, uROR_CmdLineParams, uROR_CustomContextor, uROR_CustomBroker,
  uROR_Broker;

type

  {=============================== TCCRContextor ===============================
    Overview:     Wrapper for a Vergence contextor (CCOW).
    SeeAlso:      TCCRContextorIndicator
    Description:
      The TCCRContextor class is a VistA-specific implementation of a wrapper
      for the Vergence contextor component developed by
      <a href="http://www.sentillion.com">Sentillion</a>. It works as a proxy
      and frees applications of necessity to check if the Vergence
      desktop components are installed before using them. It also introduces
      suspend/resume event handlers and enhances the contextor resume
      functionality.
  }
  TCCRContextor = class(TCCRCustomContextor)
  private

    fDFNItemName:      WideString;
    fICNItemName:      WideString;
    fOnPatientChanged: TNotifyEvent;
    fPatientChanged:   Boolean;
    fPatientDFN:       String;
    fPatientICN:       String;
    fBroker:           TCCRCustomBroker;

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns an ICN for a patient with provided DFN.
      SeeAlso:      TCCRContextor.ICNtoDFN
      Description:
        DFNtoICN calls the 'VAFCTFU CONVERT DFN TO ICN' remote procedure to get
        an Integration Control Number (ICN) from a patient record with internal
        entry number specified by the <i>aDFN</i> parameter. If the patient has
        no ICN, this function returns an empty string.
        <p>
        <b>Note:</b> The 'VAFCTFU CONVERT DFN TO ICN' remote procedure should
        be added to the context option of the application.
    }
    function DFNtoICN(aDFN: String): String;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnPending event handler dispatcher.
      SeeAlso:      TCCRCustomContextor.DoPending;
                    TCCRCustomContextor.OnPending;
                    TCCRCustomContextor.SetSurveyResponse
      Keywords:     DoPending,TCCRContextor
      Description:
        Do not call this method; DoPending is called automatically when another
        application proposes context changes. The <i>aContextItemCollection</i>
        parameter lists context items that are going to be updated and their
        proposed values. TCCRContextor overrides this method in order to
        determine whether any items with the 'patient' subject are going to be
        changed.
        <p>
        Use SetSurveyResponse to warn the user about unsaved data or another
        condition that might make the context change undesirable.
        <p>
        Descendant classes that override DoPending should always call the
        inherited method.
    }
    procedure DoPending(const aContextItemCollection: IDispatch); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns a DFN for a patient with provided ICD.
      SeeAlso:      TCCRContextor.DFNtoICN
      Description:
        ICNtoDFN calls the 'VAFCTFU CONVERT ICN TO DFN' remote procedure to get
        an internal number (DFN) of a patient record that corresponds to the
        Integration Control Number specified by the <i>anICN</i> parameter. If
        the ICN is not found, this function returns an empty string.
        <p>
        <b>Note:</b> The 'VAFCTFU CONVERT ICN TO DFN' remote procedure should
        be added to the context option of the application.
    }
    function ICNtoDFN(anICN: String): String;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Responds to notifications that components are being created
                    or destroyed.
      SeeAlso:      TCCRCustomContextor.Notification
      Keywords:     Notification,TCCRContextor
      Description:
        Do not call the Notification method in an application.  Notification is
        called automatically when the component specified by <i>aComponent</i>
        is about to be inserted or removed, as specified by <i>Operation</i>.
        <p>
        TCCRContextor overrides this method in order to clear its RPCBroker
        property when the control it refers to is destroyed.
    }
    procedure Notification(aComponent: TComponent;
      anOperation: TOperation); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the RPCBroker property.
      SeeAlso:      TCCRContextor.RPCBroker
      Description:
        Descendant classes can override this method in order to implement
        additional post-processing.
        <p>
        When overriding SetBroker, be sure to call the inherited method.
    }
    procedure SetBroker(aValue: TCCRCustomBroker); virtual;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRContextor.
      SeeAlso:      TComponent.Owner; TCCRCustomContextor.Create
      Keywords:     Create,TCCRContextor
      Description:
        Create initializes an instance of the TCCRContextor. <i>anOwner</i>
        is the component, typically a form, that is responsible for freeing the
        contextor.
    }
    constructor Create(anOwner: TComponent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Extracts patient DFN from a context.
      SeeAlso:      TCCRContextor.DFNItemName
      Description:
        PatientDFNFromContext searches for an item that specifies an internal
        number of a patient record (DFN) in a collection of context items
        referenced by the <i>aContext</i> parameter and returns the item value.
        If the item is not found, this function returns an empty string.
    }
    function PatientDFNFromContext(aContext: IContextItemCollection;
      var aValue: String): Boolean;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Extracts patient ICN from a context.
      SeeAlso:      TCCRContextor.ICNItemName
      Description:
        PatientICNFromContext searches for an item that specifies a patient
        Integration Control Number (ICN) in a collection of context items
        referenced by the <i>aContext</i> parameter and returns the item value.
        If the item is not found, this function returns an empty string.
    }
    function PatientICNFromContext(aContext: IContextItemCollection;
      var aValue: String): Boolean;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Processes known subjects and assign item values to the
                    corresponding properties.
      SeeAlso:      TCCRContextor.OnPatientChanged; TCCRContextor.PatientDFN;
                    TCCRContextor.PatientICN; TCCRCustomContextor.CurrentContext;
                    TCCRCustomContextor.ProcessKnownSubjects;
      Keywords:     ProcessKnownSubjects,TCCRContextor
      Description:
        When a contextor is notified about changes in the clinical context, it
        calls ProcessKnownSubjects. This method should examine items of the
        CurrentContext collection, extract values of known items (i.e. patient
        ICN), and assign them to the corresponding properties.
        <p>
        If the <i>Enforce</i> parameter is True, the properties should be
        updated (cleared) even if there are no corresponding items in the
        current context. If this parameter is False (default), those properties
        should remain unchanged.
        <p>
        As implemented in TCCRContextor, ProcessKnownSubjects updates values of
        the PatientDFN and PatientICN properties and calls the OnPatientChanged
        event handler, if defined. When overriding it, be sure to call the
        inherited method.
    }
    procedure ProcessKnownSubjects(Enforce: Boolean = False); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Changes a patient in a clinical context.
      SeeAlso:      TCCRContextor.DFNtoICN; TCCRContextor.ICNtoDFN;
                    TCCRContextor.PatientDFN; TCCRContextor.PatientICN


      Description:
        SetPatientContext starts a context change transaction, prepares the
        patient context changes, and tries to change the context. If the
        transaction is successfully committed, this function updates values of
        the PatientDFN and PatientICN properties, and returns True. Otherwise,
        False is returned.
        <p>
        The <i>aDFN</i> parameter specifies an internal entry number of a
        patient record (DFN). Use the <i>anICN</i> to pass the patient's
        Integration Control Number (ICN). If only one parameter is specified,
        SetPatientContext calls DFNtoICN or ICNtoDFN to get the remaining value.
        <p>
        If <i>aDFN</i> and <i>anICN</i> have the same values as the PatientDFN
        and PatientICN properties accordingly or both parameters are empty,
        the context change transaction does not occur.
    }
    function SetPatientContext(aDFN: String = '';
      anICN: String = ''): Boolean; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     DFN of a patient currently selected in a clinical context.
      SeeAlso:      TCCRContextor.PatientICN; TCCRCustomContextor.State;
                    TCCRCustomContextor.LocalContext
      Description:
        Read a value of the PatientDFN property to get an internal entry number
        of a patient currently selected in a clinical context (common and/or
        local/application, depending on the current contextor state).
    }
    property PatientDFN: String  read fPatientDFN;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     ICN of a patient currently selected in a clinical context.
      SeeAlso:      TCCRContextor.PatientICN; TCCRCustomContextor.State;
                    TCCRCustomContextor.LocalContext
      Description:
        Read a value of the PatientICN property to get an Integration Control
        Number (ICN) of a patient currently selected in a clinical context
        (common and/or local/application, depending on the current contextor
        state).
    }
    property PatientICN: String  read fPatientICN;

  published

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property ApplicationName;
    property CmdLineParams;
    property Enabled;
    property NotificationFilter;
    property OnCanceled;
    property OnCommitted;
    property OnPending;
    property OnResumed;
    property OnSuspended;
    property PassCode;
    property Survey default True;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Name of a context item that specifies a patient DFN.
      SeeAlso:      TCCRContextor.ICNItemName
      Keywords:
      Description:
        Use DFNItemName to get or set a name of a context item that specifies
        a patient's internal entry number (DFN). Default value of this property
        is 'patient.id.mrn.dfn_'.
    }
    property DFNItemName: WideString  read fDFNItemName  write fDFNItemName;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Name of a context item that specifies a patient ICN.
      SeeAlso:      TCCRContextor.DFNItemName
      Keywords:
      Description:
        Use DFNItemName to get or set a name of a context item that specifies
        a patient's Integartion Control Number (ICN). Defautl value of this
        property is 'patient.id.mrn.nationalidnumber'.
    }
    property ICNItemName: WideString  read fICNItemName  write fICNItemName;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs after a patient is selected in a clinical context.
      SeeAlso:      TCCRContextor.PatientDFN; TCCRContextor.PatientICN;
                    TCCRContextor.ProcessKnownSubjects
      Description:
        Write an OnPatientChanged event handler to respond on selecting a
        patient in a clinical context.
    }
    property OnPatientChanged: TNotifyEvent
      read fOnPatientChanged  write fOnPatientChanged;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Reference to a VistA RPC Broker instance.
      SeeAlso:      TCCRContextor.DFNtoICN; TCCRContextor.ICNtoDFN;
                    TCCRContextor.SetBroker
      Keywords:     RPCBroker,TCCRContextor
      Description:
        Assign a reference to a RPC Broker used by an application to the
        RPCBroker property before running a contextor. The contextor uses
        the broker to call remote procedures (see DFNtoICN and ICNtoDFN).
    }
    property Broker: TCCRCustomBroker  read fBroker  write SetBroker;

  end;

  {=========================== TCCRContextorIndicator ==========================
    Overview:     A compact contextor state (clinical link status) indicator.
    SeeAlso:
    Description:
      TCCRContextorIndicator, shows a small bitmap corresponding to to the
      current contextor state (clinical link status).
  }
  TCCRContextorIndicator = class(TCCRCustomContextorIndicator)
  private

    mnuBreakLink:     TMenuItem;
    mnuUseAppData:    TMenuItem;
    mnuUseGlobalData: TMenuItem;

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates a pop-up menu for an indicator.
      SeeAlso:      TControl.PopupMenu
      Description:
        CreatePopupMenu is called internally to create a pop-up menu for a
        contextor state indicator. This menu is activated when a user clicks
        a right mouse button over the indicator. This function should return a
        reference to the created menu. As implemented in TCCRContextorIndicator,
        the menu contains 3 items: 'Break the Clinical Link', 'Rejoin and Use
        Application Data', and 'Rejoin and Use Global Data'.
    }
    function CreatePopupMenu: TPopupMenu; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Renders the image of the indicator.
      SeeAlso:      TCCRCustomContextorIndicator.Images;
                    TCCRCustomContextorIndicator.LinkStatus;
                    TCustomPanel.Paint
      Keywords:     Paint,TCCRContextorIndicator
      Description:
        Override the Paint method to change the way the indicator is drawn.
        The Paint method for TCCRContextorIndicator draws a bitmap that
        corresponds to the current value of the LinkStatus property.
    }
    procedure Paint; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnClick event handler for the Break item of the pop-up menu.
      SeeAlso:      TCCRContextorIndicator.ProcessRejoin; TMenuItem.OnClick
      Description:
        This method is called when a user selects 'Break the Clinical Link'
        from a pop-up menu associated with an indicator. <i>aSender</i> is the
        menu item object whose event handler is called.
    }
    procedure ProcessBreak(aSender: TObject); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnPopup event handler for an indicator context menu.
      SeeAlso:      TPopupMenu.OnPopup; TControl.PopupMenu
      Description:
        This method is called before an indicator pop-menu is shown. As
        implemented in TCCRContextorIndicator, ProcessOnPopup ebnables and/or
        disables menu items according to the current state of the contextor.
    }
    procedure ProcessOnPopup(aSender: TObject); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnClick event handler for the Rejoin items of the pop-up
                    menu.
      SeeAlso:      TCCRContextorIndicator.ProcessBreak; TMenuItem.OnClick
      Description:
        This method is called when a user selects 'Rejoin and Use Application
        Data' or 'Rejoin and Use Global Data' from a pop-up menu associated
        with an indicator. <i>aSender</i> is the menu item object whose event
        handler is called.
    }
    procedure ProcessRejoin(aSender: TObject); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the name of the control.
      SeeAlso:      TControl.SetName
      Keywords:     SetName,TCCRContextorIndicator
      Description:
        SetName is the protected write implementation of the Name property. As
        implemented in TCCRContextorIndicator, it clears the Caption property
        after calling the inherited method.
    }
    procedure SetName(const NewName: TComponentName); override;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of
                    TCCRContextorIndicator.
      SeeAlso:      TComponent.Owner; TCCRCustomContextorIndicator.Create
      Keywords:     Create,TCCRContextorIndicator
      Description:
        Create initializes an instance of the TCCRContextorIndicator.
        <i>anOwner</i> is the component, typically a form, that is responsible
        for freeing the indicator control.
    }
    constructor Create(anOwner: TComponent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRContextorIndicator.
      SeeAlso:      TCCRCustomContextorIndicator.Destroy; TObject.Free
      Keywords:     Destroy,TCCRContextorIndicator
      Description:
        Do not call Destroy directly in an application. Instead, call Free.
        Free verifies that the control is not nil, and only then calls Destroy.
        <p>
        Applications should only free controls explicitly when the constructor
        was called without assigning an owner to the control.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Responds after the last constructor has executed.
      SeeAlso:      TCCRCustomContextorIndicator.Create; TControl.PopupMenu;
                    TObject.AfterConstruction
      Keywords:     AfterConstruction,TCCRContextorIndicator
      Description:
        AfterConstruction is called automatically after the objects last
        constructor has executed. Do not call it explicitly in your applications.
        <p>
        The AfterConstruction method implemented in TCCRContextorIndicator
        creates a context pop-up menu. Override this method when creating a
        class that takes some action after the object is created.
    }
    procedure AfterConstruction; override;

  published

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property Align;
    //property Alignment;
    property Anchors;
    //property AutoSize;
    property BevelInner  default bvNone;
    property BevelOuter  default bvNone;
    property BevelWidth;
    //property BiDiMode;
    property BorderWidth;
    property BorderStyle;
    //property Caption;
    property Color;
    property Constraints;
    property Contextor; // TCCRCustomContextorIndicator
    property Ctl3D;
    //property UseDockManager  default True;
    //property DockSite;
    //property DragCursor;
    //property DragKind;
    //property DragMode;
    property Enabled;
    property FullRepaint;
    property Font;
    property Images; // TCCRCustomContextorIndicator
    property Locked;
    //property ParentBiDiMode;
    {$IFDEF VERSION7}
    property ParentBackground;
    {$ENDIF}
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property OnCanResize;
    property OnClick;
    property OnConstrainedResize;
    property OnContextPopup;
    //property OnDockDrop;
    //property OnDockOver;
    property OnDblClick;
    //property OnDragDrop;
    //property OnDragOver;
    //property OnEndDock;
    //property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnGetSiteInfo;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;
    //property OnStartDock;
    //property OnStartDrag;
    //property OnUnDock;
    property OnUpdate; // TCCRCustomContextorIndicator

  end;

///////////////////////////////// Implementation \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

implementation

uses
  uROR_Resources, uROR_Utilities;

////////////////////////////////// TCCRContextor \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRContextor.Create(anOwner: TComponent);
begin
  inherited;
  fDFNItemName := 'patient.id.mrn.dfn_';
  fICNItemName := 'patient.id.mrn.nationalidnumber';
end;

function TCCRContextor.DFNtoICN(aDFN: String): String;
var
  rpcRes: TStringList;
begin
  rpcRes := TStringList.Create;
  try
    if Broker.CallProc('VAFCTFU CONVERT DFN TO ICN',
      [aDFN], nil, rpcRes, [rpcNoErrorCheck]) then
      {$IFDEF CCOW_USE_ICN_CHECKSUM}
      Result := Piece(rpcRes[0], '^');
      {$ELSE}
      Result := Piece(Piece(rpcRes[0], '^'), 'V');
      {$ENDIF}
      if StrToIntDef(Result, 0) < 0 then
        Result := '';
  except
    Result := '';
  end;
  rpcRes.Free;
end;

procedure TCCRContextor.DoPending(const aContextItemCollection: IDispatch);
var
  i: Integer;
begin
  fPatientChanged := False;
  if Assigned(aContextItemCollection) then
    with IContextItemCollection(aContextItemCollection) do
      for i:=1 to Count do
        with Item(i) do
          if WideSameText(Subject, 'Patient') then
            fPatientChanged := True;
  inherited;
end;

function TCCRContextor.ICNtoDFN(anICN: String): String;
var
  rpcRes: TStringList;
begin
  rpcRes := TStringList.Create;
  try
    if Broker.CallProc('VAFCTFU CONVERT ICN TO DFN',
      [Piece(anICN,'V')], nil, rpcRes, [rpcNoErrorCheck]) then
      Result := Piece(rpcRes[0], '^');
      if StrToIntDef(Result, 0) < 0 then
        Result := '';
  except
    Result := '';
  end;
  rpcRes.Free;
end;

procedure TCCRContextor.Notification(aComponent: TComponent;
  anOperation: TOperation);
begin
  inherited;
  if (aComponent = Broker) and (anOperation = opRemove) then
    Broker := nil;
end;

function TCCRContextor.PatientDFNFromContext(aContext: IContextItemCollection;
  var aValue: String): Boolean;
var
  ci: IContextItem;
begin
  ci := aContext.Present(DFNItemName);
  Result := (ci <> nil);
  if Result then
    aValue := ci.Value
  else
    aValue := '';
end;

function TCCRContextor.PatientICNFromContext(aContext: IContextItemCollection;
  var aValue: String): Boolean;
var
  ci: IContextItem;
begin
  ci := aContext.Present(ICNItemName);
  Result := (ci <> nil);
  if Result then
    {$IFDEF CCOW_USE_ICN_CHECKSUM}
    aValue := ci.Value
    {$ELSE}
    aValue := Piece(ci.Value, 'V')
    {$ENDIF}
  else
    aValue := '';
end;

procedure TCCRContextor.ProcessKnownSubjects(Enforce: Boolean);
var
  pc: Integer;
begin
  if State = csParticipating then
    begin
      if Enforce or (not Survey) then
        fPatientChanged := True;
      pc := 0;
      if PatientDFNFromContext(CurrentContext, fPatientDFN) then Inc(pc);
      if PatientICNFromContext(CurrentContext, fPatientICN) then Inc(pc);
      if fPatientChanged then
        begin
          // If the context has only one patient identifier then
          // try to get the other from VistA
          if (pc = 1) and Assigned(Broker) then
            begin
              if fPatientDFN = '' then
                fPatientDFN := ICNtoDFN(fPatientICN);
              if fPatientICN = '' then
                fPatientICN := DFNtoICN(fPatientDFN);
            end;
          // Call the user-defined event handler
          if Assigned(OnPatientChanged) then
            OnPatientChanged(Self);
          // Reset the flag
          fPatientChanged := False;
        end;
    end;
end;

function TCCRContextor.SetPatientContext(aDFN: String; anICN: String): Boolean;
var
  pc: Integer;
  commit: Boolean;
begin
  Result := True;
  {$IFNDEF CCOW_USE_ICN_CHECKSUM}
  anICN := Piece(anICN, 'V');
  {$ENDIF}

  pc := 0;
  if anICN <> '' then Inc(pc);
  if aDFN  <> '' then Inc(pc);

  if pc > 0 then
    begin
      if aDFN  = '' then aDFN  := ICNtoDFN(anICN);
      if anICN = '' then anICN := DFNtoICN(aDFN);
    end;

  if (anICN <> fPatientICN) or (aDFN <> fPatientDFN) then
    begin
      commit := False;
      if StartContextChange then
        try
          if pc > 0 then
            begin
              if anICN <> '' then
                AddTXItem(ICNItemName, Piece(anICN, 'V'));
              if aDFN <> '' then
                AddTXItem(DFNItemName, aDFN);
            end
          else
            AddTXItem(ICNItemName, '');
          commit := True;
        finally
          Result := (EndContextChange(commit) <> urCancel);
        end;

      if Result then
        begin
          fPatientICN := anICN;
          fPatientDFN := aDFN;
        end;
    end;
end;

procedure TCCRContextor.SetBroker(aValue: TCCRCustomBroker);
begin
  if aValue <> fBroker then
    begin
      if Assigned(fBroker) then
        fBroker.RemoveFreeNotification(Self);

      fBroker := aValue;

      if Assigned(fBroker) then
        begin
          fBroker.FreeNotification(Self);
          if fBroker is TCCRBroker then
            TCCRBroker(fBroker).Contextor := Self;
        end;
    end;
end;

//////////////////////////// TCCRContextorIndicator \\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRContextorIndicator.Create(anOwner: TComponent);
begin
  inherited;

  Height    := 24;
  Width     := 24;
end;

destructor TCCRContextorIndicator.Destroy;
begin
  PopupMenu := nil;
  inherited;
end;

procedure TCCRContextorIndicator.AfterConstruction;
begin
  inherited;
  if not Assigned(PopupMenu) and not (csDesigning in ComponentState) then
    PopupMenu := CreatePopupMenu;
end;

function TCCRContextorIndicator.CreatePopupMenu: TPopupMenu;
var
  mi: TMenuItem;
begin
  Result := TPopupMenu.Create(Self);
  with Result do
    begin
      SetSubComponent(True);
      AutoPopup := True;
      Name      := 'mnuDefaultContextPopup';
      OnPopup   := ProcessOnPopup;

      mnuUseAppData := TMenuItem.Create(Self);
      mnuUseAppData.Caption := rscUseAppData;
      mnuUseAppData.OnClick := ProcessRejoin;
      Items.Add(mnuUseAppData);

      mnuUseGlobalData := TMenuItem.Create(Self);
      mnuUseGlobalData.Caption  := rscUseGlobalData;
      mnuUseGlobalData.OnClick  := ProcessRejoin;
      Items.Add(mnuUseGlobalData);

      mi := TMenuItem.Create(Self);
      mi.Caption := '-';
      Items.Add(mi);

      mnuBreakLink := TMenuItem.Create(Self);
      mnuBreakLink.Caption := rscBreakLink;
      mnuBreakLink.OnClick := ProcessBreak;
      Items.Add(mnuBreakLink);
    end;
end;

procedure TCCRContextorIndicator.Paint;
var
  imgNdx: Integer;
begin
  inherited;
  if Assigned(Images) then
    begin
      imgNdx := Ord(LinkStatus);
      if (imgNdx >= 0) and (imgNdx < Images.Count) then
        Images.Draw(Canvas,
          ClientRect.Left + (ClientWidth  - Images.Width)   div 2,
          ClientRect.Top  + (ClientHeight - Images.Height ) div 2,
          imgNdx);
    end;
end;

procedure TCCRContextorIndicator.ProcessBreak(aSender: TObject);
begin
  if Assigned(Contextor) then
    Contextor.Suspend;
end;

procedure TCCRContextorIndicator.ProcessOnPopup(aSender: TObject);
begin
  if Assigned(Contextor) then
    case Contextor.State of
      csParticipating:
        begin
          mnuBreakLink.Enabled     := True;
          mnuUseAppData.Enabled    := False;
          mnuUseGlobalData.Enabled := False;
        end;
      csSuspended:
        begin
          mnuBreakLink.Enabled     := False;
          mnuUseAppData.Enabled    := True;
          mnuUseGlobalData.Enabled := True;
        end;
      else
        begin
          mnuBreakLink.Enabled     := False;
          mnuUseAppData.Enabled    := False;
          mnuUseGlobalData.Enabled := False;
        end;
    end;
end;

procedure TCCRContextorIndicator.ProcessRejoin(aSender: TObject);
begin
  if Assigned(Contextor) then
    if aSender = mnuUseAppData then
      Contextor.Resume(crmUseAppData)
    else
      Contextor.Resume(crmUseGlobalData);
end;

procedure TCCRContextorIndicator.SetName(const NewName: TComponentName);
begin
  inherited;
  Caption := '';
end;

end.
